// WebTalk Wrapper Bridge
// This script provides the interface between WebTalk JavaScript and Qt wrapper

(function() {
    'use strict';
    
    // Global wrapper detection
    window.isInWrapper = false;
    window.wrapperBridge = null;
    
    // Initialize wrapper bridge
    function initializeWrapperBridge() {
        console.log('WebTalk: Attempting to initialize wrapper bridge...');
        
        // Check for Qt WebChannel (WebEngine)
        if (typeof QWebChannel !== 'undefined' && typeof qt !== 'undefined' && qt.webChannelTransport) {
            console.log('WebTalk: Found QWebChannel, initializing...');
            new QWebChannel(qt.webChannelTransport, function(channel) {
                console.log('WebTalk: WebChannel callback triggered');
                if (channel.objects && channel.objects.qtBridge) {
                    window.wrapperBridge = channel.objects.qtBridge;
                    window.isInWrapper = true;
                    console.log('WebTalk: Qt WebEngine bridge initialized successfully');
                    setupWrapperMethods();
                } else {
                    console.log('WebTalk: qtBridge object not found in channel');
                }
            });
        }
        // Check for direct Qt object (WebKit)
        else if (typeof qtBridge !== 'undefined') {
            console.log('WebTalk: Found direct qtBridge object');
            window.wrapperBridge = qtBridge;
            window.isInWrapper = true;
            console.log('WebTalk: Qt WebKit bridge initialized successfully');
            setupWrapperMethods();
        }
        // No wrapper detected
        else {
            console.log('WebTalk: No wrapper bridge detected - running in browser-only mode');
            console.log('WebTalk: QWebChannel available:', typeof QWebChannel !== 'undefined');
            console.log('WebTalk: qt object available:', typeof qt !== 'undefined');
            console.log('WebTalk: qtBridge available:', typeof qtBridge !== 'undefined');
        }
    }
    
    // Setup wrapper methods
    function setupWrapperMethods() {
        if (!window.wrapperBridge) return;
        
        // Override shell command function
        window.shell = function(command) {
            if (window.wrapperBridge && window.wrapperBridge.executeShellCommandSync) {
                return window.wrapperBridge.executeShellCommandSync(command);
            }
            return "Error: Shell commands not available";
        };
        
        // Override sound playing
        window.playWrapperSound = function(soundFile) {
            if (window.wrapperBridge && window.wrapperBridge.playSound) {
                return window.wrapperBridge.playSound(soundFile);
            }
            return false;
        };
        
        // Override clipboard functions
        window.setClipboard = function(text) {
            if (window.wrapperBridge && window.wrapperBridge.setClipboardText) {
                return window.wrapperBridge.setClipboardText(text);
            }
            return false;
        };
        
        window.getClipboard = function() {
            if (window.wrapperBridge && window.wrapperBridge.getClipboardText) {
                return window.wrapperBridge.getClipboardText();
            }
            return "";
        };
        
        // Override new window function
        window.openNewWindow = function(url) {
            if (window.wrapperBridge && window.wrapperBridge.openNewWindow) {
                window.wrapperBridge.openNewWindow(url || "");
                return true;
            }
            return false;
        };
        
        // Wrapper detection functions
        window.isWrapper = function() {
            return window.isInWrapper;
        };
        
        window.canExecuteShell = function() {
            return window.isInWrapper && window.wrapperBridge && 
                   typeof window.wrapperBridge.canExecuteShell === 'function' ? 
                   window.wrapperBridge.canExecuteShell() : false;
        };
        
        window.getWrapperVersion = function() {
            return window.wrapperBridge && window.wrapperBridge.getWrapperVersion ? 
                   window.wrapperBridge.getWrapperVersion() : "Unknown";
        };
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initializeWrapperBridge);
    } else {
        initializeWrapperBridge();
    }
    
    // Also try to initialize after a short delay (for WebChannel)
    setTimeout(initializeWrapperBridge, 100);
    
})();
