class Stack {
    constructor() {
        this.cards = [];
        this.currentCard = 0;
        this.filename = null;
        this.modified = false;
    }

    createCard() {
        const card = {
            objects: [],
            scripts: new Map()
        };
        this.cards.push(card);
        return card;
    }

    saveStack() {
        // Use WebTalkFileOperations.prepareStackData() to ensure consistent handling of IDs and image types
        // This will use our fixes for ID preservation and proper image type handling
        const stackData = WebTalkFileOperations.prepareStackData();
        const blob = new Blob([JSON.stringify(stackData, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = this.filename || 'untitled.wstack';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        this.modified = false;
    }

    loadStack(file) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            
            reader.onload = async (e) => {
                try {
                    const stackData = JSON.parse(e.target.result);
                    
                    // Use the WebTalkFileOperations.loadStackFromData method to ensure
                    // consistent behavior with our fixes for ID preservation and image type handling
                    await WebTalkFileOperations.loadStackFromData(stackData);
                    
                    this.filename = file.name;
                    this.modified = false;
                    resolve();
                } catch (error) {
                    console.error('Error in Stack.loadStack:', error);
                    reject(error);
                }
            };
            
            reader.onerror = () => reject(reader.error);
            reader.readAsText(file);
        });
    }

    markModified() {
        this.modified = true;
    }

    async confirmSave() {
        if (this.modified) {
            const response = await new Promise(resolve => {
                const result = confirm('Do you want to save changes to the current stack?');
                resolve(result);
            });
            
            if (response) {
                this.saveStack();
            }
        }
    }
}
