// shell.js - Shell command implementation for WebTalk
console.log('Shell.js loaded successfully');

// Shell object that provides shell command functionality for WebTalk
window.WebTalkShell = {
    version: '1.0.1', // my version of this shell.js script. I will update this version number manually as and when required.
    
    // Flag to track if shell commands are allowed
    shellEnabled: false,
    
    // Flag to track if the user has been prompted about enabling shell commands
    hasPromptedUser: false,
    
    // Execute a shell command and return the result
    executeCommand: function(command) {
        if (!this.shellEnabled) {
            return 'Error: Shell commands are not enabled. Use "set the allowShell to true" first.';
        }
        
        console.log('Shell executing command:', command);
        
        // In the wrapper version, this communicates with the native shell
        try {
            // This is where the actual shell command execution happens
            // The wrapper intercepts this and executes the command in the OS shell
            const result = this._executeShellCommand(command);
            console.log('Shell command result:', result);
            return result; // Ensure result is returned to the interpreter
        } catch (error) {
            console.error('Error executing shell command:', error);
            return 'Error executing shell command: ' + error.message;
        }
    },
    
    // Internal method that is overridden by the wrapper
    _executeShellCommand: function(command) {
        // If allowShell is true, we must be in the wrapper since browser-only mode can't set it
        // OR check if we're running in the wrapper (window.mainWindow exists)
        if (this.shellEnabled || window.isWrapper || window.isWebTalkWrapper || 
            (window.mainWindow && typeof window.mainWindow.executeShellCommandSync === 'function')) {
            // Use the wrapper's native shell execution
            console.log('Executing shell command via wrapper:', command);
            try {
                // Ensure mainWindow is available
                if (window.mainWindow && typeof window.mainWindow.executeShellCommandSync === 'function') {
                    const result = window.mainWindow.executeShellCommandSync(command);
                    // Make sure we're returning the result properly
                    console.log('Shell command completed with result:', result);
                    return result;
                } else {
                    console.error('mainWindow.executeShellCommandSync not available');
                    return 'Error: Shell command bridge not available. Please restart the wrapper.';
                }
            } catch (error) {
                console.error('Error executing shell command via wrapper:', error);
                return 'Error executing shell command: ' + error.message;
            }
        } else {
            // Fallback for browser-only mode
            console.log('Shell command would be executed (browser-only mode):', command);
            return 'Shell command would be executed: ' + command + '\n(This is a placeholder - actual execution requires the wrapper version)';
        }
    }
};

// Add a method to enable shell commands with confirmation
window.WebTalkShell.enableShell = async function(interpreter) {
    // Check if already enabled
    if (this.shellEnabled) {
        console.log('Shell commands already enabled');
        return true;
    }
    
    // Only prompt once per session unless reset
    if (this.hasPromptedUser) {
        return this.shellEnabled;
    }
    
    this.hasPromptedUser = true;
    
    // Use the interpreter's createDialog method directly instead of interpret
    // This avoids issues with long strings and quotes in the answer command
    const message = "Are you sure you want to enable shell mode?\n\nDoing so will allow this program to communicate directly with the OS on a lower level, but could also allow for malicious scripts to run.";
    
    // Set "Leave it off" as the first button (default) and "Enable" as the second button
    const buttons = ["Leave it off", "Enable"];
    
    const result = await interpreter.createDialog(message, buttons);
    
    // Check the result to see which button was clicked
    if (result && result === "Enable") {
        // Only enable shell commands if we're in the wrapper
        // Check for explicit wrapper flags or mainWindow object
        if (window.isWrapper || window.isWebTalkWrapper || (window.mainWindow && typeof window.mainWindow.executeShellCommandSync === 'function')) {
            // Set the flag in our object
            this.shellEnabled = true;
            
            // Also set it directly on the interpreter for immediate access
            if (interpreter) {
                interpreter.allowShell = true;
            }
            
            console.log('Shell commands enabled');
            
            // Store in localStorage for persistence if available
            try {
                if (window.localStorage) {
                    window.localStorage.setItem('webTalkShellEnabled', 'true');
                }
            } catch (e) {
                console.warn('Could not store shell state in localStorage:', e);
            }
            
            return true;
        } else {
            // In browser-only mode, show a message that shell commands cannot be enabled
            console.log('Shell commands cannot be enabled in browser-only mode');
            interpreter.outputHandler("Shell commands cannot be enabled in browser-only mode. Use the wrapper version of WebTalk for this feature.");
            this.shellEnabled = false;
            if (interpreter) {
                interpreter.allowShell = false;
            }
            return false;
        }
    } else {
        // Reset the hasPromptedUser flag so the dialog shows again next time
        this.hasPromptedUser = false;
        this.shellEnabled = false;
        if (interpreter) {
            interpreter.allowShell = false;
        }
        console.log('Shell commands not enabled');
        return false;
    }
};

// Add a method to check if shell is enabled
window.WebTalkShell.isEnabled = function() {
    return this.shellEnabled;
};

// Initialize shell state from localStorage if available
try {
    if (window.localStorage && window.localStorage.getItem('webTalkShellEnabled') === 'true') {
        // Only enable if we're in the wrapper
        if (window.isWrapper || window.isWebTalkWrapper || 
            (window.mainWindow && typeof window.mainWindow.executeShellCommandSync === 'function')) {
            window.WebTalkShell.shellEnabled = true;
            console.log('Shell commands enabled from saved state');
        }
    }
} catch (e) {
    console.warn('Could not restore shell state from localStorage:', e);
}

// Log that shell.js has been loaded
console.log('WebTalkShell initialized');
