// player.js - Handles video player functionality for WebTalk

class WebTalkPlayer {
    // Initialize player functionality
    static initialize() {
        console.log('WebTalkPlayer initialized');
    }

    // Create a player element
    static createPlayer(name, id) {
        const playerContainer = document.createElement('div');
        playerContainer.className = 'player';
        playerContainer.dataset.name = name;
        playerContainer.dataset.type = 'player';
        playerContainer.id = id.toString();
        
        // Set default position and size
        playerContainer.style.left = '50px';
        playerContainer.style.top = '50px';
        playerContainer.style.width = '320px';
        playerContainer.style.height = '180px';
        playerContainer.style.position = 'absolute';
        
        // Set default styling
        playerContainer.style.backgroundColor = '#000';
        playerContainer.style.border = '1px solid #444';
        playerContainer.style.overflow = 'hidden';
        playerContainer.style.display = 'flex';
        playerContainer.style.alignItems = 'center';
        playerContainer.style.justifyContent = 'center';
        
        // Create placeholder content
        const placeholderText = document.createElement('div');
        placeholderText.textContent = 'Video Player';
        placeholderText.style.color = '#666';
        placeholderText.style.fontFamily = 'Arial, sans-serif';
        placeholderText.style.textAlign = 'center';
        playerContainer.appendChild(placeholderText);
        
        // Create iframe for video (hidden initially)
        const videoFrame = document.createElement('iframe');
        videoFrame.style.width = '100%';
        videoFrame.style.height = '100%';
        videoFrame.style.border = 'none';
        videoFrame.style.display = 'none';
        // We'll control pointer-events dynamically based on mode
        videoFrame.dataset.playerElement = 'true';
        videoFrame.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
        videoFrame.allowFullscreen = true;
        playerContainer.appendChild(videoFrame);
        
        // Initialize empty script for this player
        WebTalkObjects.scripts.set(name, '');
        
        // Initialize custom properties for this player
        const customProperties = new Map();
        customProperties.set('visible', 'true');
        WebTalkObjects.customProperties.set(name, customProperties);
        
        // Add event listeners for selection and interaction
        let mouseDownOnPlayer = false;
        
        playerContainer.addEventListener('mousedown', (e) => {
            // Only handle selection in edit mode
            if (window.webTalkApp.interpreter.mode !== 'edit') {
                return;
            }
            
            // Prevent default to avoid text selection
            e.preventDefault();
            
            // Select this player
            WebTalkObjects.selectObject(playerContainer);
            
            // Set flag that mouse is down on this player
            mouseDownOnPlayer = true;
            
            // Execute the mouseDown handler in the player's script
            if (window.webTalkApp.interpreter.mode === 'browse') {
                window.webTalkApp.interpreter.executeObjectScript(playerContainer.dataset.name, 'mouseDown', [e.button + 1]);
            }
        });
        
        playerContainer.addEventListener('mouseup', (e) => {
            // Only handle events in edit mode
            if (window.webTalkApp.interpreter.mode !== 'edit') {
                return;
            }
            
            // Only execute if the mouse was down on this player
            if (!mouseDownOnPlayer) return;
            
            mouseDownOnPlayer = false;
            
            // Execute the mouseUp handler in the player's script
            if (window.webTalkApp.interpreter.mode === 'browse') {
                window.webTalkApp.interpreter.executeObjectScript(playerContainer.dataset.name, 'mouseUp', [e.button + 1]);
            }
        });
        
        playerContainer.addEventListener('mousemove', (e) => {
            // Only handle events in edit mode
            if (window.webTalkApp.interpreter.mode !== 'edit') {
                return;
            }
            
            // Execute the mouseMove handler in the player's script
            if (window.webTalkApp.interpreter.mode === 'browse') {
                window.webTalkApp.interpreter.executeObjectScript(playerContainer.dataset.name, 'mouseMove');
            }
        });
        
        return playerContainer;
    }
    
    // Set the video source for a player
    static setPlayerSource(element, url) {
        if (!element) return;
        
        // Store the original URL
        const customProps = WebTalkObjects.customProperties.get(element.dataset.name);
        if (customProps) {
            customProps.set('filename', url);
        }
        
        // Clear any existing content
        const placeholderText = element.querySelector('div');
        const videoFrame = element.querySelector('iframe');
        
        if (!url || url.trim() === '') {
            // Reset to placeholder if URL is empty
            if (placeholderText) placeholderText.style.display = 'block';
            if (videoFrame) videoFrame.style.display = 'none';
            return;
        }
        
        // Process different video URLs
        let embedUrl = '';
        
        // YouTube
        if (url.includes('youtube.com') || url.includes('youtu.be')) {
            let videoId = '';
            
            // Extract video ID from various YouTube URL formats
            if (url.includes('youtube.com/watch')) {
                const urlParams = new URLSearchParams(new URL(url).search);
                videoId = urlParams.get('v');
            } else if (url.includes('youtu.be/')) {
                videoId = url.split('youtu.be/')[1];
                if (videoId.includes('?')) {
                    videoId = videoId.split('?')[0];
                }
            } else if (url.includes('youtube.com/embed/')) {
                videoId = url.split('youtube.com/embed/')[1];
                if (videoId.includes('?')) {
                    videoId = videoId.split('?')[0];
                }
            }
            
            if (videoId) {
                embedUrl = `https://www.youtube.com/embed/${videoId}`;
                
                // Set the video dimensions to match standard 16:9 aspect ratio
                const width = parseInt(element.style.width);
                const height = Math.round(width * 9 / 16);
                element.style.height = `${height}px`;
            }
        } 
        // Vimeo
        else if (url.includes('vimeo.com')) {
            let videoId = '';
            
            // Extract video ID from Vimeo URL
            if (url.includes('vimeo.com/')) {
                videoId = url.split('vimeo.com/')[1];
                if (videoId.includes('?')) {
                    videoId = videoId.split('?')[0];
                }
            }
            
            if (videoId) {
                embedUrl = `https://player.vimeo.com/video/${videoId}`;
                
                // Set the video dimensions to match standard 16:9 aspect ratio
                const width = parseInt(element.style.width);
                const height = Math.round(width * 9 / 16);
                element.style.height = `${height}px`;
            }
        }
        // Direct video file
        else if (url.endsWith('.mp4') || url.endsWith('.webm') || url.endsWith('.ogg')) {
            // For direct video files, create a video element instead of iframe
            if (videoFrame) {
                videoFrame.style.display = 'none';
            }
            
            let videoElement = element.querySelector('video');
            if (!videoElement) {
                videoElement = document.createElement('video');
                videoElement.controls = true;
                videoElement.style.width = '100%';
                videoElement.style.height = '100%';
                // We'll control pointer-events dynamically based on mode
                videoElement.dataset.playerElement = 'true';
                element.appendChild(videoElement);
            }
            
            videoElement.src = url;
            videoElement.style.display = 'block';
            
            if (placeholderText) {
                placeholderText.style.display = 'none';
            }
            
            return;
        }
        
        // If we have a valid embed URL, update the iframe
        if (embedUrl && videoFrame) {
            videoFrame.src = embedUrl;
            videoFrame.style.display = 'block';
            
            if (placeholderText) {
                placeholderText.style.display = 'none';
            }
        }
    }
    
    // Ensure player maintains aspect ratio during resize
    static maintainAspectRatio(element, newWidth, newHeight) {
        if (!element) return { width: newWidth, height: newHeight };
        
        // Default to 16:9 aspect ratio if no video is loaded
        let aspectRatio = 16/9;
        
        // Get the current dimensions
        const currentWidth = parseInt(element.style.width);
        const currentHeight = parseInt(element.style.height);
        
        // Calculate the current aspect ratio if dimensions are valid
        if (currentWidth > 0 && currentHeight > 0) {
            aspectRatio = currentWidth / currentHeight;
        }
        
        // Determine which dimension changed
        if (newWidth !== currentWidth) {
            // Width changed, adjust height to maintain aspect ratio
            return {
                width: newWidth,
                height: Math.round(newWidth / aspectRatio)
            };
        } else if (newHeight !== currentHeight) {
            // Height changed, adjust width to maintain aspect ratio
            return {
                width: Math.round(newHeight * aspectRatio),
                height: newHeight
            };
        }
        
        // If neither changed, return the original dimensions
        return { width: newWidth, height: newHeight };
    }
}

class WebTalkPlayerHelper {
    // Update player elements pointer-events based on mode
    static updatePlayerElementsForMode(mode) {
        // Find all player elements
        const playerElements = document.querySelectorAll('[data-player-element="true"]');
        
        // Set pointer-events based on mode
        playerElements.forEach(element => {
            if (mode === 'browse') {
                // In browse mode, enable interaction with player controls
                element.style.pointerEvents = 'auto';
            } else {
                // In edit mode, disable interaction to allow selection of the container
                element.style.pointerEvents = 'none';
            }
        });
    }
}

// Initialize the player when the page loads
document.addEventListener('DOMContentLoaded', () => {
    WebTalkPlayer.initialize();
    
    // Listen for mode changes to update player elements
    if (window.webTalkApp && window.webTalkApp.interpreter) {
        // Set initial state based on current mode
        const currentMode = window.webTalkApp.interpreter.mode || 'edit';
        WebTalkPlayerHelper.updatePlayerElementsForMode(currentMode);
        
        // Set up a MutationObserver to detect mode changes
        const observer = new MutationObserver((mutations) => {
            for (const mutation of mutations) {
                if (mutation.type === 'attributes' && mutation.attributeName === 'data-mode') {
                    const newMode = document.body.dataset.mode;
                    WebTalkPlayerHelper.updatePlayerElementsForMode(newMode);
                }
            }
        });
        
        // Start observing the body element for mode changes
        observer.observe(document.body, { attributes: true });
    }
});
