// overview.js - Object Overview Palette for WebTalk
// Provides a floating palette showing all objects in the current stack

class ObjectOverview {
    constructor() {
        this.isVisible = false;
        this.currentSortColumn = 'name';
        this.sortDirection = 'asc';
        this.createOverviewPalette();
        this.setupEventListeners();
    }

    createOverviewPalette() {
        // Create the main overview palette container
        const palette = document.createElement('div');
        palette.id = 'object-overview';
        palette.className = 'object-inspector'; // Reuse inspector styling
        palette.style.width = '400px';
        palette.style.top = '150px';
        palette.style.left = '150px';
        palette.style.display = 'none';

        // Create header with drag handle and close button
        const header = document.createElement('div');
        header.className = 'inspector-header';

        const dragHandle = document.createElement('div');
        dragHandle.className = 'inspector-drag-handle';
        dragHandle.innerHTML = '<span>&#8942;</span><span class="inspector-title">Object Overview</span>';

        const closeButton = document.createElement('button');
        closeButton.className = 'inspector-close-button';
        closeButton.innerHTML = '×';
        closeButton.onclick = () => this.hide();

        header.appendChild(dragHandle);
        header.appendChild(closeButton);

        // Create content area
        const content = document.createElement('div');
        content.className = 'inspector-content';
        content.style.maxHeight = '500px';

        // Create table for object list
        const table = document.createElement('table');
        table.className = 'inspector-table';
        table.id = 'overview-table';

        // Create table header
        const thead = document.createElement('thead');
        const headerRow = document.createElement('tr');

        const columns = [
            { key: 'name', label: 'Name' },
            { key: 'type', label: 'Type' },
            { key: 'card', label: 'On Card' },
            { key: 'id', label: 'ID' }
        ];

        columns.forEach(col => {
            const th = document.createElement('th');
            th.textContent = col.label;
            th.style.cursor = 'pointer';
            th.style.userSelect = 'none';
            th.onclick = () => this.sortBy(col.key);
            th.setAttribute('data-column', col.key);
            headerRow.appendChild(th);
        });

        thead.appendChild(headerRow);
        table.appendChild(thead);

        // Create table body
        const tbody = document.createElement('tbody');
        tbody.id = 'overview-tbody';
        table.appendChild(tbody);

        content.appendChild(table);
        palette.appendChild(header);
        palette.appendChild(content);

        // Add to document
        document.body.appendChild(palette);

        // Make draggable
        this.makeDraggable(palette, dragHandle);
    }

    makeDraggable(element, handle) {
        let isDragging = false;
        let startX, startY, startLeft, startTop;

        handle.addEventListener('mousedown', (e) => {
            isDragging = true;
            startX = e.clientX;
            startY = e.clientY;
            startLeft = parseInt(window.getComputedStyle(element).left, 10);
            startTop = parseInt(window.getComputedStyle(element).top, 10);
            handle.style.cursor = 'grabbing';
            e.preventDefault();
        });

        document.addEventListener('mousemove', (e) => {
            if (!isDragging) return;
            const deltaX = e.clientX - startX;
            const deltaY = e.clientY - startY;
            element.style.left = (startLeft + deltaX) + 'px';
            element.style.top = (startTop + deltaY) + 'px';
        });

        document.addEventListener('mouseup', () => {
            if (isDragging) {
                isDragging = false;
                handle.style.cursor = 'grab';
            }
        });
    }

    setupEventListeners() {
        // Listen for object row clicks to open inspector
        document.addEventListener('click', (e) => {
            if (e.target.closest('#overview-tbody tr')) {
                const row = e.target.closest('tr');
                const objectName = row.getAttribute('data-object-name');
                if (objectName && WebTalkObjects) {
                    // Get the object and its card information
                    const obj = WebTalkObjects.objects.get(objectName);
                    if (obj) {
                        // Step 1: Switch to edit mode first if not already in edit mode
                        if (window.webTalkApp && window.webTalkApp.interpreter) {
                            const currentMode = window.webTalkApp.interpreter.mode;
                            if (currentMode !== 'edit') {
                                window.webTalkApp.interpreter.interpret('set the mode to edit');
                            }
                            
                            // Step 2: Find which card contains this object by checking parent elements
                            let targetCardId = null;
                            let targetCardName = null;
                            
                            // Check which card element contains this object
                            let parentElement = obj.parentNode;
                            while (parentElement && parentElement !== document.body) {
                                if (parentElement.id === 'card') {
                                    targetCardId = 1;
                                    targetCardName = parentElement.dataset.name || 'Card 1';
                                    break;
                                } else if (parentElement.id && parentElement.id.startsWith('card-')) {
                                    targetCardId = parseInt(parentElement.id.replace('card-', ''));
                                    targetCardName = parentElement.dataset.name || `Card ${targetCardId}`;
                                    break;
                                }
                                parentElement = parentElement.parentNode;
                            }
                            
                            // Navigate to the target card if it's different from current card
                            const currentCardId = window.currentCardId || 1;
                            if (targetCardId && targetCardId !== currentCardId) {
                                window.webTalkApp.interpreter.interpret(`go card ${targetCardId}`);
                            }
                            
                            // Step 3: Select the object and open inspector after navigation
                            setTimeout(() => {
                                // Ensure we're in edit mode before selecting
                                const currentMode = window.webTalkApp.interpreter.mode;
                                if (currentMode !== 'edit') {
                                    window.webTalkApp.interpreter.interpret('set the mode to edit');
                                }
                                
                                // Select the object to show selection handles
                                if (WebTalkObjects.selectObject) {
                                    WebTalkObjects.selectObject(obj);
                                }
                                
                                // Step 4: Open inspector directly instead of simulating double-click
                                setTimeout(() => {
                                    if (window.WebTalkInspector) {
                                        window.WebTalkInspector.showInspector(obj);
                                    } else if (WebTalkInspector) {
                                        WebTalkInspector.showInspector(obj);
                                    }
                                }, 100);
                            }, targetCardId !== currentCardId ? 200 : 50); // Allow more time for card navigation if needed
                        }
                    }
                }
            }
        });

        // Listen for right-click on object rows to show context menu
        document.addEventListener('contextmenu', (e) => {
            if (e.target.closest('#overview-tbody tr')) {
                e.preventDefault();
                e.stopPropagation();
                
                const row = e.target.closest('tr');
                const objectName = row.getAttribute('data-object-name');
                if (objectName && WebTalkObjects) {
                    this.showContextMenu(e, objectName);
                }
            }
        });

    }

    showContextMenu(e, objectName) {
        // Remove any existing context menus
        const existingMenu = document.querySelector('.context-menu');
        if (existingMenu) {
            existingMenu.remove();
        }

        // Create context menu
        const contextMenu = document.createElement('div');
        contextMenu.className = 'context-menu';
        contextMenu.style.top = `${e.clientY}px`;
        contextMenu.style.left = `${e.clientX}px`;

        // Add edit script menu item
        const editScriptItem = document.createElement('div');
        editScriptItem.className = 'context-menu-item';
        editScriptItem.textContent = 'Edit Script';
        editScriptItem.addEventListener('click', () => {
            WebTalkObjects.openScriptEditor(objectName);
            contextMenu.remove();
        });

        contextMenu.appendChild(editScriptItem);

        // Add context menu to document
        document.body.appendChild(contextMenu);

        // Close menu when clicking elsewhere
        const closeMenu = (e) => {
            if (!contextMenu.contains(e.target)) {
                contextMenu.remove();
                document.removeEventListener('mousedown', closeMenu);
            }
        };

        document.addEventListener('mousedown', closeMenu);
    }

    show() {
        const palette = document.getElementById('object-overview');
        if (palette) {
            palette.style.display = 'block';
            this.isVisible = true;
            this.updateObjectList();
        }
    }

    hide() {
        const palette = document.getElementById('object-overview');
        if (palette) {
            palette.style.display = 'none';
            this.isVisible = false;
        }
    }

    toggle() {
        if (this.isVisible) {
            this.hide();
        } else {
            this.show();
        }
    }

    updateObjectList() {
        if (!this.isVisible) return;

        const tbody = document.getElementById('overview-tbody');
        if (!tbody) {
            console.log('Overview update failed: tbody element not found');
            return;
        }
        if (!WebTalkObjects) {
            console.log('Overview update failed: WebTalkObjects not available');
            return;
        }

        // Clear existing rows
        tbody.innerHTML = '';

        // Get all objects
        const objects = [];
        WebTalkObjects.objects.forEach((obj, name) => {
            const objectType = obj.dataset.type || 'unknown';
            
            // Skip objects with type 'unknown' (like cards)
            if (objectType === 'unknown') {
                return;
            }
            
            // Determine which card this object is on by checking parent elements
            let cardName = 'Unknown';
            let parentElement = obj.parentNode;
            while (parentElement && parentElement !== document.body) {
                if (parentElement.id === 'card') {
                    cardName = parentElement.dataset.name || 'Card 1';
                    break;
                } else if (parentElement.id && parentElement.id.startsWith('card-')) {
                    const cardId = parseInt(parentElement.id.replace('card-', ''));
                    cardName = parentElement.dataset.name || `Card ${cardId}`;
                    break;
                }
                parentElement = parentElement.parentNode;
            }

            objects.push({
                name: name,
                type: objectType,
                card: cardName,
                id: obj.id || 'N/A',
                element: obj
            });
        });

        // Sort objects
        this.sortObjects(objects);

        // Create table rows
        objects.forEach(obj => {
            const row = document.createElement('tr');
            row.setAttribute('data-object-name', obj.name);
            row.style.cursor = 'pointer';

            // Name column
            const nameCell = document.createElement('td');
            nameCell.textContent = obj.name;
            nameCell.className = 'property-name';
            row.appendChild(nameCell);

            // Type column
            const typeCell = document.createElement('td');
            typeCell.textContent = obj.type;
            typeCell.className = 'property-value';
            row.appendChild(typeCell);

            // Card column
            const cardCell = document.createElement('td');
            cardCell.textContent = obj.card;
            cardCell.className = 'property-value';
            row.appendChild(cardCell);

            // ID column
            const idCell = document.createElement('td');
            idCell.textContent = obj.id;
            idCell.className = 'property-value';
            row.appendChild(idCell);

            tbody.appendChild(row);
        });

        // Update sort indicators
        this.updateSortIndicators();
    }

    sortBy(column) {
        if (this.currentSortColumn === column) {
            // Toggle direction if same column
            this.sortDirection = this.sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            // New column, default to ascending
            this.currentSortColumn = column;
            this.sortDirection = 'asc';
        }
        this.updateObjectList();
    }

    sortObjects(objects) {
        objects.sort((a, b) => {
            let valueA = a[this.currentSortColumn];
            let valueB = b[this.currentSortColumn];

            // Handle numeric ID sorting
            if (this.currentSortColumn === 'id') {
                valueA = parseInt(valueA) || 0;
                valueB = parseInt(valueB) || 0;
            } else {
                // String comparison
                valueA = String(valueA).toLowerCase();
                valueB = String(valueB).toLowerCase();
            }

            let comparison = 0;
            if (valueA > valueB) {
                comparison = 1;
            } else if (valueA < valueB) {
                comparison = -1;
            }

            return this.sortDirection === 'desc' ? -comparison : comparison;
        });
    }

    updateSortIndicators() {
        // Remove existing sort indicators
        const headers = document.querySelectorAll('#overview-table th');
        headers.forEach(th => {
            th.textContent = th.textContent.replace(' ↑', '').replace(' ↓', '');
        });

        // Add sort indicator to current column
        const currentHeader = document.querySelector(`#overview-table th[data-column="${this.currentSortColumn}"]`);
        if (currentHeader) {
            const indicator = this.sortDirection === 'asc' ? ' ↑' : ' ↓';
            currentHeader.textContent += indicator;
        }
    }

    // Method to be called when objects are created, deleted, or modified
    refresh() {
        if (this.isVisible) {
            this.updateObjectList();
        }
    }
}

// Global instance
let objectOverview = null;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    objectOverview = new ObjectOverview();
});

// Global functions for external access
function showObjectOverview() {
    if (objectOverview) {
        objectOverview.show();
    }
}

function hideObjectOverview() {
    if (objectOverview) {
        objectOverview.hide();
    }
}

function toggleObjectOverview() {
    if (objectOverview) {
        objectOverview.toggle();
    }
}

function refreshObjectOverview() {
    if (objectOverview) {
        objectOverview.refresh();
    }
}
