/**
 * menu.js - Menu implementation for WebTalk buttons
 * 
 * This file implements menu functionality for WebTalk buttons.
 * When a button has its isMenu property set to true, clicking it
 * will display a menu based on its menuText property.
 */

// Define WebTalkMenu in global scope
window.WebTalkMenu = class WebTalkMenu {
    static activeMenu = null;
    static activeSubmenus = [];
    static activeButton = null; // Store reference to the button that triggered the menu
    
    /**
     * Initialize the menu system
     */
    static initialize() {
        // Add a global click handler to close menus when clicking outside
        document.addEventListener('click', (event) => {
            // If we're not clicking on a menu or a menu button, close all menus
            if (!event.target.closest('.webtalk-menu') && 
                !event.target.closest('.webtalk-menu-button')) {
                this.closeAllMenus();
            }
        });
    }
    
    /**
     * Show a menu for a button
     * @param {HTMLElement} buttonElement - The button element that triggered the menu
     * @param {string} menuText - The text content for the menu items
     */
    static showMenu(buttonElement, menuText) {
        // Close any existing menus first
        this.closeAllMenus();
        
        // Store reference to the button that triggered the menu
        this.activeButton = buttonElement;
        
        // Create the menu element
        const menu = document.createElement('div');
        menu.className = 'webtalk-menu';
        
        // Position the menu below the button
        const buttonRect = buttonElement.getBoundingClientRect();
        menu.style.position = 'absolute';
        menu.style.left = `${buttonRect.left}px`;
        menu.style.top = `${buttonRect.bottom}px`;
        menu.style.zIndex = '1000';
        
        // Parse the menu text and create menu items
        const lines = menuText.split('\n');
        this.buildMenuItems(menu, lines, 0, lines.length - 1, 0);
        
        // Add the menu to the document
        document.body.appendChild(menu);
        
        // Store reference to the active menu
        this.activeMenu = menu;
        
        // Prevent the event from bubbling up
        return false;
    }
    
    /**
     * Build menu items from text lines
     * @param {HTMLElement} parentMenu - The parent menu element
     * @param {Array<string>} lines - Array of text lines
     * @param {number} startIndex - Start index in the lines array
     * @param {number} endIndex - End index in the lines array
     * @param {number} indentLevel - Current indentation level
     * @returns {number} The index after processing
     */
    static buildMenuItems(parentMenu, lines, startIndex, endIndex, indentLevel) {
        let i = startIndex;
        
        while (i <= endIndex) {
            const line = lines[i];
            
            // Calculate the indentation of this line
            const lineIndent = this.getIndentLevel(line);
            
            // If this line has a deeper indent than current level, it belongs to a submenu
            if (lineIndent > indentLevel) {
                // Skip this line as it will be processed in a submenu context
                i++;
                continue;
            }
            
            // If this line has a lower indent than current level, return to parent
            if (lineIndent < indentLevel) {
                return i;
            }
            
            // Get the trimmed text for this menu item
            const itemText = line.trim();
            
            // Skip empty lines
            if (itemText === '') {
                i++;
                continue;
            }
            
            // Check if this is a separator
            if (itemText === '-') {
                const separator = document.createElement('div');
                separator.className = 'webtalk-menu-separator';
                parentMenu.appendChild(separator);
                i++;
                continue;
            }
            
            // Create the menu item
            const menuItem = document.createElement('div');
            menuItem.className = 'webtalk-menu-item';
            
            // Check if this is a submenu item by looking for leading spaces
            const thisIndent = this.getIndentLevel(itemText);
            // If this item has indentation, strip it for display
            if (thisIndent > 0) {
                itemText = itemText.substring(thisIndent * 3);
            }
            
            // Set the menu item text after potential indentation stripping
            menuItem.textContent = itemText;
            
            // Check if the next line is indented (submenu)
            const hasSubmenu = i < endIndex && this.getIndentLevel(lines[i + 1]) > indentLevel;
            
            if (hasSubmenu) {
                // Add submenu indicator
                menuItem.classList.add('webtalk-menu-item-with-submenu');
                
                // Create submenu container
                const submenu = document.createElement('div');
                submenu.className = 'webtalk-submenu';
                
                // Add hover handler to show submenu
                menuItem.addEventListener('mouseenter', () => {
                    // Position the submenu to the right of the parent item
                    const itemRect = menuItem.getBoundingClientRect();
                    
                    // Calculate position relative to viewport
                    let left = itemRect.right;
                    let top = itemRect.top;
                    
                    // Check if submenu would go off the right edge of the screen
                    const submenuWidth = 200; // Approximate width of submenu
                    if (left + submenuWidth > window.innerWidth) {
                        // Position to the left of the parent item instead
                        left = itemRect.left - submenuWidth;
                    }
                    
                    // Apply the position
                    submenu.style.position = 'fixed'; // Use fixed positioning for viewport coordinates
                    submenu.style.left = `${left}px`;
                    submenu.style.top = `${top}px`;
                    submenu.style.display = 'block';
                    
                    // Add to active submenus
                    this.activeSubmenus.push(submenu);
                });
                
                menuItem.addEventListener('mouseleave', (event) => {
                    // Check if we're moving to the submenu
                    const relatedTarget = event.relatedTarget;
                    if (!submenu.contains(relatedTarget)) {
                        submenu.style.display = 'none';
                        
                        // Remove from active submenus
                        const index = this.activeSubmenus.indexOf(submenu);
                        if (index !== -1) {
                            this.activeSubmenus.splice(index, 1);
                        }
                    }
                });
                
                // Add the menu item to the parent
                parentMenu.appendChild(menuItem);
                
                // Add the submenu to the document body instead of the parent menu
                // This allows proper positioning and stacking
                document.body.appendChild(submenu);
                
                // Initially hide the submenu
                submenu.style.display = 'none';
                
                // Process submenu items
                i = this.buildMenuItems(submenu, lines, i + 1, endIndex, indentLevel + 1);
            } else {
                // Regular menu item - add click handler
                menuItem.addEventListener('click', () => {
                    // Store the item text for the action
                    const selectedItemText = itemText;
                    
                    // Execute the menu item action
                    this.executeMenuAction(selectedItemText);
                    
                    // Close all menus
                    this.closeAllMenus();
                });
                
                // Add the menu item to the parent
                parentMenu.appendChild(menuItem);
                i++;
            }
        }
        
        return i;
    }
    
    /**
     * Get the indentation level of a line
     * @param {string} line - The text line
     * @returns {number} The indentation level (0 for no indent)
     */
    static getIndentLevel(line) {
        // Count groups of 3 spaces at the beginning of the line
        let count = 0;
        let i = 0;
        
        while (i < line.length) {
            if (line.substr(i, 3) === '   ') {
                count++;
                i += 3;
            } else {
                break;
            }
        }
        
        return count;
    }
    
    /**
     * Execute a menu item action
     * @param {string} itemText - The text of the selected menu item
     */
    static executeMenuAction(itemText) {
        // Get the interpreter instance
        const interpreter = window.webTalkApp?.interpreter;
        if (!interpreter) {
            console.error('WebTalk interpreter not found');
            return;
        }
        
        // If we have an active button, send the menuPick handler to it
        if (this.activeButton && this.activeButton.dataset && this.activeButton.dataset.name) {
            try {
                // Send menuPick handler to the button with the selected item as parameter
                interpreter.executeObjectScript(this.activeButton.dataset.name, 'menuPick', [itemText]);
            } catch (e) {
                console.error('Error executing menuPick handler:', e);
            }
        } else {
            // Fallback: Send a menuItemSelected message to the card
            try {
                interpreter.executeObjectScript('card', 'menuItemSelected', [itemText]);
            } catch (e) {
                console.error('Error executing menuItemSelected handler:', e);
            }
        }
    }
    
    /**
     * Close all open menus
     */
    static closeAllMenus() {
        // Close all active submenus
        this.activeSubmenus.forEach(submenu => {
            if (submenu && submenu.parentNode) {
                submenu.parentNode.removeChild(submenu);
            }
        });
        this.activeSubmenus = [];
        
        // Close the main menu
        if (this.activeMenu && this.activeMenu.parentNode) {
            this.activeMenu.parentNode.removeChild(this.activeMenu);
            this.activeMenu = null;
        }
    }
}

// Initialize the menu system when the page loads
document.addEventListener('DOMContentLoaded', () => {
    WebTalkMenu.initialize();
});
