/**
 * Inspector Tasks - Handles actions when inspector properties are clicked
 */

// Self-executing function to check if the module is loaded
(function() {
    console.log('inspector-tasks.js loaded');
})();

const InspectorTasks = {
    // Reference to the interpreter
    interpreter: null,

    // Flag to prevent multiple executions of handleVisibleToggle
    _isHandlingVisibleToggle: false,

    // Flag to prevent multiple executions of handleTextSize
    _isHandlingTextSize: false,

    // Flag to prevent multiple executions of handleShowBorderToggle
    _isHandlingShowBorderToggle: false,

    // Flag to prevent multiple executions of handleLineSize
    _isHandlingLineSize: false,

    // Flag to prevent multiple executions of handleBorderWidth
    _isHandlingBorderWidth: false,

    // Flag to prevent multiple executions of handleTextStyleEdit
    _isHandlingTextStyleEdit: false,

    // Flag to prevent multiple executions of handleFontEdit
    _isHandlingFontEdit: false,

    // Flag to prevent multiple executions of handleFilenameEdit
    _isHandlingFilenameEdit: false,

    // Flag to prevent multiple executions of handleAngleEdit
    _isHandlingAngleEdit: false,

    // Flag to prevent multiple executions of handleNameEdit
    _isHandlingNameEdit: false,

    // Flag to prevent multiple executions of handleColorPropertyEdit
    _isHandlingColorPropertyEdit: false,

    // Flag to prevent multiple executions of handleLayerEdit
    _isHandlingLayerEdit: false,

    // Flag to prevent multiple executions of handleRestoreFieldFormatting
    _isHandlingRestoreFieldFormatting: false,

    // Initialize with the interpreter instance
    initialize: function() {
        this.interpreter = new InterpreterClass();
    },

    // Run a script (array of commands)
    runScript: async function(commands) {
        if (!this.interpreter) {
            this.initialize();
        }

        // Ensure commands is an array
        if (!Array.isArray(commands)) {
            commands = [commands];
        }

        // Execute each command
        for (const command of commands) {
            if (command.trim()) {
                await this.interpreter.interpret(command);
            }
        }
    },

    // Get the correct property name for setting properties
    getCorrectPropertyName: function(property) {
        // Map of display property names to their correct internal names
        const propertyMap = {
            // Text properties
            'textfont': 'textFont',
            'textsize': 'textSize',
            'textcolor': 'textColor',
            'textstyle': 'textStyle',
            'fontfamily': 'textFont',
            'fontsize': 'fontSize',

            // Color properties
            'backgroundcolor': 'backgroundColor',
            'bordercolor': 'borderColor',
            'foregroundcolor': 'foregroundColor',
            'progresscolor': 'progressColor',
            'progresscolour': 'progressColor',

            // Other properties
            'borderwidth': 'borderWidth',
            'linesize': 'lineSize',
            'startarrow': 'startArrow',
            'endarrow': 'endArrow',
            'linepattern': 'linePattern',
            'multiline': 'multiLine',
            'autotab': 'autoTab',
            'locktext': 'lockText',
            'sharedtext': 'sharedText',
            'showname': 'showName',
            'showborder': 'showBorder',
            'dontwrap': 'dontWrap'
        };

        // Convert property to lowercase for case-insensitive matching
        const lowerProperty = property.toLowerCase();

        // Return the mapped property name or the original if no mapping exists
        return propertyMap[lowerProperty] || property;
    },

    // Process a click on a property in the inspector
    handlePropertyClick: function(property, value, objectType, objectId) {
        console.log(`InspectorTasks.handlePropertyClick called with: ${property}, ${value}, ${objectType}, ${objectId}`);

        // Skip boolean properties that should only use checkbox toggles
        const booleanProperties = ['visible', 'hilited', 'highlighted', 'ischeckbox', 'ismenu', 'disabled', 'scrollable', 'locktext', 'multiline', 'autotab', 'widemargins', 'sharedtext', 'dontwrap', 'showborder', 'hasslider'];
        if (booleanProperties.includes(property.toLowerCase())) {
            console.log(`Skipping property click for boolean property: ${property}`);
            return;
        }

        // Choose the appropriate action based on the property
        switch(property.toLowerCase()) {
            case 'blendlevel':
                this.handleBlendLevelEdit(value, objectType, objectId);
                break;
            case 'name':
                this.handleNameEdit(value, objectType, objectId);
                break;
            case 'visible':
                this.handleVisibleToggle(value, objectType, objectId);
                break;
            case 'locktext':
                if (objectType === 'field') {
                    this.handleLocktextToggle(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'showfocusborder':
                if (objectType === 'field') {
                    this.handleShowFocusBorderToggle(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'showborder':
                this.handleShowBorderToggle(value, objectType, objectId);
                break;
            case 'textstyle':
                this.handleTextStyleEdit(value, objectType, objectId);
                break;
            case 'layer':
                this.handleLayerEdit(value, objectType, objectId);
                break;
            case 'textfont':
            case 'fontfamily':
                this.handleFontEdit(value, objectType, objectId);
                break;
            case 'textsize':
            case 'fontsize':
                this.handleTextSize(value, objectType, objectId, property);
                break;
            case 'textalign':
                this.handleTextAlignEdit(value, objectType, objectId);
                break;
            case 'filename':
                this.handleFilenameEdit(value, objectType, objectId);
                break;
            case 'textcolor':
            case 'backgroundcolor':
            case 'bordercolor':
            case 'foregroundcolor':
            case 'progresscolor':
            case 'progresscolour':
                this.handleColorPropertyEdit(property, value, objectType, objectId);
                break;
            case 'linesize':
                this.handleLineSizeEdit(value, objectType, objectId);
                break;
            case 'borderwidth':
                this.handleBorderWidthEdit(value, objectType, objectId);
                break;
            case 'angle':
            case 'rotation':
                this.handleAngleEdit(value, objectType, objectId);
                break;
            case 'script':
                this.handleScriptEdit(objectType, objectId);
                break;
            // Scrollbar-specific properties
            case 'startvalue':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarStartValueEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'endvalue':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarEndValueEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'thumbposition':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarThumbPositionEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'scrolltype':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarScrollTypeEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'showvalue':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarShowValueEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'fontfamily':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarFontFamilyEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'fontsize':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarFontSizeEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            case 'textalign':
                if (objectType === 'scrollbar') {
                    this.handleScrollbarTextAlignEdit(value, objectType, objectId);
                } else {
                    this.handleGenericPropertyEdit(property, value, objectType, objectId);
                }
                break;
            // Add more property handlers here as needed
            default:
                this.handleGenericPropertyEdit(property, value, objectType, objectId);
                break;
        }
    },

    // Handle changing the font property
    handleFontEdit: function(currentName, objectType, objectId) {
        console.log(`InspectorTasks.handleFontEdit called with: ${objectType}, ${objectId}`);
        
        // Prevent multiple executions
        if (this._isHandlingFontEdit) {
            console.log('Already handling font edit, ignoring duplicate call');
            return;
        }
        
        this._isHandlingFontEdit = true;
        
        try {
            // Get the correct property name for consistent handling
            const propertyName = this.getCorrectPropertyName('textfont');
            
            // Script for changing an object's font
            const changeFontScript = [
                `put the fontNames into tFonts`,
                `sort lines of tFonts ascending`,
                `ask list tFonts`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(changeFontScript).then(() => {
                // Reselect the object after renaming
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingFontEdit = false; // Reset the flag after completion
                }, 200); // Small delay
            });
        } catch (error) {
            console.error('Error executing change of font script:', error);
            this._isHandlingFontEdit = false; // Reset the flag in case of error
        }
    },

    // Handle editing the name property
    handleNameEdit: function(currentName, objectType, objectId) {
        console.log(`InspectorTasks.handleNameEdit called with: ${currentName}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingNameEdit) {
            return;
        }
        
        this._isHandlingNameEdit = true;
        
        try {
            let renameScript;
            
            // Special handling for scrollbars and cards - use object name instead of ID
            if (objectType === 'scrollbar' || objectType === 'card') {
                const objectName = WebTalkObjects.getObject(objectId).dataset.name;
                
                // First get the new name using the dialog
                renameScript = [
                    `ask "Rename to:" with "${currentName}"`
                ];
                
                // Run the script to get the new name
                this.runScript(renameScript).then(() => {
                    // Get the result from 'it' variable
                    const newName = this.interpreter.it;
                    
                    if (newName && newName !== '') {
                        // Directly set the name using a HyperTalk script
                        let setNameScript;
                        if (objectType === 'scrollbar') {
                            setNameScript = [`set the name of scrollbar "${objectName}" to "${newName}"`];
                        } else if (objectType === 'card') {
                            setNameScript = [`set the name of card "${objectName}" to "${newName}"`];
                        }
                        this.runScript(setNameScript).then(() => {
                            // Refresh the inspector and overview palette
                            setTimeout(() => {
                                this.reselectObjectAndUpdateInspector(objectId);
                                // Refresh overview palette if it's visible
                                if (window.objectOverview && window.objectOverview.isVisible) {
                                    window.objectOverview.updateObjectList();
                                }
                                this._isHandlingNameEdit = false;
                            }, 200);
                        }).catch(() => {
                            this._isHandlingNameEdit = false;
                        });
                    } else {
                        this._isHandlingNameEdit = false;
                    }
                });
                
                return; // Return early since we're handling this case separately
            } else {
                // Standard handling for other objects
                renameScript = [
                    `ask "Rename to:" with "${currentName}"`,
                    `if it is "" then`,
                    `  -- ignore`,
                    `else`,
                    `  set the name of ${objectType} id ${objectId} to it`,
                    `end if`
                ];
                
                // Run the script
                this.runScript(renameScript).then(() => {
                    // Reselect the object after renaming
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                        // Refresh overview palette if it's visible
                        if (window.objectOverview && window.objectOverview.isVisible) {
                            window.objectOverview.updateObjectList();
                        }
                        this._isHandlingNameEdit = false;
                    }, 200); // Small delay to ensure the rename has completed
                }).catch(() => {
                    this._isHandlingNameEdit = false;
                });
            }
        } catch (error) {
            console.error('Error executing rename script:', error);
            this._isHandlingNameEdit = false;
        }
    },

    // Handle toggling the visible property
    handleVisibleToggle: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleVisibleToggle called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingVisibleToggle) {
            return;
        }
        
        this._isHandlingVisibleToggle = true;
        
        try {
            // Convert the current value to a boolean
            const isCurrentlyVisible = currentValue.toLowerCase() === 'true';

            // Script for toggling visibility
            const toggleScript = [
                'answer "Change visibility:" with "true" or "false"',
                'put line 1 of it into tChosen',
                'set the visible of ' + objectType + ' id ' + objectId + ' to tChosen'
            ];

            // Run the script
            this.runScript(toggleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingVisibleToggle = false;
                }, 100);
            }).catch(() => {
                this._isHandlingVisibleToggle = false;
            });
        } catch (error) {
            console.error('Error executing visibility toggle script:', error);
            this._isHandlingVisibleToggle = false;
        }
    },

    // Handle toggling the locktext property for fields
    handleLocktextToggle: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleLocktextToggle called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingLocktextToggle) {
            return;
        }
        
        this._isHandlingLocktextToggle = true;
        
        try {
            // Convert the current value to a boolean
            const isCurrentlyLocked = currentValue.toLowerCase() === 'true';

            // Script for toggling locktext
            const toggleScript = [
                'answer "Change locktext:" with "true" or "false"',
                'put line 1 of it into tChosen',
                'put tChosen',
                'set the locktext of ' + objectType + ' id ' + objectId + ' to tChosen'
            ];

            // Run the script
            this.runScript(toggleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingLocktextToggle = false;
                }, 100);
            }).catch(() => {
                this._isHandlingLocktextToggle = false;
            });
        } catch (error) {
            console.error('Error executing locktext toggle script:', error);
            this._isHandlingLocktextToggle = false;
        }
    },

    // Handle toggling the showFocusBorder property for fields
    handleShowFocusBorderToggle: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleShowFocusBorderToggle called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingShowFocusBorderToggle) {
            return;
        }
        
        this._isHandlingShowFocusBorderToggle = true;
        
        try {
            // Convert the current value to a boolean
            const isCurrentlyShowing = currentValue.toLowerCase() === 'true';

            // Script for toggling showFocusBorder
            const toggleScript = [
                'answer "Show focus border (currently: ' + currentValue + ')?" with "true" or "false"',
                'set the showFocusBorder of ' + objectType + ' id ' + objectId + ' to word 1 of it'
            ];

            // Run the script
            this.runScript(toggleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingShowFocusBorderToggle = false;
                }, 100);
            }).catch(() => {
                this._isHandlingShowFocusBorderToggle = false;
            });
        } catch (error) {
            console.error('Error executing showFocusBorder toggle script:', error);
            this._isHandlingShowFocusBorderToggle = false;
        }
    },

    // Handle toggling the showBorder property
    handleShowBorderToggle: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleShowBorderToggle called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingShowBorderToggle) {
            return;
        }
        
        this._isHandlingShowBorderToggle = true;
        
        try {
            // Convert the current value to a boolean
            const isCurrentlyShowing = currentValue.toLowerCase() === 'true';

            // Script for toggling border visibility
            const toggleScript = [
                'answer "Show border (currently: ' + currentValue + '):" with "true" or "false"',
                'put line 1 of the result into tChosen',
                'set the showBorder of ' + objectType + ' id ' + objectId + ' to tChosen'
            ];

            // Run the script
            this.runScript(toggleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingShowBorderToggle = false;
                }, 100);
            }).catch(() => {
                this._isHandlingShowBorderToggle = false;
            });
        } catch (error) {
            console.error('Error executing show border toggle script:', error);
            this._isHandlingShowBorderToggle = false;
        }
    },

    // Handle editing text properties (font, size, style)
    handleTextPropertyEdit: function(property, currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleTextPropertyEdit called with: ${property}, ${currentValue}, ${objectType}, ${objectId}`);

        try {
            let promptText = '';
            let propertyName = this.getCorrectPropertyName(property);

            switch(property.toLowerCase()) {
                case 'textfont':
                case 'fontfamily':
                    promptText = 'Enter new font:';
                    break;
            }

            // Script for editing the text property
            const editScript = [
                `ask "${promptText}"`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(editScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                }, 100);
            });
        } catch (error) {
            console.error(`Error executing ${property} edit script:`, error);
        }
    },

    // Handle editing color properties
    handleColorPropertyEdit: function(property, currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleColorPropertyEdit called with: ${property}, ${currentValue}, ${objectType}, ${objectId}`);
        
        // Prevent multiple executions
        if (this._isHandlingColorPropertyEdit) {
            console.log('Already handling color property edit, ignoring duplicate call');
            return;
        }
        
        // Set flag to prevent multiple executions
        this._isHandlingColorPropertyEdit = true;
        
        try {
            let promptText = '';
            let propertyName = this.getCorrectPropertyName(property);

            switch(property.toLowerCase()) {
                case 'textcolor':
                    promptText = 'Select new text color:';
                    break;
                case 'backgroundcolor':
                    promptText = 'Select new background color:';
                    break;
                case 'bordercolor':
                    promptText = 'Select new border color:';
                    break;
                case 'foregroundcolor':
                    promptText = 'Select new foreground color:';
                    break;
                case 'progresscolor':
                    promptText = 'Select new progress color:';
                    break;
            }

            // Get current RGB values
            let rgbValues = "0,0,0"; // Default black
            
            // Try to parse current color value if it exists
            if (currentValue && currentValue !== "undefined") {
                // Handle different color formats
                if (currentValue.startsWith("rgb")) {
                    // Extract RGB values from rgb(r,g,b) format
                    const rgbMatch = currentValue.match(/rgb\((\d+),\s*(\d+),\s*(\d+)\)/);
                    if (rgbMatch) {
                        rgbValues = `${rgbMatch[1]},${rgbMatch[2]},${rgbMatch[3]}`;
                    }
                } else if (currentValue.includes(",")) {
                    // Already in R,G,B format
                    rgbValues = currentValue;
                } else if (currentValue.startsWith("#")) {
                    // Convert hex to RGB
                    const hex = currentValue.substring(1);
                    const r = parseInt(hex.substring(0, 2), 16);
                    const g = parseInt(hex.substring(2, 4), 16);
                    const b = parseInt(hex.substring(4, 6), 16);
                    rgbValues = `${r},${g},${b}`;
                }
            }

            // Script for editing the color property using the color dialog
            let editScript;
        
            // Special handling for card background color
            console.log(`Handling color property for object type: ${objectType}, property: ${property}`);
            if (objectType === 'card' && (property.toLowerCase() === 'backgroundcolor' || property.toLowerCase() === 'backgroundcolour')) {
                console.log('Using special card background color handling');
                // For card background color, use a simpler approach
                // First get the color using the dialog
                editScript = [
                    `put "${promptText}"`,
                    `answer color with "${rgbValues}" show transparent`
                ];
                
                // Run the script to get the color
                this.runScript(editScript).then(() => {
                    // Get the result from 'it' variable
                    const colorResult = this.interpreter.it;
                    console.log('Color dialog result:', colorResult);
                    
                    if (colorResult && colorResult !== '') {
                        // Process the color value
                        let finalColor = colorResult;
                        
                        // Strip rgb() format if present
                        if (finalColor.startsWith('rgb(')) {
                            finalColor = finalColor.substring(4, finalColor.length - 1);
                        }
                        
                        console.log('Setting card background color to:', finalColor);
                        
                        // Set the card background color directly
                        // Determine the correct card reference from objectId
                        let cardReference = 'this card';
                        if (objectId) {
                            const element = document.getElementById(objectId);
                            if (element && (element.id === 'card' || element.id.startsWith('card-'))) {
                                // For multi-card stacks, use the specific card reference
                                const cardId = element.id === 'card' ? 1 : parseInt(element.id.replace('card-', ''));
                                cardReference = cardId === 1 ? 'card 1' : `card ${cardId}`;
                            }
                        }
                        
                        // Always quote the color value to ensure proper handling
                        // This handles both direct RGB values and variables
                        let quotedColor = finalColor;
                        if (!finalColor.startsWith('"') && !finalColor.startsWith("'")) {
                            quotedColor = `"${finalColor}"`;
                        }
                        
                        const setColorScript = [`set the backgroundcolor of ${cardReference} to ${quotedColor}`];
                        this.runScript(setColorScript).then(() => {
                            // Refresh the inspector
                            setTimeout(() => {
                                this.reselectObjectAndUpdateInspector(objectId);
                            }, 100);
                        });
                    }
                    
                    // Reset the flag
                    this._isHandlingColorPropertyEdit = false;
                });
                
                // Return early since we're handling this case separately
                return;
            } else if (objectType === 'scrollbar') {
                // Special handling for scrollbars - use object name instead of ID
                const objectName = WebTalkObjects.getObject(objectId).dataset.name;
                
                // First get the color using the dialog
                editScript = [
                    `put "${promptText}"`,
                    property.toLowerCase() === 'backgroundcolor' || property.toLowerCase() === 'fillcolor' 
                        ? `answer color with "${rgbValues}" show transparent`
                        : `answer color with "${rgbValues}"`
                ];
                
                // Run the script to get the color
                this.runScript(editScript).then(() => {
                    // Get the result from 'it' variable
                    const colorResult = this.interpreter.it;
                    console.log('Color dialog result for scrollbar:', colorResult);
                    
                    if (colorResult && colorResult !== '') {
                        // Process the color value
                        let finalColor = colorResult;
                        
                        // Strip rgb() format if present
                        if (finalColor.startsWith('rgb(')) {
                            finalColor = finalColor.substring(4, finalColor.length - 1);
                        }
                        
                        console.log(`Setting scrollbar ${propertyName} to:`, finalColor);
                        
                        // Set the scrollbar color directly using HyperTalk script
                        const setColorScript = [`set the ${propertyName} of scrollbar "${objectName}" to "${finalColor}"`];
                        this.runScript(setColorScript).then(() => {
                            // Refresh the inspector
                            setTimeout(() => {
                                this.reselectObjectAndUpdateInspector(objectId);
                            }, 100);
                        });
                    }
                    
                    // Reset the flag
                    this._isHandlingColorPropertyEdit = false;
                });
                
                // Return early since we're handling this case separately
                return;
            } else {
                // Standard handling for other objects
                editScript = [
                    `put "${promptText}"`,
                    property.toLowerCase() === 'backgroundcolor' || property.toLowerCase() === 'fillcolor' 
                        ? `answer color with "${rgbValues}" show transparent`
                        : `answer color with "${rgbValues}"`,
                    `if it is "" then`,
                    `  -- ignore`,
                    `else`,
                    `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                    `end if`
                ];
            }

            // Run the script
            this.runScript(editScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    // Reset the flag after operation is complete
                    this._isHandlingColorPropertyEdit = false;
                }, 100);
            });
        } catch (error) {
            console.error(`Error executing ${property} edit script:`, error);
            // Reset the flag in case of error
            this._isHandlingColorPropertyEdit = false;
        }
    },

    // Handle editing text style property with specific options
    handleTextStyleEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleTextStyleEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingTextStyleEdit) {
            return;
        }
        
        this._isHandlingTextStyleEdit = true;
        
        try {
            // Use the correct property name directly to avoid lowercase issues
            const propertyName = 'textStyle';

            // Script for editing text style with buttons
            const styleScript = [
                `answer "Change text style:" with "Plain" or "Bold" or "Italic" or "Underline"`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(styleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingTextStyleEdit = false;
                }, 100);
            }).catch(() => {
                this._isHandlingTextStyleEdit = false;
            });
        } catch (error) {
            console.error('Error executing text style edit script:', error);
            this._isHandlingTextStyleEdit = false;
        }
    },

    // Handle editing the script property
    handleScriptEdit: function(objectType, objectId) {
        console.log(`InspectorTasks.handleScriptEdit called with: ${objectType}, ${objectId}`);

        try {
            // Use the WebTalkObjects.openScriptEditor if available
            if (typeof WebTalkObjects !== 'undefined' && WebTalkObjects.openScriptEditor) {
                const object = document.getElementById(objectId);
                if (object) {
                    WebTalkObjects.openScriptEditor(object);
                }
            } else {
                console.error('Script editor not available');
            }
        } catch (error) {
            console.error('Error opening script editor:', error);
        }
    },

    // Handle editing generic properties
    handleGenericPropertyEdit: function(property, currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleGenericPropertyEdit called with: ${property}, ${currentValue}, ${objectType}, ${objectId}`);

        try {
            // Get the correct property name
            const propertyName = this.getCorrectPropertyName(property);

            // Script for editing a generic property
            const editScript = [
                `ask "Enter new value for ${property}:"`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(editScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                }, 100);
            });
        } catch (error) {
            console.error(`Error executing ${property} edit script:`, error);
        }
    },

    // Handle editing text size property with numerical dialog
    handleTextSize: function(currentValue, objectType, objectId, property) {
        console.log(`InspectorTasks.handleTextSize called with: ${currentValue}, ${objectType}, ${objectId}, ${property}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingTextSize) {
            return;
        }
        
        this._isHandlingTextSize = true;
        
        try {
            // Get the correct property name based on which property was clicked
            const propertyName = this.getCorrectPropertyName(property || 'textsize');
            
            // Script for editing text size with numerical dialog
            const sizeScript = [
                `answer numerical with "${currentValue}"`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the textSize of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(sizeScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingTextSize = false;
                }, 100);
            }).catch(() => {
                this._isHandlingTextSize = false;
            });
        } catch (error) {
            console.error('Error executing text size edit script:', error);
            this._isHandlingTextSize = false;
        }
    },

    // Handle editing line size property with numerical dialog
    handleLineSizeEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleLineSizeEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingLineSize) {
            return;
        }
        
        this._isHandlingLineSize = true;
        
        try {
            // The property is stored as 'linesize' in objects.js
            const propertyName = 'linesize';
            
            // Extract numeric value from current value (removing 'px' if present)
            let numericValue = currentValue;
            if (typeof currentValue === 'string' && currentValue.endsWith('px')) {
                numericValue = currentValue.replace('px', '');
            }
            
            // Default to 1 if no valid value
            if (!numericValue || isNaN(parseInt(numericValue))) {
                numericValue = "1";
            }
            
            // Script for editing line size with numerical dialog
            const sizeScript = [
                `answer numerical with "${numericValue}"`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                `end if`
            ];

            // Run the script
            this.runScript(sizeScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingLineSize = false;
                }, 100);
            }).catch(() => {
                this._isHandlingLineSize = false;
            });
        } catch (error) {
            console.error('Error executing line size edit script:', error);
            this._isHandlingLineSize = false;
        }
    },

    // Handle editing border width property with numerical dialog
    handleBorderWidthEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleBorderWidthEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingBorderWidth) {
            return;
        }
        
        this._isHandlingBorderWidth = true;
        
        try {
            // Get the correct property name using our helper function
            const propertyName = this.getCorrectPropertyName('borderwidth');
            
            // Convert the current value to a numeric value
            let numericValue = 1;
            if (currentValue) {
                // Strip any non-numeric characters (e.g., 'px')
                const match = currentValue.toString().match(/(\d+)/);
                if (match) {
                    numericValue = parseInt(match[1], 10);
                }
            }
            
            // Create a script to edit the border width
            let sizeScript;
            
            if (objectType === 'scrollbar') {
                // For scrollbars, use the object name instead of ID
                const objectName = WebTalkObjects.getObject(objectId).dataset.name;
                
                // Create a script to ask for the new border width value
                sizeScript = [
                    `put "Enter border width:"`,
                    `answer numerical with "${numericValue}"`
                ];
                
                // Run the script to get the value
                this.runScript(sizeScript).then(() => {
                    // Get the result from 'it' variable
                    const widthResult = this.interpreter.it;
                    
                    if (widthResult && widthResult !== '') {
                        // Directly set the property using a HyperTalk script
                        const setBorderWidthScript = [`set the ${propertyName} of scrollbar "${objectName}" to ${widthResult}`];
                        this.runScript(setBorderWidthScript).then(() => {
                            // Refresh the inspector
                            setTimeout(() => {
                                this.reselectObjectAndUpdateInspector(objectId);
                                this._isHandlingBorderWidth = false;
                            }, 100);
                        });
                    } else {
                        this._isHandlingBorderWidth = false;
                    }
                });
                
                // Return early since we're handling this case separately
                return;
            } else {
                // For other objects, use the original approach with ID
                sizeScript = [
                    `put "Enter border width:"`,
                    `answer numerical with "${numericValue}"`,
                    `if it is "" then`,
                    `  -- ignore`,
                    `else`,
                    `  set the ${propertyName} of ${objectType} id ${objectId} to it`,
                    `end if`
                ];
            }
            
            // Run the script
            this.runScript(sizeScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingBorderWidth = false;
                }, 100);
            }).catch(() => {
                this._isHandlingBorderWidth = false;
            });
        } catch (error) {
            console.error('Error executing border width edit script:', error);
            this._isHandlingBorderWidth = false;
        }
    },

    // Handle editing angle property with numeric dialog
    handleAngleEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleAngleEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingAngleEdit) {
            return;
        }
        
        this._isHandlingAngleEdit = true;
        
        try {
            // Convert the current value to a numeric value
            let numericValue = 0;
            if (currentValue) {
                // Strip any non-numeric characters
                const match = currentValue.toString().match(/(-?\d+(?:\.\d+)?)/); // Allow negative and decimal values
                if (match) {
                    numericValue = parseFloat(match[1]);
                }
            }
            
            // Create a script to edit the angle
            const angleScript = [
                `answer slider with "${numericValue},360"`,
                `if it is "" then`,
                `  -- do nothing`,
                `else`,
                `  set the angle of ${objectType} id ${objectId} to it`,
                `end if`
            ];
            
            // Run the script
            this.runScript(angleScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingAngleEdit = false;
                }, 100);
            }).catch(() => {
                this._isHandlingAngleEdit = false;
            });
        } catch (error) {
            console.error('Error executing angle edit script:', error);
            this._isHandlingAngleEdit = false;
        }
    },

    // Handle editing filename property with ask dialog
    handleFilenameEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleFilenameEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingFilenameEdit) {
            return;
        }
        
        this._isHandlingFilenameEdit = true;
        
        try {
            // Create a script to edit the filename
            const filenameScript = [
                `ask "Please choose a new filename: " with "${currentValue || ''}"`,
                `if it is "" then`,
                `  -- do nothing`,
                `else`,
                `  set the filename of ${objectType} id ${objectId} to it`,
                `end if`
            ];
            
            // Run the script
            this.runScript(filenameScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingFilenameEdit = false;
                }, 100);
            }).catch(() => {
                this._isHandlingFilenameEdit = false;
            });
        } catch (error) {
            console.error('Error executing filename edit script:', error);
            this._isHandlingFilenameEdit = false;
        }
    },

    // Handle editing layer property with numeric dialog
    handleLayerEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleLayerEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingLayerEdit) {
            return;
        }
        
        this._isHandlingLayerEdit = true;
        
        try {
            // Create a script to edit the layer using answer numerical
            const layerScript = [
                `answer numerical with "${currentValue || 1}"`,
                `if it is "" then`,
                `-- cancelled`,
                `else`,
                `  put it into tLayerOrder`,
                `  set the layer of ${objectType} "${WebTalkObjects.getObject(objectId).dataset.name}" to tLayerOrder`,
                `end if`
            ];
            
            // Run the script
            this.runScript(layerScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingLayerEdit = false;
                }, 100);
            }).catch(() => {
                this._isHandlingLayerEdit = false;
            });
        } catch (error) {
            console.error('Error executing layer edit script:', error);
            this._isHandlingLayerEdit = false;
        }
    },

    // Handle scrollbar-specific properties
    handleScrollbarPropertyEdit: function(property, currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarPropertyEdit called with: ${property}, ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Get the correct property name
        const propertyName = this.getCorrectPropertyName(property);
        
        // Default numeric value
        let numericValue = 0;
        if (currentValue !== undefined && currentValue !== null) {
            // Convert to number if possible
            numericValue = parseInt(currentValue) || 0;
        }
        
        // Determine prompt text based on property
        let promptText = '';
        let minValue = 0;
        let maxValue = 100;
        
        switch(property.toLowerCase()) {
            case 'thumbposition':
                promptText = 'Enter thumb position:';
                // Get startValue and endValue to set min/max
                const startValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'startValue')) || 0;
                const endValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'endValue')) || 100;
                minValue = startValue;
                maxValue = endValue;
                break;
            case 'startvalue':
                promptText = 'Enter start value:';
                // Start value should be less than end value
                maxValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'endValue')) - 1 || 99;
                break;
            case 'endvalue':
                promptText = 'Enter end value:';
                // End value should be greater than start value
                minValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'startValue')) + 1 || 1;
                break;
            case 'scrolltype':
                // For scrollType, show a list of options instead of a numeric input
                const scrollTypeScript = [
                    `put "Select scrollbar type:"`,
                    `answer "bar" or "round" or "horizontal" or "vertical"`
                ];
                
                // Run the script to get the value
                this.runScript(scrollTypeScript).then(() => {
                    // Get the result from 'it' variable
                    const typeResult = this.interpreter.it;
                    
                    if (typeResult && typeResult !== '') {
                        // Directly set the property using a HyperTalk script
                        const setTypeScript = [`set the scrollType of scrollbar "${objectName}" to "${typeResult}"`];
                        this.runScript(setTypeScript).then(() => {
                            // Refresh the inspector
                            setTimeout(() => {
                                this.reselectObjectAndUpdateInspector(objectId);
                            }, 100);
                        });
                    }
                });
                
                // Return early since we're handling scrollType differently
                return;
        }
        
        // Create script to get the numeric value
        const script = [
            `put "${promptText}"`,
            `answer numerical with "${numericValue}" min ${minValue} max ${maxValue}`
        ];
        
        // Run the script to get the value
        this.runScript(script).then(() => {
            // Get the result from 'it' variable
            const valueResult = this.interpreter.it;
            
            if (valueResult && valueResult !== '') {
                // Directly set the property using a HyperTalk script
                const setPropertyScript = [`set the ${propertyName} of scrollbar "${objectName}" to ${valueResult}`];
                this.runScript(setPropertyScript).then(() => {
                    // Refresh the inspector
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                    }, 100);
                });
            }
        });
    },
    
    // Handle startValue property for scrollbars
    handleScrollbarStartValueEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarStartValueEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Convert current value to number
        let numericValue = 0;
        if (currentValue !== undefined && currentValue !== null) {
            numericValue = parseInt(currentValue) || 0;
        }
        
        // Create script to get the new value
        const script = [
            `answer numerical with "${numericValue}"`
        ];
        
        // Run the script to get the value
        this.runScript(script).then(() => {
            // Get the result from 'it' variable
            const valueResult = this.interpreter.it;
            
            if (valueResult && valueResult !== '') {
                // Directly set the property using a HyperTalk script
                const setPropertyScript = [`set the startValue of scrollbar "${objectName}" to ${valueResult}`];
                this.runScript(setPropertyScript).then(() => {
                    // Refresh the inspector
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                    }, 100);
                });
            }
        });
    },
    
    // Handle endValue property for scrollbars
    handleScrollbarEndValueEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarEndValueEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Convert current value to number
        let numericValue = 0;
        if (currentValue !== undefined && currentValue !== null) {
            numericValue = parseInt(currentValue) || 0;
        }
        
        // Create script to get the new value
        const script = [
            `answer numerical with "${numericValue}"`
        ];
        
        // Run the script to get the value
        this.runScript(script).then(() => {
            // Get the result from 'it' variable
            const valueResult = this.interpreter.it;
            
            if (valueResult && valueResult !== '') {
                // Directly set the property using a HyperTalk script
                const setPropertyScript = [`set the endValue of scrollbar "${objectName}" to ${valueResult}`];
                this.runScript(setPropertyScript).then(() => {
                    // Refresh the inspector
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                    }, 100);
                });
            }
        });
    },
    
    // Handle thumbPosition property for scrollbars
    handleScrollbarThumbPositionEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarThumbPositionEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Convert current value to number
        let numericValue = 0;
        if (currentValue !== undefined && currentValue !== null) {
            numericValue = parseInt(currentValue) || 0;
        }
        
        // Get startValue and endValue to set min/max
        const startValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'startValue')) || 0;
        const endValue = parseInt(WebTalkObjects.getObjectProperty(objectId, 'endValue')) || 100;
        
        // Create script to get the new value
        const script = [
            `answer numerical with "${numericValue}"`
        ];
        
        // Run the script to get the value
        this.runScript(script).then(() => {
            // Get the result from 'it' variable
            const valueResult = this.interpreter.it;
            
            if (valueResult && valueResult !== '') {
                // Directly set the property using a HyperTalk script
                const setPropertyScript = [`set the thumbPosition of scrollbar "${objectName}" to ${valueResult}`];
                this.runScript(setPropertyScript).then(() => {
                    // Refresh the inspector
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                    }, 100);
                });
            }
        });
    },
    
    // Handle scrollType property for scrollbars
    handleScrollbarScrollTypeEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarScrollTypeEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Create script to get the new value
        const script = [
            `answer "Choose progress bar type:" with "bar" or "round"`,
            `set the scrollType of scrollbar "${objectName}" to it`
        ];
        
        // Run the script to get the value
        this.runScript(script).then(() => {
            // Get the result from 'it' variable
            const typeResult = this.interpreter.it;
            
            if (typeResult && typeResult !== '') {
                // Directly set the property using a HyperTalk script
                const setPropertyScript = [`set the scrollType of scrollbar "${objectName}" to "${typeResult}"`];
                this.runScript(setPropertyScript).then(() => {
                    // Refresh the inspector
                    setTimeout(() => {
                        this.reselectObjectAndUpdateInspector(objectId);
                    }, 100);
                });
            }
        });
    },
    
    // Helper method to reselect an object and update the inspector
    reselectObjectAndUpdateInspector: function(objectId) {
        // Get the object by ID
        const object = document.getElementById(objectId);
        if (object) {
            // Trigger a click on the object to reselect it
            if (typeof WebTalkObjects !== 'undefined' && WebTalkObjects.selectObject) {
                WebTalkObjects.selectObject(object);
            } else {
                // Fallback: simulate a click
                const event = new MouseEvent('click', {
                    bubbles: true,
                    cancelable: true,
                    view: window
                });
                object.dispatchEvent(event);
            }

            // Refresh the inspector if it exists
            if (typeof WebTalkInspector !== 'undefined') {
                const objectName = object.dataset.name;
                if (objectName) {
                    // Update the inspector content
                    if (WebTalkInspector.updateInspectorContent) {
                        WebTalkInspector.updateInspectorContent(objectName);
                    }
                    
                    // Update the inspector title
                    if (WebTalkInspector.inspectorWindow) {
                        const title = WebTalkInspector.inspectorWindow.querySelector('.inspector-title');
                        if (title) {
                            title.textContent = `Object Inspector: ${objectName}`;
                        }
                    }
                }
            }
        } else {
            // If object not found, just update the inspector
            if (typeof WebTalkInspector !== 'undefined' && WebTalkInspector.updateInspectorContent) {
                WebTalkInspector.updateInspectorContent();
            }
        }
    },

    // Handle showValue property for scrollbars
    handleScrollbarShowValueEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarShowValueEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name from the ID
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Script for toggling showValue
        const toggleShowValueScript = [
            `Answer "Show text value?" with "Yes" or "No"`,
            `if it is "Yes" then`,
            `  set the showvalue of scrollbar "${objectName}" to "true"`,
            `else`,
            `  set the showvalue of scrollbar "${objectName}" to "false"`,
            `end if`
        ];
        
        // Run the script
        this.runScript(toggleShowValueScript).then(() => {
            // Reselect the object after changing the property
            setTimeout(() => {
                this.reselectObjectAndUpdateInspector(objectId);
            }, 200); // Small delay
        });
    },
    
    // Handle fontFamily property for scrollbars
    handleScrollbarFontFamilyEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarFontFamilyEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name from the ID
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Prevent multiple executions
        if (this._isHandlingFontEdit) {
            console.log('Already handling font edit, ignoring duplicate call');
            return;
        }
        
        this._isHandlingFontEdit = true;
        
        try {
            // Script for changing the scrollbar's font family
            const changeFontScript = [
                `put the fontNames into tFonts`,
                `sort lines of tFonts ascending`,
                `ask list tFonts`,
                `if it is "" then`,
                `  -- ignore`,
                `else`,
                `  set the fontfamily of scrollbar "${objectName}" to it`,
                `end if`
            ];

            // Run the script
            this.runScript(changeFontScript).then(() => {
                // Reselect the object after changing the property
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingFontEdit = false; // Reset the flag after completion
                }, 200); // Small delay
            });
        } catch (error) {
            console.error('Error executing change of font script:', error);
            this._isHandlingFontEdit = false; // Reset the flag in case of error
        }
    },
    
    // Handle fontSize property for scrollbars
    handleScrollbarFontSizeEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarFontSizeEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name from the ID
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Script for changing the scrollbar's font size
        const changeFontSizeScript = [
            `ask "Enter font size:" with "${currentValue}"`,
            `if it is not "" then`,
            `  if isNumber(it) then`,
            `    set the fontsize of scrollbar "${objectName}" to it`,
            `  else`,
            `    answer "Font size must be a number."`,
            `  end if`,
            `end if`
        ];
        
        // Run the script
        this.runScript(changeFontSizeScript).then(() => {
            // Reselect the object after changing the property
            setTimeout(() => {
                this.reselectObjectAndUpdateInspector(objectId);
            }, 200); // Small delay
        });
    },
    
    // Handle textAlign property for scrollbars
    handleScrollbarTextAlignEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleScrollbarTextAlignEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Get the object name from the ID
        const objectName = WebTalkObjects.getObject(objectId).dataset.name;
        if (!objectName) {
            console.error('Could not find object name for scrollbar');
            return;
        }
        
        // Script for changing the scrollbar's text alignment
        const changeTextAlignScript = [
            `answer "Choose text alignment:" with "left" or "center" or "right"`,
            `if it is not "" then`,
            `  set the textalign of scrollbar "${objectName}" to it`,
            `end if`
        ];
        
        // Run the script
        this.runScript(changeTextAlignScript).then(() => {
            // Reselect the object after changing the property
            setTimeout(() => {
                this.reselectObjectAndUpdateInspector(objectId);
            }, 200); // Small delay
        });
    },

    // Handle editing blendLevel property with slider dialog
    _isHandlingBlendLevelEdit: false,
    
    handleBlendLevelEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleBlendLevelEdit called with: ${currentValue}, ${objectType}, ${objectId}`);
        
        // Use a flag to prevent multiple executions
        if (this._isHandlingBlendLevelEdit) {
            return;
        }
        
        this._isHandlingBlendLevelEdit = true;
        
        try {
            // Get the object name from the ID
            const object = WebTalkObjects.getObject(objectId);
            if (!object || !object.dataset || !object.dataset.name) {
                console.error('Could not find object name');
                this._isHandlingBlendLevelEdit = false;
                return;
            }
            const objectName = object.dataset.name;
            
            // Parse the current value as a number (default to 0 if not a valid number)
            const currentBlendLevel = parseInt(currentValue) || 0;
            
            // Script for changing the blendLevel with the slider dialog
            const changeBlendLevelScript = [
                `answer slider with "${currentBlendLevel},100"`,
                `if it is not "" then`,
                `  set the blendLevel of ${objectType} "${objectName}" to it`,
                `end if`
            ];
            
            // Run the script
            this.runScript(changeBlendLevelScript).then(() => {
                // Reselect the object after changing the property
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingBlendLevelEdit = false;
                }, 200); // Small delay
            }).catch((error) => {
                console.error(`Error in blendLevel edit script: ${error}`);
                this._isHandlingBlendLevelEdit = false;
            });
        } catch (error) {
            console.error('Error executing blendLevel edit:', error);
            this._isHandlingBlendLevelEdit = false;
        }
    },
    
    // Handle restoring field formatting using the arrayData property and do command
    handleRestoreFieldFormatting: function(objectType, objectId, fieldName) {
        console.log(`InspectorTasks.handleRestoreFieldFormatting called for ${objectType} ${fieldName || objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingRestoreFieldFormatting) {
            return;
        }
        
        this._isHandlingRestoreFieldFormatting = true;
        
        try {
            // Use the field name if provided, otherwise get it from the object ID
            const targetField = fieldName || (objectType === 'field' ? 
                WebTalkObjects.getObjectNameById(objectId) : null);
            
            if (!targetField) {
                console.error('No valid field name provided for formatting restoration');
                this._isHandlingRestoreFieldFormatting = false;
                return;
            }

            // Script for applying arrayData for formattingCommands
            const restoreScript = [
                `put the arrayData("formattingCommands") of field "${targetField}" into tCommands`,
                'put the number of lines in tCommands into tRepeat',
                'put 0 into tcount',
                'repeat tRepeat',
                '  add 1 to tcount',
                '  put line tcount of tCommands into tVarToDo',
                '  do tVarToDo',
                'end repeat'
            ];

            // Run the script
            this.runScript(restoreScript).then(() => {
                console.log(`Formatting restoration completed for field "${targetField}"`);
                this._isHandlingRestoreFieldFormatting = false;
            }).catch((error) => {
                console.error(`Error in formatting restoration script: ${error}`);
                this._isHandlingRestoreFieldFormatting = false;
            });
        } catch (error) {
            console.error('Error executing field formatting restoration:', error);
            this._isHandlingRestoreFieldFormatting = false;
        }
    },

    // Handle editing text alignment property with custom dialog
    handleTextAlignEdit: function(currentValue, objectType, objectId) {
        console.log(`InspectorTasks.handleTextAlignEdit called with: ${currentValue}, ${objectType}, ${objectId}`);

        // Use a flag to prevent multiple executions
        if (this._isHandlingTextAlign) {
            return;
        }
        
        this._isHandlingTextAlign = true;
        
        try {
            // Script for setting text alignment
            const alignScript = [
                'answer "Set textalign to:" with "left" or "center" or "right"',
                'put word 1 of it into tChosen',
                'set the textalign of ' + objectType + ' id ' + objectId + ' to tChosen'
            ];

            // Run the script
            this.runScript(alignScript).then(() => {
                // Refresh the inspector
                setTimeout(() => {
                    this.reselectObjectAndUpdateInspector(objectId);
                    this._isHandlingTextAlign = false;
                }, 100);
            }).catch(() => {
                this._isHandlingTextAlign = false;
            });
        } catch (error) {
            console.error('Error executing text align edit script:', error);
            this._isHandlingTextAlign = false;
        }
    }
};

// Initialize the module
InspectorTasks.initialize();

// Make available globally
if (typeof window !== 'undefined') {
    window.InspectorTasks = InspectorTasks;
    console.log('InspectorTasks module loaded and available globally');
}
