// Field fix script - the purpose of this is to stop browsers inserting extra <p>, <br> and <div> tags. We should ne using \n for newlines.
// Flag to temporarily disable field fixing when setting content programmatically
window.disableFieldFix = false;

// Flag to track if we're currently in the middle of a field edit operation
window.fieldEditInProgress = false;

/**
 * Calculate which line was clicked in a field, taking into account vertical scrolling
 * and varying lineheights for individual lines
 * @param {HTMLElement} field - The field element
 * @returns {number} - The line number that was clicked (1-based)
 */
window.getClickLine = function(field) {
    // Get the field content element
    const fieldContent = field.querySelector('.field-content');
    if (!fieldContent) return 1; // Default to first line if no content element
    
    // Get the mouse position relative to the field
    const fieldRect = field.getBoundingClientRect();
    const mouseY = window.event.clientY - fieldRect.top;
    
    // CRITICAL: Always get the scroll position directly from the DOM
    // Force a layout reflow to ensure we have the most current value
    field.getBoundingClientRect();
    const vScroll = fieldContent.scrollTop; // Use fieldContent.scrollTop instead of field.scrollTop
    
    // Also check the stored property for debugging
    let storedVScroll = 0;
    if (field.dataset && field.dataset.name && window.WebTalkObjects && window.WebTalkObjects.customProperties && 
        window.WebTalkObjects.customProperties.has(field.dataset.name)) {
        storedVScroll = window.WebTalkObjects.customProperties.get(field.dataset.name).get('vScroll') || 0;
    }
    
    // Get the computed style of the field content to determine default line height and padding
    const computedStyle = window.getComputedStyle(fieldContent);
    const fontSize = parseInt(computedStyle.fontSize) || 16;
    const defaultLineHeight = parseInt(computedStyle.lineHeight) || Math.ceil(fontSize * 1.2);
    
    // Get padding values from the field (not fieldContent, as padding is often on the outer field element)
    const fieldStyle = window.getComputedStyle(field);
    const paddingTop = parseInt(fieldStyle.paddingTop) || 0;
    const paddingBottom = parseInt(fieldStyle.paddingBottom) || 0;
    
    // Calculate the absolute Y position (including scroll)
    // Subtract paddingTop to account for the field's top padding
    const absoluteY = mouseY + vScroll - paddingTop;
    
    // Get all child nodes to analyze line by line, accounting for custom lineheights
    const childNodes = fieldContent.childNodes;
    let currentY = 0;
    let lineNumber = 1;
    
    // Debug information
    console.log(`Click at Y=${mouseY}, DOM Scroll=${vScroll}, Stored Scroll=${storedVScroll}, Absolute Y=${absoluteY}, Default Line Height=${defaultLineHeight}`);
    
    // If there are no child nodes, use the simple calculation
    if (childNodes.length === 0) {
        return 1;
    }
    
    // Process each node to calculate cumulative height
    for (let i = 0; i < childNodes.length; i++) {
        const node = childNodes[i];
        
        // For text nodes, each newline character represents a line
        if (node.nodeType === Node.TEXT_NODE) {
            const lines = node.textContent.split('\n');
            for (let j = 0; j < lines.length; j++) {
                // Skip the last empty line if it's followed by another node
                if (j === lines.length - 1 && lines[j] === '' && i < childNodes.length - 1) {
                    continue;
                }
                
                // Add the height of this line
                currentY += defaultLineHeight;
                
                // Check if we've passed the clicked position
                if (currentY > absoluteY) {
                    console.log(`Found line ${lineNumber} at Y=${currentY} (text node)`);
                    return lineNumber;
                }
                
                lineNumber++;
            }
        }
        // For span elements with custom lineheight
        else if (node.nodeType === Node.ELEMENT_NODE && node.tagName.toLowerCase() === 'span') {
            // Get the custom lineheight if specified
            let customLineHeight = defaultLineHeight;
            const style = node.getAttribute('style');
            if (style && style.includes('line-height')) {
                const match = style.match(/line-height:\s*([\d.]+)/i);
                if (match && match[1]) {
                    // The line-height can be a multiplier (e.g., 4) or a pixel value
                    const lineHeightValue = parseFloat(match[1]);
                    if (style.includes('line-height: ' + lineHeightValue + 'px')) {
                        customLineHeight = lineHeightValue;
                    } else {
                        // It's a multiplier of the font size
                        customLineHeight = fontSize * lineHeightValue;
                    }
                }
            }
            
            // Count lines within the span (could contain multiple lines)
            const lines = node.textContent.split('\n');
            for (let j = 0; j < lines.length; j++) {
                // Skip the last empty line if it's followed by another node
                if (j === lines.length - 1 && lines[j] === '' && i < childNodes.length - 1) {
                    continue;
                }
                
                // Add the custom height of this line
                currentY += customLineHeight;
                
                // Check if we've passed the clicked position
                if (currentY > absoluteY) {
                    console.log(`Found line ${lineNumber} at Y=${currentY} (span with custom lineheight: ${customLineHeight}px)`);
                    return lineNumber;
                }
                
                lineNumber++;
            }
        }
        // For other element nodes (like br)
        else if (node.nodeType === Node.ELEMENT_NODE) {
            // Add the default height for this element
            currentY += defaultLineHeight;
            
            // Check if we've passed the clicked position
            if (currentY > absoluteY) {
                console.log(`Found line ${lineNumber} at Y=${currentY} (element node)`);
                return lineNumber;
            }
            
            lineNumber++;
        }
    }
    
    // If we've gone through all nodes and haven't found the line,
    // return the last line number
    console.log(`Reached end of content, returning last line ${lineNumber - 1}`);
    return Math.max(1, lineNumber - 1);
};

/**
 * Get the clickLine property of a field
 * @param {HTMLElement} field - The field element
 * @returns {number} - The line number that was clicked (1-based)
 */
window.getClickLineProperty = function(field) {
    // Check if the field has a stored clickLine property
    const fieldName = field.dataset.name;
    if (WebTalkObjects.customProperties.has(fieldName)) {
        const props = WebTalkObjects.customProperties.get(fieldName);
        // Check both camelCase and lowercase versions for compatibility
        if (props.has('clickLine')) {
            return parseInt(props.get('clickLine'));
        } else if (props.has('clickline')) {
            return parseInt(props.get('clickline'));
        }
    }
    return 1; // Default to first line if not set
};

/**
 * Set the clickLine property of a field
 * @param {HTMLElement} field - The field element
 * @param {number} lineNumber - The line number to set (1-based)
 */
window.setClickLineProperty = function(field, lineNumber) {
    const fieldName = field.dataset.name;
    if (!WebTalkObjects.customProperties.has(fieldName)) {
        WebTalkObjects.customProperties.set(fieldName, new Map());
    }
    // Set both camelCase and lowercase versions for compatibility
    const props = WebTalkObjects.customProperties.get(fieldName);
    props.set('clickLine', lineNumber.toString());
    props.set('clickline', lineNumber.toString());
};

// Helper function to safely update field content without interference from browsers
window.safeUpdateFieldContent = function(field, content, callback) {
    // Store the field's current editability state
    let wasEditable = false;
    if (field && field.contentEditable) {
        wasEditable = field.contentEditable === 'true';
    }
    
    // Disable field fixing during this operation
    window.disableFieldFix = true;
    window.fieldEditInProgress = true;
    
    try {
        // Execute the content update
        callback();
    } finally {
        // Re-enable field fixing after a short delay
        setTimeout(() => {
            window.disableFieldFix = false;
            window.fieldEditInProgress = false;
            
            // Restore the field's editability if it was editable before
            if (wasEditable && field) {
                field.contentEditable = 'true';
                
                // Also ensure the field-content div is editable
                // Important!: I will need to modify this when I implement locktext of fields via inspector or via script.
                // (as in 'set the locktext of field "myfield" to true')
                const fieldContent = field.querySelector('.field-content');
                if (fieldContent) {
                    fieldContent.contentEditable = 'true';
                }
            }
        }, 50);
    }
};

// Make clickLine property available globally for HyperTalk scripts
if (typeof window !== 'undefined') {
    window.clickLine = {
        get: window.getClickLineProperty,
        set: window.setClickLineProperty
    };
}

document.addEventListener('DOMContentLoaded', function() {
    // Function to fix duplicate field-content divs
    // this gets rid of multiple div tags. We should only have the initial first one. (the outer HTML)
    function fixDuplicateFieldContents() {
        // If field fixing is disabled, don't process
        if (window.disableFieldFix) {
            return;
        }
        
        // Skip if we're in the middle of a field edit
        if (window.fieldEditInProgress) {
            return;
        }
        
        // Store the current active element to restore focus later if needed
        const activeElement = document.activeElement;
        const wasEditable = activeElement && activeElement.contentEditable === 'true';
        
        // Find all fields
        const fields = document.querySelectorAll('.field');
        
        // Store editable states before processing
        const editableFields = new Map();
        
        fields.forEach(field => {
            // Store the field's editability state
            if (field.contentEditable === 'true') {
                editableFields.set(field, true);
            }
            // Get all field-content divs in this field
            const contentDivs = field.querySelectorAll('.field-content');
            
            // If there's more than one field-content div, we need to fix it
            if (contentDivs.length > 1) {
                console.log(`Found ${contentDivs.length} field-content divs in field ${field.dataset.name || 'unnamed'}, fixing...`);
                
                // Keep the first div and merge content from others
                // (This ensures we don't delete a div's content, we remove just the erronious tags)
                const mainContentDiv = contentDivs[0];
                
                // Start from the second div and merge content
                for (let i = 1; i < contentDivs.length; i++) {
                    // Extract the text content (without HTML) from the extra div
                    const extraText = contentDivs[i].textContent;
                    
                    // Special handling for Enter or return key presses to prevent unwanted blank lines
                    if (window.handlingReturn && mainContentDiv.textContent === '' && extraText === '') {
                        // Don't add any content for empty lines when handling return key
                    } else {
                        // Only add a newline if needed - when appending non-empty content to non-empty content
                        // that doesn't already end with a newline
                        if (!mainContentDiv.textContent.endsWith('\n') && 
                            extraText !== '' && 
                            mainContentDiv.textContent !== '') {
                            mainContentDiv.textContent += '\n';
                        }
                        
                        // Append the text content (without HTML) to the main div
                        mainContentDiv.textContent += extraText;
                    }
                    
                    // Remove any text nodes that might have been added outside the field-content div
                    // (This is generally text nodes that are being inserted by browsers)
                    if (contentDivs[i].nextSibling && contentDivs[i].nextSibling.nodeType === Node.TEXT_NODE) {
                        contentDivs[i].nextSibling.remove();
                    }
                    
                    // Remove the extra div
                    contentDivs[i].remove();
                }
                
                console.log(`Fixed field ${field.dataset.name || 'unnamed'}, now has 1 field-content div with merged content`);
            }
            
            // Also check for any direct text nodes in the field that aren't in the field-content div
            Array.from(field.childNodes).forEach(node => {
                if (node.nodeType === Node.TEXT_NODE) {
                    // Skip if we're in the middle of a programmatic edit
                    if (window.disableFieldFix || window.fieldEditInProgress) {
                        return;
                    }
                    
                    // Get the field-content div (should be the only one by this point)
                    const fieldContent = field.querySelector('.field-content');
                    if (fieldContent) {
                        // Special handling for Enter or return key presses to prevent unwanted blank lines
                        if (window.handlingReturn && fieldContent.textContent === '' && node.textContent === '') {
                            // Don't add empty content when handling return key
                        } else {
                            // Only add a newline if needed - when appending non-empty content to non-empty content
                            // (stuff that doesn't already end with a newline)
                            if (!fieldContent.textContent.endsWith('\n') && 
                                node.textContent !== '' && 
                                fieldContent.textContent !== '') {
                                fieldContent.textContent += '\n';
                            }
                            
                            // Add the text to the field-content, preserving special characters
                            // This makes sure we don't delete unicode characters and others
                            fieldContent.textContent += node.textContent;
                        }
                    }
                    
                    // Remove the text node
                    node.remove();
                }
            });
        });
        
        // Restore editability for fields that were editable before
        editableFields.forEach((isEditable, field) => {
            if (isEditable && field) {
                // Restore the field's editability
                // Important!: Need to modify this part too when I come to implement locktext for fields (as mentioned above)
                field.contentEditable = 'true';
                
                // Also ensure the field-content div is editable
                const fieldContent = field.querySelector('.field-content');
                if (fieldContent) {
                    fieldContent.contentEditable = 'true';
                }
            }
        });
    }
    
    // Add a mutation observer to detect when fields are edited
    // note to self: 'mutation observers' replace depreciated 'mutation events'
    // mutation events are not supported at all in Edge, Firefox 14+, Chrome 26+, Safari 6+ ...ect
    const observer = new MutationObserver(mutations => {
        // If field fixing is disabled or we're in the middle of a programmatic edit, don't process mutations
        if (window.disableFieldFix || window.fieldEditInProgress) {
            return;
        }
        
        let shouldFix = false;
        let isSpecialCharacterUpdate = false;
        
        mutations.forEach(mutation => {
            // Check if any mutation involves a field
            if (mutation.target.classList && 
                (mutation.target.classList.contains('field') || 
                 mutation.target.closest('.field'))) {
                shouldFix = true;
                
                // Check if this might be a special character update
                if (mutation.type === 'characterData' || 
                    (mutation.type === 'childList' && mutation.addedNodes.length === 1 && 
                     mutation.addedNodes[0].nodeType === Node.TEXT_NODE)) {
                    isSpecialCharacterUpdate = true;
                }
            }
            
            // Also check added nodes
            if (mutation.addedNodes.length > 0) {
                Array.from(mutation.addedNodes).forEach(node => {
                    if (node.classList && node.classList.contains('field-content')) {
                        shouldFix = true;
                    }
                });
            }
        });
        
        if (shouldFix) {
            // For special character updates, use a longer delay to ensure content is preserved
            const delay = isSpecialCharacterUpdate ? 50 : 0;
            
            // Fix any duplicate field-content divs
            setTimeout(fixDuplicateFieldContents, delay);
        }
    });
    
    // Start observing the document with the configured parameters
    observer.observe(document.body, { 
        childList: true,
        subtree: true,
        characterData: true,
        attributes: false
    });
    
    // Also run the fix on page load
    setTimeout(fixDuplicateFieldContents, 500);
    
    // And run it again when a stack is loaded
    document.addEventListener('stackLoaded', function() {
        setTimeout(fixDuplicateFieldContents, 500);
    });
    
    // Add keydown event listeners to all fields
    document.addEventListener('keydown', function(event) {
        // If field fixing is disabled, don't process the event
        if (window.disableFieldFix) {
            return;
        }
        
        // Only process for fields
        if (event.target.closest('.field')) {
            // For Enter key, we need special handling to prevent unwanted blank line insertion
            if (event.key === 'Enter') {
                // Mark that we're handling a return key press
                window.handlingReturn = true;
                
                // Run the fix after a delay to allow the browser to update the DOM
                setTimeout(() => {
                    fixDuplicateFieldContents();
                    window.handlingReturn = false;
                }, 10);
            } else {
                // For other keys, just run the fix normally
                setTimeout(fixDuplicateFieldContents, 10);
            }
        }
    }, true);
    
    // Add mousedown event listener to update clickLine property
    document.addEventListener('mousedown', function(event) {
        // Only process for fields in browse mode
        const field = event.target.closest('.field');
        if (field && window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
            // Calculate which line was clicked
            const clickedLine = window.getClickLine(field);
            
            // Store the clickLine property
            window.setClickLineProperty(field, clickedLine);
        }
    }, true);
    
    // Add scroll event listeners to fields to trigger scrollbarDrag messages
    function addScrollListenersToFields() {
        const fields = document.querySelectorAll('.field');
        fields.forEach(field => {
            const fieldName = field.dataset.name;
            if (!fieldName) return;
            
            // Check if this field is multiline
            let isMultiline = false;
            if (window.WebTalkObjects && window.WebTalkObjects.customProperties && 
                window.WebTalkObjects.customProperties.has(fieldName)) {
                const multilineProperty = window.WebTalkObjects.customProperties.get(fieldName).get('multiline');
                isMultiline = multilineProperty === true || multilineProperty === 'true';
            }
            
            // Remove any existing scroll listener
            if (field._scrollHandler) {
                field.removeEventListener('scroll', field._scrollHandler);
                field._scrollHandler = null;
            }
            
            // Only add scroll listener if field is multiline
            if (isMultiline) {
                // Check if we already have a listener to avoid duplicates
                if (!field._hasScrollListener) {
                    // Create a new scroll handler
                    field._scrollHandler = function() {
                        console.log(`Scroll event triggered for field ${this.dataset.name}`);
                        
                        // Get the current scroll position
                        const scrollValue = getScrollTop(this);
                        const fieldName = this.dataset.name;
                        
                        console.log(`Field ${fieldName} scrolled to ${scrollValue}px`);
                        
                        // Update the vScroll property using the setter function
                        if (fieldName) {
                            console.log(`Calling setVScrollProperty for field ${fieldName} with value ${scrollValue}`);
                            window.setVScrollProperty(this, scrollValue);
                        }
                        
                        // Only execute the scrollbarDrag handler in browse mode
                        if (window.webTalkApp && window.webTalkApp.interpreter && 
                            window.webTalkApp.interpreter.mode === 'browse') {
                            // Execute the scrollbarDrag handler in the field's script
                            try {
                                console.log(`Executing scrollbarDrag handler for field ${fieldName} with value ${scrollValue}`);
                                // Use sendMessage instead of executeObjectScript for more reliable message handling
                                window.webTalkApp.interpreter.sendMessage({
                                    target: fieldName,
                                    handler: 'scrollbarDrag',
                                    params: [scrollValue]
                                });
                                console.log(`scrollbarDrag message sent successfully`);
                            } catch (e) {
                                console.error('Error executing scrollbarDrag handler:', e);
                            }
                        }
                    };
                    
                    // Add the new scroll listener
                    field.addEventListener('scroll', field._scrollHandler);
                    field._hasScrollListener = true;
                    console.log(`Added scroll listener to multiline field ${fieldName}`);
                }
            } else {
                // Mark that this field doesn't have a scroll listener
                field._hasScrollListener = false;
            }
        });
    }
    
    // Function to update scroll listener when multiline property changes
    window.updateFieldScrollListener = function(fieldName) {
        const field = document.querySelector(`.field[data-name="${fieldName}"]`);
        if (field) {
            // Force a re-evaluation of scroll listeners for this field
            field._hasScrollListener = false;
            addScrollListenersToFields();
        }
    };
    
    // Add scroll listeners to all fields on page load
    setTimeout(addScrollListenersToFields, 500);
    
    // Also add scroll listeners when a stack is loaded
    document.addEventListener('stackLoaded', function() {
        setTimeout(addScrollListenersToFields, 500);
    });
    
    // Periodically check for new fields that might need scroll listeners
    setInterval(addScrollListenersToFields, 2000);
    
    // Debug function to manually trigger scroll events for testing
    window.triggerScrollEvent = function(fieldName) {
        const field = WebTalkObjects.getObject(fieldName);
        if (field) {
            console.log(`Manually triggering scroll event for field ${fieldName}`);
            const event = new Event('scroll');
            field.dispatchEvent(event);
            return `Triggered scroll event on field ${fieldName}`;
        }
        return `Field ${fieldName} not found`;
    };
    
    // Debug function to test scrolling and clickLine calculation
    window.testFieldScroll = function(fieldName, scrollValue) {
        const field = WebTalkObjects.getObject(fieldName);
        if (!field) return `Field ${fieldName} not found`;
        
        // Set the scroll position
        field.scrollTop = scrollValue;
        
        // Force a layout reflow
        field.getBoundingClientRect();
        
        // Get the actual scroll position
        const actualScroll = field.scrollTop;
        
        // Check if the scroll position was applied
        if (actualScroll !== scrollValue) {
            console.warn(`Failed to set scroll position. Requested: ${scrollValue}, Actual: ${actualScroll}`);
        }
        
        // Manually trigger the scroll event
        field._scrollHandler?.call(field);
        
        return `Set field ${fieldName} scroll to ${actualScroll}px`;
    };
    
    // Function to manually update the vScroll property and trigger a scrollbarDrag message
    window.updateFieldScroll = function(fieldName, scrollValue) {
        const field = WebTalkObjects.getObject(fieldName);
        if (field) {
            // Update the DOM
            field.scrollTop = scrollValue;
            console.log(`Manually setting field ${fieldName} scroll to ${scrollValue}px`);
            
            // Update the stored property
            if (WebTalkObjects.customProperties.has(fieldName)) {
                WebTalkObjects.customProperties.get(fieldName).set('vScroll', scrollValue);
                WebTalkObjects.customProperties.get(fieldName).set('vscroll', scrollValue);
            }
            
            // Trigger the scrollbarDrag message
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(fieldName, 'scrollbarDrag', [scrollValue]);
            }
            
            return `Updated scroll position of field ${fieldName} to ${scrollValue}px`;
        }
        return `Field ${fieldName} not found`;
    };
    
    // Also run the fix after any paste operation
    // This ensures the browser doesn't try and modify our fields on paste operations
    document.addEventListener('paste', function(event) {
        if (event.target.closest('.field')) {
            setTimeout(fixDuplicateFieldContents, 10);
        }
    }, true);
    
    // Register the properties with the interpreter when it's available
    const registerProperties = function() {
        console.log('Attempting to register properties with interpreter');
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            console.log('Interpreter found, registering properties');
            // Register the clickLine property getter (camelCase version)
            window.webTalkApp.interpreter.registerPropertyGetter('clickLine', function(obj) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    return window.getClickLineProperty(obj);
                }
                return 1; // Default to first line for non-field objects
            });
            
            // Register the clickLine property setter (camelCase version)
            window.webTalkApp.interpreter.registerPropertySetter('clickLine', function(obj, value) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    const lineNumber = parseInt(value);
                    if (!isNaN(lineNumber) && lineNumber > 0) {
                        window.setClickLineProperty(obj, lineNumber);
                        return lineNumber;
                    }
                }
                return 1; // Default to first line for invalid values
            });
            
            // Register the clickline property getter (lowercase version)
            window.webTalkApp.interpreter.registerPropertyGetter('clickline', function(obj) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    return window.getClickLineProperty(obj);
                }
                return 1; // Default to first line for non-field objects
            });
            
            // Register the clickline property setter (lowercase version)
            window.webTalkApp.interpreter.registerPropertySetter('clickline', function(obj, value) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    const lineNumber = parseInt(value);
                    if (!isNaN(lineNumber) && lineNumber > 0) {
                        window.setClickLineProperty(obj, lineNumber);
                        return lineNumber;
                    }
                }
                return 1; // Default to first line for invalid values
            });
            
            // Register the vScroll property getter (camelCase version)
            window.webTalkApp.interpreter.registerPropertyGetter('vScroll', function(obj) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    // Always get the scroll position directly from the DOM
                    // This is more reliable than using stored properties
                    return getScrollTop(obj) || 0;
                }
                return 0;
            });
            
            // Register the vScroll property setter (camelCase version)
            window.webTalkApp.interpreter.registerPropertySetter('vScroll', function(obj, value) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    const scrollValue = parseInt(value);
                    if (!isNaN(scrollValue) && scrollValue >= 0) {
                        // Use the setter function to update both DOM and stored property
                        window.setVScrollProperty(obj, scrollValue);
                        return scrollValue;
                    }
                }
                return 0;
            });
            
            // Register the vscroll property getter (lowercase version)
            window.webTalkApp.interpreter.registerPropertyGetter('vscroll', function(obj) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    // Always get the scroll position directly from the DOM
                    // This is more reliable than using stored properties
                    return getScrollTop(obj) || 0;
                }
                return 0;
            });
            
            // Register the vscroll property setter (lowercase version)
            window.webTalkApp.interpreter.registerPropertySetter('vscroll', function(obj, value) {
                if (obj && obj.dataset && obj.dataset.type === 'field') {
                    const scrollValue = parseInt(value);
                    if (!isNaN(scrollValue) && scrollValue >= 0) {
                        // Use the setter function to update both DOM and stored property
                        window.setVScrollProperty(obj, scrollValue);
                        return scrollValue;
                    }
                }
                return 0;
            });
            
            console.log('clickLine, clickline, vScroll, and vscroll properties registered with interpreter');
            
            // Test if the properties are actually registered
            if (typeof window.webTalkApp.interpreter.getPropertyGetter === 'function') {
                console.log('vScroll getter registered:', !!window.webTalkApp.interpreter.getPropertyGetter('vScroll'));
                console.log('vscroll getter registered:', !!window.webTalkApp.interpreter.getPropertyGetter('vscroll'));
            }
        } else {
            console.log('Interpreter not ready, will try again in 100ms');
            // Try again in a moment if the interpreter isn't ready yet
            setTimeout(registerProperties, 100);
        }
    };
    
    // Start the registration process
    setTimeout(registerProperties, 500);
});

/**
 * Get the vScroll property of a field
 * @param {HTMLElement} field - The field element
 * @returns {number} - The vertical scroll position in pixels
 */
window.getVScrollProperty = function(field) {
    // Always get the scroll position directly from the DOM
    // Force a layout reflow to ensure we have the most current value
    field.getBoundingClientRect();
    return getScrollTop(field) || 0;
};

/**
 * Set the vScroll property of a field
 * @param {HTMLElement} field - The field element
 * @param {number} scrollValue - The scroll position to set in pixels
 */
window.setVScrollProperty = function(field, scrollValue) {
    const fieldName = field.dataset.name;
    console.log(`setVScrollProperty called for field ${fieldName} with value ${scrollValue}`);
    
    // Ensure scrollValue is a valid number
    scrollValue = Math.max(0, parseInt(scrollValue) || 0);
    
    // Update the DOM
    field.scrollTop = scrollValue;
    console.log(`Set field.scrollTop to ${scrollValue}, actual value now: ${field.scrollTop}`);
    
    // Update the stored property
    if (WebTalkObjects && WebTalkObjects.customProperties) {
        console.log(`WebTalkObjects.customProperties exists: ${WebTalkObjects.customProperties !== undefined}`);
        if (WebTalkObjects.customProperties.has(fieldName)) {
            console.log(`WebTalkObjects.customProperties has entry for ${fieldName}`);
            WebTalkObjects.customProperties.get(fieldName).set('vScroll', scrollValue);
            WebTalkObjects.customProperties.get(fieldName).set('vscroll', scrollValue);
            console.log(`Updated vScroll property for ${fieldName} to ${scrollValue}`);
        } else {
            console.log(`WebTalkObjects.customProperties does NOT have entry for ${fieldName}, creating one`);
            WebTalkObjects.customProperties.set(fieldName, new Map());
            WebTalkObjects.customProperties.get(fieldName).set('vScroll', scrollValue);
            WebTalkObjects.customProperties.get(fieldName).set('vscroll', scrollValue);
        }
    } else {
        console.error('WebTalkObjects.customProperties is not defined!');
    }
    
    return scrollValue;
};

// Fix for Firefox which doesn't support scrollTop on contenteditable elements
function getScrollTop(element) {
    let scrollValue = 0;
    
    if (element.scrollTop !== undefined) {
        scrollValue = Math.max(0, Math.round(element.scrollTop));
    } else {
        // For Firefox, we need to get the scroll position differently
        const selection = window.getSelection();
        if (selection.rangeCount > 0) {
            const range = selection.getRangeAt(0);
            const rect = range.getBoundingClientRect();
            const containerRect = element.getBoundingClientRect();
            scrollValue = Math.max(0, Math.round(containerRect.top - rect.top));
        }
    }
    
    console.log(`getScrollTop called for element ${element.dataset?.name || 'unknown'}, returning: ${scrollValue}`);
    return scrollValue;
}

// Make vScroll property available globally for HyperTalk scripts
if (typeof window !== 'undefined') {
    window.vScroll = {
        get: window.getVScrollProperty,
        set: window.setVScrollProperty
    };
}

// Export the clickLine and vScroll functions for use in other modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        getClickLine: window.getClickLine,
        getClickLineProperty: window.getClickLineProperty,
        setClickLineProperty: window.setClickLineProperty,
        getVScrollProperty: window.getVScrollProperty,
        setVScrollProperty: window.setVScrollProperty
    };
}
