/**
 * HyperTalk Expression Parser
 * 
 * A dedicated parser for HyperTalk expressions with support for:
 * - Logical operators (and, or, not)
 * - Comparison operators (=, <>, <, >, <=, >=, contains, is in, etc.)
 * - Compound expressions with proper operator precedence
 * - Space-containing operators (starts with, ends with, etc.)
 * - Parenthesized expressions
 */

// Use a self-executing function to avoid global namespace pollution
(function(global) {

class ExpressionParser {
    constructor(interpreter) {
        this.interpreter = interpreter;
    }

    /**
     * Parse and evaluate a logical expression with proper operator precedence
     * @param {string} expression - The expression to evaluate
     * @returns {boolean} - The result of the evaluation
     */
    evaluateLogicalExpression(expression) {
        try {
            // Trim the expression
            expression = expression.trim();
            
            // Handle parenthesized expressions first
            const parenResult = this.handleParentheses(expression);
            if (parenResult !== expression) {
                return this.evaluateLogicalExpression(parenResult);
            }
            
            // Handle 'not' operator
            if (/^not\s+/i.test(expression)) {
                const restExpression = expression.replace(/^not\s+/i, '').trim();
                return !this.evaluateLogicalExpression(restExpression);
            }
            
            // Handle 'and' operator (higher precedence than 'or')
            const andParts = this.splitByOperator(expression, /\s+and\s+/i);
            if (andParts.length > 1) {
                return andParts.every(part => this.evaluateLogicalExpression(part));
            }
            
            // Handle 'or' operator
            const orParts = this.splitByOperator(expression, /\s+or\s+/i);
            if (orParts.length > 1) {
                return orParts.some(part => this.evaluateLogicalExpression(part));
            }
            
            // Handle comparison expressions (e.g., "x contains y", "x starts with y")
            return this.evaluateComparisonExpression(expression);
        } catch (error) {
            console.error("Error evaluating logical expression:", error);
            return false;
        }
    }
    
    /**
     * Handle parenthesized expressions by evaluating them first
     * @param {string} expression - The expression to evaluate
     * @returns {string} - The expression with parenthesized parts evaluated
     */
    handleParentheses(expression) {
        // Check for outermost parentheses
        if (expression.startsWith('(') && expression.endsWith(')')) {
            const inner = expression.substring(1, expression.length - 1).trim();
            if (this.hasBalancedParentheses(inner)) {
                return this.evaluateLogicalExpression(inner).toString();
            }
        }
        
        // Find and evaluate nested parenthesized expressions
        let result = expression;
        let parenMatch;
        const parenRegex = /\(([^()]*)\)/g;
        
        while ((parenMatch = parenRegex.exec(expression)) !== null) {
            const [fullMatch, innerExpr] = parenMatch;
            const evaluated = this.evaluateLogicalExpression(innerExpr);
            result = result.replace(fullMatch, evaluated.toString());
        }
        
        return result;
    }
    
    /**
     * Check if a string has balanced parentheses
     * @param {string} str - The string to check
     * @returns {boolean} - True if parentheses are balanced
     */
    hasBalancedParentheses(str) {
        let count = 0;
        for (let i = 0; i < str.length; i++) {
            if (str[i] === '(') count++;
            if (str[i] === ')') count--;
            if (count < 0) return false;
        }
        return count === 0;
    }
    
    /**
     * Split an expression by an operator, respecting parentheses and quotes
     * @param {string} expression - The expression to split
     * @param {RegExp} operatorRegex - The regex for the operator
     * @returns {string[]} - The parts of the expression
     */
    splitByOperator(expression, operatorRegex) {
        // Handle simple case with no parentheses or quotes
        if (!expression.includes('(') && !expression.includes('"') && !expression.includes("'")) {
            return expression.split(operatorRegex);
        }
        
        // More complex case with parentheses and/or quotes
        const parts = [];
        let currentPart = '';
        let inQuotes = false;
        let quoteChar = '';
        let parenCount = 0;
        
        for (let i = 0; i < expression.length; i++) {
            const char = expression[i];
            
            // Handle quotes
            if ((char === '"' || char === "'") && (i === 0 || expression[i-1] !== '\\')) {
                if (inQuotes && char === quoteChar) {
                    inQuotes = false;
                } else if (!inQuotes) {
                    inQuotes = true;
                    quoteChar = char;
                }
                currentPart += char;
                continue;
            }
            
            // Handle parentheses
            if (char === '(' && !inQuotes) {
                parenCount++;
            } else if (char === ')' && !inQuotes) {
                parenCount--;
            }
            
            // Check for operator match
            if (parenCount === 0 && !inQuotes) {
                // Look ahead to check for operator
                const restOfString = expression.substring(i);
                const operatorMatch = restOfString.match(operatorRegex);
                
                if (operatorMatch && operatorMatch.index === 0) {
                    parts.push(currentPart.trim());
                    currentPart = '';
                    i += operatorMatch[0].length - 1; // Skip the operator
                    continue;
                }
            }
            
            currentPart += char;
        }
        
        // Add the last part
        if (currentPart.trim()) {
            parts.push(currentPart.trim());
        }
        
        return parts;
    }
    
    /**
     * Evaluate a comparison expression (e.g., "x contains y", "x starts with y")
     * @param {string} expression - The expression to evaluate
     * @returns {boolean} - The result of the evaluation
     */
    evaluateComparisonExpression(expression) {
        // Handle "starts with" operator
        const startsWithMatch = expression.match(/^(.+?)\s+starts\s+with\s+(.+)$/i);
        if (startsWithMatch) {
            const [_, text, pattern] = startsWithMatch;
            const value = String(this.interpreter.getTextValue(text));
            const patternValue = String(this.interpreter.getTextValue(pattern));
            return value.startsWith(patternValue);
        }
        
        // Handle "ends with" operator
        const endsWithMatch = expression.match(/^(.+?)\s+ends\s+with\s+(.+)$/i);
        if (endsWithMatch) {
            const [_, text, pattern] = endsWithMatch;
            const value = String(this.interpreter.getTextValue(text));
            const patternValue = String(this.interpreter.getTextValue(pattern));
            return value.endsWith(patternValue);
        }
        
        // Handle "contains" operator
        const containsMatch = expression.match(/^(.+?)\s+contains\s+(.+)$/i);
        if (containsMatch) {
            const [_, text, pattern] = containsMatch;
            const value = String(this.interpreter.getTextValue(text));
            const patternValue = String(this.interpreter.getTextValue(pattern));
            return value.includes(patternValue);
        }
        
        // Handle "is not" operator (MUST come before "is" operator to match correctly)
        // Use non-greedy match and stop at keywords like "then", "and", "or"
        const isNotMatch = expression.match(/^(.+?)\s+is\s+not\s+(.+?)(?:\s+(?:then|and|or)\s+|$)/i);
        if (isNotMatch) {
            const [fullMatch, left, right] = isNotMatch;
            // Use evaluateExpression to handle complex expressions with variables
            const leftValue = this.interpreter.evaluateExpression(left);
            const rightValue = this.interpreter.evaluateExpression(right);
            return String(leftValue) !== String(rightValue);
        }
        
        // Handle "is" operator
        const isMatch = expression.match(/^(.+?)\s+is\s+(.+)$/i);
        if (isMatch) {
            const [_, left, right] = isMatch;
            // Use evaluateExpression to handle complex expressions with variables
            const leftValue = this.interpreter.evaluateExpression(left);
            const rightValue = this.interpreter.evaluateExpression(right);
            
            // Special handling for true/false comparisons
            if (String(rightValue).toLowerCase() === 'true') {
                // When comparing with 'true', check if leftValue is exactly 'true'
                // This handles function results that return string boolean values
                if (String(leftValue).toLowerCase() === 'true' || String(leftValue).toLowerCase() === 'false') {
                    return String(leftValue).toLowerCase() === 'true';
                }
                // Otherwise use the standard truthiness check
                return this.interpreter.isTruthy(String(leftValue));
            } else if (String(rightValue).toLowerCase() === 'false') {
                // When comparing with 'false', check if leftValue is exactly 'false'
                // This handles function results that return string boolean values
                if (String(leftValue).toLowerCase() === 'true' || String(leftValue).toLowerCase() === 'false') {
                    return String(leftValue).toLowerCase() === 'false';
                }
                // Otherwise use the standard truthiness check
                return !this.interpreter.isTruthy(String(leftValue));
            }
            
            // Default string comparison
            return String(leftValue) === String(rightValue);
        }
        
        // Handle "is in" operator (as a synonym for "contains" with operands swapped)
        const isInMatch = expression.match(/^(.+?)\s+is\s+in\s+(.+)$/i);
        if (isInMatch) {
            const [_, needle, haystack] = isInMatch;
            // Swap the operands to match the behavior of the contains operator
            // "needle is in haystack" is equivalent to "haystack contains needle"
            const needleValue = String(this.interpreter.getTextValue(needle));
            const haystackValue = String(this.interpreter.getTextValue(haystack));
            console.log('ExpressionParser: "is in" operator');
            console.log('  needle:', needle, '→', needleValue);
            console.log('  haystack:', haystack, '→', haystackValue);
            console.log('  result:', haystackValue.includes(needleValue));
            return haystackValue.includes(needleValue);
        }
        
        // Handle "does not contain" operator
        const doesNotContainMatch = expression.match(/^(.+?)\s+does\s+not\s+contain\s+(.+)$/i);
        if (doesNotContainMatch) {
            const [_, text, pattern] = doesNotContainMatch;
            const value = String(this.interpreter.getTextValue(text));
            const patternValue = String(this.interpreter.getTextValue(pattern));
            return !value.includes(patternValue);
        }
        
        // Handle standard comparison operators (=, <>, <, >, <=, >=)
        const comparisonMatch = expression.match(/^(.+?)\s*([=<>]+)\s*(.+)$/);
        if (comparisonMatch) {
            const [_, left, operator, right] = comparisonMatch;
            const leftValue = this.interpreter.getTextValue(left);
            const rightValue = this.interpreter.getTextValue(right);
            
            // Try numeric comparison first
            const leftNum = Number(leftValue);
            const rightNum = Number(rightValue);
            const isNumeric = !isNaN(leftNum) && !isNaN(rightNum);
            
            if (isNumeric) {
                switch (operator) {
                    case '=': return leftNum === rightNum;
                    case '<>': return leftNum !== rightNum;
                    case '<': return leftNum < rightNum;
                    case '>': return leftNum > rightNum;
                    case '<=': return leftNum <= rightNum;
                    case '>=': return leftNum >= rightNum;
                }
            }
            
            // Fall back to string comparison
            const leftStr = String(leftValue);
            const rightStr = String(rightValue);
            
            switch (operator) {
                case '=': return leftStr === rightStr;
                case '<>': return leftStr !== rightStr;
                case '<': return leftStr < rightStr;
                case '>': return leftStr > rightStr;
                case '<=': return leftStr <= rightStr;
                case '>=': return leftStr >= rightStr;
            }
        }
        
        // If we get here, try to evaluate the expression using the interpreter
        const result = this.interpreter.evaluateExpression(expression);
        return this.interpreter.isTruthy(result);
    }
}

// Export the class
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ExpressionParser;
} else if (typeof global !== 'undefined') {
    global.ExpressionParser = ExpressionParser;
}

})(typeof window !== 'undefined' ? window : (typeof global !== 'undefined' ? global : this));
