class HyperTalkTestSuite {
    constructor(interpreter) {
        this.interpreter = interpreter;
        this.results = [];
        this.showDialogs = false; // Default to not showing dialogs
        this.playSounds = false; // Default to not playing sounds
        this.testsPassed = 0;
        this.testsFailed = 0;
        this.testsSkipped = 0;
        // Set up the output handler
        this.interpreter.outputHandler = (text) => {
            if (this.currentTest) {
                this.currentTest.outputs.push(text);
            }
        };
    }

    async runTest(description, script, extraDescription = '', expectedOutput = null, customValidator = null) {
        this.currentTest = {
            description,
            script: Array.isArray(script) ? script : script.split('\n'),
            outputs: [],
            status: 'running',
            extraDescription,
            name: description + (extraDescription ? ` (${extraDescription})` : ''),
            expectedOutput,
            customValidator
        };
        
        try {
            // For dialog tests, check if we should skip
            if (!this.showDialogs && 
                (description.toLowerCase().includes('dialog') || 
                 script.toString().toLowerCase().includes('answer') || 
                 script.toString().toLowerCase().includes('ask'))) {
                this.currentTest.status = 'skipped';
                this.currentTest.outputs.push('Dialog test skipped');
                this.results.push(this.currentTest);
                this.testsSkipped++;
                this.currentTest = null;
                return;
            }

            // For sound tests, check if we should skip
            if (!this.playSounds && 
                (description.toLowerCase().includes('sound') || 
                 description.toLowerCase().includes('musical') ||
                 description.toLowerCase().includes('tempo') ||
                 script.toString().toLowerCase().includes('play '))) {
                this.currentTest.status = 'skipped';
                this.currentTest.outputs.push('Sound test skipped');
                this.results.push(this.currentTest);
                this.testsSkipped++;
                this.currentTest = null;
                return;
            }

            // Split the script into lines and run each line
            for (const line of this.currentTest.script) {
                if (line.trim()) {
                    await this.interpreter.interpret(line);
                }
            }
            
            // Validate output if expectedOutput or customValidator is provided
            if (expectedOutput !== null || customValidator !== null) {
                const outputText = this.currentTest.outputs.join('\n');
                let passed = true;
                
                if (typeof customValidator === 'function') {
                    try {
                        passed = customValidator(outputText);
                    } catch (e) {
                        passed = false;
                        this.currentTest.validationError = e.message;
                    }
                } else if (expectedOutput !== null) {
                    passed = (outputText.trim() === expectedOutput.trim());
                }
                
                if (!passed) {
                    this.currentTest.status = 'failed';
                    this.currentTest.error = 'Output validation failed';
                    if (expectedOutput !== null) {
                        this.currentTest.expectedOutput = expectedOutput;
                    }
                    this.testsFailed++;
                } else {
                    this.currentTest.status = 'passed';
                    this.testsPassed++;
                }
            } else {
                this.currentTest.status = 'passed';
                this.testsPassed++;
            }
        } catch (error) {
            this.currentTest.status = 'failed';
            this.currentTest.error = error.message;
            this.testsFailed++;
        }
        
        this.results.push(this.currentTest);
        this.currentTest = null;
    }
    async runAllTests() {
        this.results = [];
        this.testsPassed = 0;
        this.testsFailed = 0;
        this.testsSkipped = 0;
        const tests = [
            { description: 'Get test', script: 'get "a"' },
            { description: 'Put test', script: 'put it' },
            { description: 'Date test', script: 'put the date' },
            { description: 'Time test', script: 'put the time' },
            { description: 'Put word x of string test', script: 'put word 1 of "this is a test"' },
            { description: 'Put word x of variable test', script: 'put word 1 of the date' },
            { description: 'Put char of variable test', script: 'put char 1 of the time' },
            { description: 'Put char of string test', script: 'put char 1 of "try"' },
            { description: 'Length of variable test', script: 'put the length of the time' },
            { description: 'Length of string test', script: 'put the length of "a sample string"' },
            { description: 'Number of words test', script: 'put the number of words of "this is another test"' },
            { description: 'Number of chars in variable', script: 'put the number of chars of the long date' },
            { description: 'Number of chars in string', script: 'put the number of chars of "abcdefghijklmnop"' },
            { description: 'String join test', script: 'put "this " & "that"' },
            { description: '"is in" variable test', script: 'if "1" is in the date then put "1 is in the date"' },
            { description: '"is in" string test', script: 'if "0" is in "hell0" then put "0 is in our string"' },
            { description: 'First char of string', script: 'put the first char of "test"' },
            { description: 'Last char of string', script: 'put the last char of "testing"' },
            { description: 'First char of variable', script: 'put the first char of the time' },
            { description: 'Last char of variable', script: 'put the last char of the time' },
            { description: 'Numeric if > then', script: 'if 5 > 2 then put "5 is more than 2"' },
            { description: 'Math addition', script: 'put 6 + 2' },
            { description: 'Math multiplication', script: 'put 6 * 2' },
            { description: 'Math division', script: 'put 6 / 2' },
            { description: 'To uppercase', script: 'put toUpper("this should be uppercase")' },
            { description: 'To lowercase', script: 'put toLower("ThIs ShoUlD bE lOweRcaSe")' },
            { description: 'Items setup', script: 'put "a, b, c, d, e" into tItems' },
            { description: 'Count items', script: 'put the number of items of tItems' },
            { description: 'Delete last char', script: 'put "strings" into tstring\ndelete last char of tstring\nput tstring' },
            { description: 'Offset test', script: 'put offset("needle" in "this is the needle somewhere in a haystack")' },
            { description: 'Repeat with random', script: 'repeat 4\nput random(20)\nend repeat' },
            { description: 'If else test', script: 'if char 1 of the time > 4 then\nput "hour more than 4"\nelse\nput "hour less than 4"\nend if' },
            { description: 'Repeat until', script: 'put true into tVar\nrepeat until tVar is false\nput false into tVar\nend repeat\nput "done"' },
            { description: 'Sort items', script: 'put "f, b, d, c, e, a" into tItems\nsort items of tItems ascending' },
            { description: 'DateItems test', script: 'put the dateItems into tDateBits\nput tDateBits' },
            { description: 'Add to variable', script: 'add 1 to item 1 of tDateBits' },
            { description: 'Convert dateItems', script: 'convert tDateBits to the long date\nput tDateBits' },
            { description: 'Char x of string', script: 'put char 3 of "this"' },
            { description: 'Item x of variable', script: 'put item 3 of tItems' },
            { description: 'Starts with test', script: 'if tItems starts with "a" then\nput "starts with working"\nelse\nput tItems & " does not start with a"\nend if' },
            { description: 'Add to var', script: 'put 3 into tNum\nadd 1 to tNum\nput tNum' },
            { description: 'Subtract from var', script: 'subtract 1 from tNum\nput tNum' },
            { description: 'isNumber Function', script: 'put isNumber("5")' },
            { description: 'Variable Assignment', script: ['put 42 into x', 'put x'] },
            
            // Dialog tests
            { description: 'Ask dialog test', script: [
                'ask "What is your name?"',
                'put it'
            ]},
            { description: 'Answer dialog test', script: [
                'answer "An answer dialog test"'
            ]},
            { description: 'Answer with buttons test', script: [
                'answer "Choose an option" with "Yes" or "No" or "Maybe"'
            ]},
            { description: 'Ask password dialog with username and password', script: [
                'ask password "Please enter your credentials:" with "username,password"',
                'put it'
            ]},
            { description: 'Ask password dialog with single field', script: [
                'ask password "Enter password:" with "password"',
                'put it'
            ]},
            { description: 'Ask password dialog with multiple fields', script: [
                'ask password "Complete registration:" with "username,email,password,confirm password"',
                'put it'
            ]},
            
            // List dialog tests
            { description: 'Ask list dialog test', script: [
                'put "item 1" & return & "item 2" & return & "item 3" into treturndelimitedList',
                'ask list "Please pick an item:" with treturndelimitedList',
                'put it'
            ]},
            { description: 'Ask list dialog with variable message', script: [
                'put "Select your favorite color:" into tPrompt',
                'put "Red" & return & "Green" & return & "Blue" into tColors',
                'ask list tPrompt with tColors',
                'put it'
            ]},
            
            // Repeat with lines count tests
            { description: 'repeat with number of lines test', script: [
                'put "/home/tom/Desktop/sort/PPC-logo.png" & return & "/home/tom/Desktop/sort/screenshot.png" into tFilesList',
                'put 0 into tCount',
                'repeat the number of lines in tFilesList',
                'add 1 to tCount',
                'delete char 7 to 18 of line tCount of tFilesList',
                'end repeat',
                'put tFilesList'
            ]},

            // Item delimiter tests
            { description: 'Set itemdelimiter to slash', script: [
                'set the itemdelimiter to slash',
                'put "a/b/c" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to literal slash', script: [
                'set the itemdelimiter to "/"',
                'put "x/y/z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to custom character', script: [
                'set the itemdelimiter to "|"',
                'put "1|2|3" into tVar',
                'put item 2 of tVar'
            ]},
            { description: 'Reset itemdelimiter to comma', script: [
                'set the itemdelimiter to ","',
                'put "a,b,c" into tVar',
                'put item 3 of tVar'
            ]},
            { description: 'Set itemdelimiter to comma keyword', script: [
                'set the itemdelimiter to comma',
                'put "x,y,z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Compare comma keyword with literal comma', script: [
                'set the itemdelimiter to comma',
                'put "1,2,3" into tVar1',
                'set the itemdelimiter to ","',
                'put "1,2,3" into tVar2',
                'put item 2 of tVar1',
                'put item 2 of tVar2'
            ]},
            { description: 'Set itemdelimiter using itemdel', script: [
                'put "a-b,b-c,c-d,d-e" into tString',
                'set the itemdel to "-"',
                'put item 2 of tString',
                'set the itemdel to ","',
                'put item 2 of tString'
            ]},
            
            // Repeat and control flow tests
            { description: 'Repeat with random', script: 'repeat 4\nput random(20)\nend repeat' },
            { description: 'If else test', script: 'if char 1 of the time > 4 then\nput "hour more than 4"\nelse\nput "hour less than 4"\nend if' },
            { description: 'Repeat until', script: 'put true into tVar\nrepeat until tVar is false\nput false into tVar\nend repeat\nput "done"' },
            
            // ... average tests
            { description: 'Average of two numbers', script: [
                'get average(12,23)',
                'put it'
            ]},
            { description: 'Average direct output', script: [
                'put average(34,23)'
            ]},
            { description: 'Average into variable', script: [
                'put average(32,43,23) into tAverageValue',
                'put tAverageValue'
            ]},
            { description: 'Average with multiple numbers', script: [
                'put average(4,36,45,50,75)'
            ]},
            { description: 'Average with variables', script: [
                'put 10 into tNum1',
                'put 20 into tNum2',
                'put average(tNum1,tNum2)'
            ]},
            { description: 'Average with expressions', script: [
                'put average(5+5, 10*2)',
                'put average(2*3, 4+5, 10/2)'
            ]},

            // Character conversion tests
            { description: 'charToNum basic test', script: [
                'put charToNum("A")'
            ]},
            { description: 'charToNum with variable', script: [
                'put "B" into tChar',
                'put charToNum(tChar)'
            ]},
            { description: 'numToChar basic test', script: [
                'put numToChar(65)'
            ]},
            { description: 'numToChar with variable', script: [
                'put 66 into tNum',
                'put numToChar(tNum)'
            ]},
            { description: 'Character conversion round trip', script: [
                'put "Z" into tChar',
                'put charToNum(tChar) into tNum',
                'put numToChar(tNum)'
            ]},
            
            // Paragraph tests
            { description: 'Get number of paragraphs', script: [
                'put "Line 1" & return & "Line 2" & return & "Line 3" into tText',
                'put tText',
                'put the number of paragraphs of tText'
            ]},

            // Exponential tests
            { description: 'exp of 1 (e)', script: [
                'put exp(1)'
            ]},
            { description: 'exp of 0 (should be 1)', script: [
                'put exp(0)'
            ]},
            { description: 'exp with variable', script: [
                'put 2 into tNum',
                'put exp(tNum)'
            ]},
            { description: 'exp with expression', script: [
                'put exp(1 + 1)'
            ]},
            
            // Value function tests
            { description: 'value of arithmetic expression', script: [
                'put the value of 22 + 3'
            ]},
            { description: 'value of complex arithmetic', script: [
                'put the value of 5 * 3 + 2'
            ]},
            { description: 'value of arithmetic with parentheses', script: [
                'put the value of (5 * 3) + 2'
            ]},
            { description: 'value of nested parentheses', script: [
                'put the value of (5 * (3 + 2))'
            ]},
            { description: 'value of variable arithmetic', script: [
                'put 10 into x',
                'put the value of x + 5'
            ]},
            { description: 'value of string concatenation', script: [
                'put the value of "hello" & " " & "world"'
            ]},

            // Item delimiter tests
            { description: 'Set itemdelimiter to slash', script: [
                'set the itemdelimiter to slash',
                'put "a/b/c" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to literal slash', script: [
                'set the itemdelimiter to "/"',
                'put "x/y/z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to custom character', script: [
                'set the itemdelimiter to "|"',
                'put "1|2|3" into tVar',
                'put item 2 of tVar'
            ]},
            { description: 'Reset itemdelimiter to comma', script: [
                'set the itemdelimiter to ","',
                'put "a,b,c" into tVar',
                'put item 3 of tVar'
            ]},
            { description: 'Set itemdelimiter to comma keyword', script: [
                'set the itemdelimiter to comma',
                'put "x,y,z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Compare comma keyword with literal comma', script: [
                'set the itemdelimiter to comma',
                'put "1,2,3" into tVar1',
                'set the itemdelimiter to ","',
                'put "1,2,3" into tVar2',
                'put item 2 of tVar1',
                'put item 2 of tVar2'
            ]},
            
            // Repeat and control flow tests
            { description: 'Repeat with random', script: 'repeat 4\nput random(20)\nend repeat' },
            { description: 'If else test', script: 'if char 1 of the time > 4 then\nput "hour more than 4"\nelse\nput "hour less than 4"\nend if' },
            { description: 'Repeat until', script: 'put true into tVar\nrepeat until tVar is false\nput false into tVar\nend repeat\nput "done"' },
            
            // ... average tests
            { description: 'Average of two numbers', script: [
                'get average(12,23)',
                'put it'
            ]},
            { description: 'Average direct output', script: [
                'put average(34,23)'
            ]},
            { description: 'Average into variable', script: [
                'put average(32,43,23) into tAverageValue',
                'put tAverageValue'
            ]},
            { description: 'Average with multiple numbers', script: [
                'put average(4,36,45,50,75)'
            ]},
            { description: 'Average with variables', script: [
                'put 10 into tNum1',
                'put 20 into tNum2',
                'put average(tNum1,tNum2)'
            ]},
            { description: 'Average with expressions', script: [
                'put average(5+5, 10*2)',
                'put average(2*3, 4+5, 10/2)'
            ]},

            // Character conversion tests
            { description: 'charToNum basic test', script: [
                'put charToNum("A")'
            ]},
            { description: 'charToNum with variable', script: [
                'put "B" into tChar',
                'put charToNum(tChar)'
            ]},
            { description: 'numToChar basic test', script: [
                'put numToChar(65)'
            ]},
            { description: 'numToChar with variable', script: [
                'put 66 into tNum',
                'put numToChar(tNum)'
            ]},
            { description: 'Character conversion round trip', script: [
                'put "Z" into tChar',
                'put charToNum(tChar) into tNum',
                'put numToChar(tNum)'
            ]},
            
            // Paragraph tests
            { description: 'Get number of paragraphs', script: [
                'put "Line 1" & return & "Line 2" & return & "Line 3" into tText',
                'put tText',
                'put the number of paragraphs of tText'
            ]},

            // Exponential tests
            { description: 'exp of 1 (e)', script: [
                'put exp(1)'
            ]},
            { description: 'exp of 0 (should be 1)', script: [
                'put exp(0)'
            ]},
            { description: 'exp with variable', script: [
                'put 2 into tNum',
                'put exp(tNum)'
            ]},
            { description: 'exp with expression', script: [
                'put exp(1 + 1)'
            ]},
            
            // Value function tests
            { description: 'value of arithmetic expression', script: [
                'put the value of 22 + 3'
            ]},
            { description: 'value of complex arithmetic', script: [
                'put the value of 5 * 3 + 2'
            ]},
            { description: 'value of arithmetic with parentheses', script: [
                'put the value of (5 * 3) + 2'
            ]},
            { description: 'value of nested parentheses', script: [
                'put the value of (5 * (3 + 2))'
            ]},
            { description: 'value of variable arithmetic', script: [
                'put 10 into x',
                'put the value of x + 5'
            ]},
            { description: 'value of string concatenation', script: [
                'put the value of "hello" & " " & "world"'
            ]},

            // Item delimiter tests
            { description: 'Set itemdelimiter to slash', script: [
                'set the itemdelimiter to slash',
                'put "a/b/c" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to literal slash', script: [
                'set the itemdelimiter to "/"',
                'put "x/y/z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to custom character', script: [
                'set the itemdelimiter to "|"',
                'put "1|2|3" into tVar',
                'put item 2 of tVar'
            ]},
            { description: 'Reset itemdelimiter to comma', script: [
                'set the itemdelimiter to ","',
                'put "a,b,c" into tVar',
                'put item 3 of tVar'
            ]},
            { description: 'Set itemdelimiter to comma keyword', script: [
                'set the itemdelimiter to comma',
                'put "x,y,z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Compare comma keyword with literal comma', script: [
                'set the itemdelimiter to comma',
                'put "1,2,3" into tVar1',
                'set the itemdelimiter to ","',
                'put "1,2,3" into tVar2',
                'put item 2 of tVar1',
                'put item 2 of tVar2'
            ]},
            
            // Repeat and control flow tests
            { description: 'Repeat with random', script: 'repeat 4\nput random(20)\nend repeat' },
            { description: 'If else test', script: 'if char 1 of the time > 4 then\nput "hour more than 4"\nelse\nput "hour less than 4"\nend if' },
            { description: 'Repeat until', script: 'put true into tVar\nrepeat until tVar is false\nput false into tVar\nend repeat\nput "done"' },
            
            // ... average tests
            { description: 'Average of two numbers', script: [
                'get average(12,23)',
                'put it'
            ]},
            { description: 'Average direct output', script: [
                'put average(34,23)'
            ]},
            { description: 'Average into variable', script: [
                'put average(32,43,23) into tAverageValue',
                'put tAverageValue'
            ]},
            { description: 'Average with multiple numbers', script: [
                'put average(4,36,45,50,75)'
            ]},
            { description: 'Average with variables', script: [
                'put 10 into tNum1',
                'put 20 into tNum2',
                'put average(tNum1,tNum2)'
            ]},
            { description: 'Average with expressions', script: [
                'put average(5+5, 10*2)',
                'put average(2*3, 4+5, 10/2)'
            ]},

            // Character conversion tests
            { description: 'charToNum basic test', script: [
                'put charToNum("A")'
            ]},
            { description: 'charToNum with variable', script: [
                'put "B" into tChar',
                'put charToNum(tChar)'
            ]},
            { description: 'numToChar basic test', script: [
                'put numToChar(65)'
            ]},
            { description: 'numToChar with variable', script: [
                'put 66 into tNum',
                'put numToChar(tNum)'
            ]},
            { description: 'Character conversion round trip', script: [
                'put "Z" into tChar',
                'put charToNum(tChar) into tNum',
                'put numToChar(tNum)'
            ]},
            
            // Paragraph tests
            { description: 'Get number of paragraphs', script: [
                'put "Line 1" & return & "Line 2" & return & "Line 3" into tText',
                'put tText',
                'put the number of paragraphs of tText'
            ]},

            // Exponential tests
            { description: 'exp of 1 (e)', script: [
                'put exp(1)'
            ]},
            { description: 'exp of 0 (should be 1)', script: [
                'put exp(0)'
            ]},
            { description: 'exp with variable', script: [
                'put 2 into tNum',
                'put exp(tNum)'
            ]},
            { description: 'exp with expression', script: [
                'put exp(1 + 1)'
            ]},
            
            // Value function tests
            { description: 'value of arithmetic expression', script: [
                'put the value of 22 + 3'
            ]},
            { description: 'value of complex arithmetic', script: [
                'put the value of 5 * 3 + 2'
            ]},
            { description: 'value of arithmetic with parentheses', script: [
                'put the value of (5 * 3) + 2'
            ]},
            { description: 'value of nested parentheses', script: [
                'put the value of (5 * (3 + 2))'
            ]},
            { description: 'value of variable arithmetic', script: [
                'put 10 into x',
                'put the value of x + 5'
            ]},
            { description: 'value of string concatenation', script: [
                'put the value of "hello" & " " & "world"'
            ]},

            // stripList tests
            { description: 'stripList function test', script: [
                'put "/home/tom/Desktop/sort/PPC-logo.png" & return & "/home/tom/Desktop/sort/screenshot.png" into tFilesList',
                'put stripList(tFilesList,7,18) into tStrippedList',
                'put tStrippedList'
            ]},
            
            // Repeat with lines count tests
            { description: 'repeat with number of lines test', script: [
                'put "/home/tom/Desktop/sort/PPC-logo.png" & return & "/home/tom/Desktop/sort/screenshot.png" into tFilesList',
                'put 0 into tCount',
                'repeat the number of lines in tFilesList',
                'add 1 to tCount',
                'delete char 7 to 18 of line tCount of tFilesList',
                'end repeat',
                'put tFilesList'
            ]},

            // Item delimiter tests
            { description: 'Set itemdelimiter to slash', script: [
                'set the itemdelimiter to slash',
                'put "a/b/c" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to literal slash', script: [
                'set the itemdelimiter to "/"',
                'put "x/y/z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Set itemdelimiter to custom character', script: [
                'set the itemdelimiter to "|"',
                'put "1|2|3" into tVar',
                'put item 2 of tVar'
            ]},
            { description: 'Reset itemdelimiter to comma', script: [
                'set the itemdelimiter to ","',
                'put "a,b,c" into tVar',
                'put item 3 of tVar'
            ]},
            { description: 'Set itemdelimiter to comma keyword', script: [
                'set the itemdelimiter to comma',
                'put "x,y,z" into tVar',
                'put item 1 of tVar'
            ]},
            { description: 'Compare comma keyword with literal comma', script: [
                'set the itemdelimiter to comma',
                'put "1,2,3" into tVar1',
                'set the itemdelimiter to ","',
                'put "1,2,3" into tVar2',
                'put item 2 of tVar1',
                'put item 2 of tVar2'
            ]},
            
            // Repeat and control flow tests
            { description: 'Repeat with random', script: 'repeat 4\nput random(20)\nend repeat' },
            { description: 'If else test', script: 'if char 1 of the time > 4 then\nput "hour more than 4"\nelse\nput "hour less than 4"\nend if' },
            { description: 'Repeat until', script: 'put true into tVar\nrepeat until tVar is false\nput false into tVar\nend repeat\nput "done"' },
            
            // ... average tests
            { description: 'Average of two numbers', script: [
                'get average(12,23)',
                'put it'
            ]},
            { description: 'Average direct output', script: [
                'put average(34,23)'
            ]},
            { description: 'Average into variable', script: [
                'put average(32,43,23) into tAverageValue',
                'put tAverageValue'
            ]},
            { description: 'Average with multiple numbers', script: [
                'put average(4,36,45,50,75)'
            ]},
            { description: 'Average with variables', script: [
                'put 10 into tNum1',
                'put 20 into tNum2',
                'put average(tNum1,tNum2)'
            ]},
            { description: 'Average with expressions', script: [
                'put average(5+5, 10*2)',
                'put average(2*3, 4+5, 10/2)'
            ]},

            // Character conversion tests
            { description: 'charToNum basic test', script: [
                'put charToNum("A")'
            ]},
            { description: 'charToNum with variable', script: [
                'put "B" into tChar',
                'put charToNum(tChar)'
            ]},
            { description: 'numToChar basic test', script: [
                'put numToChar(65)'
            ]},
            { description: 'numToChar with variable', script: [
                'put 66 into tNum',
                'put numToChar(tNum)'
            ]},
            { description: 'Character conversion round trip', script: [
                'put "Z" into tChar',
                'put charToNum(tChar) into tNum',
                'put numToChar(tNum)'
            ]},
            
            // Paragraph tests
            { description: 'Get number of paragraphs', script: [
                'put "Line 1" & return & "Line 2" & return & "Line 3" into tText',
                'put tText',
                'put the number of paragraphs of tText'
            ]},

            // Exponential tests
            { description: 'exp of 1 (e)', script: [
                'put exp(1)'
            ]},
            { description: 'exp of 0 (should be 1)', script: [
                'put exp(0)'
            ]},
            { description: 'exp with variable', script: [
                'put 2 into tNum',
                'put exp(tNum)'
            ]},
            { description: 'exp with expression', script: [
                'put exp(1 + 1)'
            ]},
        ];

        for (const test of tests) {
            await this.runTest(test.description, test.script);
        }

        // Test sin function
        await this.runTest("Sin function test", [
            'put sin(90) into tResult',
            'put tResult',
            'put 30 into tAngle',
            'put sin(tAngle) into tResult',
            'put tResult'
        ], "Testing sin function with both literal and variable input");

        // Test cos function
        await this.runTest("Cos function test", [
            'put cos(0) into tResult',
            'put tResult',
            'put 60 into tAngle',
            'put cos(tAngle) into tResult',
            'put tResult'
        ], "Testing cos function with both literal and variable input");

        // Test tan function
        await this.runTest("Tan function test", [
            'put tan(45) into tResult',
            'put tResult',
            'put 0 into tAngle',
            'put tan(tAngle) into tResult',
            'put tResult'
        ], "Testing tan function with both literal and variable input. tan(45) should be 1, tan(0) should be 0");

        // Test sqrt function
        await this.runTest("Square root function test", [
            'put sqrt(16) into tResult',
            'put tResult',
            'put 25 into tNumber',
            'put sqrt(tNumber) into tResult',
            'put tResult',
            'put sqrt(0) into tResult',
            'put tResult'
        ], "Testing sqrt function with both literal and variable input. sqrt(16)=4, sqrt(25)=5, sqrt(0)=0");

        // Test round function
        await this.runTest("Round function test", [
            'put round(3.7) into tResult',
            'put tResult',
            'put round(3.2) into tResult',
            'put tResult',
            'put 4.6 into tNumber',
            'put round(tNumber) into tResult',
            'put tResult'
        ], "Testing round function with both literal and variable input. round(3.7)=4, round(3.2)=3, round(4.6)=5");

        // Test abs function
        await this.runTest("Absolute value function test", [
            'put abs(-5) into tResult',
            'put tResult',
            'put abs(3.2) into tResult',
            'put tResult',
            'put -7.5 into tNumber',
            'put abs(tNumber) into tResult',
            'put tResult'
        ], "Testing abs function with both literal and variable input. abs(-5)=5, abs(3.2)=3.2, abs(-7.5)=7.5");

        // Test contains
        await this.runTest("Contains test", [
            'put "Hello World" contains "World" into tResult',
            'put tResult',
            'put "Hello" contains "xyz" into tResult',
            'put tResult',
            'put "Hello" into tString',
            'put tString contains "el" into tResult',
            'put tResult'
        ], "Testing contains with both literal strings and variables. Should handle true and false cases");

        // Test reverse
        await this.runTest("Reverse function test", [
            'put reverse("Hello") into tResult',
            'put tResult',
            'put "World" into tString',
            'put reverse(tString) into tResult',
            'put tResult',
            'put reverse("A man a plan") into tResult',
            'put tResult'
        ], "Testing reverse function with both literal strings and variables. Should reverse character order.");

        // Test answer dialog
        await this.runTest("Answer dialog test", [
            'answer "Choose an option" with "Yes" or "No"',
            'put the result',
            'answer "Pick a number" with "1" or "2" or "3"',
            'put the result'
        ], "Testing answer command with both simple messages and multiple choice buttons");

        // Test ask dialog
        await this.runTest("Ask dialog test", [
            'ask "What is your name?"',
            'put it',
            'put "Enter your age:" into tPrompt',
            'ask tPrompt',
            'put it'
        ], "Testing ask command with both literal strings and variables. Response stored in 'it'");

        // Test replace function
        await this.runTest("Replace function test", [
            'put replace("hello world", "world", "there") into tResult',
            'put tResult',
            'put "hello hello hello" into tString',
            'put replace(tString, "hello", "hi") into tResult',
            'put tResult',
            'put "no matches here" into tString',
            'put replace(tString, "xyz", "abc") into tResult',
            'put tResult'
        ], "Testing replace function with both literal strings and variables. Should replace all occurrences.");

        // Test sort command
        await this.runTest("Sort command test", [
            'put "banana,apple,cherry" into tFruits',
            'sort items of tFruits',
            'put tFruits',
            'put "30,10,20" into tNumbers',
            'sort items of tNumbers ascending',
            'put tNumbers',
            'put "cat,dog,bird" into tAnimals',
            'sort items of tAnimals descending',
            'put tAnimals'
        ], "Testing sort command with different types of lists, both ascending and descending");

        // Test convert date formats
        await this.runTest("Convert date formats test", [
            'put the date into tDate',
            'convert tDate from date to long date',
            'put tDate',
            'convert tDate to abbreviated date',
            'put tDate',
            'put the seconds into tToday',
            'convert tToday from seconds to date',
            'put the seconds into tToday',
            'put tToday + 240 into tToday -- we add 4 minutes',
            'convert tToday to dateitems',
            'put tToday'
        ], "Testing date conversion between different formats: date, long, abbreviated, and dateitems");

        // Test date conversion to dateitems and seconds
        await this.runTest("Date conversion to dateitems and seconds", [
            'put the seconds into tDate',
            'convert tDate to dateitems',
            'put tDate',
            'put "Number of items in dateitems:" && the number of items of tDate',
             'put "Year (item 1):" && item 1 of tDate',
            'put "Month (item 2):" && item 2 of tDate',
            'put "Day (item 3):" && item 3 of tDate',
            'put "Hour (item 4):" && item 4 of tDate',
            'put "Minute (item 5):" && item 5 of tDate',
            'put "Second (item 6):" && item 6 of tDate',
            'put "Day of week (item 7):" && item 7 of tDate',
            'put the date into tDate2',
            'convert tDate2 to seconds',
            'put "Date in seconds:" && tDate2',
            'put "Current seconds:" && the seconds'
        ], "Testing date conversion to dateitems and seconds");


        // Test power function
        await this.runTest("Power function test", [
            'put power(2, 3) into tResult -- 2^3',
            'put tResult',
            'put 5 into tBase',
            'put 2 into tExponent',
            'put power(tBase, tExponent) into tResult -- 5^2',
            'put tResult',
            'put power(3, 0) into tResult -- 3^0',
            'put tResult',
            'put power(2, -2) into tResult -- 2^-2',
            'put tResult'
        ], "Testing power function with literal numbers and variables, including edge cases");

        // Test mod function
        await this.runTest("Mod function test", [
            'put mod(7, 3) into tResult -- 7 mod 3 = 1',
            'put tResult',
            'put 17 into tNumber',
            'put 5 into tDivisor',
            'put mod(tNumber, tDivisor) into tResult -- 17 mod 5 = 2',
            'put tResult',
            'put mod(10, 2) into tResult -- 10 mod 2 = 0',
            'put tResult',
            'put mod(-7, 3) into tResult -- -7 mod 3 = 2',
            'put tResult'
        ], "Testing mod function with literal numbers and variables, including edge cases");

        // Test round and trunc functions
        await this.runTest("Round and trunc functions test", [
            'put round(3.7) into tResult -- rounds to 4',
            'put tResult',
            'put round(3.2) into tResult -- rounds to 3',
            'put tResult',
            'put round(-3.7) into tResult -- rounds to -4',
            'put tResult',
            'put 3.7 into tNumber',
            'put round(tNumber) into tResult -- rounds to 4',
            'put tResult',
            'put trunc(3.7) into tResult -- truncates to 3',
            'put tResult',
            'put trunc(-3.7) into tResult -- truncates to -3',
            'put tResult',
            'put 3.7 into tNumber',
            'put trunc(tNumber) into tResult -- truncates to 3',
            'put tResult'
        ], "Testing round and trunc functions with positive and negative numbers, both literal and variables");

        // Test play command
        await this.runTest("Play sound test", [
            'play "beep.ogg"',
            'put "beep.ogg" into tSound',
            'play tSound',
            'play "beep.ogg" until done',
            'play tSound until done'
        ], "Testing play command with both literal strings and variables, including wait until done option");

        // Test play command with sound files
        await this.runTest('play "boing.ogg"', '', 'Playing sound file');
        await this.runTest('play "boing.ogg" until done', '', 'Playing sound file with wait');
        await this.runTest('play "boing.ogg" tempo 60', '', 'Playing sound file at half speed');
        await this.runTest('play "boing.ogg" tempo 240', '', 'Playing sound file at double speed');
        await this.runTest('play "boing.ogg" tempo 120 until done', '', 'Playing sound file at normal speed with wait');

        // Test musical notation
        await this.runTest("Musical notation test", [
            'play "c d e f g" tempo 120',
            'play "c4 d4 e4" tempo 60',
            'play "c e g" tempo 240 until done'
        ], "Testing musical notation with different tempos, octaves, and note sequences");

        // Test mathematical functions
        await this.runTest("put the factors of 1000", [
            'put the factors of 1000'
        ], "Demonstrates finding all factors of a number");
        // Expected result: 1,2,4,5,8,10,20,25,40,50,100,125,200,250,500,1000

        // Test mouse location
        await this.runTest("Mouse location test", [
            'put the mouseloc',
            'put the mouseH',
            'put the mouseV'
        ], "Gets mouse coordinates (full, horizontal, and vertical)");
        // Expected results:
        // x,y coordinates (e.g. "100,200")
        // horizontal position (e.g. "100")
        // vertical position (e.g. "200")

        // Test platform detection
        await this.runTest("Platform detection test", [
            'put the platform'
        ], "Gets the current operating system platform");
        // Expected result: One of: MacOS, Windows, Linux, Android, iOS, or unknown

        // Test screen dimensions
        await this.runTest("Screen dimensions test", [
            'put the screenrect'
        ], "Gets the screen dimensions (left,top,width,height)");
        // Expected result: Four comma-separated values (e.g. "0,0,1920,1080")

        // Test browser name
        await this.runTest("Browser name test", [
            'put the browsername'
        ], "Gets the browser's user agent string");
        // Expected result: Full user agent string (e.g. "Mozilla/5.0 ...")

        // Test browser version
        await this.runTest("Browser version test", [
            'put the browserversion'
        ], "Gets the browser's version number");
        // Expected result: Version number (e.g. "120.0" for Chrome)

        // Test screen orientation
        await this.runTest("Screen orientation test", [
            'put the screenorientation'
        ], "Gets the screen orientation (landscape or portrait)");
        // Expected result: Either "landscape" or "portrait" based on screen dimensions

        // Test touch screen detection
        await this.runTest("Touch screen detection test", [
            'put the hasTouchscreen'
        ], "Checks if device has a touch screen");
        // Expected result: "true" if touch screen available, "false" if not

        // Test last click position
        await this.runTest("Last click position test", [
            'put the lastclickloc',
            'put the lastclickh',
            'put the lastclickv'
        ], "Gets the last mouse click coordinates");
        // Expected result: x,y coordinates (e.g. "100,200")
        // Expected result: x coordinate (e.g. "100")
        // Expected result: y coordinate (e.g. "200")

        // Test title get and set
        await this.runTest("Title get and set test", [
            'put the title',
            'put "Testing page 1" into testTitle',
            'set the title to testTitle',
            'put the title'
        ], "Gets and sets the page title");
        // Expected results:
        // 1. Current page title
        // 2. No output (variable assignment)
        // 3. No output (title set)
        // 4. "Testing page 1"

        // Test itemDelimiter functionality
        await this.runTest("Item delimiter test", [
            'put "a,b,c" into test1',
            'put item 2 of test1',
            'set the itemdelimiter to "/"',
            'put "x/y/z" into test2',
            'put item 2 of test2',
            'put the itemdelimiter',
            'set the itemdelimiter to ","',
            'put item 3 of test1'
        ], "Tests itemDelimiter get/set and item extraction");
        // Expected results:
        // 1. No output (variable assignment)
        // 2. "b" (using default comma delimiter)
        // 3. No output (setting delimiter)
        // 4. No output (variable assignment)
        // 5. "y" (using slash delimiter)
        // 6. "/" (current delimiter value)
        // 7. No output (resetting delimiter)
        // 8. "c" (back to using comma)

        // Test seconds
        await this.runTest("Seconds test", [
            'put the seconds',
            'put the seconds > 0'
        ], "Gets Unix epoch seconds");
        // Expected results:
        // 1. Current Unix timestamp in seconds (e.g. "1707564573")
        // 2. "true" (seconds should always be positive)

        // Test ticks functionality
        await this.runTest("ticks test", [
            'wait 1 tick'
        ], "Check to see that ticks work as expected");

        // Test exit repeat functionality
        await this.runTest("exit repeat test", [
            'repeat 30',
            'exit repeat',
            'end repeat'
        ], "Make sure exit repeat works");

        // Test isNumber function
        await this.runTest("isNumber test", [
            'put isNumber("20")'
        ], "returns true if a number, false if not");

        // Test isUpper functionality
        await this.runTest("isUpper test", [
            'put "Wed" into tString',
            'put isUpper(char 1 of tString)',
            'put isUpper(char 2 of tString)'
        ], "Test isUpper function with various inputs");

        // Test clipboard functionality
        await this.runTest("clipboard test", [
            'put "Hello World" into tString',
            'set the clipboardText to tString',
            'set the clipboardText to "Test clipboard"'
        ], "Test setting clipboard text with both variables and literals");

        // Test replaceRegex function - basic usage
        await this.runTest("replaceRegex basic test", [
            'put replaceRegex("a+", "X", "aaa bbb aaa")',
            'put replaceRegex("\\d+", "#", "abc123def456")',
            'put replaceRegex("\\s+", "-", "hello  world   test")'
        ], "Test basic regex replacement with literal strings");
        // Expected results:
        // 1. "X bbb X" (replaces all 'a' sequences with 'X')
        // 2. "abc#def#" (replaces all digit sequences with '#')
        // 3. "hello-world-test" (replaces all whitespace sequences with '-')

        // Test replaceRegex function - with variables
        await this.runTest("replaceRegex with variables test", [
            'put "[a-z]+" into pattern',
            'put "#" into replacement',
            'put "abc123def456" into text',
            'put replaceRegex(pattern, replacement, text)'
        ], "Test regex replacement with variables");
        // Expected result: "#123#456" (replaces all lowercase letter sequences with '#')

        // Test replaceRegex function - with flags
        await this.runTest("replaceRegex with flags test", [
            'put replaceRegex("a", "X", "aAa", "i")',
            'put replaceRegex("a", "X", "aAa", "ignorecase")',
            'put replaceRegex("a", "X", "aAa", "noglobal")',
            'put replaceRegex("^.", "X", "abc\ndef", "m")'
        ], "Test regex replacement with different flags");
        // Expected results:
        // 1. "XXX" (case-insensitive matching with 'i' flag)
        // 2. "XXX" (case-insensitive matching with 'ignorecase' keyword)
        // 3. "XAa" (only replaces first match with 'noglobal')
        // 4. "Xbc\nXef" (multiline mode with 'm' flag)

        // Test replaceRegex function - edge cases
        await this.runTest("replaceRegex edge cases test", [
            'put replaceRegex("a", "X", "")',
            'put replaceRegex("a", "", "banana")',
            'try',
            '  put replaceRegex("[", "X", "abc")',
            'catch error',
            '  put "Error caught: " & error',
            'end try'
        ], "Test regex replacement edge cases");
        // Expected results:
        // 1. "" (empty string remains empty)
        // 2. "bnn" (all 'a' characters removed)
        // 3. Error message about invalid regex pattern

        // Test return character handling
        await this.runTest("return character test", [
            'put "this is " & return & "my text" into tVar',
            'put tVar',
            'put "A$B" into tVar2',
            'replace "$" with return in tVar2',
            'put tVar2',
            'put "line1" & return & "line2" & return & "line3" into tVar3',
            'put tVar3'
        ], "Test return character handling in strings");

        // Test replace functionality
        await this.runTest("replace test", [
            'put replace("hello world", "world", "there") into tResult',
            'put tResult',
            'put "hello world" into tVar',
            'replace "world" with "there" in tVar',
            'put tVar'
        ], "Test both forms of replace");

        // Test paragraphs
        await this.runTest("paragraphs test", [
            'put "this is a piece of text" & return & "another piece of text" into tParagraph',
            'put the number of paragraphs in tParagraph',
            'put "single paragraph" into tSingle',
            'put the number of paragraphs in tSingle',
            'put "p1" & return & "p2" & return & "p3" into tThree',
            'put the number of paragraphs in tThree'
        ], "Test paragraph counting");

        // Test sum function
        await this.runTest("Basic sum test", [
            'put sum(1,2,3,4) into tResult',
            'get tResult',
            'put it = 10'
        ], "Testing basic sum function");

        await this.runTest("Single number sum test", [
            'put sum(42) into tResult',
            'get tResult',
            'put it = 42'
        ], "Testing sum with single number");

        await this.runTest("Empty sum test", [
            'put sum() into tResult',
            'get tResult',
            'put it = 0'
        ], "Testing sum with empty list");

        await this.runTest("Variable sum test", [
            'put 5 into x',
            'put 3 into y',
            'put sum(x,y,2) into tResult',
            'get tResult',
            'put it = 10'
        ], "Testing sum with variables");

        // Test repeat with
        await this.runTest("Repeat with test", [
            'repeat with tVar = 1 to 5',
            'put tVar',
            'end repeat'
        ], "Testing repeat with loop");

        // howmany function tests
        await this.runTest("howmany with literal strings test", 
            'put howmany("thing","this text contains something")',
            "Testing howmany with literal strings"
        );
        
        await this.runTest("howmany with variables test", [
            'put "a" into tSearch',
            'put "A string containing a few a characters." into tString',
            'put howmany(tSearch,tString)'
        ], "Testing howmany with variables");

        // adding all test cases just above this line
        console.log("All tests completed!");
        return this.results;
    }

    displayResults(container) {
        // Create a summary section at the top
        const summaryDiv = document.createElement('div');
        summaryDiv.className = 'test-summary';
        summaryDiv.innerHTML = `
            <h2>Test Summary</h2>
            <div class="summary-stats">
                <div class="stat-item">
                    <span class="stat-label">Total Tests:</span>
                    <span class="stat-value">${this.testsPassed + this.testsFailed + this.testsSkipped}</span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Passed:</span>
                    <span class="stat-value passed">${this.testsPassed}</span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Failed:</span>
                    <span class="stat-value failed">${this.testsFailed}</span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Skipped:</span>
                    <span class="stat-value skipped">${this.testsSkipped}</span>
                </div>
            </div>
        `;
        container.innerHTML = '';
        container.appendChild(summaryDiv);

        // If there are failed tests, display them first
        if (this.testsFailed > 0) {
            const failedTests = this.results.filter(test => test.status === 'failed')
                .sort((a, b) => a.description.localeCompare(b.description));
            
            const failedSection = document.createElement('div');
            failedSection.className = 'failed-tests-section';
            failedSection.innerHTML = '<h2>Failed Tests</h2>';
            container.appendChild(failedSection);

            failedTests.forEach(test => {
                const testDiv = this.createTestResultElement(test);
                failedSection.appendChild(testDiv);
            });
        }

        // Display all tests
        const allTestsSection = document.createElement('div');
        allTestsSection.className = 'all-tests-section';
        allTestsSection.innerHTML = '<h2>All Tests</h2>';
        container.appendChild(allTestsSection);

        this.results.forEach(test => {
            const testDiv = this.createTestResultElement(test);
            allTestsSection.appendChild(testDiv);
        });
    }

    createTestResultElement(test) {
        const testDiv = document.createElement('div');
        testDiv.className = `test-result ${test.status}`;
        
        // Add skipped-test class for skipped tests to apply the orange border styling
        if (test.status === 'skipped') {
            testDiv.classList.add('skipped-test');
        }

        const headerDiv = document.createElement('div');
        headerDiv.className = 'test-header';
        headerDiv.innerHTML = `
            <span class="test-name">${test.description}</span>
            <span class="test-status">${test.status}</span>
        `;
        testDiv.appendChild(headerDiv);
        
        const scriptDiv = document.createElement('div');
        scriptDiv.className = 'test-script';
        scriptDiv.textContent = Array.isArray(test.script) ? test.script.join('\n') : test.script;
        testDiv.appendChild(scriptDiv);
        
        if (test.outputs.length > 0) {
            const outputsDiv = document.createElement('div');
            outputsDiv.className = 'test-output';
            test.outputs.forEach(output => {
                const outputLine = document.createElement('div');
                outputLine.className = 'output-line';
                outputLine.textContent = output;
                outputsDiv.appendChild(outputLine);
            });
            testDiv.appendChild(outputsDiv);
        }
        if (test.expectedOutput && test.status === 'failed') {
            const expectedDiv = document.createElement('div');
            expectedDiv.className = 'test-expected';
            expectedDiv.innerHTML = '<div class="output-header">Expected Output:</div>';
            const expectedLine = document.createElement('div');
            expectedLine.className = 'output-line';
            expectedLine.textContent = test.expectedOutput;
            expectedDiv.appendChild(expectedLine);
            testDiv.appendChild(expectedDiv);
        }
        
        if (test.error) {
            const errorDiv = document.createElement('div');
            errorDiv.className = 'test-error';
            errorDiv.textContent = test.error;
            testDiv.appendChild(errorDiv);
        }
        
        return testDiv;
    }
}
