
    /**
     * Updates the visual state of palette buttons based on the current mode
     * @param {string} mode - The current mode ('browse' or 'edit')
     */
    function updatePaletteMode(mode) {
        const browseBtn = document.getElementById('palette-browse-btn');
        const editBtn = document.getElementById('palette-edit-btn');
        
        if (!browseBtn || !editBtn) return;
        
        // Remove active class from both buttons
        browseBtn.classList.remove('active');
        editBtn.classList.remove('active');
        
        // Add active class to the appropriate button
        if (mode === 'browse') {
            browseBtn.classList.add('active');
        } else if (mode === 'edit') {
            editBtn.classList.add('active');
        }
    }
    
    function runPaletteCommand(script) {
        // Run the script directly through the webTalkApp interpreter
        // without affecting the message box
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            // Split the script into lines and run each line
            const lines = script.split('\n');
            
            // Create a temporary silent output handler
            const originalOutputHandler = window.webTalkApp.interpreter.outputHandler;
            window.webTalkApp.interpreter.outputHandler = () => {}; // Silent output
            
            // Run each line of the script
            (async function runLines() {
                for (const line of lines) {
                    if (line.trim()) {
                        try {
                            await window.webTalkApp.interpreter.interpret(line);
                        } catch (error) {
                            console.error('Palette command error:', error);
                            // Restore original output handler before showing error
                            window.webTalkApp.interpreter.outputHandler = originalOutputHandler;
                            window.webTalkApp.appendOutput(`Error: ${error.message}`, true);
                            return;
                        }
                    }
                }
                // Restore the original output handler after all lines are executed
                window.webTalkApp.interpreter.outputHandler = originalOutputHandler;
            })();
        } else if (window.interpreter && typeof window.interpreter.interpret === 'function') {
            // Fallback for direct interpreter access (less common)
            window.interpreter.interpret(script);
        } else {
            alert('Interpreter not available');
        }
    }
    
    
    // Global flag to prevent duplicate shape creation from palette
    window.allowPaletteGraphics = true;
    
    document.addEventListener('DOMContentLoaded', function() {
        const shapesBtn = document.getElementById('shapes-btn');
        const shapesPopup = document.getElementById('shapes-popup');
        const rotateBtn = document.getElementById('palette-rotate-btn');
        const editPalette = document.getElementById('edit-palette');
        
        // Track popup state with a variable for more reliability
        let isPopupVisible = false;
        
        // Track palette orientation (false = horizontal, true = vertical)
        let isVerticalPalette = false;
        
        // Restore palette orientation from localStorage if present
        const savedOrientation = localStorage.getItem('paletteOrientation');
        if (savedOrientation === 'vertical') {
            isVerticalPalette = true;
        }
        
        // Ensure initial orientation class so positioning is correct on first load
        if (editPalette) {
            if (isVerticalPalette) {
                editPalette.classList.add('vertical-palette');
                editPalette.classList.remove('horizontal-palette');
            } else {
                editPalette.classList.add('horizontal-palette');
                editPalette.classList.remove('vertical-palette');
            }
        }

        // Add click event for palette rotation
        rotateBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            togglePaletteOrientation();
        });
        
        // Toggle palette orientation between horizontal and vertical
        function togglePaletteOrientation() {
            isVerticalPalette = !isVerticalPalette;
            
            if (isVerticalPalette) {
                editPalette.classList.add('vertical-palette');
                editPalette.classList.remove('horizontal-palette');
            } else {
                editPalette.classList.add('horizontal-palette');
                editPalette.classList.remove('vertical-palette');
            }
            
            // Save orientation to localStorage
            localStorage.setItem('paletteOrientation', isVerticalPalette ? 'vertical' : 'horizontal');
            
            // Update stack container padding after orientation change
            updateStackContainerPadding();
            
            // Hide popup if it's visible during rotation
            if (isPopupVisible) {
                hideShapesPopup();
            }
        }
        
        // Function to update stack-container padding based on palette position
        function updateStackContainerPadding() {
            const stackContainer = document.getElementById('stack-container');
            if (!stackContainer || !editPalette) return;
            
            const paletteRect = editPalette.getBoundingClientRect();
            const windowWidth = window.innerWidth;
            const isNearLeftEdge = paletteRect.left < 10;
            const isNearRightEdge = paletteRect.right > (windowWidth - 10);
            
            // Handle left edge padding
            if (isVerticalPalette && isNearLeftEdge) {
                // Store original padding if not already stored
                if (!stackContainer.dataset.originalPaddingLeft) {
                    const computedStyle = window.getComputedStyle(stackContainer);
                    stackContainer.dataset.originalPaddingLeft = computedStyle.paddingLeft;
                }
                stackContainer.style.paddingLeft = '120px';
            } else {
                // Restore original left padding
                if (stackContainer.dataset.originalPaddingLeft) {
                    stackContainer.style.paddingLeft = stackContainer.dataset.originalPaddingLeft;
                }
            }
            
            // Handle right edge padding
            if (isVerticalPalette && isNearRightEdge) {
                // Store original padding if not already stored
                if (!stackContainer.dataset.originalPaddingRight) {
                    const computedStyle = window.getComputedStyle(stackContainer);
                    stackContainer.dataset.originalPaddingRight = computedStyle.paddingRight;
                }
                stackContainer.style.paddingRight = '120px';
            } else {
                // Restore original right padding
                if (stackContainer.dataset.originalPaddingRight) {
                    stackContainer.style.paddingRight = stackContainer.dataset.originalPaddingRight;
                }
            }
        }
        
        // Call initial padding update on page load
        updateStackContainerPadding();
        
        // Make the update function globally accessible so main.js can call it during drag
        window.updatePaletteStackPadding = updateStackContainerPadding;
        
        // Toggle shapes popup when the shapes button is clicked
        shapesBtn.addEventListener('click', function(e) {
            e.stopPropagation(); // Prevent document click from immediately closing it
            
            // Toggle visibility using our tracked state instead of checking style
            if (isPopupVisible) {
                hideShapesPopup();
            } else {
                showShapesPopup();
            }
        });
        
        // Hide popup when clicking outside
        document.addEventListener('click', function(e) {
            if (isPopupVisible && 
                !shapesPopup.contains(e.target) && 
                e.target !== shapesBtn) {
                hideShapesPopup();
            }
        });
        
        // Handle shape item clicks
        const shapeItems = document.querySelectorAll('.shape-item');
        shapeItems.forEach(item => {
            // Using once:true to prevent duplicate shape creation when clicking
            // This still allows scripting to create multiple objects if needed
            item.addEventListener('click', function() {
                const shape = this.getAttribute('data-shape');
                createGraphicShape(shape);
                hideShapesPopup();
                
                // Re-add the event listener for future use
                // This maintains the once:true behavior for each click but allows
                // the button to be used again after the current operation completes
                setTimeout(() => {
                    item.addEventListener('click', function() {
                        const shape = item.getAttribute('data-shape');
                        createGraphicShape(shape);
                        hideShapesPopup();
                    }, { once: true });
                }, 2000); // Small delay to prevent accidental double-clicks
            }, { once: true });
        });
        
        // Show shapes popup
        function showShapesPopup() {
            // Reset the flag to allow shape creation when popup is shown
            window.allowPaletteGraphics = true;
            
            shapesPopup.style.display = 'grid';
            shapesBtn.classList.add('active');
            isPopupVisible = true;
            
            // Get the edit-palette position
            const editPalette = document.getElementById('edit-palette');
            if (!editPalette) return;
            const paletteRect = editPalette.getBoundingClientRect();
            
            // Clear all positioning classes first
            shapesPopup.classList.remove('popup-above', 'popup-below', 'popup-left', 'popup-right');
            
            if (isVerticalPalette) {
                // Vertical palette - position popup to the right or left
                const windowWidth = window.innerWidth;
                
                // Reset vertical positioning styles for horizontal positioning
                shapesPopup.style.top = 'auto';
                shapesPopup.style.bottom = 'auto';
                shapesPopup.style.marginTop = '-60px';
                shapesPopup.style.marginBottom = '0';
                
                // Check if there's enough space on the right
                if (paletteRect.right + 300 < windowWidth) {
                    // Position popup to the right of the palette
                    shapesPopup.style.left = '100%';
                    shapesPopup.style.right = 'auto';
                    shapesPopup.style.marginLeft = '8px';
                    shapesPopup.style.marginRight = '0';
                    shapesPopup.classList.add('popup-right');
                } else {
                    // Position popup to the left of the palette
                    shapesPopup.style.right = '100%';
                    shapesPopup.style.left = 'auto';
                    shapesPopup.style.marginRight = '8px';
                    shapesPopup.style.marginLeft = '0';
                    shapesPopup.classList.add('popup-left');
                }
            } else {
                // Horizontal palette - position popup above or below
                // Reset horizontal positioning styles for vertical positioning
                shapesPopup.style.left = 'auto';
                shapesPopup.style.right = 'auto';
                shapesPopup.style.marginLeft = '0';
                shapesPopup.style.marginRight = '0';
                
                // Check if the palette is less than 300px from the top
                if (paletteRect.top < 300) {
                    // Position popup below the palette
                    shapesPopup.style.bottom = 'auto';
                    shapesPopup.style.top = '100%';
                    shapesPopup.style.marginBottom = '0';
                    shapesPopup.style.marginTop = '8px';
                    shapesPopup.classList.add('popup-below');
                } else {
                    // Position popup above the palette (default behavior)
                    shapesPopup.style.bottom = '100%';
                    shapesPopup.style.top = 'auto';
                    shapesPopup.style.marginBottom = '8px';
                    shapesPopup.style.marginTop = '0';
                    shapesPopup.classList.add('popup-above');
                }
            }
        }
        
        // Hide shapes popup
        function hideShapesPopup() {
            shapesPopup.style.display = 'none';
            shapesBtn.classList.remove('active');
            isPopupVisible = false;
        }
        
        // Create a graphic with the selected shape
        function createGraphicShape(shape) {
            // Check if we should allow shape creation from palette
            // If flag is false, don't create the shape (prevents duplicates)
            if (!window.allowPaletteGraphics) {
                console.log('Preventing duplicate shape creation from palette');
                return;
            }
            
            // Set flag to false to prevent duplicate creation
            window.allowPaletteGraphics = false;
            
            // Reset flag after a delay to allow future shape creation
            setTimeout(() => {
                window.allowPaletteGraphics = true;
            }, 1000);
            // Define scripts for each shape type
            const shapeScripts = {
                // Basic shapes
                'line': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "oval " & tNewGraphic into tName
                    create graphic tName
                    set the height of graphic tName to 10
                    set the width of graphic tname to 200
                    set the points of graphic tName to "M0,5 L200,5"
                `,
                'oval': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "oval " & tNewGraphic into tName
                    create oval graphic tName
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the fill of graphic tName to tNewColour
                    -- set the fill of graphic tName to transparent -- commented out
                `,
                'rectangle': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "rectangle " & tNewGraphic into tName
                    create rectangle graphic tName
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                    -- set the backgroundcolour of graphic tName to "" -- commented out
                `,
                'square': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "square " & tNewGraphic into tName
                    create square graphic tName
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                    -- set the backgroundcolour of graphic tName to "" -- commented out
                `,
                'roundrect': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "roundedSquare " & tNewGraphic into tName
                    create graphic tName
                    set the points of graphic tName to "M10,0 L190,0 Q200,0 200,10 L200,190 Q200,200 190,200 L10,200 Q0,200 0,190 L0,10 Q0,0 10,0 Z"
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                    -- set the backgroundcolour of graphic tName to "" -- commented out
                `,
                'triangle': `
                    put randomstring(6) into tNewGraphic -- pick random name
                    put "triangle " & tNewGraphic into tName
                    create triangle graphic tName
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                    -- set the backgroundcolour of graphic tName to "" -- commented out
                `,
                
                // Arrows
                'arrow-right': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-right " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 180
                    set the points of graphic tName to "M78.320755,68.763158 L78.320755,22.921053 L97.900943,22.921053 L156.641509,91.684211 L97.900943,160.447368 L78.320755,160.447368 L78.320755,114.605263 L-0.000002,114.605263 L-0.000003,68.763158 L78.320755,68.763158 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-left': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-left " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 180
                    set the points of graphic tName to "M77.483602,68.973684 V22.991228 h-18.882075 L1.9553,91.964912 58.601527,160.938596 h18.882075 v-45.982456 h75.528307 l0,-45.982456 z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-up': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-up " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 180
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M66.521474,66.935633 L22.173825,66.935633 L22.173825,50.201725 L88.695299,0.000005 L155.216774,50.201725 L155.216774,66.935633 L110.869124,66.935633 L110.869124,133.871268 L66.521474,133.871268 L66.521474,66.935633 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-down': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-down " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 180
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M65.165094,64.561408 H21.721698 V80.701762 L86.886792,129.122807 152.051887,80.701762 V64.561408 H108.608491 V0.000004 H65.165094 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-up-right': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-up-right " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M135.38,19.25 H53.18529 L38.68029,33.6875 L66.102283,60.981979 L5.664831,121.138325 L33.015701,148.361675 L93.453201,88.205521 L120.875,115.5 L135.38,101.0625 L135.38,19.25 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-up-left': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-up-left " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M19.79,19.79 V103.8975 L34.6325,118.74 L62.692642,90.679858 L124.536491,152.523509 L152.523509,124.536491 L90.679858,62.692642 L118.74,34.6325 L103.8975,19.79 H19.79 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-down-right': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-down-right " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M97.945755,69.281847 L34.60089,5.936931 L5.934762,34.603018 L69.279617,97.947883 L40.540203,126.6875 L55.742703,141.89 H141.89 V55.742399 L126.6875,40.539899 L97.945755,69.281847 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-down-left': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-down-left " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 160
                    set the height of graphic tName to 160
                    set the points of graphic tName to "M92.741805,97.801805 L155.991704,34.551805 L127.368296,5.928144 L64.118195,69.178094 L35.42,40.48 L20.24,55.66 V141.68 H106.26 L121.44,126.5 L92.741805,97.801805 Z"
                    set the linesize of graphic tName to 3
                    set the fill of graphic tName to red
                `,
                'arrow-sort-az': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-sort-az " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 48
                    set the height of graphic tName to 48
                    set the points of graphic tName to "M11.095,4.755 V33.285 M11.095,4.755 L17.435,11.095 M11.095,4.755 L4.755,11.095 M24.5675,22.19 H32.4925 L24.5675,33.285 H32.4925 M25.36,14.265 H31.7 M23.775,15.85 L28.53,4.755 L33.285,15.85"
                    set the linesize of graphic tName to 2
                `,
                'arrow-sort-decending': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-sort-decending " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 48
                    set the height of graphic tName to 48
                    set the points of graphic tName to "M10.535,4.515 V31.605 M10.535,4.515 L16.555,10.535 M10.535,4.515 L4.515,10.535 M21.07,4.515 H31.605 M21.07,13.545 H28.595 M21.07,22.575 H25.585 M21.07,31.605 H22.575"
                    set the linesize of graphic tName to 2
                `,
                'arrow-sort-ascending': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-sort-ascending " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 48
                    set the height of graphic tName to 48
                    set the points of graphic tName to "M10.465,4.485 V31.395 M10.465,4.485 L16.445,10.465 M10.465,4.485 L4.485,10.465 M20.93,4.485 H22.425 M20.93,13.455 H25.415 M20.93,22.425 H28.405 M20.93,31.395 H31.395"
                    set the linesize of graphic tName to 2
                `,
                'arrow-reload': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrow-reload " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 36
                    set the height of graphic tName to 36
                    set the points of graphic tName to "M20.09,22.96 H27.265 V30.135 M14.35,11.48 H7.175 V4.305 M27.864256,12.919879 C27.059652,10.928472 25.712473,9.202712 23.97598,7.93875 C22.239487,6.674773 20.183562,5.922991 18.041251,5.769432 C15.898939,5.615873 13.755996,6.066319 11.856975,7.069685 C9.957939,8.073052 8.378707,9.589445 7.298324,11.445732 M6.575213,21.520838 C7.379803,23.512188 8.726895,25.237919 10.463403,26.501867 C12.199925,27.765958 14.257988,28.516751 16.400328,28.670295 C18.542639,28.823841 20.683803,28.373538 22.582739,27.370042 C24.481818,26.36669 26.06003,24.850613 27.140299,22.994297"
                    set the linesize of graphic tName to 2
                `,
                // Special shapes
                'star-5-point': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "star-5-point " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 128
                    set the height of graphic tName to 128
                    set the points of graphic tName to "M60.796794,23.798154 C62.045737,20.996099 62.670478,19.595072 63.540152,19.163633 C64.295452,18.788789 65.18239,18.788789 65.93769,19.163633 C66.807364,19.595072 67.432105,20.996099 68.681047,23.798154 L78.629427,46.116892 C78.998985,46.94524 79.183494,47.359414 79.469429,47.676586 C79.721915,47.957234 80.031048,48.18172 80.375789,48.33537 C80.765847,48.509089 81.216869,48.556672 82.118913,48.65184 L106.419612,51.216677 C109.470484,51.538651 110.995651,51.699638 111.674881,52.393381 C112.264555,52.995948 112.538621,53.839726 112.415615,54.674009 C112.274266,55.634319 111.134842,56.660987 108.855994,58.714864 L90.703437,75.073583 C90.030141,75.680521 89.692954,75.984259 89.479851,76.353816 C89.291026,76.681293 89.172876,77.044376 89.133492,77.419868 C89.088714,77.844455 89.182587,78.287924 89.370872,79.175401 L94.441093,103.079028 C95.077703,106.080266 95.396008,107.580616 94.946065,108.440579 C94.554928,109.188326 93.837393,109.709483 93.006023,109.850292 C92.048951,110.012142 90.720162,109.246053 88.062585,107.713333 L66.895302,95.504448 C66.10979,95.051268 65.717034,94.825217 65.299461,94.7362 C64.929904,94.657972 64.547938,94.657972 64.17838,94.7362 C63.760807,94.825217 63.368051,95.051268 62.582539,95.504448 L41.415365,107.713333 C38.757896,109.246053 37.429161,110.012142 36.472034,109.850292 C35.640611,109.709483 34.92286,109.188326 34.532046,108.440579 C34.082157,107.580616 34.400408,106.080266 35.036964,103.079028 L40.106969,79.175401 C40.295147,78.287924 40.389236,77.844455 40.344565,77.419868 C40.30502,77.044376 40.187031,76.681293 39.998206,76.353816 C39.784834,75.984259 39.44797,75.680521 38.774297,75.073583 L20.622172,58.714864 C18.34327,56.660987 17.203792,55.634319 17.062119,54.674009 C16.939113,53.839726 17.213233,52.995948 17.803122,52.393381 C18.482245,51.699638 20.007681,51.538651 23.058554,51.216677 L47.359198,48.65184 C48.261134,48.556672 48.712048,48.509089 49.101999,48.33537 C49.447009,48.18172 49.755927,47.957234 50.008737,47.676586 C50.294402,47.359414 50.479019,46.94524 50.848307,46.116892 L60.796794,23.798154 Z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to Yellow
                `,
                'star-6-point': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "star-6-point " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 164
                    set the height of graphic tName to 164
                    set the points of graphic tName to "M78.429582,25.029982 C79.99299,22.481197 80.774694,21.20677 81.780237,20.771913 C82.657575,20.392218 83.653416,20.392218 84.531447,20.771913 C85.536297,21.20677 86.318001,22.481197 87.881409,25.029982 L101.637459,47.457541 C102.226509,48.418663 102.521727,48.899189 102.923667,49.248669 C103.279176,49.557885 103.694283,49.791357 104.143347,49.9346 C104.651316,50.096485 105.214725,50.098633 106.342236,50.10293 L133.404579,50.205355 C136.602081,50.217413 138.201525,50.223512 139.113513,50.897662 C139.909077,51.485534 140.414967,52.385325 140.504364,53.370355 C140.607621,54.499945 139.782951,55.86966 138.132918,58.609228 L125.071947,80.299989 C124.445475,81.339489 124.132932,81.859239 124.010271,82.414332 C123.902163,82.905669 123.902163,83.414331 124.010271,83.904975 C124.132932,84.460068 124.445475,84.979818 125.071947,86.020011 L138.132918,107.710218 C139.782951,110.449647 140.607621,111.819708 140.504364,112.949298 C140.414967,113.934051 139.909077,114.834258 139.113513,115.421922 C138.201525,116.096211 136.602081,116.102448 133.404579,116.114229 L106.342236,116.216793 C105.214725,116.220951 104.651316,116.22303 104.143347,116.385192 C103.694283,116.52795 103.279176,116.761491 102.923667,117.071262 C102.521727,117.420534 102.226509,117.900783 101.637459,118.861974 L87.881409,141.289533 C86.318001,143.838387 85.536297,145.112814 84.531447,145.548018 C83.653416,145.927089 82.657575,145.927089 81.780237,145.548018 C80.774694,145.112814 79.99299,143.838387 78.429582,141.289533 L64.673948,118.861974 C64.084482,117.900783 63.789749,117.420534 63.38774,117.071262 C63.031954,116.761491 62.616847,116.52795 62.167713,116.385192 C61.660299,116.22303 61.096543,116.220951 59.969032,116.216793 L32.906689,116.114229 C29.708841,116.102448 28.109951,116.096211 27.197824,115.421922 C26.402399,114.834258 25.896163,113.934051 25.806558,112.949298 C25.703855,111.819708 26.528664,110.449647 28.178281,107.710218 L41.239529,86.020011 C41.865516,84.979818 42.178544,84.460068 42.300789,83.904975 C42.408967,83.414331 42.408967,82.905669 42.300789,82.414332 C42.178544,81.859239 41.865516,81.339489 41.239529,80.299989 L28.178281,58.609158 C26.528664,55.86966 25.703855,54.499945 25.806558,53.370355 C25.896163,52.385325 26.402399,51.485534 27.197824,50.897662 C28.109951,50.223512 29.708841,50.217413 32.906689,50.205355 L59.969032,50.10293 C61.096543,50.098633 61.660299,50.096485 62.167713,49.9346 C62.616847,49.791357 63.031954,49.557885 63.38774,49.248669 C63.789749,48.899189 64.084482,48.418663 64.673948,47.457541 L78.429582,25.029982 Z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to Yellow
                `,
                'burst': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "burst " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 168
                    set the height of graphic tName to 152
                    set the points of graphic tName to "M135.77704,76.28614 C150.06789,71.57528 158.76343,66.90166 158.13035,63.45696 157.49727,60.01226 147.72177,58.52266 132.65819,58.89506 144.43534,49.67816 150.97096,42.3605 149.17413,39.33475 147.41454,36.309 137.68559,38.20824 123.64611,43.59873 131.52237,30.99299 135.09741,21.93436 132.37889,19.68134 129.66037,17.42832 121.17896,22.46503 109.88593,32.26846 112.86513,17.77279 113.06995,8.04384 109.71835,6.85216 106.39468,5.66048 100.18491,13.23882 92.98828,26.21696 90.74457,11.60026 87.53262,2.37405 83.98551,2.37405 80.41978,2.37405 77.21714,11.59095 74.98274,26.21696 67.79542,13.22951 61.58565,5.65117 58.22474,6.83354 54.87314,8.03453 55.08727,17.76348 58.07578,32.24984 46.79206,22.46503 38.30134,17.42832 35.57351,19.66272 32.84568,21.89712 36.43003,30.98368 44.29698,43.58942 30.26681,38.18962 20.52855,36.29969 18.75965,39.32544 16.98144,42.35119 23.50775,49.66885 35.30352,58.89506 20.23063,58.52266 10.4272,60.00295 9.81274,63.45696 9.18897,66.88304 17.88451,71.57528 32.15674,76.28614 17.88451,80.997 9.18897,85.68924 9.81274,89.12463 c0.62377,3.43539 10.41789,4.9343 25.49078,4.55259 -11.80508,9.20759 -18.35001,16.51594 -16.54387,19.56031 1.7689,3.03506 11.50716,1.14513 25.51871,-4.26398 -7.86695,12.61505 -11.44199,21.68299 -8.72347,23.93601 2.74645,2.25302 11.21855,-2.80231 22.50227,-12.57781 -2.98851,14.48636 -3.19333,24.21531 0.15827,25.40699 3.34229,1.20099 9.57068,-6.38666 16.76731,-19.38342 2.2344,14.6167 5.43704,23.84291 9.00277,23.84291 3.54711,0 6.75906,-9.2169 9.00277,-23.84291 7.18732,12.98745 13.4064,20.56579 16.74869,19.37411 3.3516,-1.19168 3.14678,-10.92063 0.14896,-25.40699 11.29303,9.78481 19.77444,14.83083 22.49296,12.57781 2.74645,-2.25302 -0.8379,-11.32096 -8.71416,-23.91739 14.02086,5.40911 23.74981,7.28042 25.52802,4.26398 1.77821,-3.03506 -4.76672,-10.34341 -16.52525,-19.56031 15.05427,0.3724 24.83908,-1.1172 25.46285,-4.57121 C158.76343,85.68924 150.06789,80.997 135.77704,76.28614 Z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to Yellow
                `,
                'speech-bubble-square-left': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "speech-bubble-square-left " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 158
                    set the height of graphic tName to 146
                    set the points of graphic tName to "M36.3,132.87615 V102.85 H31.7625 C20.908729,102.836666 12.113334,94.041273 12.1,83.1875 V31.7625 C12.113334,20.908729 20.908729,12.113334 31.7625,12.1 h87.725 c10.853773,0.01331 19.649166,8.808729 19.6625,19.6625 v51.425 c-0.01331,10.853773 -8.808727,19.649166 -19.6625,19.6625 h-41.9023 z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to white
                `,
                'speech-bubble-square-right': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "speech-bubble-square-right " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 158
                    set the height of graphic tName to 146
                    set the points of graphic tName to "M115.045,132.985965 V102.935 h4.54125 c10.862743,-0.013321 19.665405,-8.816007 19.67875,-19.67875 V31.78875 C139.251655,20.926009 130.448993,12.123345 119.58625,12.11 H31.78875 C20.926007,12.123321 12.123345,20.926009 12.11,31.78875 v51.4675 c0.013321,10.862743 8.816007,19.665405 19.67875,19.67875 h41.93693 z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to white
                `,
                'diamond': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "warning " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 80
                    set the height of graphic tName to 80
                    set the points of graphic tName to "M35.07,80.16 H45.09 L80.16,45.09 V35.07 L45.09,0 H35.07 L0,35.07 V45.09 L35.07,80.16 Z"
                    set the linesize of graphic tName to 2
                    set the fill of graphic tName to cyan
                `,
                'pentagon': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "pentagon " & tNewGraphic into tName
                    create pentagon graphic tName
                    set the width of graphic tName to 200
                    set the height of graphic tName to 200
                    set the linesize of graphic tName to 2
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                `,
                'hexagon': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "hexagon " & tNewGraphic into tName
                    create hexagon graphic tName
                    set the width of graphic tName to 200
                    set the height of graphic tName to 200
                    set the linesize of graphic tName to 2
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                `,
                'octagon': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "octagon " & tNewGraphic into tName
                    create octagon graphic tName
                    set the width of graphic tName to 200
                    set the height of graphic tName to 200
                    set the linesize of graphic tName to 2
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                `,
                'decagon': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "decagon " & tNewGraphic into tName
                    create decagon graphic tName
                    set the width of graphic tName to 200
                    set the height of graphic tName to 200
                    set the linesize of graphic tName to 2
                    put random(255) into tRGBa
                    put random(255) into tRGBb
                    put random(255) into tRGBc
                    put tRGBa & "," & tRGBb & "," & tRGBc into tNewColour
                    set the backgroundcolour of graphic tname to tNewColour
                `,
                'arrows-vert': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrows-vert " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 38
                    set the height of graphic tName to 38
                    set the points of graphic tName to "M22.005,24.45 H31.785 V26.895 L19.56,39.12 L7.335,26.895 L7.335,24.45 H17.115 L17.114976,14.67 L7.335,14.67 L7.335,12.225 L19.56,0 L31.785,12.225 V14.67 L22.005,14.67 L22.005,24.45 Z"
                    set the linesize of graphic tName to 0
                    set the fill of graphic tName to black
                `,
                'circle-neg': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "Circle-neg " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 22
                    set the height of graphic tName to 22
                    set the points of graphic tName to "M8 12H16M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z"
                `,
                'circle-pos': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "Circle-pos " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 22
                    set the height of graphic tName to 22
                    set the points of graphic tName to "M8 12H16M12 8V16M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z"
                `,
                'circle-cross': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "Circle-cross " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 22
                    set the height of graphic tName to 22
                    set the points of graphic tName to "M9 9L15 15M15 9L9 15M21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3C16.9706 3 21 7.02944 21 12Z"
                `,
                'arrows-horz': `
                    put the number of graphics of this card into tNewGraphic
                    add 1 to tNewGraphic
                    put "arrows-horz " & tNewGraphic into tName
                    create graphic tName
                    set the width of graphic tName to 38
                    set the height of graphic tName to 38
                    set the points of graphic tName to "M13.17,19.755 L13.17,28.535 L10.975,28.535 L-0.000007,17.56 L10.975,6.585 L13.17,6.585 L13.17,15.365 L21.95,15.365 L21.95,6.585 L24.145,6.585 L35.12,17.56 L24.145,28.535 L21.95,28.535 L21.95,19.755 L13.17,19.755 Z"
                    set the linesize of graphic tName to 0
                    set the fill of graphic tName to black
                `
            };
            
            // Get the script for this shape, or use a default if not defined
            const script = shapeScripts[shape] || `
                put the number of graphics of this card into tNewGraphic
                add 1 to tNewGraphic
                put "${shape} " & tNewGraphic into tName
                create ${shape} graphic tName
            `;
            
            // Run the script silently
            runPaletteCommand(script);
        }
    });
