/**
 * objectscaler-calc.js
 * Provides utilities for scaling SVG paths while maintaining proportions
 */

const ObjectScaler = {
    /**
     * Parse an SVG path string into a structured array of commands
     * @param {string} pathString - The SVG path data string (e.g., "M10,0 L190,0 Q200,0 200,10...")
     * @returns {Array} Array of parsed path commands
     */
    parsePath: function(pathString) {
        if (!pathString) return [];
        
        // Regular expression to match SVG path commands with their parameters
        const pathRegex = /([MLHVCSQTAZmlhvcsqtaz])([^MLHVCSQTAZmlhvcsqtaz]*)/g;
        const commands = [];
        let match;
        
        while ((match = pathRegex.exec(pathString)) !== null) {
            const command = match[1];
            const params = match[2].trim()
                .split(/[\s,]+/)
                .filter(param => param !== '')
                .map(param => parseFloat(param));
            
            commands.push({ command, params });
        }
        
        return commands;
    },
    
    /**
     * Detect if a path represents a rounded rectangle
     * @param {Array} commands - Array of parsed path commands
     * @returns {boolean} True if the path is a rounded rectangle
     */
    isRoundedRectangle: function(commands) {
        // A rounded rectangle typically follows this pattern:
        // M x,y L x,y Q x,y x,y L x,y Q x,y x,y L x,y Q x,y x,y L x,y Q x,y x,y Z
        
        if (commands.length !== 9 && commands.length !== 10) return false;
        
        // Check if the last command is Z (path closure)
        const hasClosingZ = commands.length === 10 && commands[9].command.toUpperCase() === 'Z';
        
        // If we have 10 commands, the last one should be Z
        if (commands.length === 10 && !hasClosingZ) return false;
        
        // Check the pattern of the first 9 commands
        const expectedCommands = ['M', 'L', 'Q', 'L', 'Q', 'L', 'Q', 'L', 'Q'];
        for (let i = 0; i < 9; i++) {
            if (commands[i].command.toUpperCase() !== expectedCommands[i]) {
                return false;
            }
        }
        
        return true;
    },
    
    /**
     * Extract dimensions and corner radius from a rounded rectangle path
     * @param {Array} commands - Array of parsed path commands
     * @returns {Object} Dimensions and radius {width, height, radius}
     */
    getRoundedRectDimensions: function(commands) {
        if (!this.isRoundedRectangle(commands)) {
            return null;
        }
        
        // For a standard rounded rectangle:
        // M topLeft.x,0 L topRight.x-radius,0 Q topRight.x,0 topRight.x,radius ...
        
        // Extract width from the right-most X coordinate
        const width = Math.max(
            commands[2].params[0], // Q control point X (top right)
            commands[4].params[0]  // Q control point X (bottom right)
        );
        
        // Extract height from the bottom-most Y coordinate
        const height = Math.max(
            commands[4].params[1], // Q control point Y (bottom right)
            commands[6].params[1]  // Q control point Y (bottom left)
        );
        
        // Calculate radius (difference between L endpoint and Q control point)
        const radiusX = width - commands[1].params[0]; // topRight.x - (topRight.x - radius)
        
        return {
            width: width,
            height: height,
            radius: radiusX
        };
    },
    
    /**
     * Create a new rounded rectangle path with specified dimensions and radius
     * @param {number} width - Width of the rectangle
     * @param {number} height - Height of the rectangle
     * @param {number} radius - Corner radius
     * @returns {string} SVG path string for the rounded rectangle
     */
    createRoundedRectPath: function(width, height, radius) {
        // Ensure radius isn't too large for the rectangle
        const maxRadius = Math.min(width / 2, height / 2);
        const r = Math.min(radius, maxRadius);
        
        // Create the path
        return [
            `M${r},0`,                              // Start at top-left corner + radius
            `L${width - r},0`,                      // Line to top-right corner - radius
            `Q${width},0 ${width},${r}`,            // Top-right corner curve
            `L${width},${height - r}`,              // Line to bottom-right corner - radius
            `Q${width},${height} ${width - r},${height}`, // Bottom-right corner curve
            `L${r},${height}`,                      // Line to bottom-left corner + radius
            `Q0,${height} 0,${height - r}`,         // Bottom-left corner curve
            `L0,${r}`,                              // Line to top-left corner + radius
            `Q0,0 ${r},0`,                          // Top-left corner curve
            'Z'                                     // Close path
        ].join(' ');
    },
    
    /**
     * Scale a path based on original points and new dimensions
     * @param {string} originalPoints - Original SVG path string
     * @param {number} newWidth - New width
     * @param {number} newHeight - New height
     * @param {number} originalWidth - Original width
     * @param {number} originalHeight - Original height
     * @returns {string} Scaled SVG path string
     */
    scalePath: function(originalPoints, newWidth, newHeight, originalWidth, originalHeight) {
        // Ensure we have valid inputs
        if (!originalPoints || !newWidth || !newHeight || !originalWidth || !originalHeight) {
            return originalPoints; // Return original if invalid inputs
        }
        
        // Parse the original path
        const commands = this.parsePath(originalPoints);
        
        // Check if this is a rounded rectangle and use specialized scaling
        if (this.isRoundedRectangle(commands)) {
            const dimensions = this.getRoundedRectDimensions(commands);
            if (dimensions) {
                // Calculate new radius proportionally
                const radiusScale = Math.min(
                    newWidth / originalWidth,
                    newHeight / originalHeight
                );
                const newRadius = dimensions.radius * radiusScale;
                
                // Create a new rounded rectangle with the new dimensions and radius
                return this.createRoundedRectPath(newWidth, newHeight, newRadius);
            }
        }
        
        // For other path types, scale each command
        return this.scaleGeneralPath(originalPoints, newWidth, newHeight, originalWidth, originalHeight);
    },
    
    /**
     * Scale a general SVG path
     * @param {string} pathString - Original SVG path string
     * @param {number} newWidth - New width
     * @param {number} newHeight - New height
     * @param {number} originalWidth - Original width
     * @param {number} originalHeight - Original height
     * @returns {string} Scaled SVG path string
     */
    scaleGeneralPath: function(pathString, newWidth, newHeight, originalWidth, originalHeight) {
        // Calculate scaling factors
        const scaleX = newWidth / originalWidth;
        const scaleY = newHeight / originalHeight;
        
        // Parse the path
        const commands = this.parsePath(pathString);
        
        // Scale the commands
        const scaledCommands = this.scalePathCommands(commands, scaleX, scaleY);
        
        // Convert back to a path string
        return this.commandsToPathString(scaledCommands);
    },
    
    /**
     * Scale parsed path commands by the given factors
     * @param {Array} commands - Array of parsed path commands
     * @param {number} scaleX - X-axis scale factor
     * @param {number} scaleY - Y-axis scale factor
     * @returns {Array} Scaled path commands
     */
    scalePathCommands: function(commands, scaleX, scaleY) {
        return commands.map(cmd => {
            const scaledCmd = { command: cmd.command, params: [] };
            
            switch (cmd.command) {
                case 'M': // Move to (absolute)
                case 'L': // Line to (absolute)
                case 'C': // Cubic Bézier curve (absolute)
                case 'S': // Smooth cubic Bézier curve (absolute)
                case 'Q': // Quadratic Bézier curve (absolute)
                case 'T': // Smooth quadratic Bézier curve (absolute)
                    // For absolute commands, scale all coordinates
                    for (let i = 0; i < cmd.params.length; i += 2) {
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                        if (i + 1 < cmd.params.length) {
                            scaledCmd.params.push(cmd.params[i + 1] * scaleY);
                        }
                    }
                    break;
                    
                case 'H': // Horizontal line to (absolute)
                    // Scale X coordinates
                    for (let i = 0; i < cmd.params.length; i++) {
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                    }
                    break;
                    
                case 'V': // Vertical line to (absolute)
                    // Scale Y coordinates
                    for (let i = 0; i < cmd.params.length; i++) {
                        scaledCmd.params.push(cmd.params[i] * scaleY);
                    }
                    break;
                    
                case 'A': // Elliptical Arc (absolute)
                    // Format: rx, ry, x-axis-rotation, large-arc-flag, sweep-flag, x, y
                    for (let i = 0; i < cmd.params.length; i += 7) {
                        // Radii
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                        scaledCmd.params.push(cmd.params[i + 1] * scaleY);
                        
                        // x-axis-rotation, large-arc-flag, sweep-flag (unchanged)
                        scaledCmd.params.push(cmd.params[i + 2]);
                        scaledCmd.params.push(cmd.params[i + 3]);
                        scaledCmd.params.push(cmd.params[i + 4]);
                        
                        // End point
                        scaledCmd.params.push(cmd.params[i + 5] * scaleX);
                        scaledCmd.params.push(cmd.params[i + 6] * scaleY);
                    }
                    break;
                    
                case 'Z': // Close path (no parameters)
                case 'z':
                    // No parameters to scale
                    break;
                    
                // Relative commands
                case 'm': // Move to (relative)
                case 'l': // Line to (relative)
                case 'q': // Quadratic Bézier curve (relative)
                case 't': // Smooth quadratic Bézier curve (relative)
                    // For relative commands, scale the distances
                    for (let i = 0; i < cmd.params.length; i += 2) {
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                        if (i + 1 < cmd.params.length) {
                            scaledCmd.params.push(cmd.params[i + 1] * scaleY);
                        }
                    }
                    break;
                    
                case 'h': // Horizontal line to (relative)
                    // Scale X distances
                    for (let i = 0; i < cmd.params.length; i++) {
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                    }
                    break;
                    
                case 'v': // Vertical line to (relative)
                    // Scale Y distances
                    for (let i = 0; i < cmd.params.length; i++) {
                        scaledCmd.params.push(cmd.params[i] * scaleY);
                    }
                    break;
                    
                case 'c': // Cubic Bézier curve (relative)
                case 's': // Smooth cubic Bézier curve (relative)
                    // For relative commands, scale the distances
                    for (let i = 0; i < cmd.params.length; i += 2) {
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                        if (i + 1 < cmd.params.length) {
                            scaledCmd.params.push(cmd.params[i + 1] * scaleY);
                        }
                    }
                    break;
                    
                case 'a': // Elliptical Arc (relative)
                    // Format: rx, ry, x-axis-rotation, large-arc-flag, sweep-flag, dx, dy
                    for (let i = 0; i < cmd.params.length; i += 7) {
                        // Radii
                        scaledCmd.params.push(cmd.params[i] * scaleX);
                        scaledCmd.params.push(cmd.params[i + 1] * scaleY);
                        
                        // x-axis-rotation, large-arc-flag, sweep-flag (unchanged)
                        scaledCmd.params.push(cmd.params[i + 2]);
                        scaledCmd.params.push(cmd.params[i + 3]);
                        scaledCmd.params.push(cmd.params[i + 4]);
                        
                        // Relative end point
                        scaledCmd.params.push(cmd.params[i + 5] * scaleX);
                        scaledCmd.params.push(cmd.params[i + 6] * scaleY);
                    }
                    break;
            }
            
            return scaledCmd;
        });
    },
    
    /**
     * Convert scaled path commands back to an SVG path string
     * @param {Array} commands - Array of path commands
     * @returns {string} SVG path string
     */
    commandsToPathString: function(commands) {
        return commands.map(cmd => {
            // For each command, format the parameters appropriately
            let paramString = '';
            
            switch (cmd.command) {
                case 'A':
                case 'a':
                    // Special handling for arc commands to ensure proper formatting
                    for (let i = 0; i < cmd.params.length; i += 7) {
                        if (i > 0) paramString += ' ';
                        
                        // rx, ry
                        paramString += `${cmd.params[i]},${cmd.params[i + 1]} `;
                        
                        // x-axis-rotation
                        paramString += `${cmd.params[i + 2]} `;
                        
                        // large-arc-flag, sweep-flag
                        paramString += `${cmd.params[i + 3]},${cmd.params[i + 4]} `;
                        
                        // end point
                        paramString += `${cmd.params[i + 5]},${cmd.params[i + 6]}`;
                    }
                    break;
                    
                default:
                    // For most commands, parameters are pairs or single values
                    for (let i = 0; i < cmd.params.length; i++) {
                        if (i > 0 && (i % 2 === 0)) {
                            // Add space between coordinate pairs
                            paramString += ' ';
                        } else if (i > 0) {
                            // Add comma between x and y in a pair
                            paramString += ',';
                        }
                        
                        // Round to 6 decimal places to avoid floating point issues
                        paramString += parseFloat(cmd.params[i].toFixed(6));
                    }
            }
            
            return cmd.command + paramString;
        }).join(' ');
    }
};

// Export the ObjectScaler object
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ObjectScaler;
} else {
    window.ObjectScaler = ObjectScaler;
}
