class WebTalkObjects {
    static objects = new Map();
    static objectsById = new Map(); // Map object IDs to names
    static nextId = 1;
    static scripts = new Map(); // Store scripts for objects
    static customProperties = new Map(); // Store custom properties for objects
    static selectedObject = null; // Track the currently selected object
    static highestLayer = 1; // Track the highest layer value for z-index ordering
    static isPhysicsLoopRunning = false; // Flag to track if the physics loop is running
    static lastPhysicsFrameTime = 0; // Last timestamp for physics calculations
    static collisionRate = 60; // Default collision check rate in milliseconds
    static lastCollisionCheckTime = 0; // Last timestamp for collision checks
    static lastObject = { type: null, name: null }; // Track the last created object

    // Update button appearance based on properties
    static updateButtonAppearance(button, buttonName) {
        if (!button || !buttonName || !this.customProperties.has(buttonName)) {
            return;
        }
        
        const properties = this.customProperties.get(buttonName);
        const isCheckbox = properties.get('isCheckbox') === true;
        const isDisabled = properties.get('disabled') === true;
        // Get the original button name from the dataset to avoid accumulating tick symbols
        const originalText = button.dataset.name;
        
        // Clear existing content
        button.innerHTML = '';
        
        if (isCheckbox) {
            // Create checkbox appearance
            const checkboxContainer = document.createElement('div');
            checkboxContainer.className = 'checkbox-container';
            
            // Create the checkbox element
            const checkbox = document.createElement('div');
            checkbox.className = 'checkbox';
            
            // Create the checkmark element (initially hidden)
            const checkmark = document.createElement('div');
            checkmark.className = 'checkmark';
            checkmark.innerHTML = '✓';
            
            // Add checked state if needed
            // For checkbox buttons, we use the hilighted property to determine checked state
            if (properties.get('hilighted') === true) {
                checkbox.classList.add('checked');
                checkmark.style.display = 'block';
            } else {
                checkbox.classList.remove('checked');
                checkmark.style.display = 'none';
            }
            
            // Add checkbox and checkmark to container
            checkbox.appendChild(checkmark);
            checkboxContainer.appendChild(checkbox);
            
            // Add label text
            const label = document.createElement('span');
            label.className = 'checkbox-label';
            label.textContent = originalText;
            checkboxContainer.appendChild(label);
            
            // Add container to button
            button.appendChild(checkboxContainer);
            
            // Apply checkbox styling
            button.classList.add('checkbox-button');
        } else {
            // Standard button appearance
            button.textContent = originalText;
            button.classList.remove('checkbox-button');
            
            // Apply inner glow effect if hilighted is true
            if (properties.get('hilighted') === true) {
                button.classList.add('button-hilighted');
            } else {
                button.classList.remove('button-hilighted');
            }
        }
        
        // Apply disabled styling
        if (isDisabled) {
            button.classList.add('button-disabled');
            button.style.cursor = 'not-allowed';
        } else {
            button.classList.remove('button-disabled');
            button.style.cursor = 'pointer';
        }
        
        // Apply text alignment
        const textAlign = properties.get('textAlign') || 'center';
        if (isCheckbox) {
            // For checkbox buttons, apply alignment to the container
            button.style.justifyContent = textAlign === 'left' ? 'flex-start' : 
                                         textAlign === 'right' ? 'flex-end' : 'center';
        } else {
            // For regular buttons, apply text alignment directly
            button.style.textAlign = textAlign;
            button.style.justifyContent = textAlign === 'left' ? 'flex-start' : 
                                         textAlign === 'right' ? 'flex-end' : 'center';
        }
    }
    
    static createObject(type, name = null, shape = null, customId = null) {
        // Use customId if provided, otherwise generate a new ID
        const id = customId !== null ? parseInt(customId, 10) : this.nextId++;
        
        // Update nextId if necessary to avoid ID conflicts
        if (customId !== null && id >= this.nextId) {
            this.nextId = id + 1;
        }
        
        const objectName = name || `${type}${id}`;

        let element;
        switch (type.toLowerCase()) {
            case 'button':
                element = this.createButton(objectName, id);
                break;
            case 'field':
                element = this.createField(objectName, id);
                break;
            case 'graphic':
                element = this.createGraphic(objectName, id, shape);
                break;
            case 'image':
                element = this.createImage(objectName, id);
                break;
            case 'player':
                element = WebTalkPlayer.createPlayer(objectName, id);
                break;
            case 'scrollbar':
                element = this.createScrollbar(objectName, id);
                break;
            default:
                throw new Error(`Unknown object type: ${type}`);
        }
        
        // Store the last created object's type and name
        this.lastObject = { type: type.toLowerCase(), name: objectName };

        // Use current card context for object creation, fallback to main card
        const card = window.currentCard || document.getElementById('card');
        if (card) {
            card.appendChild(element);
            this.objects.set(objectName, element);
            this.objectsById.set(id.toString(), objectName);

            // Add direct click handler for selection in edit mode
            element.addEventListener('click', (e) => {
                const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
                if (isEditMode) {
                    console.log('Direct click in edit mode on:', objectName);
                    this.selectObject(element);
                    e.preventDefault();
                    e.stopPropagation();
                }
            });

            // Add double-click handler to open inspector in edit mode
            element.addEventListener('dblclick', (e) => {
                const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
                if (isEditMode) {
                    console.log('Double-click in edit mode on:', objectName);
                    this.selectObject(element);
                    if (WebTalkInspector) {
                        WebTalkInspector.showInspector(element);
                    }
                    e.preventDefault();
                    e.stopPropagation();
                }
            });

            // Make object draggable
            this.makeObjectDraggable(element);

            // Add context menu for objects
            this.addContextMenu(element);
        }

        // Refresh overview palette if visible
        if (window.refreshObjectOverview) {
            window.refreshObjectOverview();
        }

        return element;
    }

    static createButton(name, id) {
        const button = document.createElement('div');
        button.className = 'button';
        button.textContent = name;
        button.dataset.name = name;
        button.dataset.type = 'button';
        button.id = id.toString();

        // Set default position and size
        button.style.left = '50px';
        button.style.top = '50px';
        button.style.width = '150px';
        button.style.height = '30px';

        // Initialize empty script for this button
        this.scripts.set(name, '');

        // Initialize custom properties
        this.customProperties.set(name, new Map([
            ['visible', true],
            ['hilighted', false],
            ['isCheckbox', false],
            ['isMenu', false],
            ['menuText', ''],
            ['disabled', false],
            ['textAlign', 'center'],
            ['textSize', '14px'],
            ['textColor', '#000'],
            ['textFont', 'inherit'],
            ['textStyle', 'normal']
        ]));
        
        // Set default layer property
        this.highestLayer++;
        this.customProperties.get(name).set('layer', this.highestLayer.toString());
        button.style.zIndex = this.highestLayer.toString();
        
        // Initialize the button appearance based on properties
        this.updateButtonAppearance(button, name);

        // Track if mouse is down on this button
        let mouseDownOnButton = false;
        
        // Variables for double-click detection
        let lastMouseDownTime = 0;
        let lastMouseUpTime = 0;
        let lastClickButton = -1;
        const doubleClickThreshold = 500; // milliseconds

         // Add mouse event handlers
         button.addEventListener('mousedown', (e) => {
            // Prevent default browser behavior for all mouse buttons
            e.preventDefault();
            // Stop propagation to prevent the card from receiving the event
            e.stopPropagation();

            // Set flag that mouse is down on this button
            mouseDownOnButton = true;

            // Only execute in browse mode and if button is not disabled
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Check if button is disabled
                const properties = WebTalkObjects.customProperties.get(button.dataset.name);
                const isDisabled = properties && properties.get('disabled') === true;
                
                if (!isDisabled) {
                    const currentTime = Date.now();
                    const buttonNumber = e.button + 1;
                    
                    // Check for double-click (same button within threshold time)
                    if (currentTime - lastMouseDownTime <= doubleClickThreshold && buttonNumber === lastClickButton) {
                        // Execute the mouseDoubleDown handler in the button's script
                        window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseDoubleDown', [buttonNumber]);
                    } else {
                        // Execute the regular mouseDown handler in the button's script
                        window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseDown', [buttonNumber]);
                    }
                    
                    // Update last mousedown time and button for next comparison
                    lastMouseDownTime = currentTime;
                    lastClickButton = buttonNumber;
                }
            }
        });

        button.addEventListener('mouseup', async (e) => {
            // Only execute if the mouse was down on this button
            if (!mouseDownOnButton) return;

            // Prevent default browser behavior and stop propagation
            e.preventDefault();
            e.stopPropagation();
            
            mouseDownOnButton = false;
            
            // Get the current mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            const isBrowseMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse';
            
            // In edit mode, ensure any dragging is properly terminated
            if (isEditMode) {
                // This ensures the object is released if it was being dragged
                if (document.onmouseup) document.onmouseup();
            }
            
            // Execute script handlers in browse mode and if button is not disabled
            if (isBrowseMode) {
                // Check if button is disabled
                const buttonName = button.dataset.name;
                const properties = WebTalkObjects.customProperties.get(buttonName);
                const isDisabled = properties && properties.get('disabled') === true;
                
                if (!isDisabled) {
                    const currentTime = Date.now();
                    const buttonNumber = e.button + 1;
                    
                    // Check if this is a checkbox button
                    const isCheckbox = WebTalkObjects.customProperties.has(buttonName) && 
                                       WebTalkObjects.customProperties.get(buttonName).get('isCheckbox') === true;
                    
                    // Toggle checkbox state if this is a checkbox button and left mouse button was clicked
                    if (isCheckbox && buttonNumber === 1) {
                        // Get current hilighted state
                        const currentState = properties.get('hilighted') === true;
                        
                        // Toggle hilighted state
                        properties.set('hilighted', !currentState);
                        
                        // Update button appearance
                        WebTalkObjects.updateButtonAppearance(button, buttonName);
                    }
                    
                    // Check for double-click (same button within threshold time)
                    try {
                        if (currentTime - lastMouseUpTime <= doubleClickThreshold && buttonNumber === lastClickButton) {
                            // Execute the mouseDoubleUp handler in the button's script
                            await window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseDoubleUp', [buttonNumber]);
                        } else {
                            // Execute the regular mouseUp handler in the button's script
                            await window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseUp', [buttonNumber]);
                        }
                    } catch (error) {
                        // Log the error but don't rethrow to prevent uncaught promise rejection
                        console.error('Error in button script execution:', error);
                    }
                    
                    // Update last mouseup time for next comparison
                    lastMouseUpTime = currentTime;
                }
            }
        });

        // Add document-level mouseup listener to detect mouseupoutside
        document.addEventListener('mouseup', (e) => {
            // Only handle left-click (button 0)
            if (e.button !== 0) return;

            // If mouse was down on this button but mouseup is not on the button
            if (mouseDownOnButton && !button.contains(e.target)) {
                // Reset flag
                mouseDownOnButton = false;
                
                // Get the current mode
                const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
                const isBrowseMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse';
                
                // In edit mode, ensure any dragging is properly terminated
                if (isEditMode) {
                    // This ensures the object is released if it was being dragged
                    if (document.onmouseup) document.onmouseup();
                }

                // Only execute script handlers in browse mode
                if (isBrowseMode) {
                    // Execute the mouseUpOutside handler in the button's script
                    window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseUpOutside');
                }
            }
        });

        button.addEventListener('mouseenter', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseEnter handler in the button's script
                window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseEnter');
                
                // Check if this is a menu button
                const properties = WebTalkObjects.customProperties.get(name);
                if (properties && properties.get('isMenu') === true && properties.get('menuText')) {
                    // Check if the button has a menuPick handler by examining its script
                    const script = WebTalkObjects.scripts.get(name) || '';
                    const hasMenuPickHandler = script.includes('on menuPick') || script.includes('function menuPick');
                    
                    // Store the menuPick handler status on the button element for later use
                    button.dataset.hasMenuPickHandler = hasMenuPickHandler.toString();
                    
                    if (!hasMenuPickHandler) {
                        console.log(`Button "${name}" is a menu button but has no menuPick handler`);
                    }
                }
            }
        });

        button.addEventListener('mouseleave', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseLeave handler in the button's script
                window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseLeave');
            }
        });

        // For mouseWithin, we need to track when the mouse is inside the button
        button.addEventListener('mousemove', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseWithin handler in the button's script
                window.webTalkApp.interpreter.executeObjectScript(button.dataset.name, 'mouseWithin');
            }
        });
        
        // Add click handler for menu functionality
        button.addEventListener('click', (e) => {
            // Check if we're in edit mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit') {
                return;
            }
            
            // In browse mode, handle button click
            // Use button.dataset.name instead of closure variable to handle renames correctly
            const currentName = button.dataset.name;
            const properties = WebTalkObjects.customProperties.get(currentName);
            console.log(`Button ${currentName} clicked. Properties:`, properties ? Object.fromEntries(properties) : 'none');
            
            if (properties && properties.get('disabled') !== true) {
                // Check if this is a menu button
                const isMenuValue = properties.get('isMenu');
                const menuTextValue = properties.get('menuText');
                console.log(`Button ${currentName}: isMenu=${isMenuValue}, menuText="${menuTextValue}", WebTalkMenu available=${!!window.WebTalkMenu}`);
                
                if (isMenuValue === true) {
                    // Add menu-specific class for styling
                    button.classList.add('webtalk-menu-button');
                    
                    // Show the menu
                    if (window.WebTalkMenu) {
                        const menuText = menuTextValue || '';
                        console.log(`Showing menu for ${currentName} with text: "${menuText}"`);
                        WebTalkMenu.showMenu(button, menuText);
                        
                        // Prevent further event propagation
                        e.stopPropagation();
                        return false;
                    } else {
                        console.error(`WebTalkMenu not available for button ${currentName}`);
                    }
                } else {
                    console.log(`Button ${currentName} is not a menu button (isMenu=${isMenuValue})`);
                }
                
                // Execute the mouseClick handler if it exists
                if (window.webTalkApp && window.webTalkApp.interpreter) {
                    window.webTalkApp.interpreter.executeObjectScript(currentName, 'mouseClick');
                }
            }
        });

        return button;
    }

    static createField(name, id) {
        // Create the main field container
        const field = document.createElement('div');
        field.className = 'field';
        field.contentEditable = false; // The container is not editable
        field.dataset.name = name;
        field.dataset.type = 'field';
        field.id = id.toString();
        
        // Create the field content div (editable)
        const fieldContent = document.createElement('div');
        fieldContent.className = 'field-content';
        fieldContent.contentEditable = true;
        fieldContent.spellcheck = false;
        fieldContent.setAttribute('autocapitalize', 'none');
        fieldContent.setAttribute('autocorrect', 'off');
        fieldContent.setAttribute('autocomplete', 'off');
        fieldContent.style.width = '100%';
        fieldContent.style.height = '100%';
        fieldContent.style.overflow = 'auto';
        fieldContent.style.outline = 'none';
        fieldContent.style.whiteSpace = 'pre-wrap';
        
        // Configure field content for consistent line handling
        fieldContent.style.whiteSpace = 'pre-wrap';
        
        // By default, remove focus border (can be enabled later with showFocusBorder property)
        fieldContent.style.outline = 'none !important';
        field.style.outline = 'none !important';
        
        // Initialize formattingCommands array in customProperties
        if (!this.customProperties.has(name)) {
            this.customProperties.set(name, new Map());
        }
        this.customProperties.get(name).set('formattingCommands', []);
        this.customProperties.get(name).set('showFocusBorder', false);
        this.customProperties.get(name).set('styledtext', false);
        
        // Disable middle-button paste - apply to both the field and its content
        const preventMiddleClickDefault = function(e) {
            // Middle button is usually button 1 (0 is left, 2 is right)
            if (e.button === 1) {
                // Only prevent default behavior (auto-inserting text)
                // but allow event propagation to the card for toggling edit/browse mode
                e.preventDefault();
                // Do NOT call stopPropagation() to allow event bubbling to the card
            }
        };
        
        // Apply to both the field container and the content element
        fieldContent.addEventListener('mousedown', preventMiddleClickDefault, true);
        field.addEventListener('mousedown', preventMiddleClickDefault, true);
        
        // Handle the auxclick event which can trigger paste on middle click
        // Only prevent default behavior but allow event propagation
        fieldContent.addEventListener('auxclick', function(e) {
            if (e.button === 1) {
                e.preventDefault();
                // Do NOT call stopPropagation() to allow event bubbling to the card
            }
        }, true);
        field.addEventListener('auxclick', function(e) {
            if (e.button === 1) {
                e.preventDefault();
                // Do NOT call stopPropagation() to allow event bubbling to the card
            }
        }, true);
        
        // Disable paste event directly
        fieldContent.addEventListener('paste', function(e) {
            // Allow paste only from keyboard (Ctrl+V) or context menu
            if (e.clipboardData && !e.isTrusted) {
                e.preventDefault();
                return false;
            }
        }, true);
        
        // Make sure the field is properly focusable and can receive keyboard events
        fieldContent.setAttribute('tabindex', '0');
        
        // Ensure field content is properly editable
        fieldContent.contentEditable = 'true';
        
        // Fix for arrow key navigation and focus border management
        fieldContent.addEventListener('focus', function() {
            // When the field gets focus, ensure it's in a state where arrow keys will work
            // This is a good place to clean up the DOM if needed
            // But we'll keep it minimal to avoid interfering with cursor positioning
            
            // Show focus border if showFocusBorder is true
            const objectName = field.dataset.name;
            if (WebTalkObjects.customProperties.has(objectName) && 
                WebTalkObjects.customProperties.get(objectName).get('showFocusBorder') === true) {
                field.style.outline = '2px solid #4d90fe';
                field.style.outlineOffset = '2px';
            }
            
            // Set up selection if needed
            const selection = window.getSelection();
            if (!selection.rangeCount) {
                const range = document.createRange();
                if (fieldContent.firstChild) {
                    range.setStart(fieldContent.firstChild, 0);
                    range.setEnd(fieldContent.firstChild, 0);
                } else {
                    // If the field is empty, create a text node to focus on
                    const textNode = document.createTextNode('');
                    fieldContent.appendChild(textNode);
                    range.setStart(textNode, 0);
                    range.setEnd(textNode, 0);
                }
                selection.removeAllRanges();
                selection.addRange(range);
            }
        });
        
        // Handle blur event to hide focus border
        fieldContent.addEventListener('blur', function() {
            // Hide focus border when field loses focus
            field.style.outline = 'none';
        });
        
        // Ensure arrow keys work properly for navigation
        fieldContent.addEventListener('keydown', function(e) {
            // Only handle arrow keys
            if (e.key.startsWith('Arrow')) {
                // Make sure the field is focused
                if (document.activeElement !== fieldContent) {
                    fieldContent.focus();
                }
                
                // Let the browser handle arrow key navigation naturally
                // No setTimeout or DOM manipulation here to avoid interfering with navigation
            }
        });
        
        // Add click handler to ensure proper focus
        fieldContent.addEventListener('click', function() {
            // Ensure the field has focus when clicked
            if (document.activeElement !== fieldContent) {
                fieldContent.focus();
            }
        });
        
        // Prevent default Enter key behavior which creates divs
        fieldContent.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                // Prevent the default behavior which creates divs with large spacing
                e.preventDefault();
                
                // Get current selection
                const selection = window.getSelection();
                if (!selection.rangeCount) return false;
                const range = selection.getRangeAt(0);
                
                // Create a text node with a newline character
                const newlineNode = document.createTextNode('\n');
                
                // Insert the newline at the current position
                range.deleteContents(); // Delete any selected text first
                range.insertNode(newlineNode);
                
                // Move the cursor after the inserted newline
                range.setStartAfter(newlineNode);
                range.setEndAfter(newlineNode);
                
                // Update the selection
                selection.removeAllRanges();
                selection.addRange(range);
                
                // Normalize the DOM to clean up any adjacent text nodes
                fieldContent.normalize();
                
                // Ensure we don't have any unwanted divs
                setTimeout(() => {
                    // Fix any divs that might have been created
                    const divs = fieldContent.querySelectorAll('div');
                    if (divs.length > 0) {
                        // For each div, replace it with its text content + newline
                        divs.forEach(div => {
                            const text = div.textContent;
                            const textNode = document.createTextNode(text + '\n');
                            div.parentNode.replaceChild(textNode, div);
                        });
                        
                        // Normalize again after modifications
                        fieldContent.normalize();
                        
                        // Restore selection at the end if we can't determine where it was
                        if (!selection.rangeCount) {
                            const newRange = document.createRange();
                            const lastNode = fieldContent.lastChild || fieldContent;
                            newRange.setStartAfter(lastNode);
                            newRange.setEndAfter(lastNode);
                            selection.removeAllRanges();
                            selection.addRange(newRange);
                        }
                    }
                }, 0);
                
                return false;
            }
        });
        
        // Handle paste events to ensure plain text and prevent unwanted divs
        fieldContent.addEventListener('paste', function(e) {
            e.preventDefault();
            const text = (e.clipboardData || window.clipboardData).getData('text/plain');
            document.execCommand('insertText', false, text);
            
            // Clean up any divs that might have been created during paste
            setTimeout(() => {
                const divs = fieldContent.querySelectorAll('div');
                if (divs.length > 0) {
                    // For each div, replace it with its text content + newline
                    divs.forEach(div => {
                        const text = div.textContent;
                        const textNode = document.createTextNode(text + '\n');
                        div.parentNode.replaceChild(textNode, div);
                    });
                    fieldContent.normalize();
                }
                
                // Also clean up any br elements that might have been inserted
                const brs = fieldContent.querySelectorAll('br');
                if (brs.length > 0) {
                    brs.forEach(br => {
                        const newline = document.createTextNode('\n');
                        br.parentNode.replaceChild(newline, br);
                    });
                    fieldContent.normalize();
                }
            }, 0);
        });
        
        // We're not adding an input event listener that would strip HTML styling
        // This allows styled content (like colored text) to remain intact
        // The keydown handler for Enter and paste handler should be sufficient
        // for basic normalization without causing styling issues
        
        // Add a mutation observer to catch any DOM changes that might insert unwanted elements
        const observer = new MutationObserver(mutations => {
            let needsCleanup = false;
            
            // Check if any mutations added divs or brs
            for (const mutation of mutations) {
                if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
                    for (const node of mutation.addedNodes) {
                        if (node.nodeName === 'DIV' || node.nodeName === 'BR') {
                            needsCleanup = true;
                            break;
                        }
                    }
                    if (needsCleanup) break;
                }
            }
            
            // If we found problematic elements, clean them up
            if (needsCleanup) {
                // Temporarily disconnect observer to avoid infinite loop
                observer.disconnect();
                
                // Fix divs
                const divs = fieldContent.querySelectorAll('div');
                divs.forEach(div => {
                    const text = div.textContent;
                    const textNode = document.createTextNode(text + '\n');
                    div.parentNode.replaceChild(textNode, div);
                });
                
                // Fix brs
                const brs = fieldContent.querySelectorAll('br');
                brs.forEach(br => {
                    const newline = document.createTextNode('\n');
                    br.parentNode.replaceChild(newline, br);
                });
                
                // Normalize the DOM
                fieldContent.normalize();
                
                // Reconnect the observer
                observer.observe(fieldContent, { childList: true, subtree: true });
            }
        });
        
        // Start observing
        observer.observe(fieldContent, { childList: true, subtree: true })
        
        // Add the content div to the field container
        field.appendChild(fieldContent);

        // Set default position and size
        field.style.left = '50px';
        field.style.top = '50px';
        field.style.width = '200px';
        field.style.height = '100px';

        // Initialize empty script for this field
        this.scripts.set(name, '');

        // Initialize empty custom properties for this field
        this.customProperties.set(name, new Map());

        // Set default visible property
        this.customProperties.get(name).set('visible', 'true');
        this.customProperties.get(name).set('rotation', '0'); // Initialize rotation property
        
        // Set default layer property
        this.highestLayer++;
        this.customProperties.get(name).set('layer', this.highestLayer.toString());
        field.style.zIndex = this.highestLayer.toString();

        // Set default field-specific properties
        const fieldProperties = this.customProperties.get(name);
        fieldProperties.set('textalign', 'left');
        fieldProperties.set('scrollable', true);
        fieldProperties.set('locktext', false);
        fieldProperties.set('multiline', true);
        fieldProperties.set('autotab', false);
        fieldProperties.set('widemargins', false);
        fieldProperties.set('sharedtext', false);
        fieldProperties.set('rotation', 0);
        fieldProperties.set('dontwrap', false);
        fieldProperties.set('textsize', 16); // Default text size
        fieldProperties.set('vScroll', 0);
        fieldProperties.set('hScroll', 0);
        fieldProperties.set('visibleLines', '1');
        fieldProperties.set('formattingCommands', []); // Initialize empty array for formatting commands

        // Apply default styles
        field.style.fontSize = '16px'; // Apply default text size

        // Ensure field is actually editable when clicked
        field.addEventListener('click', function(e) {
            // Only allow editing in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Only stop propagation to prevent card selection
                // but allow default browser behavior for cursor positioning
                e.stopPropagation();
            }
        });

        // Add scroll event handler to track scroll position
        // Note: The scroll event should be on fieldContent, not field, since fieldContent is the scrollable element
        fieldContent.addEventListener('scroll', () => {
            // Store scroll position in custom properties
            const vScroll = fieldContent.scrollTop;
            const hScroll = fieldContent.scrollLeft;

            // Update custom properties
            this.customProperties.get(field.dataset.name).set('vScroll', vScroll);
            this.customProperties.get(field.dataset.name).set('hScroll', hScroll);

            // Calculate visible lines
            const lineHeight = parseInt(field.style.fontSize) || 16;
            const fieldHeight = fieldContent.clientHeight;
            const totalLines = fieldContent.textContent.split('\n').length;

            // Calculate first visible line (1-based)
            const firstLine = Math.floor(vScroll / lineHeight) + 1;
            // Calculate last visible line
            const visibleLineCount = Math.floor(fieldHeight / lineHeight);
            const lastLine = Math.min(firstLine + visibleLineCount - 1, totalLines);

            // Store visible lines range
            this.customProperties.get(field.dataset.name).set('visibleLines', `${firstLine}`);

            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the scroll handler in the field's script
                window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'scroll');
                
                // Execute the scrollbarDrag handler with the new position as parameter
                window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'scrollbarDrag', [vScroll]);
            }
        });

        // Track if mouse is down on this field
        let mouseDownOnField = false;

        // Add mouse event handlers
        field.addEventListener('mousedown', function(e) {
            // For middle mouse button (button 1), only prevent default behavior but allow propagation
            // to the card for toggling edit/browse mode
            if (e.button === 1) {
                e.preventDefault();
                
                // Set flag that mouse is down on this field
                mouseDownOnField = true;
                
                // Execute the mouseDown handler in the field's script if in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Use setTimeout to ensure browser handles cursor positioning first
                    setTimeout(() => {
                        window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseDown', [e.button + 1]);
                    }, 0);
                }
                
                // Do NOT call stopPropagation() to allow event bubbling to the card
                return;
            }
            
            // Set flag that mouse is down on this field
            mouseDownOnField = true;

            // Only execute script in browse mode, but don't interfere with default behavior
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Stop propagation to prevent card selection, but allow default cursor positioning
                e.stopPropagation();

                // Execute the mouseDown handler in the field's script
                // Use setTimeout to ensure browser handles cursor positioning first
                setTimeout(() => {
                    window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseDown', [e.button + 1]);
                }, 0);
            }
        });

        field.addEventListener('mouseup', (e) => {
            // Only execute if the mouse was down on this field
            if (!mouseDownOnField) return;

            mouseDownOnField = false;

            // Check if this is a scrollbar interaction
            const isScrollbarClick = e.offsetX > field.offsetWidth - 15; // Approximate scrollbar width
            
            // Only execute in browse mode and not a scrollbar interaction
            if (window.webTalkApp && window.webTalkApp.interpreter && 
                window.webTalkApp.interpreter.mode === 'browse' && !isScrollbarClick) {
                // Check if field is locked
                const isLocked = this.customProperties.get(field.dataset.name)?.get('locktext');
                if (!isLocked) {
                    // Execute the mouseUp handler in the field's script
                    window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseUp', [e.button + 1]);
                }
            }
        });

        // Add document-level mouseup listener to detect mouseupoutside
        document.addEventListener('mouseup', (e) => {
            // Only handle left-click (button 0)
            if (e.button !== 0) return;

            // If mouse was down on this field but mouseup is not on the field
            if (mouseDownOnField && !field.contains(e.target)) {
                // Reset flag
                mouseDownOnField = false;

                // Only execute in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Execute the mouseUpOutside handler in the field's script
                    window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseUpOutside');
                }
            }
        });

        field.addEventListener('mouseenter', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseEnter handler in the field's script
                window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseEnter');
            }
        });

        field.addEventListener('mouseleave', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseLeave handler in the field's script
                window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseLeave');
            }
        });

        // For mouseWithin, we need to track when the mouse is inside the field
        field.addEventListener('mousemove', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseWithin handler in the field's script
                window.webTalkApp.interpreter.executeObjectScript(field.dataset.name, 'mouseWithin');
            }
        });

        return field;
    }

    // Helper function to apply dontwrap property to a field
    static applyDontWrapToField(fieldName, dontWrap) {
        const field = this.getObject(fieldName);
        if (!field || field.dataset.type !== 'field') {
            return false;
        }

        const fieldContent = field.querySelector('.field-content');
        if (!fieldContent) {
            return false;
        }

        if (dontWrap) {
            // Enable horizontal scrolling, disable text wrapping
            fieldContent.style.whiteSpace = 'pre';
            fieldContent.style.overflowX = 'auto';
            fieldContent.style.overflowY = 'auto';
        } else {
            // Default behavior: allow text wrapping, vertical scrolling only
            fieldContent.style.whiteSpace = 'pre-wrap';
            fieldContent.style.overflowX = 'hidden';
            fieldContent.style.overflowY = 'auto';
        }

        return true;
    }

    static createGraphic(name, id, shape = null) {
        const graphicContainer = document.createElement('div');
        graphicContainer.className = 'graphic';
        graphicContainer.dataset.name = name;
        graphicContainer.dataset.type = 'graphic';
        graphicContainer.id = id.toString();

        // Initialize empty script for this graphic
        this.scripts.set(name, '');

        // Initialize custom properties
        this.customProperties.set(name, new Map());
        
        // Set default visible property
        this.customProperties.get(name).set('visible', 'true');
        this.customProperties.get(name).set('rotation', '0'); // Initialize rotation property
        
        // Set default layer property
        this.highestLayer++;
        this.customProperties.get(name).set('layer', this.highestLayer.toString());
        graphicContainer.style.zIndex = this.highestLayer.toString();

        // Create SVG element
        const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
        svg.setAttribute('width', '100%');
        svg.setAttribute('height', '100%');

        // Add this line to prevent the SVG from capturing the right-click event
        svg.style.pointerEvents = 'none';

        // Default width and height for the graphic
        let width = 200;
        let height = 200;

        // Handle different shape types
        if (shape === 'oval') {
            // Create an ellipse for oval shape
            const ellipse = document.createElementNS('http://www.w3.org/2000/svg', 'ellipse');
            ellipse.setAttribute('cx', '50%');
            ellipse.setAttribute('cy', '50%');
            ellipse.setAttribute('rx', '45%');
            ellipse.setAttribute('ry', '45%');
            ellipse.setAttribute('fill', 'none');
            ellipse.setAttribute('stroke', 'black');
            ellipse.setAttribute('stroke-width', '2');

            svg.appendChild(ellipse);
            graphicContainer.dataset.graphicType = 'oval';

            // Set oval-specific dimensions
            width = 200;
            height = 100;
        } else if (shape === 'circle') {
            // Create a circle shape
            const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
            circle.setAttribute('cx', '50%');
            circle.setAttribute('cy', '50%');
            circle.setAttribute('r', '45%');
            circle.setAttribute('fill', 'none');
            circle.setAttribute('stroke', 'black');
            circle.setAttribute('stroke-width', '2');

            svg.appendChild(circle);
            graphicContainer.dataset.graphicType = 'circle';

            // Set circle-specific dimensions (equal width and height)
            width = 200;
            height = 200;
        } else if (shape === 'rectangle') {
            // Create a rectangle shape
            const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
            rect.setAttribute('x', '5%');
            rect.setAttribute('y', '5%');
            rect.setAttribute('width', '90%');
            rect.setAttribute('height', '90%');
            rect.setAttribute('fill', 'none');
            rect.setAttribute('stroke', 'black');
            rect.setAttribute('stroke-width', '2');

            svg.appendChild(rect);
            graphicContainer.dataset.graphicType = 'rectangle';

            // Set rectangle-specific dimensions
            width = 200;
            height = 100;
        } else if (shape === 'square') {
            // Create a square shape
            const rect = document.createElementNS('http://www.w3.org/2000/svg', 'rect');
            rect.setAttribute('x', '5%');
            rect.setAttribute('y', '5%');
            rect.setAttribute('width', '90%');
            rect.setAttribute('height', '90%');
            rect.setAttribute('fill', 'none');
            rect.setAttribute('stroke', 'black');
            rect.setAttribute('stroke-width', '2');

            svg.appendChild(rect);
            graphicContainer.dataset.graphicType = 'square';

            // Set square-specific dimensions (equal width and height)
            width = 200;
            height = 200;
        } else if (shape === 'triangle') {
            // Create a triangle shape (equilateral)
            const polygon = document.createElementNS('http://www.w3.org/2000/svg', 'polygon');
            polygon.setAttribute('points', '100,20 180,180 20,180');
            polygon.setAttribute('fill', 'none');
            polygon.setAttribute('stroke', 'black');
            polygon.setAttribute('stroke-width', '2');

            svg.appendChild(polygon);
            graphicContainer.dataset.graphicType = 'triangle';

            // Set triangle-specific dimensions
            width = 200;
            height = 200;
        } else if (shape === 'pentagon') {
            // Create a pentagon shape
            const polygon = document.createElementNS('http://www.w3.org/2000/svg', 'polygon');
            polygon.setAttribute('points', '100,20 180,80 150,180 50,180 20,80');
            polygon.setAttribute('fill', 'none');
            polygon.setAttribute('stroke', 'black');
            polygon.setAttribute('stroke-width', '2');

            svg.appendChild(polygon);
            graphicContainer.dataset.graphicType = 'pentagon';

            // Set pentagon-specific dimensions
            width = 200;
            height = 200;
        } else if (shape === 'hexagon') {
            // Create a hexagon shape
            const polygon = document.createElementNS('http://www.w3.org/2000/svg', 'polygon');
            polygon.setAttribute('points', '100,20 180,60 180,140 100,180 20,140 20,60');
            polygon.setAttribute('fill', 'none');
            polygon.setAttribute('stroke', 'black');
            polygon.setAttribute('stroke-width', '2');

            svg.appendChild(polygon);
            graphicContainer.dataset.graphicType = 'hexagon';

            // Set hexagon-specific dimensions
            width = 200;
            height = 200;
        } else if (shape === 'octagon') {
            // Create an octagon shape
            const polygon = document.createElementNS('http://www.w3.org/2000/svg', 'polygon');
            polygon.setAttribute('points', '60,20 140,20 180,60 180,140 140,180 60,180 20,140 20,60');
            polygon.setAttribute('fill', 'none');
            polygon.setAttribute('stroke', 'black');
            polygon.setAttribute('stroke-width', '2');

            svg.appendChild(polygon);
            graphicContainer.dataset.graphicType = 'octagon';

            // Set octagon-specific dimensions
            width = 200;
            height = 200;
        } else if (shape === 'decagon') {
            // Create a decagon shape
            const polygon = document.createElementNS('http://www.w3.org/2000/svg', 'polygon');
            polygon.setAttribute('points', '100,20 150,30 180,70 180,130 150,170 100,180 50,170 20,130 20,70 50,30');
            polygon.setAttribute('fill', 'none');
            polygon.setAttribute('stroke', 'black');
            polygon.setAttribute('stroke-width', '2');

            svg.appendChild(polygon);
            graphicContainer.dataset.graphicType = 'decagon';

            // Set decagon-specific dimensions
            width = 200;
            height = 200;
        } else {
            // Create a line by default
            const line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
            line.setAttribute('x1', '10%');
            line.setAttribute('y1', '10%');
            line.setAttribute('x2', '90%');
            line.setAttribute('y2', '90%');
            line.setAttribute('stroke', 'black');
            line.setAttribute('stroke-width', '2');

            svg.appendChild(line);
            graphicContainer.dataset.graphicType = 'line';
        }

        graphicContainer.appendChild(svg);

        // Initialize empty script for this graphic
        this.scripts.set(name, '');

        // Set default position and size
        graphicContainer.style.left = '50px';
        graphicContainer.style.top = '50px';
        graphicContainer.style.width = `${width}px`;
        graphicContainer.style.height = `${height}px`;
        graphicContainer.style.position = 'absolute';

        // Default to no border
        graphicContainer.style.border = 'none';

        // Initialize custom properties for this graphic
        if (!this.customProperties.has(name)) {
            this.customProperties.set(name, new Map());
        }
        // Set default showBorder to false
        this.customProperties.get(name).set('showborder', false);

        // Add mouse event handlers
        let mouseDownOnGraphic = false;

        graphicContainer.addEventListener('mousedown', (e) => {
            // Only handle mouse events if we're not in edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) return;

            // Prevent default browser behavior for all mouse buttons
            e.preventDefault();

            // Set flag that mouse is down on this graphic
            mouseDownOnGraphic = true;

            // Execute the mouseDown handler in the graphic's script
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(graphicContainer.dataset.name, 'mouseDown', [e.button + 1]);
            }
        });

        graphicContainer.addEventListener('mouseup', (e) => {
            // Only handle mouse events if we're not in edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) return;

            // Only execute if the mouse was down on this graphic
            if (!mouseDownOnGraphic) return;

            mouseDownOnGraphic = false;

            // Execute the mouseUp handler in the graphic's script
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(graphicContainer.dataset.name, 'mouseUp', [e.button + 1]);
            }
        });

        return graphicContainer;
    }

    static createImage(name, id) {
        const imageContainer = document.createElement('div');
        imageContainer.className = 'image';
        imageContainer.dataset.name = name;
        imageContainer.dataset.type = 'image';
        imageContainer.id = id.toString();

        // Set default position and size
        imageContainer.style.left = '50px';
        imageContainer.style.top = '50px';
        imageContainer.style.width = '150px';
        imageContainer.style.height = '150px';
        imageContainer.style.position = 'absolute'; // Ensure absolute positioning
        
        // Add a placeholder background
        imageContainer.style.backgroundColor = '#f0f0f0';
        imageContainer.style.border = '1px dashed #ccc';
        
        // Create the img element
        const img = document.createElement('img');
        img.style.width = '100%';
        img.style.height = '100%';
        img.style.objectFit = 'contain';
        imageContainer.appendChild(img);
        
        // Initialize empty script for this image
        this.scripts.set(name, '');
        
        // Initialize custom properties for this image
        this.customProperties.set(name, new Map());
        
        // Set default visible property
        this.customProperties.get(name).set('visible', 'true');
        this.customProperties.get(name).set('rotation', '0'); // Initialize rotation property
        
        // Set default layer property
        this.highestLayer++;
        this.customProperties.get(name).set('layer', this.highestLayer.toString());
        imageContainer.style.zIndex = this.highestLayer.toString();
        
        // Add mouse event handlers
        let mouseDownOnImage = false;
        
        imageContainer.addEventListener('mousedown', (e) => {
            // Only handle mouse events if we're not in edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) return;
            
            // Prevent default browser behavior for all mouse buttons
            e.preventDefault();
            
            // Set flag that mouse is down on this image
            mouseDownOnImage = true;
            
            // Execute the mouseDown handler in the image's script
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(imageContainer.dataset.name, 'mouseDown', [e.button + 1]);
            }
        });
        
        imageContainer.addEventListener('mouseup', (e) => {
            // Only handle mouse events if we're not in edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) return;
            
            // Only execute if the mouse was down on this image
            if (!mouseDownOnImage) return;
            
            mouseDownOnImage = false;
            
            // Execute the mouseUp handler in the image's script
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(imageContainer.dataset.name, 'mouseUp', [e.button + 1]);
            }
        });
        
        // Add a separate mouseMove handler
        imageContainer.addEventListener('mousemove', (e) => {
            // Only handle mouse events if we're not in edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) return;
            
            // Execute the mouseMove handler in the image's script
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.executeObjectScript(imageContainer.dataset.name, 'mouseMove');
            }
        });
        
        return imageContainer;
    }
    

    
    static createScrollbar(name, id) {
        const scrollbar = document.createElement('div');
        scrollbar.className = 'scrollbar webtalk-object';
        scrollbar.dataset.name = name;
        scrollbar.dataset.type = 'scrollbar';
        scrollbar.id = id.toString();

        // Set default position and size
        scrollbar.style.left = '50px';
        scrollbar.style.top = '50px';
        
        // Set initial dimensions - these will be overridden later if loading from JSON
        scrollbar.style.width = '150px';
        scrollbar.style.height = '20px';
        
        scrollbar.style.position = 'absolute'; // Ensure absolute positioning
        
        // Set orientation class based on dimensions
        if (parseInt(scrollbar.style.width) > parseInt(scrollbar.style.height)) {
            scrollbar.classList.add('horizontal');
        } else {
            scrollbar.classList.add('vertical');
        }

        // Create scrollbar thumb (the draggable part)
        const thumb = document.createElement('div');
        thumb.className = 'scrollbar-thumb';
        scrollbar.appendChild(thumb);

        // Initialize empty script for this scrollbar
        this.scripts.set(name, '');

        // Initialize custom properties
        this.customProperties.set(name, new Map());
        
        // Set default visible property
        this.customProperties.get(name).set('visible', 'true');
        
        // Set default layer property
        this.highestLayer++;
        this.customProperties.get(name).set('layer', this.highestLayer.toString());
        scrollbar.style.zIndex = this.highestLayer.toString();
        
        // Set scrollbar specific properties in customProperties
        this.customProperties.get(name).set('startValue', '0');
        this.customProperties.get(name).set('endValue', '100');
        this.customProperties.get(name).set('thumbPosition', '0');
        this.customProperties.get(name).set('scrollType', 'bar');
        this.customProperties.get(name).set('progressColor', '0,0,255'); // Default blue color
        this.customProperties.get(name).set('fontFamily', 'Arial'); // Default font family
        this.customProperties.get(name).set('fontSize', '12'); // Default font size
        this.customProperties.get(name).set('textAlign', 'center'); // Default text alignment
        this.customProperties.get(name).set('rotation', '0'); // Initialize rotation property
        this.customProperties.get(name).set('hasSlider', false); // Initialize hasSlider property
        
        // Track if mouse is down on this scrollbar
        let mouseDownOnScrollbar = false;
        
        // Add mouse event handlers for the scrollbar (not just the thumb)
        scrollbar.addEventListener('mousedown', (e) => {
            // Prevent default browser behavior for all mouse buttons
            e.preventDefault();
            
            // Set flag that mouse is down on this scrollbar
            mouseDownOnScrollbar = true;
            
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseDown handler in the scrollbar's script
                window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseDown', [e.button + 1]);
            }
        });
        
        scrollbar.addEventListener('mouseup', (e) => {
            // Only execute if the mouse was down on this scrollbar
            if (!mouseDownOnScrollbar) return;
            
            mouseDownOnScrollbar = false;
            
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseUp handler in the scrollbar's script
                window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseUp', [e.button + 1]);
            }
        });
        
        // Add document-level mouseup listener to detect mouseupoutside
        document.addEventListener('mouseup', (e) => {
            // Only handle left-click (button 0)
            if (e.button !== 0) return;
            
            // If mouse was down on this scrollbar but mouseup is not on the scrollbar
            if (mouseDownOnScrollbar && !scrollbar.contains(e.target)) {
                // Reset flag
                mouseDownOnScrollbar = false;
                
                // Only execute in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Execute the mouseUpOutside handler in the scrollbar's script
                    window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseUpOutside');
                }
            }
        });
        
        scrollbar.addEventListener('mouseenter', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseEnter handler in the scrollbar's script
                window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseEnter');
            }
        });
        
        scrollbar.addEventListener('mouseleave', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseLeave handler in the scrollbar's script
                window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseLeave');
            }
        });
        
        // For mouseWithin, we need to track when the mouse is inside the scrollbar
        scrollbar.addEventListener('mousemove', () => {
            // Only execute in browse mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                // Execute the mouseWithin handler in the scrollbar's script
                window.webTalkApp.interpreter.executeObjectScript(scrollbar.dataset.name, 'mouseWithin');
            }
        });
        
        // Add event listener for thumb dragging
        thumb.addEventListener('mousedown', (e) => {
            e.stopPropagation();
            
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if (isEditMode) {
                return; // Don't allow dragging in edit mode
            }
            
            const scrollbarRect = scrollbar.getBoundingClientRect();
            const isVertical = scrollbar.classList.contains('vertical');
            const startValue = parseInt(this.customProperties.get(name).get('startValue') || '0');
            const endValue = parseInt(this.customProperties.get(name).get('endValue') || '100');
            const range = endValue - startValue;
            
            const onMouseMove = (moveEvent) => {
                moveEvent.preventDefault();
                
                let percentage;
                if (isVertical) {
                    const scrollbarHeight = scrollbarRect.height;
                    const mouseY = moveEvent.clientY - scrollbarRect.top;
                    percentage = Math.max(0, Math.min(1, mouseY / scrollbarHeight));
                } else {
                    const scrollbarWidth = scrollbarRect.width;
                    const mouseX = moveEvent.clientX - scrollbarRect.left;
                    percentage = Math.max(0, Math.min(1, mouseX / scrollbarWidth));
                }
                
                const newValue = Math.round(startValue + (percentage * range));
                this.customProperties.get(name).set('thumbPosition', newValue.toString());
                
                // Update thumb position
                this.updateScrollbarThumb(scrollbar, name);
            };
            
            const onMouseUp = () => {
                document.removeEventListener('mousemove', onMouseMove);
                document.removeEventListener('mouseup', onMouseUp);
            };
            
            document.addEventListener('mousemove', onMouseMove);
            document.addEventListener('mouseup', onMouseUp);
        });
        
        // Initial thumb position update
        this.updateScrollbarThumb(scrollbar, name);

        return scrollbar;
    }
    
    static updateScrollbarThumb(scrollbar, name, interpreter = null) {
        const startValue = parseInt(this.customProperties.get(name).get('startValue') || '0');
        const endValue = parseInt(this.customProperties.get(name).get('endValue') || '100');
        const thumbPosition = parseInt(this.customProperties.get(name).get('thumbPosition') || '0');
        const scrollType = this.customProperties.get(name).get('scrollType') || 'bar';
        const hasSliderValue = this.customProperties.get(name).get('hasSlider');
        const hasSlider = hasSliderValue === true || hasSliderValue === 'true';
        
        const range = endValue - startValue;
        const percentage = range > 0 ? (thumbPosition - startValue) / range : 0;
        
        // For scrollbars loaded from JSON, get the original dimensions from customProperties
        const isLoadedFromJSON = this.customProperties.get(name).has('loadedFromJSON');
        const jsonWidth = isLoadedFromJSON ? this.customProperties.get(name).get('originalWidth') : null;
        const jsonHeight = isLoadedFromJSON ? this.customProperties.get(name).get('originalHeight') : null;
        
        // Store current dimensions before clearing content
        const currentWidth = scrollbar.style.width;
        const currentHeight = scrollbar.style.height;
        
        // Clear existing content except for the thumb element
        const thumb = scrollbar.querySelector('.scrollbar-thumb');
        if (thumb) {
            thumb.remove();
        }
        scrollbar.innerHTML = '';
        
        // Handle different scrollbar types
        if (scrollType === 'round') {
            // For loaded scrollbars, respect the dimensions from the JSON file
            if (isLoadedFromJSON) {
                // Get the original dimensions from customProperties
                const originalWidth = WebTalkObjects.customProperties.get(name).get('originalWidth');
                const originalHeight = WebTalkObjects.customProperties.get(name).get('originalHeight');
                
                if (originalWidth && originalHeight) {
                    console.log(`Applying JSON dimensions to ${name}: ${originalWidth} × ${originalHeight}`);
                    // Apply the original dimensions from JSON and override CSS minimums
                    // Defer updates if screen is locked
                    if (interpreter && interpreter.isScreenLocked) {
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'width',
                            value: originalWidth
                        });
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'height',
                            value: originalHeight
                        });
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'minWidth',
                            value: originalWidth
                        });
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'minHeight',
                            value: originalHeight
                        });
                    } else {
                        scrollbar.style.width = originalWidth;
                        scrollbar.style.height = originalHeight;
                        scrollbar.style.minWidth = originalWidth;
                        scrollbar.style.minHeight = originalHeight;
                    }
                    
                    // Store current dimensions for debugging
                    console.log(`After applying: width=${scrollbar.style.width}, height=${scrollbar.style.height}`);
                } else {
                    console.log(`No original dimensions found for ${name}, using current dimensions`);
                    // Use current dimensions if no original dimensions are available
                    const size = Math.max(scrollbar.offsetWidth, scrollbar.offsetHeight);
                    const sizeValue = `${size}px`;
                    if (interpreter && interpreter.isScreenLocked) {
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'width',
                            value: sizeValue
                        });
                        interpreter.deferredUpdates.push({
                            type: 'style',
                            element: scrollbar,
                            property: 'height',
                            value: sizeValue
                        });
                    } else {
                        scrollbar.style.width = sizeValue;
                        scrollbar.style.height = sizeValue;
                    }
                }
            } else {
                // For new scrollbars, ensure they are perfectly round
                const size = Math.max(scrollbar.offsetWidth, scrollbar.offsetHeight);
                const sizeValue = `${size}px`;
                if (interpreter && interpreter.isScreenLocked) {
                    interpreter.deferredUpdates.push({
                        type: 'style',
                        element: scrollbar,
                        property: 'width',
                        value: sizeValue
                    });
                    interpreter.deferredUpdates.push({
                        type: 'style',
                        element: scrollbar,
                        property: 'height',
                        value: sizeValue
                    });
                } else {
                    scrollbar.style.width = sizeValue;
                    scrollbar.style.height = sizeValue;
                }
            }
            // Always set border radius to make it circular
            if (interpreter && interpreter.isScreenLocked) {
                interpreter.deferredUpdates.push({
                    type: 'style',
                    element: scrollbar,
                    property: 'borderRadius',
                    value: '50%'
                });
            } else {
                scrollbar.style.borderRadius = '50%';
            }
            
            // Create SVG for circular progress indicator
            const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            svg.setAttribute('width', '100%');
            svg.setAttribute('height', '100%');
            svg.setAttribute('viewBox', '0 0 100 100');
            svg.style.position = 'absolute';
            svg.style.top = '0';
            svg.style.left = '0';
            
            // Create circular background
            const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
            circle.setAttribute('cx', '50');
            circle.setAttribute('cy', '50');
            circle.setAttribute('r', '45');
            circle.setAttribute('fill', 'transparent');
            circle.setAttribute('stroke', '#ccc');
            circle.setAttribute('stroke-width', '5');
            svg.appendChild(circle);
            
            // Only draw progress arc if there is progress to show
            if (percentage > 0) {
                // Create progress arc
                const progressArc = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                const progressColor = this.customProperties.get(name).get('progressColor');
                
                // For circular progress, we want to draw an arc starting at 12 o'clock (top)
                // and moving clockwise based on the percentage
                const startAngle = -90; // Start at 12 o'clock (-90 degrees)
                
                // When percentage is exactly 1 (100%), use 99.9% to make the circle appear complete
                // This prevents the arc from disappearing when start and end points are identical
                const adjustedPercentage = percentage >= 0.999 ? 0.999 : percentage;
                const endAngle = startAngle + (adjustedPercentage * 360); // End angle based on percentage
                
                // Convert angles to radians for calculations
                const startRad = (startAngle * Math.PI) / 180;
                const endRad = (endAngle * Math.PI) / 180;
                
                // Calculate points on the circle for the arc
                const radius = 45;
                const startX = 50 + radius * Math.cos(startRad);
                const startY = 50 + radius * Math.sin(startRad);
                const endX = 50 + radius * Math.cos(endRad);
                const endY = 50 + radius * Math.sin(endRad);
                
                // Create the arc path
                // Use the 'large-arc-flag' (0 for arcs less than 180 degrees, 1 for arcs greater than 180 degrees)
                const largeArcFlag = percentage > 0.5 ? 1 : 0;
                const arcPath = `M ${startX} ${startY} A ${radius} ${radius} 0 ${largeArcFlag} 1 ${endX} ${endY}`;
                
                progressArc.setAttribute('d', arcPath);
                progressArc.setAttribute('stroke', progressColor ? this.parseColor(progressColor) : 'rgb(0,0,255)');
                progressArc.setAttribute('stroke-width', '5');
                progressArc.setAttribute('fill', 'none');
                svg.appendChild(progressArc);
            }
            
            scrollbar.appendChild(svg);
            
            // Re-create the thumb element for event handling
            const newThumb = document.createElement('div');
            newThumb.className = 'scrollbar-thumb';
            newThumb.style.position = 'absolute';
            newThumb.style.width = '100%';
            newThumb.style.height = '100%';
            newThumb.style.opacity = '0'; // Make it invisible
            newThumb.style.cursor = 'pointer';
            scrollbar.appendChild(newThumb);
        } else {
            // Standard bar type scrollbar
            // Reset any inline styles that might have been set for round scrollbars
            scrollbar.style.borderRadius = '';
            
            // Check if this scrollbar has custom dimensions stored in customProperties
            const customWidth = this.customProperties.get(name).get('width');
            const customHeight = this.customProperties.get(name).get('height');
            
            // If no custom dimensions are stored, use defaults only for new scrollbars
            // This preserves any dimensions that were set via HyperTalk commands
            if (!customWidth && !customHeight) {
                // Only set default dimensions for newly created scrollbars
                // that haven't had their dimensions explicitly set
                const hasExplicitDimensions = scrollbar.style.width && scrollbar.style.width !== '150px' && 
                                             scrollbar.style.height && scrollbar.style.height !== '20px' && 
                                             scrollbar.style.height !== '150px';
                
                if (!hasExplicitDimensions) {
                    if (scrollbar.classList.contains('horizontal')) {
                        // Default horizontal scrollbar dimensions
                        scrollbar.style.width = '150px';
                        scrollbar.style.height = '20px';
                    } else {
                        // Default vertical scrollbar dimensions
                        scrollbar.style.width = '20px';
                        scrollbar.style.height = '150px';
                    }
                }
            } else {
                // Apply stored custom dimensions if available
                if (customWidth) scrollbar.style.width = customWidth;
                if (customHeight) scrollbar.style.height = customHeight;
            }
            
            if (hasSlider) {
                // Create draggable slider element instead of progress indicator
                // this is where we style our draggable "slider" element when setting the "hasSlider" of a scrollbar to true
                const slider = document.createElement('div');
                slider.className = 'slider';
                slider.style.position = 'absolute';
                slider.style.width = '20px';
                slider.style.height = '100%';
                slider.style.backgroundColor = this.parseColor('255,255,255'); // white slider
                slider.style.cursor = 'grab';
                slider.style.borderRadius = '3px 3px 12px 12px'; // 3px top corners, 12px bottom corners
                slider.style.border = '1px solid #999999';
                slider.style.boxSizing = 'border-box';
                
                // Position the slider based on thumbPosition and orientation
                if (scrollbar.classList.contains('vertical')) {
                    slider.style.width = '100%';
                    slider.style.height = '30px';
                    slider.style.top = `${percentage * (100 - (30 / scrollbar.offsetHeight * 100))}%`;
                    slider.style.left = '0';
                } else {
                    slider.style.width = '20px';
                    slider.style.height = '140%';
                    slider.style.left = `${percentage * (100 - (30 / scrollbar.offsetWidth * 100))}%`;
                    slider.style.top = '-4px';
                }
                
                scrollbar.appendChild(slider);
                
                // Debug slider properties
                console.log('Slider created with dimensions:', {
                    width: slider.offsetWidth,
                    height: slider.offsetHeight,
                    position: slider.style.position,
                    left: slider.style.left,
                    top: slider.style.top,
                    zIndex: slider.style.zIndex,
                    pointerEvents: slider.style.pointerEvents
                });
                
                // Ensure slider is clickable
                slider.style.pointerEvents = 'auto';
                slider.style.zIndex = '10';
                
                // Add drag functionality to the slider
                this.makeSliderDraggable(slider, scrollbar, name);
            } else {
                // Create progress element (fills from start to position)
                const progress = document.createElement('div');
                progress.className = 'scrollbar-progress';
                progress.style.position = 'absolute';
                progress.style.top = '0';
                progress.style.left = '0';
                
                // Apply progress color
                const progressColor = this.customProperties.get(name).get('progressColor');
                if (progressColor) {
                    progress.style.setProperty('background-color', this.parseColor(progressColor), 'important');
                } else {
                    progress.style.setProperty('background-color', 'rgb(0,0,255)', 'important');
                }
                
                // Set the size based on orientation
                if (scrollbar.classList.contains('vertical')) {
                    progress.style.width = '100%';
                    progress.style.height = `${percentage * 100}%`;
                } else {
                    progress.style.height = '100%';
                    progress.style.width = `${percentage * 100}%`;
                }
                
                scrollbar.appendChild(progress);
            }
            
            // Re-create the thumb element for event handling
            const newThumb = document.createElement('div');
            newThumb.className = 'scrollbar-thumb';
            newThumb.style.position = 'absolute';
            newThumb.style.width = '100%';
            newThumb.style.height = '100%';
            newThumb.style.opacity = '0'; // Make it invisible
            newThumb.style.cursor = 'pointer';
            scrollbar.appendChild(newThumb);
        }
    }

    static makeSliderDraggable(slider, scrollbar, name) {
        console.log('makeSliderDraggable called for:', name, 'slider element:', slider);
        
        slider.addEventListener('mousedown', (e) => {
            console.log('Slider mousedown event fired!');
            e.preventDefault();
            e.stopPropagation();
            
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            
            // Debug logging to check mode
            console.log('Slider drag attempt - mode:', window.webTalkApp?.interpreter?.mode, 'isEditMode:', isEditMode);
            
            if (isEditMode) {
                console.log('Blocking drag - in edit mode');
                return; // Don't allow dragging in edit mode
            }
            
            console.log('Allowing drag - in browse mode');
            slider.style.cursor = 'grabbing';
            
            const scrollbarRect = scrollbar.getBoundingClientRect();
            const isVertical = scrollbar.classList.contains('vertical');
            const startValue = parseInt(this.customProperties.get(name).get('startValue') || '0');
            const endValue = parseInt(this.customProperties.get(name).get('endValue') || '100');
            const range = endValue - startValue;
            
            // Get actual slider dimensions
            const sliderRect = slider.getBoundingClientRect();
            const sliderWidth = sliderRect.width;
            const sliderHeight = sliderRect.height;
            
            // Calculate initial mouse offset within the slider
            const initialMouseX = e.clientX - sliderRect.left;
            const initialMouseY = e.clientY - sliderRect.top;
            
            const onMouseMove = (moveEvent) => {
                moveEvent.preventDefault();
                
                let percentage;
                if (isVertical) {
                    const scrollbarHeight = scrollbarRect.height;
                    const mouseY = moveEvent.clientY - scrollbarRect.top - initialMouseY;
                    const availableHeight = scrollbarHeight - sliderHeight;
                    percentage = Math.max(0, Math.min(1, mouseY / availableHeight));
                } else {
                    const scrollbarWidth = scrollbarRect.width;
                    const mouseX = moveEvent.clientX - scrollbarRect.left - initialMouseX;
                    const availableWidth = scrollbarWidth - sliderWidth;
                    percentage = Math.max(0, Math.min(1, mouseX / availableWidth));
                }
                
                const newValue = Math.round(startValue + (percentage * range));
                this.customProperties.get(name).set('thumbPosition', newValue.toString());
                
                // Update slider position immediately
                if (isVertical) {
                    slider.style.top = `${percentage * (100 - (sliderHeight / scrollbar.offsetHeight * 100))}%`;
                } else {
                    slider.style.left = `${percentage * (100 - (sliderWidth / scrollbar.offsetWidth * 100))}%`;
                }
            };
            
            const onMouseUp = () => {
                slider.style.cursor = 'grab';
                document.removeEventListener('mousemove', onMouseMove);
                document.removeEventListener('mouseup', onMouseUp);
            };
            
            document.addEventListener('mousemove', onMouseMove);
            document.addEventListener('mouseup', onMouseUp);
        });
    }

    static getObject(nameOrId, cardId = null, options = {}) {
        if (!nameOrId) return null;

        // Special handling for 'card' references
        if (nameOrId === 'card' || nameOrId === 'this card' || nameOrId === 'current card') {
            if (cardId && cardId > 1) {
                return document.getElementById(`card-${cardId}`);
            }
            return document.getElementById('card');
        }
        
        // Handle card references by number (e.g., 'card 1')
        const cardMatch = typeof nameOrId === 'string' && nameOrId.match(/^card\s+(\d+)$/i);
        if (cardMatch) {
            const matchedCardId = parseInt(cardMatch[1]);
            // If cardId is provided, it takes precedence over the matched card ID
            const targetCardId = cardId || matchedCardId;
            
            if (targetCardId === 1) {
                return document.getElementById('card');
            } else {
                return document.getElementById(`card-${targetCardId}`);
            }
        }

        // First try to get by name
        const objectByName = this.objects.get(nameOrId);
        if (objectByName) {
            return objectByName;
        }

        // If not found by name, try to get by ID
        const objectName = this.objectsById.get(nameOrId);
        if (objectName) {
            return this.objects.get(objectName);
        }

        // If we need to check for similar names (case-insensitive)
        if (options.checkSimilar) {
            const lowerNameOrId = nameOrId.toLowerCase();
            let similarName = null;
            
            // Check for case-insensitive match
            for (const [name, obj] of this.objects.entries()) {
                if (name.toLowerCase() === lowerNameOrId) {
                    similarName = name;
                    break;
                }
            }
            
            if (similarName) {
                options.similarFound = similarName;
            }
        }

        return null;
    }
    
    static getObjectByType(name, type, cardId = null, options = {}) {
        const object = this.getObject(name, cardId, options);
        if (!object) return null;
        
        // Check if the object is of the specified type
        if (!object.dataset || !object.dataset.type) return null;
        
        // Case-insensitive comparison of types
        if (object.dataset.type.toLowerCase() !== type.toLowerCase()) return null;
        
        return object;
    }
    
    /**
     * Get an object by name and type, returning an error message if not found
     * @param {string} name - Object name
     * @param {string} type - Object type (field, button, etc.)
     * @param {Object} options - Additional options
     * @returns {HTMLElement|string} - The object if found, or an error message if not found
     */
    static getObjectByTypeWithErrorMessage(name, type, cardId = null, options = {}) {
        const object = this.getObject(name, cardId, options);
        if (!object) {
            return `${type.charAt(0).toUpperCase() + type.slice(1)} "${name}" not found`;
        }
        
        // Check if the object is of the specified type
        if (!object.dataset || !object.dataset.type) {
            return `Invalid ${type} object "${name}"`;
        }
        
        // Case-insensitive comparison of types
        if (object.dataset.type.toLowerCase() !== type.toLowerCase()) {
            return `Object "${name}" is not a ${type}`;
        }
        
        return object;
    }

    static deleteObject(name) {
        const element = this.getObject(name);
        if (element) {
            // Remove from DOM
            element.parentNode.removeChild(element);

            // Clean up maps
            this.objects.delete(name);
            this.scripts.delete(name);
            this.customProperties.delete(name);
            this.objectsById.delete(element.id);

            // Refresh overview palette if visible
            if (window.refreshObjectOverview) {
                window.refreshObjectOverview();
            }
        }
    }

    /**
     * Updates the visual position of a scrollbar's thumb based on its properties
     * @param {HTMLElement} scrollbarElement - The scrollbar DOM element
     */
    static updateScrollbarThumbPosition(scrollbarElement) {
        if (!scrollbarElement || scrollbarElement.dataset.type !== 'scrollbar') return;
        
        const name = scrollbarElement.dataset.name;
        if (!name || !this.customProperties.has(name)) return;
        
        // Get the current values from customProperties
        const thumbPosition = parseFloat(this.customProperties.get(name).get('thumbPosition') || 0);
        const startValue = parseFloat(this.customProperties.get(name).get('startValue') || 0);
        const endValue = parseFloat(this.customProperties.get(name).get('endValue') || 100);
        const scrollType = this.customProperties.get(name).get('scrollType') || 'bar';
        const isHorizontal = scrollbarElement.classList.contains('horizontal');
        
        // Find the thumb element
        const thumbElement = scrollbarElement.querySelector('.scrollbar-thumb');
        if (!thumbElement) return;
        
        // Calculate the percentage position (0 to 1)
        const range = endValue - startValue;
        const percentage = range === 0 ? 0 : (thumbPosition - startValue) / range;
        
        // Apply the position based on orientation
        if (isHorizontal) {
            thumbElement.style.left = `${percentage * 100}%`;
        } else {
            thumbElement.style.top = `${percentage * 100}%`;
        }
    }
    
    static setObjectProperty(object, property, value, options = null, interpreter = null, cardId = null) {
        // Handle arrayData property - special case for setting array data from strings
        if (property && property.startsWith('arrayData(') && property.endsWith(')')) {
            const arrayPropName = property.substring(10, property.length - 1).trim();
            // Remove quotes if present
            const cleanPropName = arrayPropName.replace(/^["']|["']$/g, '');
            
            let objectName;
            if (typeof object === 'string') {
                objectName = object;
            } else if (object && object.dataset && object.dataset.name) {
                objectName = object.dataset.name;
            } else {
                throw new Error(`Cannot set arrayData: Invalid object ${object}`);
            }
            
            // Ensure custom properties map exists for this object
            if (!this.customProperties.has(objectName)) {
                this.customProperties.set(objectName, new Map());
            }
            
            // Handle special case for 'empty'
            if (value === 'empty' || value === '"empty"' || value === "'empty'") {
                // Clear the array
                this.customProperties.get(objectName).set(cleanPropName, []);
                return;
            }
            
            // Convert string value to array by splitting on newlines
            if (typeof value === 'string') {
                const arrayValue = value.split('\n');
                this.customProperties.get(objectName).set(cleanPropName, arrayValue);
                return;
            }
            
            // If value is already an array, store it directly
            if (Array.isArray(value)) {
                this.customProperties.get(objectName).set(cleanPropName, value);
                return;
            }
            
            // For other types, convert to string and store as single-item array
            this.customProperties.get(objectName).set(cleanPropName, [value.toString()]);
            return;
        }
        
        // Helper function to defer a style update when screen is locked
        const deferStyleUpdate = (element, property, value) => {
            if (interpreter && interpreter.isScreenLocked) {
                console.log(`Deferring style update: ${property} = ${value}`);
                interpreter.deferredUpdates.push({
                    type: 'style',
                    element: element,
                    property: property,
                    value: value
                });
                return true; // Update was deferred
            }
            return false; // Update was not deferred
        };
        
        // Helper function to defer an attribute update when screen is locked
        const deferAttributeUpdate = (element, attribute, value) => {
            if (interpreter && interpreter.isScreenLocked) {
                console.log(`Deferring attribute update: ${attribute} = ${value}`);
                interpreter.deferredUpdates.push({
                    type: 'attribute',
                    element: element,
                    attribute: attribute,
                    value: value
                });
                return true; // Update was deferred
            }
            return false; // Update was not deferred
        };
        
        // Handle global properties first, regardless of object
        if (property.toLowerCase() === 'collisionrate') {
            // Parse and validate the value
            const rate = parseInt(value);
            if (isNaN(rate) || rate < 0) {
                throw new Error('collisionRate must be a non-negative number');
            }
            
            // Set the global collisionRate property
            this.collisionRate = rate;
            console.log(`Set global collisionRate to ${rate}ms`);
            return value;
        }
        // Special handling for 'card' object
        if (typeof object === 'string' &&
            (object === 'card' || object === 'this card' || object === 'current card' || object.match(/^card\s+\d+$/i))) {
            // Get the card element and id
            let card;
            let targetCardId = '1';
            let oldName = 'card 1';
            
            // If cardId parameter is provided, it takes precedence
            if (cardId) {
                targetCardId = cardId.toString();
                card = cardId === 1 ? document.getElementById('card') : document.getElementById(`card-${cardId}`);
                if (card && card.dataset.name) {
                    oldName = card.dataset.name;
                }
            } else if (object === 'card' || object === 'this card' || object === 'current card') {
                card = document.getElementById('card');
                if (card && card.dataset.id) {
                    targetCardId = card.dataset.id;
                }
                if (card && card.dataset.name) {
                    oldName = card.dataset.name;
                }
            } else {
                // Handle 'card N' format
                const match = object.match(/^card\s+(\d+)$/i);
                if (match) {
                    targetCardId = match[1];
                    const matchedCardId = parseInt(targetCardId);
                    card = matchedCardId === 1 ? document.getElementById('card') : document.getElementById(`card-${matchedCardId}`);
                    if (card && card.dataset.name) {
                        oldName = card.dataset.name;
                    }
                }
            }
            
            if (!card) {
                card = document.getElementById('card');
            }
            
            if (card) {
                // Handle card-specific properties
                switch (property.toLowerCase()) {
                    case 'name':
                        // Check if a card with this name already exists
                        if (this.objects.has(value) && value !== oldName) {
                            throw new Error(`An object with the name "${value}" already exists`);
                        }
                        
                        // Update the objects map
                        this.objects.delete(oldName);
                        this.objects.set(value, card);
                        
                        // Update the objectsById map
                        this.objectsById.set(targetCardId, value);
                        
                        // Update the element's dataset
                        card.dataset.name = value;
                        
                        // Update any scripts
                        if (this.scripts.has(oldName)) {
                            const script = this.scripts.get(oldName);
                            this.scripts.delete(oldName);
                            this.scripts.set(value, script);
                        }
                        
                        // Update any custom properties
                        if (this.customProperties.has(oldName)) {
                            const props = this.customProperties.get(oldName);
                            this.customProperties.delete(oldName);
                            this.customProperties.set(value, props);
                        }
                        
                        return value;
                    case 'backgroundcolor':
                    case 'backgroundcolour':
                        const bgColor = this.parseColor(value);
                        
                        // If screen is locked, defer the update, otherwise apply immediately
                        if (!deferStyleUpdate(card, 'backgroundColor', bgColor)) {
                            card.style.backgroundColor = bgColor;
                        }
                        
                        // Clear any background image if setting background color
                        if (!deferStyleUpdate(card, 'backgroundImage', '')) {
                            card.style.backgroundImage = '';
                        }

                        // Store in custom properties
                        if (!this.customProperties.has('card')) {
                            this.customProperties.set('card', new Map());
                        }
                        this.customProperties.get('card').set('backgroundcolor', value);
                        
                        // Clear any background pattern properties when setting background color
                        if (this.customProperties.get('card').has('backgroundPattern')) {
                            this.customProperties.get('card').delete('backgroundPattern');
                            this.customProperties.get('card').delete('backgroundPatternData');
                            this.customProperties.get('card').delete('backgroundPatternType');
                        }
                        return value;
                    case 'backgroundpattern':
                        // The value might be directly the image data from another image
                        // or it might be a string reference that we need to evaluate
                        
                        let sourceImageData;
                        // Try to use existing type if available, otherwise detect from content
                        let sourceImageType = null; // Will be determined based on content or existing value
                        let sourceImageName;
                        
                        // Case 1: Direct image data from interpreter (already evaluated)
                        if (typeof value === 'object' && value.type === 'IMAGE_DATA') {
                            sourceImageName = value.sourceName;
                            sourceImageData = value.data;
                            // Try to detect SVG content from the data
                            if (value.data && typeof value.data === 'string') {
                                try {
                                    const decodedData = atob(value.data);
                                    if (decodedData.includes('<svg') || 
                                        (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                        sourceImageType = 'image/svg+xml';
                                    } else {
                                        // Use provided dataType if available, otherwise keep as null to be set later
                                        sourceImageType = value.dataType || null;
                                    }
                                } catch (e) {
                                    // Use provided dataType if available, otherwise keep as null to be set later
                                    sourceImageType = value.dataType || null;
                                }
                            } else {
                                // Use provided dataType if available, otherwise keep as null to be set later
                                sourceImageType = value.dataType || null;
                            }
                        }
                        // Case 2: String reference to another image's data
                        else if (typeof value === 'string' && value.startsWith('the imageData of image ')) {
                            // Extract the source image name
                            sourceImageName = value.substring('the imageData of image '.length).replace(/^["']|["']$/g, '');
                            
                            // Get the source image
                            const sourceImage = this.getObject(sourceImageName);
                            if (!sourceImage) {
                                throw new Error(`Source image not found: ${sourceImageName}`);
                            }
                            
                            // Get the source image data
                            if (this.customProperties.has(sourceImageName)) {
                                sourceImageData = this.customProperties.get(sourceImageName).get('data');
                                // Get the source image type, prioritizing SVG if that's what it is
                                // Get the source image type, preserving the original type if available
                                sourceImageType = this.customProperties.get(sourceImageName).get('type') || null;
                                
                                // If we have data, check if it's SVG content
                                if (sourceImageData && typeof sourceImageData === 'string') {
                                    try {
                                        const decodedData = atob(sourceImageData);
                                        if (decodedData.includes('<svg') || 
                                            (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                            sourceImageType = 'image/svg+xml';
                                        }
                                    } catch (e) {
                                        // Not base64 data, continue with existing type
                                    }
                                }
                            }
                        }
                        // Case 3: Direct data passed from another property
                        else if (typeof value === 'string' && value.length > 0) {
                            // Try to use it directly as image data
                            sourceImageData = value;
                        }
                        
                        // Validate we have image data
                        if (!sourceImageData) {
                            throw new Error(`No valid image data found${sourceImageName ? ` in image "${sourceImageName}"` : ''}`);
                        }
                        
                        // Set the background pattern on the card
                        // If sourceImageType is still null at this point, check the data for SVG content
                        if (sourceImageType === null) {
                            // Try to detect SVG content from the data
                            if (sourceImageData && typeof sourceImageData === 'string') {
                                try {
                                    const decodedData = atob(sourceImageData);
                                    if (decodedData.includes('<svg') || 
                                        (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                        sourceImageType = 'image/svg+xml';
                                        console.log('SVG content detected in backgroundpattern');
                                    } else {
                                        sourceImageType = 'image/png'; // Default as last resort
                                    }
                                } catch (e) {
                                    sourceImageType = 'image/png'; // Default if not base64
                                }
                            } else {
                                sourceImageType = 'image/png'; // Default if no data
                            }
                        }
                        
                        console.log(`Using image type for backgroundpattern: ${sourceImageType}`);
                        const bgImageUrl = `url(data:${sourceImageType};base64,${sourceImageData})`;
                        if (!deferStyleUpdate(card, 'backgroundImage', bgImageUrl)) {
                            card.style.backgroundImage = bgImageUrl;
                        }
                        
                        if (!deferStyleUpdate(card, 'backgroundRepeat', 'repeat')) {
                            card.style.backgroundRepeat = 'repeat';
                        }
                        
                        // Clear any background color when setting background pattern
                        if (!deferStyleUpdate(card, 'backgroundColor', '')) {
                            card.style.backgroundColor = '';
                        }
                        
                        // Store in custom properties
                        if (!this.customProperties.has('card')) {
                            this.customProperties.set('card', new Map());
                        }
                        this.customProperties.get('card').set('backgroundPattern', sourceImageName || 'custom');
                        this.customProperties.get('card').set('backgroundPatternData', sourceImageData);
                        this.customProperties.get('card').set('backgroundPatternType', sourceImageType);
                        
                        // Clear any background color property when setting background pattern
                        if (this.customProperties.get('card').has('backgroundcolor')) {
                            this.customProperties.get('card').delete('backgroundcolor');
                        }
                        return value;
                }
            }
        }

        // Handle card references in the form "card" without any other qualifiers
        if (typeof object === 'string' && object.toLowerCase() === 'card') {
            const card = document.getElementById('card');
            if (card) {
                // Handle card-specific properties
                switch (property.toLowerCase()) {
                    case 'backgroundcolor':
                    case 'backgroundcolour':
                        card.style.backgroundColor = this.parseColor(value);

                        // Store in custom properties
                        if (!this.customProperties.has('card')) {
                            this.customProperties.set('card', new Map());
                        }
                        this.customProperties.get('card').set('backgroundcolor', value);
                        return value;
                }
            }
        }

        const element = typeof object === 'string' ? this.getObject(object) : object;

        if (!element) {
            throw new Error(`Object not found: ${object}`);
        }

        const objectName = element.dataset.name;
        const objectType = element.dataset.type;
        const objectId = element.id;

        switch (property.toLowerCase()) {
            case 'blendlevel':
                // Handle blendLevel property (0-100, where 0 is fully visible and 100 is fully transparent)
                let blendValue = parseInt(value);
                
                // Validate the blend level value
                if (isNaN(blendValue) || blendValue < 0) {
                    blendValue = 0; // Default to fully visible if invalid
                } else if (blendValue > 100) {
                    blendValue = 100; // Cap at maximum transparency
                }
                
                // Convert blendLevel (0-100) to CSS opacity (1-0)
                // 0 blendLevel = 1.0 opacity (fully visible)
                // 100 blendLevel = 0.0 opacity (fully transparent)
                const opacity = (100 - blendValue) / 100;
                
                // Apply the opacity to the element
                if (!deferStyleUpdate(element, 'opacity', opacity)) {
                    element.style.opacity = opacity;
                }
                
                // Store in custom properties
                if (!this.customProperties.has(objectName)) {
                    this.customProperties.set(objectName, new Map());
                }
                this.customProperties.get(objectName).set('blendLevel', blendValue.toString());
                
                return value;
                
            case 'name':
            case 'shortname':
                // Rename the object
                
                // For buttons, check if it's a checkbox before renaming
                let wasCheckbox = false;
                let wasHighlighted = false;
                
                if (objectType === 'button' && this.customProperties.has(objectName)) {
                    const props = this.customProperties.get(objectName);
                    wasCheckbox = props.get('isCheckbox') === true;
                    wasHighlighted = props.get('hilighted') === true;
                    console.log(`Button rename: ${objectName} -> ${value}, wasCheckbox: ${wasCheckbox}, wasHighlighted: ${wasHighlighted}`);
                }
                
                if (this.objects.has(value)) {
                    throw new Error(`An object with the name "${value}" already exists`);
                }

                // Update the objects map
                this.objects.delete(objectName);
                this.objects.set(value, element);
                
                // Update the objectsById map
                this.objectsById.set(objectId, value);

                // Update the element's dataset
                element.dataset.name = value;

                // If it's a button, update the displayed text
                if (objectType === 'button') {
                    element.textContent = value;
                }

                // Update any scripts or custom properties
                if (this.scripts.has(objectName)) {
                    const script = this.scripts.get(objectName);
                    this.scripts.delete(objectName);
                    this.scripts.set(value, script);
                }

                if (this.customProperties.has(objectName)) {
                    const props = this.customProperties.get(objectName);
                    this.customProperties.delete(objectName);
                    this.customProperties.set(value, props);
                    
                    // Restore checkbox appearance if it was a checkbox button
                    if (wasCheckbox && objectType === 'button') {
                        // First set isCheckbox to false
                        this.setObjectProperty(value, 'isCheckbox', false);
                        
                        // Then set it back to true to restore the checkbox appearance
                        this.setObjectProperty(value, 'isCheckbox', true);
                        
                        // Also restore the hilighted state if needed
                        if (wasHighlighted) {
                            this.setObjectProperty(value, 'hilighted', true);
                        }
                        
                        console.log(`Restored checkbox appearance for ${value}`);
                    }
                }

                // Keep the ID mapping updated
                this.objectsById.set(objectId, value);

                break;
            case 'left':
                if (!deferStyleUpdate(element, 'left', `${value}px`)) {
                    element.style.left = `${value}px`;
                }
                break;
            case 'top':
                element.style.top = `${value}px`;
                break;
            case 'right':
                // Set the right edge position (left + width)
                const rightWidth = parseInt(element.style.width) || 100;
                element.style.left = `${parseInt(value) - rightWidth}px`;
                break;
            case 'bottom':
                // Set the bottom edge position (top + height)
                const bottomHeight = parseInt(element.style.height) || 50;
                element.style.top = `${parseInt(value) - bottomHeight}px`;
                break;
            case 'topleft':
                // Parse the x,y coordinates from the value
                const topLeftCoords = value.split(',');
                if (topLeftCoords.length !== 2) {
                    throw new Error('Invalid topleft format. Expected: "x,y"');
                }
                element.style.left = `${topLeftCoords[0].trim()}px`;
                element.style.top = `${topLeftCoords[1].trim()}px`;
                break;
            case 'topright':
                // Parse the x,y coordinates from the value
                const topRightCoords = value.split(',');
                if (topRightCoords.length !== 2) {
                    throw new Error('Invalid topright format. Expected: "x,y"');
                }
                const trWidth = parseInt(element.style.width) || 100;
                element.style.left = `${parseInt(topRightCoords[0].trim()) - trWidth}px`;
                element.style.top = `${topRightCoords[1].trim()}px`;
                break;
            case 'bottomleft':
                // Parse the x,y coordinates from the value
                const bottomLeftCoords = value.split(',');
                if (bottomLeftCoords.length !== 2) {
                    throw new Error('Invalid bottomleft format. Expected: "x,y"');
                }
                const blHeight = parseInt(element.style.height) || 50;
                element.style.left = `${bottomLeftCoords[0].trim()}px`;
                element.style.top = `${parseInt(bottomLeftCoords[1].trim()) - blHeight}px`;
                break;
            case 'bottomright':
                // Parse the x,y coordinates from the value
                const bottomRightCoords = value.split(',');
                if (bottomRightCoords.length !== 2) {
                    throw new Error('Invalid bottomright format. Expected: "x,y"');
                }
                const brWidth = parseInt(element.style.width) || 100;
                const brHeight = parseInt(element.style.height) || 50;
                element.style.left = `${parseInt(bottomRightCoords[0].trim()) - brWidth}px`;
                element.style.top = `${parseInt(bottomRightCoords[1].trim()) - brHeight}px`;
                break;
            case 'loc':
            case 'location':
                // Parse the x,y coordinates from the value
                let coordinates;
                if (typeof value === 'string') {
                    coordinates = value.split(',').map(coord => parseInt(coord.trim()));
                } else if (Array.isArray(value) && value.length >= 2) {
                    coordinates = [parseInt(value[0]), parseInt(value[1])];
                } else {
                    throw new Error('Location must be specified as "x,y" or [x,y]');
                }

                if (coordinates.length < 2 || isNaN(coordinates[0]) || isNaN(coordinates[1])) {
                    throw new Error('Invalid location format. Use "x,y" where x and y are numbers');
                }

                const [x, y] = coordinates;
                const width = parseInt(element.style.width) || 0;
                const height = parseInt(element.style.height) || 0;

                // Set position based on the center point
                element.style.left = `${x - width / 2}px`;
                element.style.top = `${y - height / 2}px`;
                break;
            case 'mouseloc':
                // Special property for tracking mouse without centering adjustment
                let mouseCoordinates;
                if (typeof value === 'string') {
                    mouseCoordinates = value.split(',').map(p => parseInt(p.trim()));
                } else if (Array.isArray(value) && value.length >= 2) {
                    mouseCoordinates = [parseInt(value[0]), parseInt(value[1])];
                } else {
                    throw new Error('MouseLoc must be specified as "x,y" or [x,y]');
                }

                if (mouseCoordinates.length < 2 || isNaN(mouseCoordinates[0]) || isNaN(mouseCoordinates[1])) {
                    throw new Error('Invalid mouseloc format. Use "x,y" where x and y are numbers');
                }

                const [mouseX, mouseY] = mouseCoordinates;

                // Set position directly without centering adjustment
                if (!deferStyleUpdate(element, 'left', `${mouseX}px`)) {
                    element.style.left = `${mouseX}px`;
                }
                if (!deferStyleUpdate(element, 'top', `${mouseY}px`)) {
                    element.style.top = `${mouseY}px`;
                }
                break;
            case 'width':
                if (!deferStyleUpdate(element, 'width', `${value}px`)) {
                    element.style.width = `${value}px`;
                }
                
                // Store width in customProperties for scrollbars to preserve custom dimensions
                if (element.dataset.type === 'scrollbar') {
                    const name = element.dataset.name;
                    if (name && this.customProperties.has(name)) {
                        this.customProperties.get(name).set('width', `${value}px`);
                    }
                }
                break;
            case 'height':
                if (!deferStyleUpdate(element, 'height', `${value}px`)) {
                    element.style.height = `${value}px`;
                }
                
                // Store height in customProperties for scrollbars to preserve custom dimensions
                if (element.dataset.type === 'scrollbar') {
                    const name = element.dataset.name;
                    if (name && this.customProperties.has(name)) {
                        this.customProperties.get(name).set('height', `${value}px`);
                    }
                }
                break;
            case 'rotation':
            case 'angle':
                // Set the rotation of the element
                if (isNaN(parseFloat(value))) {
                    throw new Error('Rotation must be a number (degrees)');
                }

                // Check if we have a rotation point specified
                const transformOrigin = (options && options.x !== undefined && options.y !== undefined) ?
                    `${options.x}px ${options.y}px` : 'center center';
                
                // Set or defer transform-origin
                if (!deferStyleUpdate(element, 'transformOrigin', transformOrigin)) {
                    element.style.transformOrigin = transformOrigin;
                }
                
                // Set or defer transform
                const transform = `rotate(${value}deg)`;
                if (!deferStyleUpdate(element, 'transform', transform)) {
                    element.style.transform = transform;
                }

                // Store the rotation value as a custom property
                const rotationProperties = this.customProperties.get(objectName) || new Map();
                rotationProperties.set('rotation', parseFloat(value));
                this.customProperties.set(objectName, rotationProperties);
                break;
            case 'textcolor':
            case 'textcolour':
            case 'foregroundcolor':
            case 'foregroundcolour':
                if (element.dataset.type === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (svg) {
                        // Check the graphic type
                        const graphicType = element.dataset.graphicType || 'line';
                        const parsedColor = this.parseColor(value);

                        if (graphicType === 'line') {
                            const line = svg.querySelector('line');
                            if (line) {
                                if (!deferAttributeUpdate(line, 'stroke', parsedColor)) {
                                    line.setAttribute('stroke', parsedColor);
                                }
                            }
                        } else if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                if (!deferAttributeUpdate(path, 'stroke', parsedColor)) {
                                    path.setAttribute('stroke', parsedColor);
                                }
                            }
                        } else if (graphicType === 'oval') {
                            const ellipse = svg.querySelector('ellipse');
                            if (ellipse) {
                                if (!deferAttributeUpdate(ellipse, 'stroke', parsedColor)) {
                                    ellipse.setAttribute('stroke', parsedColor);
                                }
                            }
                        } else if (graphicType === 'circle') {
                            const circle = svg.querySelector('circle');
                            if (circle) {
                                if (!deferAttributeUpdate(circle, 'stroke', parsedColor)) {
                                    circle.setAttribute('stroke', parsedColor);
                                }
                            }
                        } else if (graphicType === 'rectangle' || graphicType === 'square') {
                            const rect = svg.querySelector('rect');
                            if (rect) {
                                if (!deferAttributeUpdate(rect, 'stroke', parsedColor)) {
                                    rect.setAttribute('stroke', parsedColor);
                                }
                            }
                        } else if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(graphicType)) {
                            const polygon = svg.querySelector('polygon');
                            if (polygon) {
                                if (!deferAttributeUpdate(polygon, 'stroke', parsedColor)) {
                                    polygon.setAttribute('stroke', parsedColor);
                                }
                            }
                        }
                    }
                    // Store the foreground color in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('foregroundColor', value);
                } else {
                    // Parse and apply the color with !important to override any default styles
                    const parsedColor = this.parseColor(value);
                    
                    // For fields, we need to apply the color to both the container and the content element
                    if (element.dataset.type === 'field') {
                        // Apply to the field container or defer if screen is locked
                        if (!deferStyleUpdate(element, 'color', parsedColor)) {
                            element.style.setProperty('color', parsedColor, 'important');
                        }
                        
                        // Find and apply to the field-content element if it exists
                        const fieldContent = element.querySelector('.field-content');
                        if (fieldContent) {
                            if (!deferStyleUpdate(fieldContent, 'color', parsedColor)) {
                                fieldContent.style.setProperty('color', parsedColor, 'important');
                            }
                        }
                        
                        // Create a style tag for this specific field to ensure the color is applied
                        // with the highest possible specificity
                        const styleId = `field-color-${element.id}`;
                        let styleEl = document.getElementById(styleId);
                        
                        if (!styleEl) {
                            styleEl = document.createElement('style');
                            styleEl.id = styleId;
                            document.head.appendChild(styleEl);
                        }
                        
                        // Set the style with very high specificity
                        const styleContent = `
                            #${element.id} { color: ${parsedColor} !important; }
                            #${element.id} * { color: ${parsedColor} !important; }
                        `;
                        
                        // If screen is locked, defer the update, otherwise apply immediately
                        if (interpreter && interpreter.isScreenLocked) {
                            interpreter.deferredUpdates.push({
                                type: 'function',
                                function: () => {
                                    styleEl.textContent = styleContent;
                                }
                            });
                        } else {
                            styleEl.textContent = styleContent;
                        }
                    } else {
                        // For non-field elements, just set the color directly or defer if screen is locked
                        if (!deferStyleUpdate(element, 'color', parsedColor)) {
                            element.style.setProperty('color', parsedColor, 'important');
                        }
                    }
                    
                    // Store the foreground color in custom properties for all object types
                    // Store it with both camelCase and lowercase keys for backward compatibility
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('foregroundColor', value);
                    this.customProperties.get(objectName).set('foregroundcolor', value);
                }
                break;
            case 'backgroundcolor':
            case 'backgroundcolour': // Add UK spelling
                if (element.dataset.type === 'button' || element.dataset.type === 'field') {
                    const parsedColor = this.parseColor(value);
                    element.style.backgroundColor = parsedColor;
                    
                    // For fields, also apply to the field-content element
                    if (element.dataset.type === 'field') {
                        const fieldContent = element.querySelector('.field-content');
                        if (fieldContent) {
                            fieldContent.style.backgroundColor = parsedColor;
                        }
                    }
                } else if (element.dataset.type === 'graphic') {
                    // For graphics, we need special handling based on the graphic type
                    const svg = element.querySelector('svg');
                    if (svg) {
                        // Check the graphic type
                        const graphicType = element.dataset.graphicType || 'line';

                        // Fill only applies to path elements, not lines
                        if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    path.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    path.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        } else if (graphicType === 'oval') {
                            const ellipse = svg.querySelector('ellipse');
                            if (ellipse) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    ellipse.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    ellipse.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        } else if (graphicType === 'circle') {
                            const circle = svg.querySelector('circle');
                            if (circle) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    circle.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    circle.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        } else if (graphicType === 'rectangle' || graphicType === 'square') {
                            const rect = svg.querySelector('rect');
                            if (rect) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    rect.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    rect.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        } else if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(graphicType)) {
                            const polygon = svg.querySelector('polygon');
                            if (polygon) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    polygon.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    polygon.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        }
                        // For other graphic types, we might not want to set a background
                    }
                    // Store the background color in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('backgroundcolor', value);
                } else {
                    element.style.backgroundColor = this.parseColor(value);
                }

                // Store in custom properties if not a graphic (already stored above for graphics)
                if (element.dataset.type !== 'graphic') {
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('backgroundcolor', value);
                }
                break;
            case 'text':
            case 'content':
                // Special handling for fields with styledText enabled
                if (objectType === 'field') {
                    const fieldContent = element.querySelector('.field-content');
                    const isStyled = this.customProperties.get(objectName)?.get('styledtext') || false;
                    
                    if (fieldContent) {
                        if (interpreter && interpreter.isScreenLocked) {
                            interpreter.deferredUpdates.push({
                                type: 'function',
                                function: () => {
                                    if (isStyled) {
                                        fieldContent.innerHTML = value;
                                    } else {
                                        fieldContent.textContent = value;
                                    }
                                }
                            });
                        } else {
                            if (isStyled) {
                                fieldContent.innerHTML = value;
                            } else {
                                fieldContent.textContent = value;
                            }
                        }
                    } else {
                        // Fallback if field-content not found
                        if (interpreter && interpreter.isScreenLocked) {
                            interpreter.deferredUpdates.push({
                                type: 'function',
                                function: () => {
                                    element.textContent = value;
                                }
                            });
                        } else {
                            element.textContent = value;
                        }
                    }
                } else {
                    // For non-field objects, use textContent as before
                    if (interpreter && interpreter.isScreenLocked) {
                        interpreter.deferredUpdates.push({
                            type: 'function',
                            function: () => {
                                element.textContent = value;
                            }
                        });
                    } else {
                        element.textContent = value;
                    }
                }
                break;
            case 'script':
                this.setScript(objectName, value);
                return value;
            // Menu-specific properties for buttons
            case 'ismenu':
                if (element.dataset.type === 'button') {
                    // Convert value to boolean
                    let boolValue;
                    if (typeof value === 'boolean') {
                        boolValue = value;
                    } else if (typeof value === 'string') {
                        boolValue = value.toLowerCase() === 'true';
                    } else {
                        boolValue = Boolean(value);
                    }
                    
                    // Ensure custom properties map exists for this object
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    
                    // Set the isMenu property
                    this.customProperties.get(objectName).set('isMenu', boolValue);
                    
                    // Add or remove the menu button class immediately
                    if (boolValue === true) {
                        element.classList.add('webtalk-menu-button');
                    } else {
                        element.classList.remove('webtalk-menu-button');
                    }
                    
                    return boolValue;
                }
                return value;
            case 'menutext':
                if (element.dataset.type === 'button') {
                    // Ensure custom properties map exists for this object
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    
                    // Set the menuText property
                    this.customProperties.get(objectName).set('menuText', value);
                    
                    // Make sure the button has the menu class if it's a menu button
                    if (this.customProperties.get(objectName).get('isMenu') === true) {
                        element.classList.add('webtalk-menu-button');
                    }
                    
                    return value;
                }
                return value;
            case 'showname':
            case 'textalign':
                if (objectType === 'field') {
                    const validAlignments = ['left', 'center', 'right', 'justify'];
                    const alignment = value.toString().toLowerCase();
                    if (!validAlignments.includes(alignment)) {
                        throw new Error(`Invalid text alignment: ${value}. Must be one of: ${validAlignments.join(', ')}`);
                    }
                    element.style.textAlign = alignment;
                    this.customProperties.get(objectName).set('textalign', alignment);
                } else if (objectType === 'button') {
                    const validAlignments = ['left', 'center', 'right'];
                    const alignment = value.toString().toLowerCase();
                    if (!validAlignments.includes(alignment)) {
                        throw new Error(`Invalid text alignment: ${value}. Must be one of: ${validAlignments.join(', ')}`);
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('textAlign', alignment);
                    
                    // Update the button appearance to apply the text alignment
                    this.updateButtonAppearance(element, objectName);
                } else {
                    throw new Error(`Cannot set textalign on ${objectType} objects`);
                }
                break;
            case 'scrollable':
                if (objectType === 'field') {
                    const isScrollable = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    element.style.overflow = isScrollable ? 'auto' : 'hidden';
                    this.customProperties.get(objectName).set('scrollable', isScrollable);
                } else {
                    throw new Error(`Cannot set scrollable on ${objectType} objects`);
                }
                break;
            case 'locktext':
                if (objectType === 'field') {
                    const isLocked = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    // Set contentEditable on the inner .field-content div, not the outer field element
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        fieldContent.contentEditable = !isLocked;
                        // Set cursor style based on lock state
                        fieldContent.style.cursor = isLocked ? 'default' : 'text';
                        // Also set cursor on the field container
                        element.style.cursor = isLocked ? 'default' : 'text';
                    }
                    this.customProperties.get(objectName).set('locktext', isLocked);
                } else {
                    throw new Error(`Cannot set locktext on ${objectType} objects`);
                }
                break;
            case 'styledtext':
                if (objectType === 'field') {
                    const isStyled = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        if (isStyled) {
                            // When enabling styledText:
                            // 1. Clear all existing formatting commands
                            this.customProperties.get(objectName).set('formattingCommands', []);
                            
                            // 2. Get current content (plain text with newlines)
                            const currentText = fieldContent.textContent || '';
                            
                            // 3. Set innerHTML to render as HTML
                            fieldContent.innerHTML = currentText;
                            
                            // 4. Set data attribute for CSS styling
                            element.setAttribute('data-styled-text', 'true');
                            
                            // 5. Store the styledText property
                            this.customProperties.get(objectName).set('styledtext', true);
                        } else {
                            // When disabling styledText:
                            // 1. Get the text content (strips all HTML)
                            const plainText = fieldContent.textContent || '';
                            
                            // 2. Set as plain text
                            fieldContent.textContent = plainText;
                            
                            // 3. Remove data attribute
                            element.removeAttribute('data-styled-text');
                            
                            // 4. Store the styledText property
                            this.customProperties.get(objectName).set('styledtext', false);
                        }
                    }
                } else {
                    throw new Error(`Cannot set styledtext on ${objectType} objects`);
                }
                break;
            case 'dontwrap':
                if (objectType === 'field') {
                    const dontWrap = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    // Apply the dontwrap behavior using our helper function
                    this.applyDontWrapToField(objectName, dontWrap);
                    // Store the property value
                    this.customProperties.get(objectName).set('dontwrap', dontWrap);
                } else {
                    throw new Error(`Cannot set dontwrap on ${objectType} objects`);
                }
                break;
            case 'multiline':
                if (objectType === 'field') {
                    const isMultiLine = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    element.style.whiteSpace = isMultiLine ? 'pre-wrap' : 'nowrap';
                    if (isMultiLine) {
                        element.style.lineHeight = '24px'; // set line height to 24px for multiline fields!
                    }
                    this.customProperties.get(objectName).set('multiline', isMultiLine);
                    
                    // Update scroll listener when multiline property changes
                    if (window.updateFieldScrollListener) {
                        window.updateFieldScrollListener(objectName);
                    }
                } else {
                    throw new Error(`Cannot set multiline on ${objectType} objects`);
                }
                break;
            case 'autotab':
                if (objectType === 'field') {
                    const autoTab = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    this.customProperties.get(objectName).set('autotab', autoTab);
                } else {
                    throw new Error(`Cannot set autotab on ${objectType} objects`);
                }
                break;
            case 'lineheight':
                if (objectType === 'field') {
                    // Get the field content element
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Apply the lineHeight to the field content element
                        fieldContent.style.lineHeight = value.toString().includes('px') ? value : `${value}px`;
                    }
                    // Store in custom properties with the correct case
                    this.customProperties.get(objectName).set('lineHeight', value);
                } else {
                    throw new Error(`Cannot set lineHeight on ${objectType} objects`);
                }
                break;
            case 'widemargins':
                if (objectType === 'field') {
                    const wideMargins = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    element.style.padding = wideMargins ? '10px' : '5px';
                    this.customProperties.get(objectName).set('widemargins', wideMargins);
                } else {
                    throw new Error(`Cannot set widemargins on ${objectType} objects`);
                }
                break;
            case 'sharedtext':
                if (objectType === 'field') {
                    const sharedText = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    this.customProperties.get(objectName).set('sharedtext', sharedText);
                } else {
                    throw new Error(`Cannot set sharedtext on ${objectType} objects`);
                }
                break;
            case 'textsize':
                if (objectType === 'field' || objectType === 'button') {
                    const size = parseInt(value);
                    if (isNaN(size) || size <= 0) {
                        throw new Error('Text size must be a positive number');
                    }
                    element.style.fontSize = `${size}px`;
                    this.customProperties.get(objectName).set('textsize', size);
                } else {
                    throw new Error(`Cannot set textsize on ${objectType} objects`);
                }
                break;
            case 'textstyle':
                if (objectType === 'field' || objectType === 'button') {
                    const style = value.toString().toLowerCase();
                    // Reset all styles first
                    element.style.fontWeight = 'normal';
                    element.style.fontStyle = 'normal';
                    element.style.textDecoration = 'none';
                    
                    // Apply the requested style(s)
                    if (style.includes('bold')) {
                        element.style.fontWeight = 'bold';
                    }
                    if (style.includes('italic')) {
                        element.style.fontStyle = 'italic';
                    }
                    if (style.includes('underline')) {
                        element.style.textDecoration = 'underline';
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('textStyle', value);
                } else {
                    throw new Error(`Cannot set textStyle on ${objectType} objects`);
                }
                break;
            case 'hilighted':
            case 'hilite': // Add synonym for hilighted
            case 'highlighted': // Add another synonym for hilighted
                // Convert to boolean
                const isHighlighted = (value === true || value === 'true');
                
                if (objectType === 'field') {
                    // Apply highlight styling for fields
                    if (isHighlighted) {
                        element.classList.add('highlighted');
                    } else {
                        element.classList.remove('highlighted');
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('hilighted', isHighlighted);
                } else if (objectType === 'button') {
                    // For buttons, store the property and update appearance if it's a checkbox
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('hilighted', isHighlighted);
                    
                    // Update button appearance for all buttons when hilighted changes
                    this.updateButtonAppearance(element, objectName);
                } else {
                    throw new Error(`Cannot set hilighted on ${objectType} objects`);
                }
                break;
            case 'disabled':
                if (objectType === 'button') {
                    // Convert value to boolean
                    const isDisabled = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('disabled', isDisabled);
                    
                    // Update the button appearance
                    this.updateButtonAppearance(element, objectName);
                } else {
                    throw new Error(`Cannot set disabled on ${objectType} objects`);
                }
                break;
            case 'showfocusborder':
                if (objectType === 'field') {
                    const showBorder = value === true || value === 'true';
                    const fieldContent = element.querySelector('.field-content');
                    
                    if (fieldContent) {
                        if (showBorder) {
                            // Apply a visible focus border to the outer field element
                            element.style.outline = '2px solid #4d90fe';
                            element.style.outlineOffset = '2px';
                            // Remove the outline from the inner content to avoid double borders
                            fieldContent.style.setProperty('outline', 'none', 'important');
                            console.log(`Enabled focus border for field ${objectName}`);
                        } else {
                            // Hide the focus border on both elements
                            element.style.setProperty('outline', 'none', 'important');
                            fieldContent.style.setProperty('outline', 'none', 'important');
                            console.log(`Disabled focus border for field ${objectName}`);
                        }
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('showFocusBorder', showBorder);
                } else {
                    throw new Error(`Cannot set showFocusBorder on ${objectType} objects`);
                }
                break;
            case 'textfont':
                if (objectType === 'field' || objectType === 'button') {
                    // Set the font family
                    element.style.fontFamily = value;
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('textFont', value);
                } else {
                    throw new Error(`Cannot set textFont on ${objectType} objects`);
                }
                break;
            case 'ischeckbox':
                if (objectType === 'button') {
                    // Convert value to boolean
                    const isCheckbox = typeof value === 'boolean' ? value : value.toString().toLowerCase() === 'true';
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('isCheckbox', isCheckbox);
                    
                    // Update the button appearance
                    this.updateButtonAppearance(element, objectName);
                } else {
                    throw new Error(`Cannot set isCheckbox on ${objectType} objects`);
                }
                break;
            case 'vscroll':
                if (objectType === 'field') {
                    const vScrollValue = parseInt(value);
                       if (isNaN(vScrollValue)) {
                        throw new Error('vScroll must be a number');
                    }
                    element.scrollTop = vScrollValue;
                    this.customProperties.get(objectName).set('vScroll', vScrollValue);
                } else {
                    throw new Error(`Cannot set vScroll on ${objectType} objects`);
                }
                break;
            case 'hscroll':
                if (objectType === 'field') {
                    const hScrollValue = parseInt(value);
                    if (isNaN(hScrollValue)) {
                        throw new Error('hScroll must be a number');
                        }
                    element.scrollLeft = hScrollValue;
                    this.customProperties.get(objectName).set('hScroll', hScrollValue);
                } else {
                    throw new Error(`Cannot set hScroll on ${objectType} objects`);
                }
                break;
            // Graphic-specific properties
            case 'points':
                if (objectType === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (svg) {
                        // Check if we have a line element or a path element
                        let line = svg.querySelector('line');
                        let path = svg.querySelector('path');

                        // Parse the points string
                        const pointsStr = value.toString().trim();

                        // Check if this is a path-based curve (starts with 'M')
                        if (pointsStr.startsWith('M')) {
                            // This is a path-based curve (bezier, arc, etc.)

                            // If we have a line element, remove it and create a path instead
                            if (line) {
                                line.remove();
                                line = null;
                            }

                            // Create path element if it doesn't exist
                            if (!path) {
                                path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                                svg.appendChild(path);
                            }

                            // Set the path data
                            path.setAttribute('d', pointsStr);

                            // Set stroke properties (use existing or defaults)
                            const stroke = line ? line.getAttribute('stroke') :
                                (this.customProperties.has(objectName) &&
                                 this.customProperties.get(objectName).has('foregroundcolor')) ?
                                    this.parseColor(this.customProperties.get(objectName).get('foregroundcolor')) :
                                    'black';

                            const strokeWidth = line ? line.getAttribute('stroke-width') :
                                (this.customProperties.has(objectName) &&
                                 this.customProperties.get(objectName).has('linesize')) ?
                                    this.customProperties.get(objectName).get('linesize') :
                                    1;

                            path.setAttribute('stroke', stroke);
                            path.setAttribute('stroke-width', strokeWidth);

                            // Check if the path is closed (ends with Z or z)
                            const isClosedPath = /[Zz](\s*)$/.test(pointsStr.trim());

                            // Set fill based on whether the path is closed
                            let fill = 'none'; // Default to no fill

                            // If path is closed, check if a fill is already specified
                            if (isClosedPath) {
                                // Check custom properties for fill
                                if (this.customProperties.has(objectName) &&
                                    this.customProperties.get(objectName).has('fill')) {
                                    fill = this.customProperties.get(objectName).get('fill');
                                    // If fill is 'none' or 'transparent', keep it as none
                                    if (fill.toLowerCase() !== 'none' && fill.toLowerCase() !== 'transparent') {
                                        fill = this.parseColor(fill);
                                    }
                                } else {
                                    // Default fill for closed paths if none specified
                                    fill = 'none';
                                }
                            }

                            path.setAttribute('fill', fill);

                            // Update the graphic type
                            element.dataset.graphicType = 'path';
                        } else {
                            // This is a regular line with x1,y1,x2,y2 format

                            // If we have a path element, remove it and create a line instead
                            if (path) {
                                path.remove();
                                path = null;
                            }

                            // Create line element if it doesn't exist
                            if (!line) {
                                line = document.createElementNS('http://www.w3.org/2000/svg', 'line');
                                svg.appendChild(line);
                            }

                            // Parse the points (format: x1,y1,x2,y2)
                            const points = pointsStr.split(',').map(p => parseInt(p.trim()));
                            if (points.length !== 4) {
                                throw new Error('Invalid points format for line. Expected: "x1,y1,x2,y2"');
                            }

                            // Set line attributes
                            line.setAttribute('x1', points[0]);
                            line.setAttribute('y1', points[1]);
                            line.setAttribute('x2', points[2]);
                            line.setAttribute('y2', points[3]);

                            // Set stroke properties (use existing or defaults)
                            const stroke = path ? path.getAttribute('stroke') :
                                (this.customProperties.has(objectName) &&
                                 this.customProperties.get(objectName).has('foregroundcolor')) ?
                                    this.parseColor(this.customProperties.get(objectName).get('foregroundcolor')) :
                                    'black';

                            const strokeWidth = path ? path.getAttribute('stroke-width') :
                                (this.customProperties.has(objectName) &&
                                 this.customProperties.get(objectName).has('linesize')) ?
                                    this.customProperties.get(objectName).get('linesize') :
                                    1;

                            line.setAttribute('stroke', stroke);
                            line.setAttribute('stroke-width', strokeWidth);

                            // Update the graphic type
                            element.dataset.graphicType = 'line';
                        }

                        // Store the points in custom properties
                        if (!this.customProperties.has(objectName)) {
                            this.customProperties.set(objectName, new Map());
                        }
                        this.customProperties.get(objectName).set('points', value.toString());

                        // Store the original points for scaling if not already stored
                        if (!this.customProperties.get(objectName).has('originalpoints')) {
                            this.customProperties.get(objectName).set('originalpoints', value.toString());
                            this.customProperties.get(objectName).set('originalwidth', parseInt(element.style.width));
                            this.customProperties.get(objectName).set('originalheight', parseInt(element.style.height));
                        }
                    }
                } else {
                    throw new Error(`Cannot set points on ${objectType} objects`);
                }
                break;
            case 'linesize':
                if (objectType === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (svg) {
                        const graphicType = element.dataset.graphicType || 'line';

                        if (graphicType === 'line') {
                            const line = svg.querySelector('line');
                            if (line) {
                                line.setAttribute('stroke-width', value);
                            }
                        } else if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                path.setAttribute('stroke-width', value);
                            }
                        } else if (graphicType === 'oval') {
                            const ellipse = svg.querySelector('ellipse');
                            if (ellipse) {
                                ellipse.setAttribute('stroke-width', value);
                            }
                        }
                    }
                    // Set the image data and type in the custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    
                    // Store the linesize value in custom properties
                    this.customProperties.get(objectName).set('linesize', value);
                }
                break;
            case 'showborder':
                if (objectType === 'graphic' || objectType === 'field') {
                    // Convert value to boolean
                    const showBorder = value === true || value === 'true';

                    // Apply border style
                    element.style.border = showBorder ? '1px solid black' : 'none';

                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('showborder', showBorder);
                } else {
                    throw new Error(`Cannot set showBorder on ${objectType} objects`);
                }
                break;
            case 'bordercolor':
                // Apply border color to the element
                element.style.borderColor = this.parseColor(value);
                
                // Store in custom properties
                if (!this.customProperties.has(objectName)) {
                    this.customProperties.set(objectName, new Map());
                }
                this.customProperties.get(objectName).set('bordercolor', value);
                break;
            case 'borderwidth':
                // Convert value to number and apply border width
                const borderWidth = parseInt(value);
                if (!isNaN(borderWidth)) {
                    // Make sure there's a border style if setting width
                    if (!element.style.borderStyle || element.style.borderStyle === 'none') {
                        element.style.borderStyle = 'solid';
                    }
                    element.style.borderWidth = `${borderWidth}px`;
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('borderwidth', borderWidth.toString());
                }
                break;
            // Handle image and player-specific properties
            case 'filename':
                if (objectType === 'image') {
                    // Handle setting the image filename
                    this.setImageFile(element, value);
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('filename', value);
                } else if (objectType === 'player') {
                    // Handle setting the player video source
                    WebTalkPlayer.setPlayerSource(element, value);
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('filename', value);
                } else {
                    throw new Error(`Cannot set filename on ${objectType} objects`);
                }
                break;
            case 'src':
            case 'source':
                if (objectType === 'image') {
                    // Direct setting of the image source
                    const imgElement = element.querySelector('img');
                    if (imgElement) {
                        imgElement.src = value;
                        
                        // Remove placeholder styling
                        element.style.backgroundColor = '';
                        element.style.border = '';
                        
                        // Store in custom properties
                        if (!this.customProperties.has(objectName)) {
                            this.customProperties.set(objectName, new Map());
                        }
                        this.customProperties.get(objectName).set('src', value);
                    }
                } else {
                    throw new Error(`Cannot set src/source on ${objectType} objects`);
                }
                break;
            case 'backgroundpattern':
                if (objectType === 'image' || objectType === 'graphic') {
                    // The value might be directly the image data from another image
                    // or it might be a string reference that we need to evaluate
                    
                    let sourceImageData;
                    let sourceImageType = 'image/png'; // Default type, but we'll try to detect SVG
                    let sourceImageName;
                    
                    // Case 1: Direct image data from interpreter (already evaluated)
                    if (typeof value === 'object' && value.type === 'IMAGE_DATA') {
                        sourceImageName = value.sourceName;
                        sourceImageData = value.data;
                        // Try to detect SVG content from the data
                        if (value.data && typeof value.data === 'string') {
                            try {
                                const decodedData = atob(value.data);
                                if (decodedData.includes('<svg') || 
                                    (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                    sourceImageType = 'image/svg+xml';
                                } else {
                                    sourceImageType = value.dataType || 'image/png';
                                }
                            } catch (e) {
                                sourceImageType = value.dataType || 'image/png';
                            }
                        } else {
                            sourceImageType = value.dataType || 'image/png';
                        }
                    }
                    // Case 2: String reference to another image's data
                    else if (typeof value === 'string' && value.startsWith('the imageData of image ')) {
                        // Extract the source image name
                        sourceImageName = value.substring('the imageData of image '.length).replace(/^["']|["']$/g, '');
                        
                        // Get the source image
                        const sourceImage = this.getObject(sourceImageName);
                        if (!sourceImage) {
                            throw new Error(`Source image not found: ${sourceImageName}`);
                        }
                        
                        // Get the source image data
                        if (this.customProperties.has(sourceImageName)) {
                            sourceImageData = this.customProperties.get(sourceImageName).get('data');
                            // Get the source image type, prioritizing SVG if that's what it is
                            sourceImageType = this.customProperties.get(sourceImageName).get('type') || 'image/png';
                            
                            // If we have data, check if it's SVG content
                            if (sourceImageData && typeof sourceImageData === 'string') {
                                try {
                                    const decodedData = atob(sourceImageData);
                                    if (decodedData.includes('<svg') || 
                                        (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                        sourceImageType = 'image/svg+xml';
                                    }
                                } catch (e) {
                                    // Not base64 data, continue with existing type
                                }
                            }
                        }
                    }
                    // Case 3: Direct data passed from another property
                    else if (typeof value === 'string' && value.length > 0) {
                        // Try to use it directly as image data
                        sourceImageData = value;
                    }
                    
                    // Validate we have image data
                    if (!sourceImageData) {
                        throw new Error(`No valid image data found${sourceImageName ? ` in image "${sourceImageName}"` : ''}`);
                    }
                    
                    if (objectType === 'image') {
                        // Set the background pattern for image objects
                        const imgElement = element.querySelector('img');
                        if (imgElement) {
                            // Make the image transparent so the background shows through
                            imgElement.style.opacity = '0';
                        }
                        
                        // Set the background pattern on the container for images
                        element.style.backgroundImage = `url(data:${sourceImageType};base64,${sourceImageData})`;
                        element.style.backgroundRepeat = 'repeat';
                    } else if (objectType === 'graphic') {
                        // For graphic objects, apply the pattern to the SVG fill area
                        const svg = element.querySelector('svg');
                        if (svg) {
                            // Check the graphic type
                            const graphicType = element.dataset.graphicType || 'line';
                            
                            // Clear any backgroundColor property when setting background pattern
                            // This ensures the pattern is not overridden by a backgroundColor
                            if (this.customProperties.has(objectName)) {
                                this.customProperties.get(objectName).delete('backgroundcolor');
                                this.customProperties.get(objectName).delete('backgroundcolour');
                            }
                            
                            // Create a pattern definition in the SVG
                            let patternId = `pattern_${objectName.replace(/\s+/g, '_')}`;
                            
                            // Remove any existing pattern with this ID
                            let existingDefs = svg.querySelector('defs');
                            if (existingDefs) {
                                let existingPattern = existingDefs.querySelector(`#${patternId}`);
                                if (existingPattern) {
                                    existingDefs.removeChild(existingPattern);
                                }
                            } else {
                                // Create defs element if it doesn't exist
                                existingDefs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
                                svg.insertBefore(existingDefs, svg.firstChild);
                            }
                            
                            // Create the pattern element
                            const pattern = document.createElementNS('http://www.w3.org/2000/svg', 'pattern');
                            pattern.setAttribute('id', patternId);
                            pattern.setAttribute('patternUnits', 'userSpaceOnUse');
                            pattern.setAttribute('width', '20'); // Default size, will be overridden by image
                            pattern.setAttribute('height', '20');
                            
                            // Create an image element inside the pattern
                            const image = document.createElementNS('http://www.w3.org/2000/svg', 'image');
                            image.setAttribute('href', `data:${sourceImageType};base64,${sourceImageData}`);
                            image.setAttribute('x', '0');
                            image.setAttribute('y', '0');
                            image.setAttribute('width', '20');
                            image.setAttribute('height', '20');
                            
                            // Add the image to the pattern
                            pattern.appendChild(image);
                            
                            // Add the pattern to the defs
                            existingDefs.appendChild(pattern);
                            
                            // Pattern will be applied to the shape element after the image loads
                            // (see the tempImg.onload callback above)
                            
                            // Load the image to get its dimensions
                            const tempImg = new Image();
                            tempImg.onload = function() {
                                pattern.setAttribute('width', this.width);
                                pattern.setAttribute('height', this.height);
                                image.setAttribute('width', this.width);
                                image.setAttribute('height', this.height);
                                
                                // Apply the pattern to the appropriate shape element AFTER the image is loaded
                                if (graphicType === 'path') {
                                    const path = svg.querySelector('path');
                                    if (path) {
                                        path.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'oval') {
                                    const ellipse = svg.querySelector('ellipse');
                                    if (ellipse) {
                                        ellipse.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'rectangle') {
                                    const rect = svg.querySelector('rect');
                                    if (rect) {
                                        rect.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'polygon') {
                                    const polygon = svg.querySelector('polygon');
                                    if (polygon) {
                                        polygon.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else {
                                    // Fallback for any other graphic types
                                    // Try to find any shape element and apply the pattern
                                    const shape = svg.querySelector('rect, circle, ellipse, polygon, path');
                                    if (shape) {
                                        shape.setAttribute('fill', `url(#${patternId})`);
                                    }
                                }
                            };
                            tempImg.src = `data:${sourceImageType};base64,${sourceImageData}`;
                        }
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('backgroundPattern', sourceImageName || 'custom');
                    this.customProperties.get(objectName).set('backgroundPatternData', sourceImageData);
                    this.customProperties.get(objectName).set('backgroundPatternType', sourceImageType);
                } else {
                    throw new Error(`Cannot set backgroundPattern on ${objectType} objects`);
                }
                break;
            case 'sounddata':
                // soundData can be set on any object type
                if (!this.customProperties.has(objectName)) {
                    this.customProperties.set(objectName, new Map());
                }
                
                // Check if value is 'lastSound' reference
                if (value === 'lastSound' && interpreter && interpreter.lastSound) {
                    this.customProperties.get(objectName).set('soundData', interpreter.lastSound);
                    this.customProperties.get(objectName).set('audioContext', interpreter.lastAudioContext);
                } else if (value.startsWith('the soundData of ')) {
                    // Extract the source object name
                    const sourceObjectName = value.substring('the soundData of '.length).replace(/^["']|["']$/g, '');
                    
                    // Get the source object's sound data
                    if (!this.customProperties.has(sourceObjectName)) {
                        throw new Error(`Source object not found: ${sourceObjectName}`);
                    }
                    
                    const sourceSoundData = this.customProperties.get(sourceObjectName).get('soundData');
                    const sourceAudioContext = this.customProperties.get(sourceObjectName).get('audioContext');
                    
                    if (!sourceSoundData) {
                        throw new Error(`Source object has no sound data: ${sourceObjectName}`);
                    }
                    
                    // Copy the sound data
                    this.customProperties.get(objectName).set('soundData', sourceSoundData);
                    this.customProperties.get(objectName).set('audioContext', sourceAudioContext);
                } else {
                    throw new Error('soundData must be set to "lastSound" or "the soundData of objectName"');
                }
                break;
            case 'imagedata':
                if (objectType === 'image') {
                    // Check if value is a reference to another image
                    if (value.startsWith('the imageData of image ')) {
                        // Extract the source image name
                        const sourceImageName = value.substring('the imageData of image '.length).replace(/^["']|["']$/g, '');
                        
                        // Get the source image data
                        const sourceImage = this.getObject(sourceImageName);
                        if (!sourceImage) {
                            throw new Error(`Source image not found: ${sourceImageName}`);
                        }
                        
                        const sourceImageData = this.customProperties.get(sourceImageName).get('data');
                        const sourceImageType = this.customProperties.get(sourceImageName).get('type');
                        
                        if (!sourceImageData) {
                            throw new Error(`Source image has no data: ${sourceImageName}`);
                        }
                        
                        // Set the image data
                        if (!this.customProperties.has(objectName)) {
                            this.customProperties.set(objectName, new Map());
                        }
                        
                        this.customProperties.get(objectName).set('data', sourceImageData);
                        this.customProperties.get(objectName).set('type', sourceImageType);
                        
                        // Update the image element or defer if screen is locked
                        const imgElement = element.querySelector('img');
                        if (imgElement) {
                            const imgSrc = `data:${sourceImageType};base64,${sourceImageData}`;
                            if (interpreter && interpreter.isScreenLocked) {
                                interpreter.deferredUpdates.push({
                                    type: 'attribute',
                                    element: imgElement,
                                    attribute: 'src',
                                    value: imgSrc
                                });
                            } else {
                                imgElement.src = imgSrc;
                            }
                        }
                        
                        // Remove placeholder styling or defer if screen is locked
                        deferStyleUpdate(element, 'backgroundColor', '') || (element.style.backgroundColor = '');
                        deferStyleUpdate(element, 'border', '') || (element.style.border = '');
                    } else {
                        // Direct base64 data
                        if (!this.customProperties.has(objectName)) {
                            this.customProperties.set(objectName, new Map());
                        }
                        
                        // Get the existing image type if available, or use a default
                        // Try to preserve the existing image type if already set
                        let imageType = (this.customProperties.has(objectName) && 
                            this.customProperties.get(objectName).has('type')) ? 
                            this.customProperties.get(objectName).get('type') : 'image/png';
                        
                        // We've already checked for existing type above, this is redundant
                        // Keeping the SVG detection below
                        
                        // Check if the data is SVG content (starts with SVG XML declaration or SVG tag)
                        if (typeof value === 'string') {
                            try {
                                const decodedData = atob(value);
                                if (decodedData.includes('<svg') || 
                                    decodedData.includes('<?xml') && decodedData.includes('<svg')) {
                                    imageType = 'image/svg+xml';
                                }
                            } catch (e) {
                                // Not base64 data, continue with default type
                            }
                        }
                        
                        this.customProperties.get(objectName).set('data', value);
                        this.customProperties.get(objectName).set('type', imageType); // Use detected or existing type
                        
                        // Update the image element or defer if screen is locked
                        const imgElement = element.querySelector('img');
                        if (imgElement) {
                            const imgSrc = `data:${imageType};base64,${value}`;
                            if (interpreter && interpreter.isScreenLocked) {
                                interpreter.deferredUpdates.push({
                                    type: 'attribute',
                                    element: imgElement,
                                    attribute: 'src',
                                    value: imgSrc
                                });
                            } else {
                                imgElement.src = imgSrc;
                            }
                        }
                        
                        // Remove placeholder styling or defer if screen is locked
                        deferStyleUpdate(element, 'backgroundColor', '') || (element.style.backgroundColor = '');
                        deferStyleUpdate(element, 'border', '') || (element.style.border = '');
                    }
                } else {
                    throw new Error(`Cannot set imageData on ${objectType} objects`);
                }
                break;
            case 'visible':
                // Convert value to boolean
                const isVisible = value.toLowerCase() === 'true';
                    
                // Set visibility of the element or defer if screen is locked
                const displayValue = isVisible ? '' : 'none';
                if (!deferStyleUpdate(element, 'display', displayValue)) {
                    element.style.display = displayValue;
                }
                    
                // Store in custom properties
                if (!this.customProperties.has(objectName)) {
                    this.customProperties.set(objectName, new Map());
                }
                this.customProperties.get(objectName).set('visible', isVisible ? 'true' : 'false');
                break;
            case 'layer':
                // Handle special layer values
                // First, ensure value is treated as a string for comparison
                const valueStr = String(value).toLowerCase();
                
                if (valueStr === 'top') {
                    // Find the highest layer and set this object one higher
                    const highestLayer = this.getHighestLayer();
                    const newLayer = highestLayer + 1;
                    
                    // Set zIndex or defer if screen is locked
                    if (!deferStyleUpdate(element, 'zIndex', newLayer.toString())) {
                        element.style.zIndex = newLayer.toString();
                    }
                    
                    this.highestLayer = newLayer;
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('layer', newLayer.toString());
                } else if (valueStr === 'bottom') {
                    // Set to one higher than the card's layer (which is always 1)
                    element.style.zIndex = '2';
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('layer', '2');
                } else {
                    // Set to the specified numeric value
                    // Handle both numeric values and string representations of numbers
                    let layerValue;
                    
                    // If value is already a number, use it directly
                    if (typeof value === 'number') {
                        layerValue = value;
                    } else {
                        // Otherwise try to parse it as a number
                        layerValue = parseInt(value);
                        if (isNaN(layerValue)) {
                            throw new Error('Layer must be a number, "top", or "bottom"');
                        }
                    }
                    
                    element.style.zIndex = layerValue.toString();
                    
                    // Update highest layer if necessary
                    if (layerValue > this.highestLayer) {
                        this.highestLayer = layerValue;
                    }
                    
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('layer', layerValue.toString());
                }
                break;
            case 'fill':
            case 'fillcolor':
            case 'fillcolour':
                if (objectType === 'graphic') {
                    // Check if there's a background pattern - if so, don't apply fill
                    if (this.customProperties.has(objectName)) {
                        const customProps = this.customProperties.get(objectName);
                        if (customProps.has('backgroundPattern') || customProps.has('backgroundpattern')) {
                            console.log(`Skipping fill on ${objectName} because backgroundPattern is present`);
                            
                            // Still store the fill value in custom properties
                            if (!this.customProperties.has(objectName)) {
                                this.customProperties.set(objectName, new Map());
                            }
                            this.customProperties.get(objectName).set('fill', value.toString());
                            return value;
                        }
                    }
                    
                    const svg = element.querySelector('svg');
                    if (svg) {
                        // Check the graphic type
                        const graphicType = element.dataset.graphicType || 'line';

                        // Fill only applies to path elements, not lines
                        if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    path.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    path.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        } else if (graphicType === 'oval') {
                            const ellipse = svg.querySelector('ellipse');
                            if (ellipse) {
                                // If value is 'none' or 'transparent', set fill to none
                                if (value.toString().toLowerCase() === 'none' ||
                                    value.toString().toLowerCase() === 'transparent') {
                                    ellipse.setAttribute('fill', 'none');
                                } else {
                                    // Otherwise, set the fill color
                                    ellipse.setAttribute('fill', this.parseColor(value));
                                }
                            }
                        }
                        // For other graphic types, we might not want to set a background
                    }
                    // Store in custom properties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('fill', value.toString());
                } else {
                    throw new Error(`Cannot set fill on ${objectType} objects`);
                }
                break;
            case 'applyforce':
                // Parse the speed, direction, and optional drag values from the input string
                // Format: "speed,direction[,drag]" (e.g., "40,180,5")
                // Also supports variables: mySpeed,myDirection,myDrag or expressions
                if (typeof value === 'string') {
                    const parts = value.split(',');
                    if (parts.length >= 2 && parts.length <= 3) {
                        // Get speed, direction, and optional drag values, evaluating variables if needed
                        let speed, direction, drag = 0;
                        
                        // If interpreter is provided, try to evaluate expressions
                        if (interpreter) {
                            try {
                                // Evaluate the speed part (could be a variable or expression)
                                speed = parseFloat(interpreter.evaluateExpression(parts[0].trim()));
                                
                                // Evaluate the direction part (could be a variable or expression)
                                direction = parseFloat(interpreter.evaluateExpression(parts[1].trim()));
                                
                                // Evaluate the optional drag part if provided
                                if (parts.length === 3) {
                                    drag = parseFloat(interpreter.evaluateExpression(parts[2].trim()));
                                    // Clamp drag value between 1 and 100
                                    drag = Math.max(1, Math.min(100, drag));
                                }
                            } catch (error) {
                                // If evaluation fails, try direct parsing
                                speed = parseFloat(parts[0].trim());
                                direction = parseFloat(parts[1].trim());
                                if (parts.length === 3) {
                                    drag = parseFloat(parts[2].trim());
                                    // Clamp drag value between 1 and 100
                                    drag = Math.max(1, Math.min(100, drag));
                                }
                            }
                        } else {
                            // No interpreter provided, use direct parsing
                            speed = parseFloat(parts[0].trim());
                            direction = parseFloat(parts[1].trim());
                            if (parts.length === 3) {
                                drag = parseFloat(parts[2].trim());
                                // Clamp drag value between 1 and 100
                                drag = Math.max(1, Math.min(100, drag));
                            }
                        }
                        
                        // Validate the inputs
                        if (!isNaN(speed) && !isNaN(direction) && (parts.length === 2 || !isNaN(drag))) {
                            // Store the values in custom properties
                            if (!this.customProperties.has(objectName)) {
                                this.customProperties.set(objectName, new Map());
                            }
                            
                            // Store the original string value
                            this.customProperties.get(objectName).set('applyforce', value);
                            
                            // Store the parsed values separately for easier access
                            this.customProperties.get(objectName).set('force_speed', speed);
                            this.customProperties.get(objectName).set('force_direction', direction);
                            
                            // Calculate and store normalized direction vector components
                            const directionRadians = (direction * Math.PI) / 180;
                            const dirX = Math.sin(directionRadians);
                            const dirY = -Math.cos(directionRadians); // Negative because Y increases downward
                            this.customProperties.get(objectName).set('force_dir_x', dirX);
                            this.customProperties.get(objectName).set('force_dir_y', dirY);
                            
                            // Store the drag value if provided
                            if (parts.length === 3) {
                                this.customProperties.get(objectName).set('force_drag', drag);
                            } else {
                                // Remove any existing drag value
                                this.customProperties.get(objectName).delete('force_drag');
                            }
                            
                            // Start the physics loop if it's not already running
                            if (!this.isPhysicsLoopRunning && speed > 0) {
                                this.startPhysicsLoop();
                            }
                        } else {
                            throw new Error(`Invalid applyForce value: ${value}. Format should be "speed,direction[,drag]" with numeric values.`);
                        }
                    } else {
                        throw new Error(`Invalid applyForce format: ${value}. Expected format: "speed,direction[,drag]" (e.g., "40,180" or "40,180,5").`);
                    }
                } else {
                    throw new Error(`Invalid applyForce value type: ${typeof value}. Expected a string in format: "speed,direction[,drag]".`);
                }
                break;
            // Scrollbar-specific properties
            case 'thumbposition':
                if (objectType === 'scrollbar') {
                    // Parse and validate the value
                    let thumbPosition = parseFloat(value);
                    if (isNaN(thumbPosition)) {
                        throw new Error('thumbPosition must be a number');
                    }
                    
                    // Get current startValue and endValue from customProperties
                    const startValue = parseFloat(this.customProperties.get(objectName).get('startValue') || 0);
                    const endValue = parseFloat(this.customProperties.get(objectName).get('endValue') || 100);
                    
                    // Clamp thumbPosition between startValue and endValue
                    thumbPosition = Math.max(startValue, Math.min(endValue, thumbPosition));
                    
                    // Update customProperties
                    this.customProperties.get(objectName).set('thumbPosition', thumbPosition.toString());
                    
                    // Update the scrollbar appearance
                    this.updateScrollbarThumb(element, objectName);
                    
                    return thumbPosition;
                }
                break;
            case 'startvalue':
                if (objectType === 'scrollbar') {
                    // Parse and validate the value
                    let newStartValue = parseFloat(value);
                    if (isNaN(newStartValue)) {
                        throw new Error('startValue must be a number');
                    }
                    
                    // Get current endValue from customProperties
                    const currentEndValue = parseFloat(this.customProperties.get(objectName).get('endValue') || 100);
                    
                    // Ensure startValue is less than endValue
                    if (newStartValue >= currentEndValue) {
                        throw new Error('startValue must be less than endValue');
                    }
                    
                    // Update customProperties
                    this.customProperties.get(objectName).set('startValue', newStartValue.toString());
                    
                    // Get current thumbPosition from customProperties and clamp it if needed
                    let currentThumbPosition = parseFloat(this.customProperties.get(objectName).get('thumbPosition') || 0);
                    if (currentThumbPosition < newStartValue) {
                        this.customProperties.get(objectName).set('thumbPosition', newStartValue.toString());
                    }
                    
                    // Update the scrollbar appearance
                    this.updateScrollbarThumb(element, objectName);
                    
                    return newStartValue;
                }
                break;
            case 'endvalue':
                if (objectType === 'scrollbar') {
                    // Parse and validate the value
                    let newEndValue = parseFloat(value);
                    if (isNaN(newEndValue)) {
                        throw new Error('endValue must be a number');
                    }
                    
                    // Get current startValue from customProperties
                    const currentStartValue = parseFloat(this.customProperties.get(objectName).get('startValue') || 0);
                    
                    // Ensure endValue is greater than startValue
                    if (newEndValue <= currentStartValue) {
                        throw new Error('endValue must be greater than startValue');
                    }
                    
                    // Update customProperties
                    this.customProperties.get(objectName).set('endValue', newEndValue.toString());
                    
                    // Get current thumbPosition from customProperties and clamp it if needed
                    let thumbPos = parseFloat(this.customProperties.get(objectName).get('thumbPosition') || 0);
                    if (thumbPos > newEndValue) {
                        this.customProperties.get(objectName).set('thumbPosition', newEndValue.toString());
                    }
                    
                    // Update the scrollbar appearance
                    this.updateScrollbarThumb(element, objectName);
                    
                    return newEndValue;
                }
                break;
            case 'scrolltype':
                if (objectType === 'scrollbar') {
                    // Handle different scrollType values
                    const lowerValue = value.toString().toLowerCase();
                    
                    // Handle orientation values (horizontal/vertical)
                    if (lowerValue === 'horizontal' || lowerValue === 'vertical') {
                        // Update orientation classes
                        element.classList.remove('horizontal', 'vertical');
                        element.classList.add(lowerValue);
                        
                        // If setting to vertical and current type is 'round', change to 'bar'
                        // since round scrollbars can only be horizontal
                        if (lowerValue === 'vertical' && this.customProperties.get(objectName).get('scrollType') === 'round') {
                            this.customProperties.get(objectName).set('scrollType', 'bar');
                            element.classList.remove('round');
                            element.classList.add('bar');
                        }
                        
                        // Update the visual appearance of the thumb
                        this.updateScrollbarThumb(element, element.dataset.name);
                        
                        return lowerValue;
                    }
                    // Handle type values (bar/round)
                    else if (lowerValue === 'bar' || lowerValue === 'round') {
                        // Update type classes
                        element.classList.remove('bar', 'round');
                        element.classList.add(lowerValue);
                        
                        // If setting to 'round', ensure orientation is horizontal
                        if (lowerValue === 'round' && !element.classList.contains('horizontal')) {
                            element.classList.remove('vertical');
                            element.classList.add('horizontal');
                        }
                        
                        // Update customProperties
                        this.customProperties.get(objectName).set('scrollType', lowerValue);
                        
                        // For scrollbars loaded from JSON, ensure we preserve the original dimensions
                        if (this.customProperties.get(objectName).has('loadedFromJSON')) {
                            const originalWidth = this.customProperties.get(objectName).get('originalWidth');
                            const originalHeight = this.customProperties.get(objectName).get('originalHeight');
                            
                            if (originalWidth && originalHeight) {
                                console.log(`Preserving JSON dimensions before updating thumb: ${originalWidth} × ${originalHeight}`);
                            }
                        }
                        
                        // Update the visual appearance of the thumb
                        this.updateScrollbarThumb(element, element.dataset.name);
                        
                        return lowerValue;
                    }
                    else {
                        throw new Error(`Invalid scrollType value: ${value}. Must be 'bar', 'round', 'horizontal', or 'vertical'.`);
                    }
                }
                break;
            case 'hasslider':
                if (objectType === 'scrollbar') {
                    // Handle hasSlider property - convert to boolean
                    const boolValue = (value === true || value === 'true' || value === 1 || value === '1');
                    
                    console.log(`Setting hasSlider for ${objectName}: value =`, value, 'boolValue =', boolValue);
                    
                    // Update the custom property (use camelCase)
                    this.customProperties.get(objectName).set('hasSlider', boolValue);
                    
                    // Remove any lowercase duplicate if it exists
                    if (this.customProperties.get(objectName).has('hasslider')) {
                        this.customProperties.get(objectName).delete('hasslider');
                    }
                    
                    console.log(`After setting, hasSlider property =`, this.customProperties.get(objectName).get('hasSlider'));
                    
                    // Update the scrollbar visual representation
                    this.updateScrollbarThumb(element, objectName, interpreter);
                    
                    return boolValue;
                }
                break;
            // Scrollbar styling properties
            case 'backgroundcolor':
            case 'backgroundcolour':
                if (objectType === 'scrollbar') {
                    // Set the background color of the scrollbar track
                    element.style.backgroundColor = value;
                    return value;
                }
                break;
            case 'foregroundcolor':
            case 'foregroundcolour':
                if (objectType === 'scrollbar') {
                    // Set the color of the scrollbar thumb
                    const thumb = element.querySelector('.scrollbar-thumb');
                    if (thumb) {
                        thumb.style.backgroundColor = value;
                    }
                    return value;
                }
                break;
            case 'progresscolor':
            case 'progresscolour':
                if (objectType === 'scrollbar') {
                    // Store the progress color in customProperties
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('progressColor', this.parseColor(value));
                    
                    // Update the scrollbar appearance to reflect the new color
                    this.updateScrollbarThumb(element, objectName);
                    
                    return value;
                }
                break;
            case 'borderwidth':
                if (objectType === 'scrollbar') {
                    // Set the border width of the scrollbar
                    element.style.borderWidth = `${value}px`;
                    // Also store in customProperties for consistency
                    if (!this.customProperties.has(objectName)) {
                        this.customProperties.set(objectName, new Map());
                    }
                    this.customProperties.get(objectName).set('borderWidth', value.toString());
                    return value;
                }
                break;
            default:
                // Handle custom properties
                if (!this.customProperties.has(objectName)) {
                    this.customProperties.set(objectName, new Map());
                }
                this.customProperties.get(objectName).set(property.toLowerCase(), value);
                break;
        }

        return value;
    }

    static setLineProperty(element, objectName, lineNumber, property, value) {
        console.log(`[DEBUG] setLineProperty called with:`, {
            element: element ? { id: element.id, name: element.dataset?.name } : 'null',
            objectName,
            lineNumber,
            property,
            value
        });
        
        // Store the formatting command in the field's formattingCommands array
        this.storeFormattingCommand(element, `set the ${property} of line ${lineNumber} of field "${objectName}" to ${value}`);
        // Convert string line numbers to integers
        if (typeof lineNumber === 'string') {
            // Handle word-based line numbers
            const wordToNumber = {
                'one': 1, 'two': 2, 'three': 3, 'four': 4, 'five': 5,
                'six': 6, 'seven': 7, 'eight': 8, 'nine': 9, 'ten': 10,
                'eleven': 11, 'twelve': 12, 'thirteen': 13, 'fourteen': 14, 'fifteen': 15,
                'sixteen': 16, 'seventeen': 17, 'eighteen': 18, 'nineteen': 19, 'twenty': 20
            };
            
            if (wordToNumber[lineNumber.toLowerCase()]) {
                lineNumber = wordToNumber[lineNumber.toLowerCase()];
            } else {
                // Try to convert to number directly
                lineNumber = parseInt(lineNumber, 10);
            }
        }
        
        // Ensure lineNumber is a valid number
        if (isNaN(lineNumber) || lineNumber < 1) {
            throw new Error(`Invalid line number: ${lineNumber}`);
        }
        
        // Get the field content element
        const fieldContent = element.querySelector('.field-content');
        if (!fieldContent) {
            throw new Error(`Field content element not found for ${objectName}`);
        }
        
        // Get the plain text content (using innerText to match the number of lines command)
        const plainText = fieldContent.innerText;
        
        // Split the plain text into lines
        const plainLines = plainText.split('\n');
        
        // Check if the line number is valid using the plain text line count
        if (lineNumber > plainLines.length) {
            throw new Error(`Line number ${lineNumber} exceeds the number of lines in the field (${plainLines.length})`);
        }
        
        // Create a temporary DOM element to handle the HTML manipulation
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = fieldContent.innerHTML;
        
        // Find the line we need to modify
        const lines = tempDiv.innerHTML.split('\n');
        if (lines.length < lineNumber) {
            throw new Error(`Line number ${lineNumber} exceeds the number of lines in the HTML content (${lines.length})`);
        }
        
        // Adjust for 0-based indexing
        const index = lineNumber - 1;
        
        // Extract the line content
        let lineContent = lines[index];
        
        // Create a temporary element to manipulate the line
        const lineElement = document.createElement('div');
        lineElement.innerHTML = lineContent;
        
        // Handle different properties
        switch (property.toLowerCase()) {
            case 'lineheight':
                // Check if the line is already wrapped in a span with line-height
                let lineHeightSpan = lineElement.querySelector('span[style*="line-height"]');
                if (lineHeightSpan) {
                    // Update existing line-height
                    lineHeightSpan.style.lineHeight = value;
                } else {
                    // Wrap the content in a new span with line-height
                    lineElement.innerHTML = `<span style="line-height: ${value}">${lineElement.innerHTML}</span>`;
                }
                // Update the line in the lines array with the modified content
                lines[index] = lineElement.innerHTML;
                break;
                
            case 'textfont':
            case 'fontfamily':
                // Handle existing font-family spans
                this._updateOrCreateSpanStyle(lineElement, 'font-family', value);
                break;
                
            case 'textcolor':
            case 'textcolour':
                // Parse the color value
                const parsedColor = this.parseColor(value);
                
                // Get the plain text content of this line without any styling
                const plainLineText = plainLines[index];
                
                // Create a new span with the desired color
                lines[index] = `<span style="color: ${parsedColor} !important">${plainLineText}</span>`;
                break;
                
            case 'textsize':
                // Get the plain text content of this line without any styling
                const plainSizeText = plainLines[index];
                
                // Create a new span with the desired font size
                lines[index] = `<span style="font-size: ${value}px">${plainSizeText}</span>`;
                break;
                
            case 'textstyle':
                // Handle text style (bold, italic, underline, etc.)
                // Get the plain text content of this line without any styling
                const plainStyleText = plainLines[index];
                
                // Determine the style to apply
                let styleTag = '';
                switch(value.toString().toLowerCase()) {
                    case 'bold':
                        styleTag = 'b';
                        break;
                    case 'italic':
                        styleTag = 'i';
                        break;
                    case 'underline':
                        styleTag = 'u';
                        break;
                    case 'plain':
                        // No style tag needed, just use the plain text
                        lines[index] = plainStyleText;
                        break;
                    default:
                        throw new Error(`Unsupported text style: ${value}`);
                }
                
                // Apply the style if it's not 'plain'
                if (styleTag) {
                    lines[index] = `<${styleTag}>${plainStyleText}</${styleTag}>`;
                }
                break;
        }
        
        // Update the field content
        fieldContent.innerHTML = lines.join('\n');
        
        // Store the line-specific property in custom properties
        if (!this.customProperties.has(objectName)) {
            this.customProperties.set(objectName, new Map());
        }
        
        // Create a nested map for line properties if it doesn't exist
        if (!this.customProperties.get(objectName).has('lineProperties')) {
            this.customProperties.get(objectName).set('lineProperties', new Map());
        }
        
        // Create a nested map for the specific line if it doesn't exist
        const lineProperties = this.customProperties.get(objectName).get('lineProperties');
        if (!lineProperties.has(lineNumber)) {
            lineProperties.set(lineNumber, new Map());
        }
        
        // Store the property for the specific line
        lineProperties.get(lineNumber).set(property.toLowerCase(), value);
        
        return value;
    }

    /**
     * Apply formatting to a specific character range within a line of a field
     * @param {HTMLElement} element - The field element
     * @param {string} objectName - The name of the field
     * @param {number} lineNumber - The line number (1-based)
     * @param {number} startChar - The starting character position (1-based)
     * @param {number} endChar - The ending character position (1-based)
     * @param {string} property - The property to set (textcolor, textstyle, textfont, textsize)
     * @param {string} value - The value to set
     * @returns {string} The value that was set
     */
    static setCharRangeProperty(element, objectName, lineNumber, startChar, endChar, property, value) {
        // Ensure character positions are valid numbers
        if (isNaN(startChar) || startChar < 1) {
            throw new Error(`Invalid start character position: ${startChar}`);
        }
        if (isNaN(endChar) || endChar < startChar) {
            throw new Error(`Invalid end character position: ${endChar}`);
        }
        
        // Get the field content element
        const fieldContent = element.querySelector('.field-content');
        if (!fieldContent) {
            throw new Error(`Field content element not found for ${objectName}`);
        }
        
        // Get the plain text content
        const plainText = fieldContent.innerText;
        
        // Split the plain text into lines
        const plainLines = plainText.split('\n');
        
        // Check if the line number is valid
        if (lineNumber > plainLines.length) {
            throw new Error(`Line number ${lineNumber} exceeds the number of lines in the field (${plainLines.length})`);
        }
        
        // Adjust for 0-based indexing
        const lineIndex = lineNumber - 1;
        
        // Get the text of the specified line
        const lineText = plainLines[lineIndex];
        
        // Check if character positions are valid
        if (endChar > lineText.length) {
            throw new Error(`End character position ${endChar} exceeds the length of line ${lineNumber} (${lineText.length})`);
        }
        
        // Create a temporary DOM element to handle the HTML manipulation
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = fieldContent.innerHTML;
        
        // Find the line we need to modify
        const lines = tempDiv.innerHTML.split('\n');
        if (lines.length < lineNumber) {
            throw new Error(`Line number ${lineNumber} exceeds the number of lines in the HTML content (${lines.length})`);
        }
        
        // Extract the line content
        const lineContent = lines[lineIndex];
        
        // Create a temporary element to manipulate the line
        const lineElement = document.createElement('div');
        lineElement.innerHTML = lineContent;
        
        // Get the plain text of this line
        const plainLineText = plainLines[lineIndex];
        
        // Adjust for 0-based indexing
        const startIndex = startChar - 1;
        const endIndex = endChar - 1;
        
        // Extract the text before, during, and after the range
        const beforeText = plainLineText.substring(0, startIndex);
        const targetText = plainLineText.substring(startIndex, endIndex + 1);
        const afterText = plainLineText.substring(endIndex + 1);
        
        // Handle different properties
        let styledText;
        switch (property.toLowerCase()) {
            case 'textcolor':
            case 'textcolour':
                // Parse the color value
                const parsedColor = this.parseColor(value);
                styledText = `<span style="color: ${parsedColor} !important">${targetText}</span>`;
                break;
                
            case 'textstyle':
                // Handle different text styles
                let styleValue;
                switch (value.toLowerCase()) {
                    case 'bold':
                        styleValue = 'font-weight: bold';
                        break;
                    case 'italic':
                        styleValue = 'font-style: italic';
                        break;
                    case 'underline':
                        styleValue = 'text-decoration: underline';
                        break;
                    case 'strikethrough':
                        styleValue = 'text-decoration: line-through';
                        break;
                    case 'plain':
                        styleValue = 'font-weight: normal; font-style: normal; text-decoration: none';
                        break;
                    default:
                        throw new Error(`Unsupported text style: ${value}`);
                }
                styledText = `<span style="${styleValue}">${targetText}</span>`;
                break;
                
            case 'textfont':
            case 'fontfamily':
                styledText = `<span style="font-family: ${value}">${targetText}</span>`;
                break;
                
            case 'textsize':
                styledText = `<span style="font-size: ${value}px">${targetText}</span>`;
                break;
                
            default:
                throw new Error(`Unsupported property: ${property}`);
        }
        
        // Combine the parts back together
        lines[lineIndex] = beforeText + styledText + afterText;
        
        // Update the field content
        fieldContent.innerHTML = lines.join('\n');
        
        // Store the character-specific property in custom properties
        if (!this.customProperties.has(objectName)) {
            this.customProperties.set(objectName, new Map());
        }
        
        // Create a nested map for character properties if it doesn't exist
        if (!this.customProperties.get(objectName).has('charProperties')) {
            this.customProperties.get(objectName).set('charProperties', new Map());
        }
        
        // Create a nested map for the specific line if it doesn't exist
        const charProperties = this.customProperties.get(objectName).get('charProperties');
        if (!charProperties.has(lineNumber)) {
            charProperties.set(lineNumber, new Map());
        }
        
        // Create a nested map for the specific character range if it doesn't exist
        const lineCharProperties = charProperties.get(lineNumber);
        const rangeKey = `${startChar}-${endChar}`;
        if (!lineCharProperties.has(rangeKey)) {
            lineCharProperties.set(rangeKey, new Map());
        }
        
        // Store the property for the specific character range
        lineCharProperties.get(rangeKey).set(property.toLowerCase(), value);
        
        // Store the formatting command in the field's formattingCommands array
        this.storeFormattingCommand(element, `set the ${property} of char ${startChar} to ${endChar} of line ${lineNumber} of field "${objectName}" to ${value}`);
        
        return value;
    }

    /**
     * Apply formatting to a specific word or word range within a field
     * @param {HTMLElement} element - The field element
     * @param {string} objectName - The name of the field
     * @param {number} wordNumber - The word number (1-based) or start of word range
     * @param {number} endWordNumber - The end of word range (if applicable)
     * @param {string} property - The property to set (textcolor, textstyle, textfont, textsize)
     * @param {string} value - The value to set
     * @param {number} lineNumber - Optional line number (1-based) to specify a word within a specific line
     * @returns {string} The value that was set
     */
    static setWordProperty(element, objectName, wordNumber, endWordNumber = null, property, value, lineNumber = null) {
        // Ensure word positions are valid numbers
        if (isNaN(wordNumber) || wordNumber < 1) {
            throw new Error(`Invalid word number: ${wordNumber}`);
        }
        if (endWordNumber !== null && (isNaN(endWordNumber) || endWordNumber < wordNumber)) {
            throw new Error(`Invalid end word number: ${endWordNumber}`);
        }
        
        // Get the field content element
        const fieldContent = element.querySelector('.field-content');
        if (!fieldContent) {
            throw new Error(`Field content element not found for ${objectName}`);
        }
        
        // Get the plain text content
        const plainText = fieldContent.innerText;
        
        // Variables for line-specific processing
        let workingText = plainText;
        let charOffset = 0;
        
        // If a line number is specified, we need to work with just that line
        if (lineNumber !== null) {
            // Split the plain text into lines
            const plainLines = plainText.split('\n');
            
            // Check if the line number is valid
            if (lineNumber > plainLines.length) {
                throw new Error(`Line number ${lineNumber} exceeds the number of lines in the field (${plainLines.length})`);
            }
            
            // Adjust for 0-based indexing
            const lineIndex = lineNumber - 1;
            
            // Get the text of the specified line
            workingText = plainLines[lineIndex];
            
            // Calculate the character offset of this line in the full text
            for (let i = 0; i < lineIndex; i++) {
                // +1 for the newline character
                charOffset += plainLines[i].length + 1;
            }
        }
        
        // Split the working text into words
        const words = workingText.trim().split(/\s+/);
        
        // Check if the word number is valid
        if (wordNumber > words.length) {
            throw new Error(`Word number ${wordNumber} exceeds the number of words in the field (${words.length})`);
        }
        
        // If endWordNumber is not provided, it's a single word
        if (endWordNumber === null) {
            endWordNumber = wordNumber;
        }
        
        // Check if end word number is valid
        if (endWordNumber > words.length) {
            throw new Error(`End word number ${endWordNumber} exceeds the number of words in the field (${words.length})`);
        }
        
        // Create a temporary DOM element to handle the HTML manipulation
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = fieldContent.innerHTML;
        
        // Find the positions of the words in the working text
        let currentWordIndex = 0;
        let startPos = -1;
        let endPos = -1;
        let inWord = false;
        
        // Iterate through each character to find word boundaries
        for (let i = 0; i < workingText.length; i++) {
            const char = workingText[i];
            
            // Check if the current character is part of a word or whitespace
            if (/\S/.test(char)) { // Non-whitespace character
                if (!inWord) {
                    // Start of a new word
                    currentWordIndex++;
                    inWord = true;
                    
                    // If this is our target word (start), mark its position
                    if (currentWordIndex === wordNumber) {
                        startPos = i;
                    }
                }
                
                // If this is our target word (end), update its end position
                if (currentWordIndex === endWordNumber) {
                    endPos = i + 1; // +1 to include this character
                }
            } else { // Whitespace character
                inWord = false;
            }
        }
        
        // If we couldn't find the words, throw an error
        if (startPos === -1 || endPos === -1) {
            throw new Error(`Could not locate word(s) ${wordNumber} to ${endWordNumber} in the ${lineNumber ? 'line' : 'field'} content`);
        }
        
        // Handle styling differently based on whether we're working with a specific line or the entire field
    let styledText;
    let targetText;
    
    if (lineNumber !== null) {
        // For word in a specific line, we'll modify only that line to preserve formatting in other lines
        // Create a temporary DOM element to handle the HTML manipulation
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = fieldContent.innerHTML;
        
        // Find the line we need to modify by splitting the HTML content
        const lines = tempDiv.innerHTML.split('\n');
        if (lines.length < lineNumber) {
            throw new Error(`Line number ${lineNumber} exceeds the number of lines in the HTML content (${lines.length})`);
        }
        
        // Get the line we want to modify
        const lineIndex = lineNumber - 1;
        
        // Create a temporary element to get the plain text of just this line
        const lineElement = document.createElement('div');
        lineElement.innerHTML = lines[lineIndex];
        const lineText = lineElement.innerText;
        
        // Find the positions of the word within this line
        let currentWordIndex = 0;
        let wordStartPos = -1;
        let wordEndPos = -1;
        let inWord = false;
        
        // Iterate through each character to find word boundaries
        for (let i = 0; i < lineText.length; i++) {
            const char = lineText[i];
            
            // Check if the current character is part of a word or whitespace
            if (/\S/.test(char)) { // Non-whitespace character
                if (!inWord) {
                    // Start of a new word
                    currentWordIndex++;
                    inWord = true;
                    
                    // If this is our target word (start), mark its position
                    if (currentWordIndex === wordNumber) {
                        wordStartPos = i;
                    }
                }
                
                // If this is our target word (end), update its end position
                if (currentWordIndex === endWordNumber) {
                    wordEndPos = i + 1; // +1 to include this character
                }
            } else { // Whitespace character
                inWord = false;
            }
        }
        
        // If we couldn't find the words, throw an error
        if (wordStartPos === -1 || wordEndPos === -1) {
            throw new Error(`Could not locate word(s) ${wordNumber} to ${endWordNumber} in line ${lineNumber}`);
        }
        
        // Get the target word text
        targetText = lineText.substring(wordStartPos, wordEndPos);
        
        // We'll store these values for later use when updating the line
        window.tempLineData = {
            lines,
            lineIndex,
            lineText,
            wordStartPos,
            wordEndPos
        };
    } else {
        // For words in the entire field, use the original approach
        startPos += charOffset; // This is always 0 for whole field
        endPos += charOffset;
        targetText = plainText.substring(startPos, endPos);
    }
        switch (property.toLowerCase()) {
            case 'textcolor':
            case 'textcolour':
                // Parse the color value
                const parsedColor = this.parseColor(value);
                styledText = `<span style="color: ${parsedColor} !important">${targetText}</span>`;
                break;
                
            case 'textstyle':
                // Handle different text styles
                let styleValue;
                switch (value.toLowerCase()) {
                    case 'bold':
                        styleValue = 'font-weight: bold';
                        break;
                    case 'italic':
                        styleValue = 'font-style: italic';
                        break;
                    case 'underline':
                        styleValue = 'text-decoration: underline';
                        break;
                    case 'strikethrough':
                        styleValue = 'text-decoration: line-through';
                        break;
                    case 'plain':
                        styleValue = 'font-weight: normal; font-style: normal; text-decoration: none';
                        break;
                    default:
                        throw new Error(`Unsupported text style: ${value}`);
                }
                styledText = `<span style="${styleValue}">${targetText}</span>`;
                break;
                
            case 'textfont':
            case 'fontfamily':
                styledText = `<span style="font-family: ${value}">${targetText}</span>`;
                break;
                
            case 'textsize':
                styledText = `<span style="font-size: ${value}px">${targetText}</span>`;
                break;
                
            default:
                throw new Error(`Unsupported property: ${property}`);
        }
        
        // Update the field content based on whether we're working with a specific line or the entire field
        if (lineNumber !== null) {
            // For word in a specific line, we need to update just that line
            const { lines, lineIndex, lineText, wordStartPos, wordEndPos } = window.tempLineData;
            delete window.tempLineData; // Clean up temp data
            
            // Get the text before, during, and after the target word
            const beforeWordText = lineText.substring(0, wordStartPos);
            const afterWordText = lineText.substring(wordEndPos);
            
            // Replace the line's content with the styled version
            lines[lineIndex] = beforeWordText + styledText + afterWordText;
            
            // Combine all lines back together
            fieldContent.innerHTML = lines.join('\n');
        } else {
            // For word in the entire field, reconstruct the entire content
            const beforeText = plainText.substring(0, startPos);
            const afterText = plainText.substring(endPos);
            fieldContent.innerHTML = beforeText + styledText + afterText;
        }
        
        // Store the formatting command for later restoration
        let commandText;
        if (lineNumber !== null) {
            // Command for word in a specific line
            commandText = endWordNumber === wordNumber 
                ? `set the ${property} of word ${wordNumber} of line ${lineNumber} of field "${objectName}" to ${value}`
                : `set the ${property} of word ${wordNumber} to ${endWordNumber} of line ${lineNumber} of field "${objectName}" to ${value}`;
        } else {
            // Command for word in the entire field
            commandText = endWordNumber === wordNumber 
                ? `set the ${property} of word ${wordNumber} of field "${objectName}" to ${value}`
                : `set the ${property} of word ${wordNumber} to ${endWordNumber} of field "${objectName}" to ${value}`;
        }
        this.storeFormattingCommand(element, commandText);
        
        return value;
    }

    /**
     * Store a formatting command in a field's formattingCommands array
     * @param {HTMLElement} element - The field element
     * @param {string} commandString - The HyperTalk command to store
     * @param {HTMLElement} element - The field element
     * @param {string} commandString - The HyperTalk command to store
     */
    static storeFormattingCommand(element, commandString) {
        // Get the field's name
        const fieldName = element.getAttribute('data-name');
        if (!fieldName) {
            console.error('Field name not found on element:', element);
            return;
        }
        
        // Make sure we have customProperties for this field
        if (!this.customProperties.has(fieldName)) {
            this.customProperties.set(fieldName, new Map());
        }
        
        // Initialize formattingCommands array if it doesn't exist
        if (!this.customProperties.get(fieldName).has('formattingCommands')) {
            this.customProperties.get(fieldName).set('formattingCommands', []);
        }
        
        // Get the current formatting commands
        const formattingCommands = this.customProperties.get(fieldName).get('formattingCommands');
        
        // Parse the command to identify its target (line/char range and property)
        let targetLine = null;
        let targetProperty = null;
        let targetCharRange = null;
        
        // Match patterns like: set the textColor of line 5 of field "BCODE" to red
        const lineRegex = /set the (\w+) of line (\d+) of field "([^"]+)" to (.+)/i;
        const lineMatch = commandString.match(lineRegex);
        
        // Match patterns like: set the textColor of char 1 to 5 of line 3 of field "BCODE" to red
        const charRangeRegex = /set the (\w+) of char (\d+) to (\d+) of line (\d+) of field "([^"]+)" to (.+)/i;
        const charRangeMatch = commandString.match(charRangeRegex);
        
        // Match patterns like: set the textColor of word 1 of field "BCODE" to red
        const wordRegex = /set the (\w+) of word (\d+) of field "([^"]+)" to (.+)/i;
        const wordMatch = commandString.match(wordRegex);
        
        // Match patterns like: set the textColor of word 1 to 3 of field "BCODE" to red
        const wordRangeRegex = /set the (\w+) of word (\d+) to (\d+) of field "([^"]+)" to (.+)/i;
        const wordRangeMatch = commandString.match(wordRangeRegex);
        
        // Match patterns like: set the textColor of word 1 of line 2 of field "BCODE" to red
        const wordInLineRegex = /set the (\w+) of word (\d+) of line (\d+) of field "([^"]+)" to (.+)/i;
        const wordInLineMatch = commandString.match(wordInLineRegex);
        
        // Match patterns like: set the textColor of word 1 to 3 of line 2 of field "BCODE" to red
        const wordRangeInLineRegex = /set the (\w+) of word (\d+) to (\d+) of line (\d+) of field "([^"]+)" to (.+)/i;
        const wordRangeInLineMatch = commandString.match(wordRangeInLineRegex);
        
        if (lineMatch) {
            targetProperty = lineMatch[1].toLowerCase();
            targetLine = lineMatch[2];
        } else if (charRangeMatch) {
            targetProperty = charRangeMatch[1].toLowerCase();
            targetCharRange = `${charRangeMatch[2]}-${charRangeMatch[3]}`;
            targetLine = charRangeMatch[4];
        } else if (wordMatch) {
            targetProperty = wordMatch[1].toLowerCase();
            targetCharRange = `word-${wordMatch[2]}`;
        } else if (wordRangeMatch) {
            targetProperty = wordRangeMatch[1].toLowerCase();
            targetCharRange = `word-${wordRangeMatch[2]}-${wordRangeMatch[3]}`;
        } else if (wordInLineMatch) {
            targetProperty = wordInLineMatch[1].toLowerCase();
            targetCharRange = `word-${wordInLineMatch[2]}`;
            targetLine = wordInLineMatch[3];
        } else if (wordRangeInLineMatch) {
            targetProperty = wordRangeInLineMatch[1].toLowerCase();
            targetCharRange = `word-${wordRangeInLineMatch[2]}-${wordRangeInLineMatch[3]}`;
            targetLine = wordRangeInLineMatch[4];
        }
        
        // If we identified the target, look for existing commands that affect the same target
        let indexToReplace = -1;
        
        // Determine the type of command we're processing
        const isLineCommand = targetLine !== null && targetCharRange === null;
        const isCharRangeCommand = targetLine !== null && targetCharRange !== null && !targetCharRange.startsWith('word-');
        const isWordCommand = targetCharRange !== null && targetCharRange.startsWith('word-') && targetLine === null;
        const isWordInLineCommand = targetCharRange !== null && targetCharRange.startsWith('word-') && targetLine !== null;
        
        // Only proceed if we have a valid property
        if (targetProperty) {
            for (let i = 0; i < formattingCommands.length; i++) {
                const cmd = formattingCommands[i];
                
                // Check for line property commands (e.g., set the textsize of line 2...)
                if (isLineCommand) {
                    const match = cmd.match(lineRegex);
                    if (match && 
                        match[1].toLowerCase() === targetProperty && 
                        match[2] === targetLine && 
                        match[3] === fieldName) {
                        indexToReplace = i;
                        break;
                    }
                } 
                // Check for character range commands (e.g., set the textsize of char 1 to 5 of line 2...)
                else if (isCharRangeCommand) {
                    const match = cmd.match(charRangeRegex);
                    if (match && 
                        match[1].toLowerCase() === targetProperty && 
                        `${match[2]}-${match[3]}` === targetCharRange && 
                        match[4] === targetLine && 
                        match[5] === fieldName) {
                        indexToReplace = i;
                        break;
                    }
                }
                // Check for word in line commands (e.g., set the textsize of word 3 of line 2...)
                else if (isWordInLineCommand) {
                    if (targetCharRange.includes('-')) {
                        // Word range in line (e.g., set the textsize of word 3 to 5 of line 2...)
                        const match = cmd.match(wordRangeInLineRegex);
                        if (match && 
                            match[1].toLowerCase() === targetProperty && 
                            `word-${match[2]}-${match[3]}` === targetCharRange && 
                            match[4] === targetLine && 
                            match[5] === fieldName) {
                            indexToReplace = i;
                            break;
                        }
                    } else {
                        // Single word in line (e.g., set the textsize of word 3 of line 2...)
                        const match = cmd.match(wordInLineRegex);
                        if (match && 
                            match[1].toLowerCase() === targetProperty && 
                            `word-${match[2]}` === targetCharRange && 
                            match[3] === targetLine && 
                            match[4] === fieldName) {
                            indexToReplace = i;
                            break;
                        }
                    }
                }
                // Check for word commands with no line specified (e.g., set the textsize of word 3...)
                else if (isWordCommand) {
                    if (targetCharRange.includes('-')) {
                        // Word range (e.g., set the textsize of word 3 to 5...)
                        const match = cmd.match(wordRangeRegex);
                        if (match && 
                            match[1].toLowerCase() === targetProperty && 
                            `word-${match[2]}-${match[3]}` === targetCharRange && 
                            match[4] === fieldName) {
                            indexToReplace = i;
                            break;
                        }
                    } else {
                        // Single word (e.g., set the textsize of word 3...)
                        const match = cmd.match(wordRegex);
                        if (match && 
                            match[1].toLowerCase() === targetProperty && 
                            `word-${match[2]}` === targetCharRange && 
                            match[3] === fieldName) {
                            indexToReplace = i;
                            break;
                        }
                    }
                }
            }
            
            if (indexToReplace >= 0) {
                // Replace the existing command with the new one
                formattingCommands[indexToReplace] = commandString;
                console.log(`Replaced formatting command for ${fieldName} at index ${indexToReplace}:`, commandString);
            } else if (!formattingCommands.includes(commandString)) {
                // Add as new if no exact match exists
                formattingCommands.push(commandString);
                console.log(`Added new formatting command to ${fieldName}:`, commandString);
            }
        } else {
            // For commands we couldn't parse, fall back to simple duplicate check
            if (!formattingCommands.includes(commandString)) {
                formattingCommands.push(commandString);
                console.log(`Added formatting command to ${fieldName}:`, commandString);
            } else {
                console.log(`Command already exists for ${fieldName}, not adding duplicate:`, commandString);
            }
        }
        
        console.log(`Field ${fieldName} now has ${formattingCommands.length} formatting commands`);
    }

    /**
     * Get a property value for a specific character range within a line of a field
     * @param {HTMLElement} element - The field element
     * @param {string} objectName - The name of the field
     * @param {number} lineNumber - The line number (1-based)
     * @param {number} startChar - The starting character position (1-based)
     * @param {number} endChar - The ending character position (1-based)
     * @param {string} property - The property to get (textcolor, textstyle, textfont, textsize)
     * @returns {string} The property value or empty string if not found
     */
    static getCharRangeProperty(element, objectName, lineNumber, startChar, endChar, property) {
        // Ensure character positions are valid numbers
        if (isNaN(startChar) || startChar < 1) {
            throw new Error(`Invalid start character position: ${startChar}`);
        }
        if (isNaN(endChar) || endChar < startChar) {
            throw new Error(`Invalid end character position: ${endChar}`);
        }
        
        // Check if the field has custom properties
        if (!this.customProperties.has(objectName)) {
            return '';
        }
        
        // Check if the field has character properties
        const objectProps = this.customProperties.get(objectName);
        if (!objectProps.has('charProperties')) {
            return '';
        }
        
        // Check if the line has character properties
        const charProperties = objectProps.get('charProperties');
        if (!charProperties.has(lineNumber)) {
            return '';
        }
        
        // Get the line properties
        const lineProperties = charProperties.get(lineNumber);
        
        // Look for an exact match first
        const exactRangeKey = `${startChar}-${endChar}`;
        if (lineProperties.has(exactRangeKey)) {
            const rangeProps = lineProperties.get(exactRangeKey);
            if (rangeProps.has(property.toLowerCase())) {
                return rangeProps.get(property.toLowerCase());
            }
        }
        
        // If no exact match, we need to check if this range is within any formatted range
        // This is more complex as we need to check the DOM
        const fieldContent = element.querySelector('.field-content');
        if (!fieldContent) {
            return '';
        }
        
        // Get the line content
        const lines = fieldContent.innerHTML.split('\n');
        if (lineNumber > lines.length) {
            return '';
        }
        
        // Get the line HTML
        const lineHTML = lines[lineNumber - 1];
        
        // Create a temporary element to parse the HTML
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = lineHTML;
        
        // Get the plain text to map positions
        const plainText = tempDiv.textContent;
        
        // If the requested range is beyond the text length, return empty
        if (startChar > plainText.length) {
            return '';
        }
        
        // Adjust end character if it exceeds the text length
        const adjustedEndChar = Math.min(endChar, plainText.length);
        
        // Find the span that contains the start character
        let currentPos = 0;
        let result = '';
        
        // Helper function to process a node and find the character position
        const processNode = (node) => {
            if (node.nodeType === Node.TEXT_NODE) {
                // Text node
                const textLength = node.textContent.length;
                const nodeStartPos = currentPos;
                const nodeEndPos = currentPos + textLength;
                
                // Check if our target range overlaps with this text node
                if (nodeEndPos >= startChar && nodeStartPos <= adjustedEndChar) {
                    // This text node contains part of our target range
                    // Check if the parent has styling
                    const parentElement = node.parentElement;
                    if (parentElement && parentElement.tagName === 'SPAN') {
                        const style = parentElement.getAttribute('style');
                        if (style) {
                            // Extract the relevant style based on the property
                            switch (property.toLowerCase()) {
                                case 'textcolor':
                                case 'textcolour':
                                    const colorMatch = style.match(/color:\s*([^;!]+)/i);
                                    if (colorMatch) {
                                        result = colorMatch[1].trim();
                                    }
                                    break;
                                    
                                case 'textstyle':
                                    if (style.includes('font-weight: bold')) {
                                        result = 'bold';
                                    } else if (style.includes('font-style: italic')) {
                                        result = 'italic';
                                    } else if (style.includes('text-decoration: underline')) {
                                        result = 'underline';
                                    } else if (style.includes('text-decoration: line-through')) {
                                        result = 'strikethrough';
                                    } else if (style.includes('font-weight: normal') && 
                                               style.includes('font-style: normal') && 
                                               style.includes('text-decoration: none')) {
                                        result = 'plain';
                                    }
                                    break;
                                    
                                case 'textfont':
                                case 'fontfamily':
                                    const fontMatch = style.match(/font-family:\s*([^;!]+)/i);
                                    if (fontMatch) {
                                        result = fontMatch[1].trim();
                                    }
                                    break;
                                    
                                case 'textsize':
                                    const sizeMatch = style.match(/font-size:\s*([\d.]+)px/i);
                                    if (sizeMatch) {
                                        result = sizeMatch[1].trim();
                                    }
                                    break;
                            }
                        }
                    }
                }
                
                currentPos += textLength;
            } else if (node.nodeType === Node.ELEMENT_NODE) {
                // Element node - recurse through children
                Array.from(node.childNodes).forEach(child => {
                    if (!result) { // Only continue if we haven't found a result yet
                        processNode(child);
                    }
                });
            }
        };
        
        // Process all nodes in the line
        Array.from(tempDiv.childNodes).forEach(node => {
            if (!result) { // Only continue if we haven't found a result yet
                processNode(node);
            }
        });
        
        return result;
    }
    
    /**
     * Get a property value for a specific line of a field
     * @param {HTMLElement} element - The field element
     * @param {string} objectName - The name of the field
     * @param {number} lineNumber - The line number (1-based)
     * @param {string} property - The property to get (textcolor, textstyle, textfont, textsize)
     * @returns {string} The property value or empty string if not found
     */
    static getLineProperty(element, objectName, lineNumber, property) {
        // Check if the field has custom properties
        if (!this.customProperties.has(objectName)) {
            return '';
        }
        
        // Check if the field has line properties
        const objectProps = this.customProperties.get(objectName);
        if (!objectProps.has('lineProperties')) {
            return '';
        }
        
        // Check if the line has properties
        const lineProperties = objectProps.get('lineProperties');
        if (!lineProperties.has(lineNumber)) {
            return '';
        }
        
        // Get the line properties
        const lineProps = lineProperties.get(lineNumber);
        if (lineProps.has(property.toLowerCase())) {
            return lineProps.get(property.toLowerCase());
        }
        
        // If not found in custom properties, check the DOM
        const fieldContent = element.querySelector('.field-content');
        if (!fieldContent) {
            return '';
        }
        
        // Get the line content
        const lines = fieldContent.innerHTML.split('\n');
        if (lineNumber > lines.length) {
            return '';
        }
        
        // Get the line HTML
        const lineHTML = lines[lineNumber - 1];
        
        // Create a temporary element to parse the HTML
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = lineHTML;
        
        // Check if the line has a wrapper span with the property
        if (tempDiv.firstChild && tempDiv.firstChild.tagName === 'SPAN') {
            const style = tempDiv.firstChild.getAttribute('style');
            if (style) {
                // Extract the relevant style based on the property
                switch (property.toLowerCase()) {
                    case 'lineheight':
                        const lineHeightMatch = style.match(/line-height:\s*([^;!]+)/i);
                        if (lineHeightMatch) {
                            return lineHeightMatch[1].trim();
                        }
                        break;
                        
                    case 'textcolor':
                    case 'textcolour':
                        const colorMatch = style.match(/color:\s*([^;!]+)/i);
                        if (colorMatch) {
                            return colorMatch[1].trim();
                        }
                        break;
                        
                    case 'textstyle':
                        if (style.includes('font-weight: bold')) {
                            return 'bold';
                        } else if (style.includes('font-style: italic')) {
                            return 'italic';
                        } else if (style.includes('text-decoration: underline')) {
                            return 'underline';
                        } else if (style.includes('text-decoration: line-through')) {
                            return 'strikethrough';
                        } else if (style.includes('font-weight: normal') && 
                                   style.includes('font-style: normal') && 
                                   style.includes('text-decoration: none')) {
                            return 'plain';
                        }
                        break;
                        
                    case 'textfont':
                    case 'fontfamily':
                        const fontMatch = style.match(/font-family:\s*([^;!]+)/i);
                        if (fontMatch) {
                            return fontMatch[1].trim();
                        }
                        break;
                        
                    case 'textsize':
                        const sizeMatch = style.match(/font-size:\s*([\d.]+)px/i);
                        if (sizeMatch) {
                            return sizeMatch[1].trim();
                        }
                        break;
                }
            }
        }
        
        return '';
    }

    /**
     * Get all properties of an object as a multi-line string
     * @param {string} objectName - Name of the object
     * @param {string} objectType - Type of the object (button, field, graphic, etc.)
     * @param {number} cardId - Optional card ID
     * @returns {string} - Multi-line string with all properties
     */
    static getAllObjectProperties(objectName, objectType, cardId = null) {
        // Get the object
        const object = this.getObject(objectName, cardId);
        if (!object) {
            throw new Error(`Object "${objectName}" not found`);
        }

        const properties = [];
        
        // Get common properties for all object types
        properties.push(`name ${objectName}`);
        properties.push(`id ${object.id}`);
        properties.push(`type ${objectType}`);
        
        // Get rect and location
        const rect = this.getObjectProperty(object, 'rect', objectType, cardId);
        properties.push(`rect ${rect}`);
        
        const loc = this.getObjectProperty(object, 'loc', objectType, cardId);
        properties.push(`loc ${loc}`);
        
        const left = this.getObjectProperty(object, 'left', objectType, cardId);
        properties.push(`left ${left}`);
        
        const top = this.getObjectProperty(object, 'top', objectType, cardId);
        properties.push(`top ${top}`);
        
        const width = this.getObjectProperty(object, 'width', objectType, cardId);
        properties.push(`width ${width}`);
        
        const height = this.getObjectProperty(object, 'height', objectType, cardId);
        properties.push(`height ${height}`);
        
        // Get layer
        const layer = this.getObjectProperty(object, 'layer', objectType, cardId);
        properties.push(`layer ${layer}`);
        
        // Get rotation
        const rotation = this.getObjectProperty(object, 'rotation', objectType, cardId);
        properties.push(`rotation ${rotation}`);
        
        // Get blendLevel
        const blendLevel = this.getObjectProperty(object, 'blendLevel', objectType, cardId);
        properties.push(`blendLevel ${blendLevel}`);
        
        // Get visible property from custom properties
        if (this.customProperties.has(objectName)) {
            const customProps = this.customProperties.get(objectName);
            const visible = customProps.get('visible');
            if (visible !== undefined) {
                properties.push(`visible ${visible}`);
            }
        }
        
        // Type-specific properties
        if (objectType === 'button' || objectType === 'field') {
            // Text properties
            try {
                const textColor = this.getObjectProperty(object, 'textColor', objectType, cardId);
                if (textColor) properties.push(`textColor ${textColor}`);
            } catch (e) {}
            
            try {
                const backgroundColor = this.getObjectProperty(object, 'backgroundColor', objectType, cardId);
                if (backgroundColor) properties.push(`backgroundColor ${backgroundColor}`);
            } catch (e) {}
            
            try {
                const textSize = this.getObjectProperty(object, 'textSize', objectType, cardId);
                if (textSize) properties.push(`textSize ${textSize}`);
            } catch (e) {}
            
            try {
                const textStyle = this.getObjectProperty(object, 'textStyle', objectType, cardId);
                if (textStyle) properties.push(`textStyle ${textStyle}`);
            } catch (e) {}
            
            try {
                const textAlign = this.getObjectProperty(object, 'textAlign', objectType, cardId);
                if (textAlign) properties.push(`textAlign ${textAlign}`);
            } catch (e) {}
        }
        
        // Button-specific properties
        if (objectType === 'button') {
            try {
                const hilighted = this.getObjectProperty(object, 'hilighted', objectType, cardId);
                properties.push(`hilighted ${hilighted}`);
            } catch (e) {}
            
            if (this.customProperties.has(objectName)) {
                const customProps = this.customProperties.get(objectName);
                
                const isCheckbox = customProps.get('isCheckbox');
                if (isCheckbox !== undefined) {
                    properties.push(`isCheckbox ${isCheckbox}`);
                }
                
                const disabled = customProps.get('disabled');
                if (disabled !== undefined) {
                    properties.push(`disabled ${disabled}`);
                }
                
                const isMenu = customProps.get('isMenu');
                if (isMenu !== undefined) {
                    properties.push(`isMenu ${isMenu}`);
                }
                
                const menuText = customProps.get('menuText');
                if (menuText !== undefined && menuText !== '') {
                    properties.push(`menuText ${menuText}`);
                }
            }
        }
        
        // Field-specific properties
        if (objectType === 'field') {
            try {
                const lockText = this.getObjectProperty(object, 'lockText', objectType, cardId);
                properties.push(`lockText ${lockText}`);
            } catch (e) {}
            
            try {
                const scrollable = this.getObjectProperty(object, 'scrollable', objectType, cardId);
                properties.push(`scrollable ${scrollable}`);
            } catch (e) {}
            
            try {
                const dontWrap = this.getObjectProperty(object, 'dontWrap', objectType, cardId);
                properties.push(`dontWrap ${dontWrap}`);
            } catch (e) {}
            
            try {
                const multiline = this.getObjectProperty(object, 'multiline', objectType, cardId);
                properties.push(`multiline ${multiline}`);
            } catch (e) {}
            
            if (this.customProperties.has(objectName)) {
                const customProps = this.customProperties.get(objectName);
                
                const showFocusBorder = customProps.get('showFocusBorder');
                if (showFocusBorder !== undefined) {
                    properties.push(`showFocusBorder ${showFocusBorder}`);
                }
            }
        }
        
        // Graphic-specific properties
        if (objectType === 'graphic') {
            try {
                const foregroundColor = this.getObjectProperty(object, 'foregroundColor', objectType, cardId);
                if (foregroundColor) properties.push(`foregroundColor ${foregroundColor}`);
            } catch (e) {}
            
            try {
                const lineSize = this.getObjectProperty(object, 'lineSize', objectType, cardId);
                if (lineSize) properties.push(`lineSize ${lineSize}`);
            } catch (e) {}
            
            try {
                const fill = this.getObjectProperty(object, 'fill', objectType, cardId);
                if (fill) properties.push(`fill ${fill}`);
            } catch (e) {}
            
            if (object.dataset.graphicType) {
                properties.push(`graphicType ${object.dataset.graphicType}`);
            }
        }
        
        // Add any custom properties that aren't standard
        if (this.customProperties.has(objectName)) {
            const customProps = this.customProperties.get(objectName);
            for (const [key, value] of customProps.entries()) {
                // Skip properties we've already added
                const standardProps = ['visible', 'hilighted', 'isCheckbox', 'disabled', 'isMenu', 
                                      'menuText', 'lockText', 'scrollable', 'dontWrap', 'multiline',
                                      'showFocusBorder', 'layer', 'rotation', 'blendLevel', 'vScroll',
                                      'hScroll', 'visibleLines', 'textalign', 'textAlign', 'textsize',
                                      'textStyle', 'formattingCommands'];
                
                if (!standardProps.includes(key)) {
                    // Format the value appropriately
                    let formattedValue = value;
                    if (Array.isArray(value)) {
                        formattedValue = value.join(',');
                    } else if (typeof value === 'object') {
                        formattedValue = JSON.stringify(value);
                    }
                    properties.push(`${key} ${formattedValue}`);
                }
            }
        }
        
        return properties.join('\n');
    }
    
    /**
     * Get an object property by name for a specific card
     * @param {string} objectName - Name of the object
     * @param {string} property - Property name to get
     * @param {number} cardId - Card ID to get the object from
     * @returns {*} - Property value
     */
    static getObjectPropertyByCard(objectName, property, cardId) {
        // Get the object by name from the specific card
        const object = this.getObject(objectName, cardId);
        if (!object) {
            throw new Error(`Object "${objectName}" not found on card ${cardId}`);
        }

        // Get the property value
        return this.getObjectProperty(object, property, null, cardId);
    }
    
    /**
     * Get the content of a field on a specific card
     * @param {string} fieldName - Name of the field
     * @param {number} cardId - Card ID to get the field from
     * @returns {string} - Field content
     */
    static getFieldContentByCard(fieldName, cardId) {
        // Get the field from the specific card
        const field = this.getObject(fieldName, cardId);
        if (!field) {
            throw new Error(`Field "${fieldName}" not found on card ${cardId}`);
        }
        
        // Get the field content
        let content = '';
        if (field.querySelector('.field-content')) {
            content = field.querySelector('.field-content').textContent || '';
        } else {
            content = field.textContent || '';
        }
        
        return content;
    }
    
    static setFieldContentByCard(fieldName, cardId, content) {
        // Get the field from the specific card
        const field = this.getObject(fieldName, cardId);
        if (!field) {
            throw new Error(`Field "${fieldName}" not found on card ${cardId}`);
        }
        
        // Set the field content
        if (field.querySelector('.field-content')) {
            field.querySelector('.field-content').textContent = content;
        } else {
            field.textContent = content;
        }
        
        return true;
    }
    
    static getObjectProperty(object, property, type = null, cardId = null) {
        // Handle arrayData property - special case for accessing array data as strings
        if (property && property.startsWith('arrayData(') && property.endsWith(')')) {
            const arrayPropName = property.substring(10, property.length - 1).trim();
            // Remove quotes if present
            const cleanPropName = arrayPropName.replace(/^["']|["']$/g, '');
            
            let objectName;
            if (typeof object === 'string') {
                objectName = object;
            } else if (object && object.dataset && object.dataset.name) {
                objectName = object.dataset.name;
            } else {
                throw new Error(`Cannot get arrayData: Invalid object ${object}`);
            }
            
            // Check if the custom property exists and is an array
            if (this.customProperties.has(objectName) && 
                this.customProperties.get(objectName).has(cleanPropName)) {
                
                const propValue = this.customProperties.get(objectName).get(cleanPropName);
                
                // If it's an array, convert to multi-line string
                if (Array.isArray(propValue)) {
                    return propValue.join('\n');
                } 
                // If it's already a string, return as is
                else if (typeof propValue === 'string') {
                    return propValue;
                }
                // For other types, convert to string
                else if (propValue !== undefined && propValue !== null) {
                    return propValue.toString();
                }
            }
            
            // Return empty string if property doesn't exist
            return '';
        }
        
        let element;
        if (typeof object === 'string') {
            // Parse object name to check for card-specific references
            const cardMatch = object.match(/^(.+) of card (\d+|"[^"]+"|[a-zA-Z]+)$/i);
            
            if (cardMatch) {
                const [_, objRef, cardRef] = cardMatch;
                // Extract card ID or name
                let targetCardId;
                
                // Handle numeric card ID
                if (/^\d+$/.test(cardRef)) {
                    targetCardId = parseInt(cardRef);
                } 
                // Handle quoted card name
                else if (/^"[^"]+"$/.test(cardRef)) {
                    const cardName = cardRef.substring(1, cardRef.length - 1);
                    targetCardId = this.getCardIdByName(cardName);
                }
                // Handle ordinal card reference (e.g., "one", "two")
                else {
                    const phoneticMap = {
                        'one': 1, 'two': 2, 'three': 3, 'four': 4, 'five': 5,
                        'six': 6, 'seven': 7, 'eight': 8, 'nine': 9, 'ten': 10
                    };
                    targetCardId = phoneticMap[cardRef.toLowerCase()] || null;
                }
                
                if (targetCardId) {
                    // Override the cardId parameter with the one from the object reference
                    cardId = targetCardId;
                    // Update object to remove the card reference part
                    object = objRef;
                }
            }
            
            // Use getObjectByType if type is provided
            if (type) {
                element = this.getObjectByType(object, type, cardId);
            } else {
                element = this.getObject(object, cardId);
            }
        } else {
            element = object;
        }

        if (!element) {
            throw new Error(`Object not found: ${object}`);
        }

        const objectName = typeof object === 'string' ? object : element.dataset.name;

        // Special handling for 'card' object
        if (objectName === 'card' || objectName === 'this card' || objectName === 'current card' || 
            objectName === 'card 1' || objectName.startsWith('card-')) {
            // Determine which card to use
            let card;
            let cardPropertyName;
            
            if (cardId && cardId > 1) {
                card = document.getElementById(`card-${cardId}`);
                cardPropertyName = `card-${cardId}`;
            } else {
                card = document.getElementById('card');
                cardPropertyName = 'card';
            }
            
            if (card) {
                // Handle card-specific properties
                switch (property.toLowerCase()) {
                    case 'backgroundcolor':
                    case 'backgroundcolour':
                        return card.style.backgroundColor || '';
                    case 'backgroundpattern':
                        if (this.customProperties.has(cardPropertyName) && 
                            this.customProperties.get(cardPropertyName).has('backgroundPattern')) {
                            return this.customProperties.get(cardPropertyName).get('backgroundPattern');
                        }
                        return '';
                }
            }
        }

        const objectType = element.dataset.type;
        const objectId = element.id;

        switch (property.toLowerCase()) {
            case 'id':
                return objectId;
            case 'name':
            case 'shortname':
                return objectName;
            case 'hilite': // Add synonym for hilighted
            case 'highlighted': // Add another synonym for hilighted
                // Get the hilighted property value from custom properties
                if (this.customProperties.has(objectName) && 
                    this.customProperties.get(objectName).has('hilighted')) {
                    return this.customProperties.get(objectName).get('hilighted');
                }
                return false; // Default to false if not set
            case 'left':
                return parseInt(element.style.left);
            case 'top':
                return parseInt(element.style.top);
            case 'right':
                return parseInt(element.style.left) + parseInt(element.style.width);
            case 'bottom':
                return parseInt(element.style.top) + parseInt(element.style.height);
            case 'width':
                return parseInt(element.style.width);
           case 'height':
                return parseInt(element.style.height);
            case 'topleft':
                return `${parseInt(element.style.left)},${parseInt(element.style.top)}`;
            case 'topright':
                return `${parseInt(element.style.left) + parseInt(element.style.width)},${parseInt(element.style.top)}`;
            case 'bottomleft':
                return `${parseInt(element.style.left)},${parseInt(element.style.top) + parseInt(element.style.height)}`;
            case 'bottomright':
                return `${parseInt(element.style.left) + parseInt(element.style.width)},${parseInt(element.style.top) + parseInt(element.style.height)}`;
            case 'rect':
                const rectLeft = parseInt(element.style.left) || 0;
                const rectTop = parseInt(element.style.top) || 0;
                const rectWidth = parseInt(element.style.width) || 0;
                const rectHeight = parseInt(element.style.height) || 0;
                const rectRight = rectLeft + rectWidth;
                const rectBottom = rectTop + rectHeight;
                return `${rectLeft},${rectTop},${rectRight},${rectBottom}`;
            case 'loc':
            case 'location':
                const left = parseInt(element.style.left) || 0;
                const top = parseInt(element.style.top) || 0;
                const width = parseInt(element.style.width) || 0;
                const height = parseInt(element.style.height) || 0;
                // Calculate center point
                const centerX = left + width / 2;
                const centerY = top + height / 2;
                return `${Math.round(centerX)},${Math.round(centerY)}`;
            case 'mouseloc':
                // Return the actual mouse position from the interpreter
                if (window.webTalkApp && window.webTalkApp.interpreter) {
                    return `${window.webTalkApp.interpreter.mouseX},${window.webTalkApp.interpreter.mouseY}`;
                }
                return '0,0'; // Default if interpreter not available
            case 'layer':
                // Get the layer value from z-index or custom properties
                const zIndex = element.style.zIndex;
                if (zIndex) {
                    return zIndex;
                }
                // Fall back to custom properties
                if (this.customProperties.has(objectName) && 
                    this.customProperties.get(objectName).has('layer')) {
                    return this.customProperties.get(objectName).get('layer');
                }
                return '1'; // Default layer if not set
            case 'rotation':
            case 'angle':
                // Get the rotation value from custom properties
                if (this.customProperties.has(objectName)) {
                    return this.customProperties.get(objectName).get('rotation') || 0;
                }
                return 0;
            case 'blendlevel':
                // Get the blendLevel value from custom properties
                if (this.customProperties.has(objectName) && 
                    this.customProperties.get(objectName).has('blendLevel')) {
                    return this.customProperties.get(objectName).get('blendLevel');
                }
                // If not set, calculate from the current opacity
                if (element.style.opacity !== '') {
                    const opacity = parseFloat(element.style.opacity);
                    // Convert opacity (1-0) to blendLevel (0-100)
                    return Math.round((1 - opacity) * 100).toString();
                }
                // Default to 0 (fully visible) if not set
                return '0';
            case 'textcolor':
            case 'textcolour':
            case 'foregroundcolor':
            case 'foregroundcolour':
                if (objectType === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (svg) {
                        // Check the graphic type
                        const graphicType = element.dataset.graphicType || 'line';

                        if (graphicType === 'line') {
                            const line = svg.querySelector('line');
                            if (line) {
                                return line.getAttribute('stroke') || 'black';
                            }
                        } else if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                return path.getAttribute('stroke') || 'black';
                            }
                        }
                    }
                    // Fall back to custom properties if SVG elements not found
                    if (this.customProperties.has(objectName)) {
                        return this.customProperties.get(objectName).get('foregroundcolor') || 'black';
                    }
                    return 'black';
                } else {
                    // For non-graphic objects, check custom properties first, then fall back to style
                    if (this.customProperties.has(objectName)) {
                        const customColor = this.customProperties.get(objectName).get('foregroundColor') || 
                                           this.customProperties.get(objectName).get('foregroundcolor');
                        if (customColor) {
                            // Also ensure the color is applied to the element
                            element.style.setProperty('color', customColor, 'important');
                            return customColor;
                        }
                    }
                    // Fall back to the element's style color
                    return element.style.color || 'black';
                }
            case 'backgroundcolor':
            case 'backgroundcolour': // Add UK spelling
                if (element.dataset.type === 'button' || element.dataset.type === 'field') {
                    // First check inline style
                    if (element.style.backgroundColor) {
                        return element.style.backgroundColor;
                    }
                    // Then check custom properties
                    if (this.customProperties.has(objectName)) {
                        const customBgColor = this.customProperties.get(objectName).get('backgroundColor') || 
                                             this.customProperties.get(objectName).get('backgroundcolor');
                        if (customBgColor) {
                            return customBgColor;
                        }
                    }
                    // Finally, get computed style as fallback
                    const computedStyle = window.getComputedStyle(element);
                    return computedStyle.backgroundColor || '';
                } else if (element.dataset.type === 'graphic') {
                    // For graphics, get the fill color based on graphic type
                    const svg = element.querySelector('svg');
                    if (svg) {
                        const graphicType = element.dataset.graphicType || 'line';

                        if (graphicType === 'path') {
                            const path = svg.querySelector('path');
                            if (path) {
                                return path.getAttribute('fill') || '';
                            }
                        } else if (graphicType === 'oval') {
                            const ellipse = svg.querySelector('ellipse');
                            if (ellipse) {
                                return ellipse.getAttribute('fill') || '';
                            }
                        }
                    }

                    // Fall back to custom properties if available
                    if (this.customProperties.has(objectName)) {
                        return this.customProperties.get(objectName).get('backgroundcolor') || '';
                    }
                    return '';
                } else {
                    return element.style.backgroundColor;
                }
            case 'text':
            case 'content':
                return element.textContent;
            case 'script':
                return this.getScript(element.dataset.name);
            // Menu-specific properties for buttons
            case 'ismenu':
                if (element.dataset.type === 'button') {
                    // Get from custom properties if available
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('isMenu')) {
                        return this.customProperties.get(objectName).get('isMenu');
                    }
                    return false; // Default to false if not set
                }
                return '';
            case 'menutext':
                if (element.dataset.type === 'button') {
                    // Get from custom properties if available
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('menuText')) {
                        return this.customProperties.get(objectName).get('menuText');
                    }
                    return ''; // Default to empty string if not set
                }
                return '';
            // Field-specific properties
            case 'vscroll':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('vScroll') || 0;
                }
                throw new Error(`Cannot get vScroll from ${objectType} objects`);
            case 'hscroll':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('hScroll') || 0;
                }
                throw new Error(`Cannot get hScroll from ${objectType} objects`);
            case 'visiblelines':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('visibleLines') || '1 to 1';
                }
                throw new Error(`Cannot get visibleLines from ${objectType} objects`);
            case 'textalign':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('textalign') || 'left';
                } else if (objectType === 'button') {
                    return this.customProperties.get(objectName).get('textAlign') || 'center';
                }
                throw new Error(`Cannot get textalign from ${objectType} objects`);
            case 'scrollable':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('scrollable') || false;
                }
                throw new Error(`Cannot get scrollable from ${objectType} objects`);
            case 'locktext':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('locktext') || false;
                }
                throw new Error(`Cannot get locktext from ${objectType} objects`);
            case 'styledtext':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('styledtext') || false;
                }
                throw new Error(`Cannot get styledtext from ${objectType} objects`);
            case 'dontwrap':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('dontwrap') || false;
                }
                throw new Error(`Cannot get dontwrap from ${objectType} objects`);
            case 'multiline':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('multiline') || true;
                }
                throw new Error(`Cannot get multiline from ${objectType} objects`);
            case 'autotab':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('autotab') || false;
                }
                throw new Error(`Cannot get autotab from ${objectType} objects`);
            case 'widemargins':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('widemargins') || false;
                }
                throw new Error(`Cannot get widemargins from ${objectType} objects`);
            case 'sharedtext':
                if (objectType === 'field') {
                    return this.customProperties.get(objectName).get('sharedtext') || false;
                }
                throw new Error(`Cannot get sharedtext from ${objectType} objects`);
            case 'textsize':
                if (objectType === 'field' || objectType === 'button') {
                    return this.customProperties.get(objectName).get('textsize') || 16;
                }
                throw new Error(`Cannot get textsize from ${objectType} objects`);
            case 'textstyle':
                if (objectType === 'field' || objectType === 'button') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('textStyle')) {
                        return this.customProperties.get(objectName).get('textStyle');
                    }
                    return 'plain'; // Default to plain if not set
                }
                throw new Error(`Cannot get textStyle from ${objectType} objects`);
                case 'points':
    if (objectType === 'graphic') {
        const svg = element.querySelector('svg');
        if (!svg) return '';

        // Check the graphic type
        const graphicType = element.dataset.graphicType || 'line';

        if (graphicType === 'line' && svg.querySelector('line')) {
            // Get the current position of the graphic container
            const offsetLeft = element.offsetLeft;
            const offsetTop = element.offsetTop;

            // Get the line coordinates (relative to the container)
            const x1 = parseInt(svg.querySelector('line').getAttribute('x1'));
            const y1 = parseInt(svg.querySelector('line').getAttribute('y1'));
            const x2 = parseInt(svg.querySelector('line').getAttribute('x2'));
            const y2 = parseInt(svg.querySelector('line').getAttribute('y2'));

            // Return the absolute coordinates
            return `${x1 + offsetLeft},${y1 + offsetTop},${x2 + offsetLeft},${y2 + offsetTop}`;
        } else if (graphicType === 'path' && svg.querySelector('path')) {
            // Return the path data directly
            return svg.querySelector('path').getAttribute('d');
        }

        // Fall back to custom properties if available
        if (this.customProperties.has(objectName)) {
            return this.customProperties.get(objectName).get('points') || '';
        }

        return '';
    }
    throw new Error(`Cannot get points from ${objectType} objects`);
            case 'linesize':
                if (objectType === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (!svg) return 1;

                    // Check the graphic type
                    const graphicType = element.dataset.graphicType || 'line';

                    if (graphicType === 'line') {
                        const line = svg.querySelector('line');
                        if (line) {
                            return parseInt(line.getAttribute('stroke-width')) || 1;
                        }
                    } else if (graphicType === 'path') {
                        const path = svg.querySelector('path');
                        if (path) {
                            return parseInt(path.getAttribute('stroke-width')) || 1;
                        }
                    } else if (graphicType === 'oval') {
                        const ellipse = svg.querySelector('ellipse');
                        if (ellipse) {
                            return parseInt(ellipse.getAttribute('stroke-width')) || 1;
                        }
                    }

                    // Fall back to custom properties if available
                    if (this.customProperties.has(objectName)) {
                        return this.customProperties.get(objectName).get('linesize') || 1;
                    }

                    return 1;
                }
                throw new Error(`Cannot get linesize from ${objectType} objects`);
            case 'showborder':
                if (objectType === 'graphic' || objectType === 'field') {
    return this.customProperties.get(objectName).get('showborder') || false;
}
                throw new Error(`Cannot get showborder from ${objectType} objects`);
            case 'fill':
            case 'fillcolor':
            case 'fillcolour':
                if (objectType === 'graphic') {
                    const svg = element.querySelector('svg');
                    if (!svg) return '';

                    // Check the graphic type
                    const graphicType = element.dataset.graphicType || 'line';

                    if (graphicType === 'path') {
                        const path = svg.querySelector('path');
                        if (path) {
                            return path.getAttribute('fill') || '';
                        }
                    }

                    // Fall back to custom properties if available
                    if (this.customProperties.has(objectName)) {
                        return this.customProperties.get(objectName).get('fill') || '';
                    }

                    return '';
                }
                throw new Error(`Cannot get fill from ${objectType} objects`);
            case 'sounddata':
                // soundData can be retrieved from any object type
                if (this.customProperties.has(objectName)) {
                    return this.customProperties.get(objectName).get('soundData') || '';
                }
                return '';
            case 'imagedata':
                if (objectType === 'image') {
                    return this.customProperties.get(objectName).get('data') || '';
                }
                throw new Error(`Cannot get imageData from ${objectType} objects`);
            case 'backgroundpattern':
                if (objectType === 'image' || objectType === 'graphic') {
                    return this.customProperties.get(objectName).get('backgroundPattern') || '';
                }
                throw new Error(`Cannot get backgroundPattern from ${objectType} objects`);
            // Scrollbar-specific properties
            case 'thumbposition':
                if (objectType === 'scrollbar') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('thumbPosition')) {
                        return this.customProperties.get(objectName).get('thumbPosition');
                    }
                    return '0';
                }
                throw new Error(`Cannot get thumbPosition from ${objectType} objects`);
            case 'startvalue':
                if (objectType === 'scrollbar') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('startValue')) {
                        return this.customProperties.get(objectName).get('startValue');
                    }
                    return '0';
                }
                throw new Error(`Cannot get startValue from ${objectType} objects`);
            case 'endvalue':
                if (objectType === 'scrollbar') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('endValue')) {
                        return this.customProperties.get(objectName).get('endValue');
                    }
                    return '100';
                }
                throw new Error(`Cannot get endValue from ${objectType} objects`);
            case 'scrolltype':
                if (objectType === 'scrollbar') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('scrollType')) {
                        return this.customProperties.get(objectName).get('scrollType');
                    }
                    return 'bar';
                }
                throw new Error(`Cannot get scrollType from ${objectType} objects`);
            case 'progresscolor':
            case 'progresscolour':
                if (objectType === 'scrollbar') {
                    if (this.customProperties.has(objectName) && 
                        this.customProperties.get(objectName).has('progressColor')) {
                        return this.customProperties.get(objectName).get('progressColor');
                    }
                    return '0,0,255'; // Default blue color
                }
                throw new Error(`Cannot get progressColor from ${objectType} objects`);
            case 'visible':
                if (this.customProperties.has(objectName) &&
                    this.customProperties.get(objectName).has('visible')) {
                    return this.customProperties.get(objectName).get('visible');
                }
                // Default to true if not specified
                return 'true';
            // Field scroll position properties
            case 'vscroll':
                if (objectType === 'field') {
                    // Get the field content element (the scrollable part)
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Return the actual current scroll position
                        return fieldContent.scrollTop;
                    }
                    return 0;
                }
                throw new Error(`Cannot get vScroll from ${objectType} objects`);
            case 'hscroll':
                if (objectType === 'field') {
                    // Get the field content element (the scrollable part)
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Return the actual current scroll position
                        return fieldContent.scrollLeft;
                    }
                    return 0;
                }
                throw new Error(`Cannot get hScroll from ${objectType} objects`);
            default:
                // Handle custom properties
                if (this.customProperties.has(objectName) &&
                    this.customProperties.get(objectName).has(property.toLowerCase())) {
                    return this.customProperties.get(objectName).get(property.toLowerCase());
                }
                // For any unknown property, return empty string instead of throwing an error
                // This allows custom properties to be created on first use
                return "";
        }
    }

    static parseColor(value) {
    // Convert value to string if it's not already
    const stringValue = typeof value === 'object' ? value.toString() : String(value);
    
    // Check if the value is already in rgb(...) format
    if (stringValue.trim().startsWith('rgb(') && stringValue.trim().endsWith(')')) {
        // Already in correct format, return as is
        return stringValue;
    }
    
    // Handle RGB format "r,g,b"
    if (stringValue.includes(',')) {
        const rgbParts = stringValue.split(',');
        
        // Make sure we have exactly 3 parts for RGB
        if (rgbParts.length === 3) {
            const [r, g, b] = rgbParts.map(n => {
                const parsed = parseInt(n.trim());
                // Ensure values are in valid RGB range (0-255)
                return isNaN(parsed) ? 0 : Math.max(0, Math.min(255, parsed));
            });
            return `rgb(${r},${g},${b})`;
        }
    }

        // Handle named colors from col-references.css
        const colorName = value.trim().toLowerCase();
        const namedColors = {
            'black': 'rgb(0,0,0)',
            'white': 'rgb(255,255,255)',
            'red': 'rgb(255,0,0)',
            'green': 'rgb(0,255,0)',
            'blue': 'rgb(0,0,255)',
            'yellow': 'rgb(255,255,0)',
            'cyan': 'rgb(0,255,255)',
            'magenta': 'rgb(255,0,255)',
            'darkred': 'rgb(139,0,0)',
            'darkgreen': 'rgb(0,100,0)',
            'darkblue': 'rgb(0,0,139)',
            'lightred': 'rgb(255,102,102)',
            'lightgreen': 'rgb(144,238,144)',
            'lightblue': 'rgb(173,216,230)',
            'orange': 'rgb(255,165,0)',
            'purple': 'rgb(128,0,128)',
            'brown': 'rgb(165,42,42)',
            'gray': 'rgb(128,128,128)',
            'lightgray': 'rgb(211,211,211)',
            'darkgray': 'rgb(64,64,64)',
            'pink': 'rgb(255,192,203)',
            'gold': 'rgb(255,215,0)',
            'silver': 'rgb(192,192,192)',
            'navy': 'rgb(0,0,128)',
            'olive': 'rgb(128,128,0)',
            'teal': 'rgb(0,128,128)'
        };

        if (namedColors[colorName]) {
            return namedColors[colorName];
        }

        return value;
    }

    static setImageFile(object, filename) {
        const objectName = object.dataset.name;
        const customProps = this.customProperties.get(objectName);
        
        // Store the filename in custom properties
        customProps.set('filename', filename);
        
        console.log(`setImageFile called for ${objectName}, filename: ${filename}`);
        console.log(`Current data in customProps:`, customProps.get('data') ? 'HAS DATA' : 'NO DATA');
        
        // Check if image already has data - if so, don't prompt for file selection
        if (customProps.has('data') && customProps.get('data')) {
            console.log(`Image ${objectName} already has data, skipping file selection`);
            return;
        }
        
        console.log(`No existing data found for ${objectName}, triggering file selection dialog`);
        
        // Create a file input element
        const fileInput = document.createElement('input');
        fileInput.type = 'file';
        fileInput.accept = '.png, .jpg, .jpeg, .gif, .svg';
        fileInput.style.display = 'none';
        
        // Function to handle the selected file
        fileInput.onchange = function(event) {
            const file = event.target.files[0];
            if (!file) return;
            
            const reader = new FileReader();
            reader.onload = function(e) {
                // Get the base64 data
                const base64Data = e.target.result.split(',')[1];
                
                // Store image data in custom properties
                customProps.set('type', file.type);
                customProps.set('data', base64Data);
                
                // Add console log to debug
                console.log('Image data set (file picker):', {
                    name: objectName,
                    type: customProps.get('type'),
                    dataLength: customProps.get('data') ? customProps.get('data').length : 0,
                    dataSample: customProps.get('data') ? customProps.get('data').substring(0, 50) + '...' : ''
                });
                
                // Create a temporary image to get dimensions
                const tempImg = new Image();
                tempImg.onload = function() {
                    // Adjust container size to match image dimensions
                    object.style.width = `${tempImg.width}px`;
                    object.style.height = `${tempImg.height}px`;
                };
                tempImg.src = e.target.result;
                
                // Update the image element
                const imgElement = object.querySelector('img');
                if (imgElement) {
                    imgElement.src = e.target.result;
                }
                
                // Remove the placeholder styling
                object.style.backgroundColor = '';
                object.style.border = '';
            };
            reader.readAsDataURL(file);
            
            // Clean up
            document.body.removeChild(fileInput);
        };
        
        // Add to DOM and trigger click
        document.body.appendChild(fileInput);
        fileInput.click();
    }

    static makeObjectDraggable(element) {
        // Store the original mousedown handler if it exists
        element._originalMouseDown = element.onmousedown;

        // Add draggable functionality that can be enabled/disabled
        element.isDraggable = false; // Default to not draggable

        let pos1 = 0, pos2 = 0, pos3 = 0, pos4 = 0;

        // We don't want to interfere with the addEventListener handlers
        // So we'll only handle dragging and edit mode in our onmousedown handler
        element.onmousedown = function(e) {
            // Check for edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';

            // Don't start drag if clicking on a resize handle
            if (e.target.classList.contains('resize-handle')) {
                return;
            }

            // In edit mode, select the object and allow dragging
            if (isEditMode) {
                console.log('Edit mode detected, selecting object:', element.dataset.name);
                WebTalkObjects.selectObject(element);
                
                // In edit mode, we want to allow dragging the object
                // Don't return early - continue to the dragging code below
                e.preventDefault(); // Prevent default to ensure selection works
                e.stopPropagation(); // Stop propagation to prevent card from deselecting
                
                // Always enable dragging in edit mode
                // This is the key change - we're forcing draggable to be true in edit mode
                element.isDraggable = true;
                
                // Continue to dragging code below
            }
            
            // Only handle dragging in this handler when draggable is enabled
            if (element.isDraggable) {
                // Handle dragging logic here
                e.preventDefault();
                e.stopPropagation();
                // Dragging code will continue below
            } else {
                // If not draggable, call original handler if it exists
                if (element._originalMouseDown) {
                    element._originalMouseDown.call(element, e);
                }
                // Don't prevent default or stop propagation - let event bubble to addEventListener handlers
                return;
            }

            // Don't drag if right-click
            if (e.button === 2) {
                return;
            }

            e.preventDefault();
            pos3 = e.clientX;
            pos4 = e.clientY;
            
            // Store the intended position (for grid snapping) - start with current position
            element._intendedTop = element.offsetTop;
            element._intendedLeft = element.offsetLeft;
            
            // Make sure to clean up any existing handlers first to prevent memory leaks
            if (document.onmouseup) document.onmouseup();
            
            // Set up new handlers
            document.onmouseup = closeDragElement;
            document.onmousemove = elementDrag;
        };

        // Add touch event support
        element.ontouchstart = function(e) {
            // Check for edit mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';

            // Don't start drag if touching a resize handle
            if (e.target.classList.contains('resize-handle')) {
                return;
            }

            // In edit mode, select the object
            if (isEditMode) {
                console.log('Touch edit mode detected, selecting object:', element.dataset.name);
                WebTalkObjects.selectObject(element);
                e.preventDefault(); // Prevent default to ensure selection works
                e.stopPropagation(); // Stop propagation to prevent card from deselecting
            }

            // If not in draggable mode and not in edit mode, don't proceed with drag
            if (!element.isDraggable && !isEditMode) {
                // For fields, allow default behavior when not in draggable mode
                if (element.dataset.type === 'field') {
                    return true;
                }
                return;
            }

            // Only handle single touch for dragging
            if (e.touches.length === 1) {
                e.preventDefault();
                pos3 = e.touches[0].clientX;
                pos4 = e.touches[0].clientY;
                
                // Store the intended position (for grid snapping) - start with current position
                element._intendedTop = element.offsetTop;
                element._intendedLeft = element.offsetLeft;
                
                // Make sure to clean up any existing handlers first to prevent memory leaks
                if (document.ontouchend) document.ontouchend();
                
                // Set up new handlers
                document.ontouchend = closeDragElement;
                document.ontouchcancel = closeDragElement;
                document.ontouchmove = elementDragTouch;
            }
        };

        function snapToGrid(value, gridSize) {
            if (gridSize <= 0) return value;
            return Math.round(value / gridSize) * gridSize;
        }

        function elementDrag(e) {
            e.preventDefault();
            pos1 = pos3 - e.clientX;
            pos2 = pos4 - e.clientY;
            pos3 = e.clientX;
            pos4 = e.clientY;

            // Update the intended position based on mouse movement
            element._intendedTop = element._intendedTop - pos2;
            element._intendedLeft = element._intendedLeft - pos1;

            let newTop = element._intendedTop;
            let newLeft = element._intendedLeft;

            // Apply grid snapping if gridSize is set and we're in edit mode or draggable mode
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
            if ((isEditMode || element.isDraggable) && window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.gridSize > 0) {
                newTop = snapToGrid(newTop, window.webTalkApp.interpreter.gridSize);
                newLeft = snapToGrid(newLeft, window.webTalkApp.interpreter.gridSize);
            }

            element.style.top = newTop + "px";
            element.style.left = newLeft + "px";
        }

        function elementDragTouch(e) {
            if (e.touches.length === 1) {
                e.preventDefault();
                pos1 = pos3 - e.touches[0].clientX;
                pos2 = pos4 - e.touches[0].clientY;
                pos3 = e.touches[0].clientX;
                pos4 = e.touches[0].clientY;

                // Update the intended position based on touch movement
                element._intendedTop = element._intendedTop - pos2;
                element._intendedLeft = element._intendedLeft - pos1;

                let newTop = element._intendedTop;
                let newLeft = element._intendedLeft;

                // Apply grid snapping if gridSize is set and we're in edit mode or draggable mode
                const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit';
                if ((isEditMode || element.isDraggable) && window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.gridSize > 0) {
                    newTop = snapToGrid(newTop, window.webTalkApp.interpreter.gridSize);
                    newLeft = snapToGrid(newLeft, window.webTalkApp.interpreter.gridSize);
                }

                element.style.top = newTop + "px";
                element.style.left = newLeft + "px";
            }
        }

        function closeDragElement() {
            // Clear all event handlers to prevent memory leaks and ensure proper cleanup
            document.onmouseup = null;
            document.onmousemove = null;
            document.ontouchend = null;
            document.ontouchcancel = null;
            document.ontouchmove = null;
            
            // Check for edit mode to ensure proper cleanup
            const isEditMode = window.webTalkApp && window.webTalkApp.interpreter && 
                window.webTalkApp.interpreter.mode === 'edit';
                
            // In edit mode, ensure we properly release the object
            if (isEditMode) {
                console.log('Releasing dragged object in edit mode:', element.dataset.name);
            }

            // Update points for graphic objects after dragging
            if (element.dataset.type === 'graphic') {
                const objectName = element.dataset.name;
                const svg = element.querySelector('svg');
                if (svg) {
                    const graphicType = element.dataset.graphicType || 'line';

                    // Update points based on graphic type
                    if (graphicType === 'line' && svg.querySelector('line')) {
                        // Get the current position of the graphic container
                        const offsetLeft = element.offsetLeft;
                        const offsetTop = element.offsetTop;

                        // Get the line coordinates (relative to the container)
                        const x1 = parseInt(svg.querySelector('line').getAttribute('x1'));
                        const y1 = parseInt(svg.querySelector('line').getAttribute('y1'));
                        const x2 = parseInt(svg.querySelector('line').getAttribute('x2'));
                        const y2 = parseInt(svg.querySelector('line').getAttribute('y2'));

                        // Store the absolute coordinates in the custom properties
                        const points = `${x1},${y1},${x2},${y2}`;
                        if (!WebTalkObjects.customProperties.has(objectName)) {
                            WebTalkObjects.customProperties.set(objectName, new Map());
                        }
                        WebTalkObjects.customProperties.get(objectName).set('points', points);
                    } else if (graphicType === 'path' && svg.querySelector('path')) {
                        if (!WebTalkObjects.customProperties.has(objectName)) {
                            WebTalkObjects.customProperties.set(objectName, new Map());
                        }
                        WebTalkObjects.customProperties.get(objectName).set('points', svg.querySelector('path').getAttribute('d'));
                    }
                }
            }
        }
    }

    static updateDraggableState(draggable) {
        // Update all objects' draggable state
        this.objects.forEach((element) => {
            element.isDraggable = draggable;
        });

        // Update cursor for the card
        const card = document.getElementById('card');
        if (card) {
            card.style.cursor = draggable ? 'default' : 'pointer';
        }
    }

    static addContextMenu(element) {
        element.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            e.stopPropagation();

            // Remove any existing context menus
            const existingMenu = document.querySelector('.context-menu');
            if (existingMenu) {
                existingMenu.remove();
            }

            // Create context menu
            const contextMenu = document.createElement('div');
            contextMenu.className = 'context-menu';
            contextMenu.style.top = `${e.clientY}px`;
            contextMenu.style.left = `${e.clientX}px`;

            // Add edit script menu item
            const editScriptItem = document.createElement('div');
            editScriptItem.className = 'context-menu-item';
            editScriptItem.textContent = 'Edit Script';
            editScriptItem.addEventListener('click', () => {
                WebTalkObjects.openScriptEditor(element.dataset.name);
                contextMenu.remove();
            });

            contextMenu.appendChild(editScriptItem);

            // In edit mode, add inspect option
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit') {
                const inspectItem = document.createElement('div');
                inspectItem.className = 'context-menu-item';
                inspectItem.textContent = 'Inspect';
                inspectItem.addEventListener('click', () => {
                    WebTalkObjects.selectObject(element);
                    if (WebTalkInspector) {
                        WebTalkInspector.showInspector(element);
                    }
                    contextMenu.remove();
                });

                contextMenu.appendChild(inspectItem);
            }

            // Add duplicate menu item if in edit mode
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit') {
                const objectType = element.dataset.type; // Assuming type is stored in dataset, e.g., 'button' or 'field'
                if (objectType) {
                    const duplicateItem = document.createElement('div');
                    duplicateItem.className = 'context-menu-item';
                   duplicateItem.textContent = `Duplicate ${objectType.charAt(0).toUpperCase() + objectType.slice(1)}`; // Capitalize the first letter for display
                    duplicateItem.addEventListener('click', () => {
                        WebTalkObjects.duplicateObject(element); // Call the new duplicate method
                        contextMenu.remove(); // Close the menu after action
                    });
                    contextMenu.appendChild(duplicateItem);
                }
}

            // Add context menu to document
            document.body.appendChild(contextMenu);

            // Close menu when clicking elsewhere
            const closeMenu = (e) => {
                if (!contextMenu.contains(e.target)) {
                    contextMenu.remove();
                    document.removeEventListener('mousedown', closeMenu);
                }
            };

            document.addEventListener('mousedown', closeMenu);
        });
    }

    static setScript(objectName, script) {
        this.scripts.set(objectName, script);
    }

    static getScript(objectName) {
        return this.scripts.get(objectName) || '';
    }

    static openScriptEditor(objectName, errorLine = null) {
        // Check if script editor is already open
        const existingEditor = document.querySelector('.script-editor-overlay');
        if (existingEditor) {
            // If already open, don't open another one
            return;
        }

        // Special handling for card objects
        if (objectName === 'card' || objectName === 'this card' || objectName === 'current card') {
            // Get the current card element and ID
            const cardElement = window.currentCard || document.getElementById('card');
            const cardId = window.currentCardId || 1;
            
            if (!cardElement) return;
            
            // Use the specific card ID as the object name for script lookup
            const cardScriptName = cardId === 1 ? 'card' : `card-${cardId}`;
            
            // Ensure card script exists in the scripts map
            if (!this.scripts.has(cardScriptName)) {
                this.scripts.set(cardScriptName, '');
            }
            
            // Update objectName to use the specific card ID
            objectName = cardScriptName;
        } else if (objectName.startsWith('card-')) {
            // Direct reference to a specific card by ID
            const cardId = objectName.substring(5); // Extract the ID part
            const cardElement = document.getElementById(objectName);
            
            if (!cardElement) return;
            
            // Ensure card script exists in the scripts map
            if (!this.scripts.has(objectName)) {
                this.scripts.set(objectName, '');
            }
        } else {
            // For other objects, get the DOM element
            const object = this.getObject(objectName);
            if (!object) return;
        }

        // Create script editor overlay
        const overlay = document.createElement('div');
        overlay.className = 'script-editor-overlay';

        // Create script editor dialog
        const dialog = document.createElement('div');
        dialog.className = 'script-editor-dialog';

        // Create header
        const header = document.createElement('div');
        header.className = 'script-editor-header';

        const title = document.createElement('div');
        title.className = 'script-editor-title';
        title.textContent = `Script of ${objectName}`;

        header.appendChild(title);

        // Create content area
        const content = document.createElement('div');
        content.className = 'script-editor-content';

        const textareaContainer = document.createElement('div');
        textareaContainer.className = 'script-editor-textarea-container';

        // Create textarea for editing
        const textarea = document.createElement('textarea');
        textarea.className = 'script-editor-textarea';
        textarea.value = this.getScript(objectName);
        // fixes for iOS and Android script editor field
        textarea.spellcheck = false;
        textarea.autocapitalize="none";
        textarea.autocorrect="off";
        textarea.autocomplete="off";

        // Create highlight layer for syntax highlighting
        const highlightLayer = document.createElement('div');
        highlightLayer.className = 'script-editor-highlight';

        // Create autocomplete suggestions container
        const suggestionsContainer = document.createElement('div');
        suggestionsContainer.className = 'script-editor-suggestions';
        suggestionsContainer.style.display = 'none';

        // Flag to track if autocomplete is enabled
        let autocompleteEnabled = false;

        // Function to format the entire script with proper indentation
        const formatScript = () => {
            const text = textarea.value;
            const lines = text.split('\n');
            let formattedLines = [];
            let indentLevel = 0;
            let inIfBlock = false;

            for (let i = 0; i < lines.length; i++) {
                const line = lines[i];
                const trimmedLine = line.trim();

                // Determine indentation for this line
                let currentIndent = '';

                // Check for 'end if' or 'end' statements that should reduce indentation
                if (/^end\s+if\b/i.test(trimmedLine)) {
                    // Reduce indent for 'end if'
                    if (indentLevel > 0) indentLevel--;
                    inIfBlock = false;
                } else if (/^end\s+repeat\b/i.test(trimmedLine)) {
                    // Reduce indent for 'end repeat'
                    if (indentLevel > 0) indentLevel--;
                } else if (/^end\s+try\b/i.test(trimmedLine)) {
                    // Reduce indent for 'end try'
                    if (indentLevel > 0) indentLevel--;
                } else if (/^end\s+switch\b/i.test(trimmedLine)) {
                    // Reduce indent for 'end switch'
                    if (indentLevel > 0) indentLevel--;
                } else if (/^catch\b/i.test(trimmedLine)) {
                    // 'catch' should be at the same level as its 'try'
                    if (indentLevel > 0) indentLevel--;
                } else if (/^else\b/i.test(trimmedLine)) {
                    // 'else' should be at the same level as its 'if'
                    if (indentLevel > 0 && inIfBlock) indentLevel--;
                } else if (/^end\s+\w+$/i.test(trimmedLine)) {
                    // Any other 'end' statement (like 'end mouseup')
                    indentLevel = 0; // Reset to base level for handler end
                }

                // Apply indentation
                for (let j = 0; j < indentLevel; j++) {
                    currentIndent += '   '; // 3 spaces per indent level
                }

                // Add the formatted line
                formattedLines.push(currentIndent + trimmedLine);

                // Increase indentation for next line if needed
                if (/^on\s+\w+/i.test(trimmedLine)) {
                    // Start of handler - indent next line
                    indentLevel = 1;
                } else if (/^if\b.*\bthen\b/i.test(trimmedLine) && !/^if\b.*\bthen\b.*\bend\s+if\b/i.test(trimmedLine)) {
                    // Only increase indent if the line ends with 'then' (for multi-line if statements)
                    // If the line has a complete action after 'then', don't increase indentation
                    if (/\bthen\s*$/i.test(trimmedLine)) {
                        // Line ends with 'then', so indent the next line
                        indentLevel++;
                        inIfBlock = true;
                    }
                    // No else clause needed - if the line has 'then' followed by a command,
                    // we don't want to increase indentation
                } else if (/^else\b/i.test(trimmedLine)) {
                    // Increase indent after 'else' for the next line
                    indentLevel++;
                } else if (/^repeat\b/i.test(trimmedLine)) {
                    // Repeat loop
                    indentLevel++;
                } else if (/^try\b/i.test(trimmedLine)) {
                    // Try block
                    indentLevel++;
                } else if (/^catch\b/i.test(trimmedLine)) {
                    // Catch block
                    indentLevel++;
                } else if (/^switch\b/i.test(trimmedLine)) {
                    // Switch statement
                    indentLevel++;
                } else if (/^case\b/i.test(trimmedLine)) {
                    // Case statement - keep at same indentation level as other cases
                    // No change to indentLevel
                } else if (/^break\b/i.test(trimmedLine)) {
                    // Break statement - keep at same indentation level as case statements
                    // No change to indentLevel
                }
            }

            // Update the textarea with formatted text
            const cursorPos = textarea.selectionStart;
            const formattedText = formattedLines.join('\n');

            // Only update if there's a change
            if (formattedText !== text) {
                textarea.value = formattedText;

                // Try to maintain cursor position
                textarea.setSelectionRange(cursorPos, cursorPos);

                // Force focus to prevent ghosted text
                textarea.focus();
            }
        };

        // Update syntax highlighting when text changes
        textarea.addEventListener('input', () => {
            // Update the highlight layer with the current text
            let html = this.highlightSyntax(textarea.value);
            
            // Add multiple extra <br> tags to ensure proper scrolling and synchronization
            html += '<br><br><br><br><br>';
            
            highlightLayer.innerHTML = html;

            // Sync scroll position
            highlightLayer.scrollTop = textarea.scrollTop;
            highlightLayer.scrollLeft = textarea.scrollLeft;

            // Show autocomplete suggestions if enabled
            if (autocompleteEnabled) {
                this.showAutocompleteSuggestions(textarea, suggestionsContainer);
            }
        });

        // Ensure the highlight layer stays in sync with the textarea
        textarea.addEventListener('scroll', () => {
            highlightLayer.scrollTop = textarea.scrollTop;
            highlightLayer.scrollLeft = textarea.scrollLeft;
        });

        // Handle keydown events for autocomplete navigation and formatting
        textarea.addEventListener('keydown', (e) => {
            if (autocompleteEnabled && suggestionsContainer.style.display !== 'none') {
                const suggestions = suggestionsContainer.querySelectorAll('.suggestion-item');
                const selectedIndex = Array.from(suggestions).findIndex(item => item.classList.contains('selected'));

                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    if (selectedIndex < suggestions.length - 1) {
                        if (selectedIndex >= 0) {
                            suggestions[selectedIndex].classList.remove('selected');
                        }
                        suggestions[selectedIndex + 1].classList.add('selected');
                    }
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    if (selectedIndex > 0) {
                        suggestions[selectedIndex].classList.remove('selected');
                        suggestions[selectedIndex - 1].classList.add('selected');
                    }
                } else if (e.key === 'Tab' || e.key === 'Enter') {
                    e.preventDefault();
                    if (selectedIndex >= 0) {
                        this.applyAutocompleteSuggestion(textarea, suggestions[selectedIndex].textContent);
                        suggestionsContainer.style.display = 'none';
                        highlightLayer.innerHTML = this.highlightSyntax(textarea.value);
                    }
                } else if (e.key === 'Escape') {
                    suggestionsContainer.style.display = 'none';
                }
                return;
            }

            // Handle Enter key for auto-completion of handlers and indentation
            if (e.key === 'Enter') {
                const text = textarea.value;
                const cursorPos = textarea.selectionStart;
                const textBeforeCursor = text.substring(0, cursorPos);
                const textAfterCursor = text.substring(cursorPos);
                const lines = textBeforeCursor.split('\n');
                const currentLine = lines[lines.length - 1];

                // Check if the current line starts with 'on' to detect handler definition
                const handlerMatch = currentLine.match(/^\s*on\s+(\w+)(?:\s+(.+))?/i);
                if (handlerMatch) {
                    const handlerName = handlerMatch[1];
                    const paramName = handlerMatch[2] || '';

                    // Check if there's already an 'end handlerName' in the script
                    const endHandlerRegex = new RegExp(`end\\s+${handlerName}`, 'i');
                    if (!text.match(endHandlerRegex)) {
                        // Add an empty line and 'end handlerName'
                        const indentation = currentLine.match(/^\s*/)[0]; // Get current indentation

                        // Create the new text with proper indentation
                        const newText = textBeforeCursor + '\n' +
                                      indentation + '   ' + '\n' +
                                      indentation + 'end ' + handlerName +
                                      textAfterCursor;

                        textarea.value = newText;

                        // Position cursor after the empty line (ready for user to type handler content)
                        const newCursorPos = textBeforeCursor.length + 1 + indentation.length + 3; // +1 for '\n', +3 for '   '
                        textarea.setSelectionRange(newCursorPos, newCursorPos);

                        // Update syntax highlighting
                        let html = this.highlightSyntax(textarea.value);
                        html += '<br><br><br><br><br>';
                        highlightLayer.innerHTML = html;

                        e.preventDefault();
                        return;
                    }
                }
                
                // Check if the current line starts with 'switch' to detect switch statement
                const switchMatch = currentLine.match(/^\s*switch\s+(.+)/i);
                if (switchMatch) {
                    // Check if there's already an 'end switch' in the script
                    if (!text.match(/end\s+switch/i)) {
                        // Add just the end switch, without any case statements
                        const indentation = currentLine.match(/^\s*/)[0]; // Get current indentation
                        
                        // Create the new text with proper indentation
                        const newText = textBeforeCursor + '\n' +
                                      indentation + 'end switch' +
                                      textAfterCursor;

                        textarea.value = newText;

                        // Position cursor at the beginning of the line after switch
                        const newCursorPos = textBeforeCursor.length + 1; // Just after the newline
                        textarea.setSelectionRange(newCursorPos, newCursorPos);

                        // Update syntax highlighting
                        let html = this.highlightSyntax(textarea.value);
                        html += '<br><br><br><br><br>';
                        highlightLayer.innerHTML = html;

                        e.preventDefault();
                        return;
                    }
                }
                // Check if the current line starts with 'case' and ends with a quote to detect case statement
                const caseMatch = currentLine.match(/^(\s*)case\s+".*"\s*$/i);
                if (caseMatch && cursorPos === textBeforeCursor.length) {
                    const indentation = caseMatch[1]; // Get current indentation
                    const codeIndent = indentation + '   ';
                    
                    // Add indented line for code only, no auto-insertion of another case
                    const newText = textBeforeCursor + '\n' + codeIndent;
                    
                    textarea.value = newText + textAfterCursor;
                    
                    // Position cursor at the indented line
                    const newCursorPos = textBeforeCursor.length + 1 + codeIndent.length;
                    textarea.setSelectionRange(newCursorPos, newCursorPos);
                    
                    // Update syntax highlighting
                    let html = this.highlightSyntax(textarea.value);
                    html += '<br><br><br><br><br>';
                    highlightLayer.innerHTML = html;
                    e.preventDefault();
                    return;
                }
                
                // Check if the current line is empty or contains only whitespace
                if (/^\s*$/.test(currentLine) && cursorPos === textBeforeCursor.length) {
                    // Just insert a new line normally without any special handling
                    const newText = textBeforeCursor + '\n' + textAfterCursor;
                    textarea.value = newText;

                    // Set cursor position at the beginning of the new line
                    const newCursorPos = textBeforeCursor.length + 1;
                    textarea.setSelectionRange(newCursorPos, newCursorPos);

                    // Update syntax highlighting
                    let html = this.highlightSyntax(textarea.value);
                    html += '<br><br><br><br><br>';
                    highlightLayer.innerHTML = html;

                    e.preventDefault();
                    return;
                }
                 // Check if the current line ends with 'end handlerName'
                 const endHandlerMatch = currentLine.match(/^\s*end\s+\w+\s*$/i);
                 if (endHandlerMatch && cursorPos === textBeforeCursor.length) {
                     // Just insert a new line normally without any special handling
                     const newText = textBeforeCursor + '\n' + textAfterCursor;
                     textarea.value = newText;

                     // Set cursor position at the beginning of the new line
                     const newCursorPos = textBeforeCursor.length + 1;
                     textarea.setSelectionRange(newCursorPos, newCursorPos);

                     // Update syntax highlighting
                     highlightLayer.innerHTML = this.highlightSyntax(textarea.value);
                     e.preventDefault();
                     return;
                 }

                // Get the indentation of the current line
                const currentIndent = currentLine.match(/^\s*/)[0];

                // Determine if we need to increase indentation
                let newIndent = currentIndent;

                // Check if the current line ends with a keyword that should increase indentation
                if (/\b(then|else|repeat)\s*$/i.test(currentLine)) {
                    newIndent += '   '; // Add 3 spaces for increased indentation
                }

                // Insert a new line with the appropriate indentation
                const newText = textBeforeCursor + '\n' + newIndent + textAfterCursor;
                textarea.value = newText;

                // Set cursor position after the indentation on the new line
                const newCursorPos = textBeforeCursor.length + 1 + newIndent.length;
                textarea.setSelectionRange(newCursorPos, newCursorPos);

                // Update syntax highlighting
                let html = this.highlightSyntax(textarea.value);
                html += '<br><br><br><br><br>';
                highlightLayer.innerHTML = html;

                // Format the script to ensure proper indentation
                formatScript();

                e.preventDefault();
                return;
            }

            // Handle Tab key for indentation and formatting
            if (e.key === 'Tab') {
                e.preventDefault();

                // Format the entire script to fix any indentation issues
                formatScript();
            }
        });

        // Initial syntax highlighting
        let initialHtml = this.highlightSyntax(textarea.value);
        // Add extra lines to ensure proper scrolling
        initialHtml += '<br><br><br><br><br>';
        highlightLayer.innerHTML = initialHtml;

        // Apply error line highlighting if specified
        if (errorLine !== null && errorLine > 0) {
            const lines = textarea.value.split('\n');
            if (errorLine <= lines.length) {
                // Create a new highlight layer with error line highlighting
                const highlightedContent = this.highlightSyntax(textarea.value);
                const highlightedLines = highlightedContent.split('<br>');
                
                // Add error-line class to the specified line
                highlightedLines[errorLine - 1] = `<span class="error-line">${highlightedLines[errorLine - 1]}</span>`;
                
                // Update the highlight layer
                highlightLayer.innerHTML = highlightedLines.join('<br>');
            }
        }

        textareaContainer.appendChild(textarea);
        textareaContainer.appendChild(highlightLayer);
        textareaContainer.appendChild(suggestionsContainer);
        content.appendChild(textareaContainer);

        // Create footer with buttons
        const footer = document.createElement('div');
        footer.className = 'script-editor-footer';

        // Create autocomplete checkbox
        const autocompleteContainer = document.createElement('div');
        autocompleteContainer.className = 'script-editor-autocomplete-container';

        const autocompleteCheckbox = document.createElement('input');
        autocompleteCheckbox.type = 'checkbox';
        autocompleteCheckbox.id = 'autocomplete-checkbox';
        autocompleteCheckbox.className = 'script-editor-autocomplete-checkbox';
        autocompleteCheckbox.checked = autocompleteEnabled;

        const autocompleteLabel = document.createElement('label');
        autocompleteLabel.htmlFor = 'autocomplete-checkbox';
        autocompleteLabel.className = 'script-editor-autocomplete-label';
        autocompleteLabel.textContent = 'Autocomplete';

        autocompleteCheckbox.addEventListener('change', () => {
            autocompleteEnabled = autocompleteCheckbox.checked;
            if (!autocompleteEnabled) {
                suggestionsContainer.style.display = 'none';
            }
        });

        autocompleteContainer.appendChild(autocompleteCheckbox);
        autocompleteContainer.appendChild(autocompleteLabel);

        const formatButton = document.createElement('button');
        formatButton.className = 'script-editor-button script-editor-format';
        formatButton.textContent = 'Format';
        // this is where we specify how the format button works in the script editor.
        formatButton.addEventListener('click', () => {
            // Remember scroll position before formatting
            const scrollTop = textarea.scrollTop;
            const scrollLeft = textarea.scrollLeft;

            // Format the script
            formatScript();

            // Force refresh of the highlight layer
            let html = this.highlightSyntax(textarea.value);
            html += '<br><br><br><br><br>';
            highlightLayer.innerHTML = html;

            // Restore scroll position
            textarea.scrollTop = scrollTop;
            textarea.scrollLeft = scrollLeft;
            highlightLayer.scrollTop = scrollTop;
            highlightLayer.scrollLeft = scrollLeft;
        });

        footer.appendChild(autocompleteContainer);
        footer.appendChild(formatButton);

        // Add a spacer div to create the 40px gap
        const spacer = document.createElement('div');
        spacer.style.display = 'inline-block';
        spacer.style.width = '40px';
        footer.appendChild(spacer);

        const cancelButton = document.createElement('button');
        cancelButton.className = 'script-editor-button script-editor-cancel';
        cancelButton.textContent = 'Cancel';
        cancelButton.addEventListener('click', () => {
            // Remove all script editor related elements
            overlay.remove();
            
            // Remove any other potential script editor elements
            const existingOverlays = document.querySelectorAll('.script-editor-overlay');
            existingOverlays.forEach(el => el.remove());
            
            // Remove any background covers that might be left
            const backgroundCovers = document.querySelectorAll('.dialog-background-cover');
            backgroundCovers.forEach(el => el.remove());
        });

        const saveButton = document.createElement('button');
        saveButton.className = 'script-editor-button script-editor-save';
        saveButton.textContent = 'Save';
        saveButton.addEventListener('click', () => {
            this.setScript(objectName, textarea.value);
            overlay.remove();
        });

        footer.appendChild(cancelButton);
        footer.appendChild(saveButton);

        // Assemble dialog
        dialog.appendChild(header);
        dialog.appendChild(content);
        dialog.appendChild(footer);

        // resizer element for the script editor
        const resizer = document.createElement('div');
        resizer.className = 'script-editor-resizer';
        dialog.appendChild(resizer);
        overlay.appendChild(dialog);

        // Add to document
        document.body.appendChild(overlay);

        // Focus textarea
        textarea.focus();
    }

    static showAutocompleteSuggestions(textarea, suggestionsContainer) {
        const text = textarea.value;
        const suggestions = this.getSuggestions(text);

        // Clear existing suggestions
        suggestionsContainer.innerHTML = '';

        // Add new suggestions
        suggestions.forEach(suggestion => {
            const suggestionItem = document.createElement('div');
            suggestionItem.className = 'suggestion-item';
            suggestionItem.textContent = suggestion;
            suggestionsContainer.appendChild(suggestionItem);
        });

        // Show suggestions container
        suggestionsContainer.style.display = 'block';
    }

    static applyAutocompleteSuggestion(textarea, suggestion) {
        const text = textarea.value;
        const lastWord = text.split(' ').pop();
        const newText = text.replace(lastWord, suggestion);
        textarea.value = newText;
    }

    static getSuggestions(text) {
        if (!text) return [];

        const keywords = [
            'on', 'end', 'if', 'then', 'else', 'repeat', 'with', 'while', 'until', 'put', 'get',
            'into', 'after', 'before', 'the', 'to', 'in', 'and', 'or', 'not', 'is', 'add',
            'subtract', 'multiply', 'divide', 'exit', 'pass', 'return', 'next', 'function',
            'switch', 'case', 'break'
        ];

        const functions = [
            'abs', 'average', 'date', 'time', 'length', 'random', 'round', 'sqrt', 'sum', 'trunc',
            'word', 'char', 'item', 'line', 'offset', 'contains', 'number', 'min', 'max', 'sin',
            'cos', 'tan', 'exp', 'ln', 'log2', 'log10', 'pi', 'seconds', 'ticks', 'convert',
            'format', 'value', 'clickLoc', 'mouseLoc', 'screenRect', 'params'
        ];

        const operators = ['of'];

        const lastWord = text.split(' ').pop();
        const suggestions = [...keywords, ...functions, ...operators].filter(suggestion =>
            suggestion.toLowerCase().startsWith(lastWord.toLowerCase()));

        return suggestions;
    }

    static highlightSyntax(code) {
        if (!code) return '';

        // Split the code into lines for more precise handling
        const lines = code.split('\n');
        const processedLines = [];

        for (let line of lines) {
            // Process each line separately to avoid cross-line issues
            let processedLine = line;

            // Check if line might contain an error (simplified detection)
            const errorPatterns = [
                /\b(put|get)\s+$/i,  // put/get with nothing after
                /\bif\s+$/i,         // if with nothing after
                /\bend\s+$/i,        // end with nothing after
                /\brepeat\s+$/i,     // repeat with nothing after
                /\bswitch\s+$/i,     // switch with nothing after
                /\bcase\s+$/i,       // case with nothing after
                /^\s*then\s*$/i,     // then alone on a line (not part of an if statement)
                /^\s*else\s*$/i,     // else alone on a line
                /\s+=\s*$/,          // equals sign at the end
                /^\s*=\s+/,          // equals sign at the beginning
                /\(\s*\)/,           // empty parentheses
                /\[\s*\]/,           // empty brackets
                /\{\s*\}/,           // empty braces
                /&quot;\s*&quot;/,   // empty quotes (escaped)
                /\bend\s+\w+\s+\w+/i // end handler with extra text
            ];
            
            let hasError = false;
            
            // Check for errors using the patterns
            for (const pattern of errorPatterns) {
                if (pattern.test(line)) {
                    hasError = true;
                    break;
                }
            }
            
            // Check for comments first - they take precedence over everything
            const commentIndex = line.indexOf('--');
            if (commentIndex >= 0) {
                // Split the line into content before comment and the comment itself
                const beforeComment = line.substring(0, commentIndex);
                const comment = line.substring(commentIndex);
                
                // Process the content before the comment normally
                let processedBeforeComment = beforeComment;
                
                // Check for unclosed quotes before escaping
                const hasUnclosedQuotes = (beforeComment.match(/"/g) || []).length % 2 !== 0;
                
                // Escape HTML for display
                processedBeforeComment = processedBeforeComment.replace(/[&<>'"]/g,
                    tag => ({
                        '&': '&amp;',
                        '<': '&lt;',
                        '>': '&gt;',
                        // Only escape single quotes if not inside an unclosed double-quoted string
                        "'": hasUnclosedQuotes ? "'" : '&#39;',
                        '"': '&quot;'
                    }[tag]));
                
                // Process the part before comment with normal syntax highlighting
                let beforeResult = '';
                if (processedBeforeComment) {
                    // Extract and process strings
                    let lastIndex = 0;
                    const stringRegex = /&quot;(.*?)&quot;/g;
                    let match;
                    
                    while ((match = stringRegex.exec(processedBeforeComment)) !== null) {
                        // Add the text before the string
                        beforeResult += this.processNonStringContent(processedBeforeComment.substring(lastIndex, match.index), false);
                        
                        // Add the string with proper highlighting
                        beforeResult += '<span class="ht-string">' + match[0] + '</span>';
                        
                        lastIndex = match.index + match[0].length;
                    }
                    
                    // Add any remaining text
                    if (lastIndex < processedBeforeComment.length) {
                        beforeResult += this.processNonStringContent(processedBeforeComment.substring(lastIndex), false);
                    }
                    
                    // If no strings were found, process the whole part
                    if (beforeResult === '') {
                        beforeResult = this.processNonStringContent(processedBeforeComment, false);
                    }
                }
                
                // Escape the comment part and wrap in comment span
                const escapedComment = comment.replace(/[&<>'"]/g,
                    tag => ({
                        '&': '&amp;',
                        '<': '&lt;',
                        '>': '&gt;',
                        "'": '&#39;',
                        '"': '&quot;'
                    }[tag]));
                
                // Combine the processed parts
                let result = beforeResult + '<span class="ht-comment">' + escapedComment + '</span>';
                
                // Apply error highlighting if needed
                if (hasError) {
                    result = '<span class="ht-error">' + result + '</span>';
                }
                
                processedLines.push(result);
                continue; // Skip the rest of the processing for this line
            }
            
            // If no comment, process normally
            // Check for unclosed quotes before escaping
            const hasUnclosedQuotes = (line.match(/"/g) || []).length % 2 !== 0;

            // Escape HTML for display
            processedLine = processedLine.replace(/[&<>'"]/g,
                tag => ({
                    '&': '&amp;',
                    '<': '&lt;',
                    '>': '&gt;',
                    // Only escape single quotes if not inside an unclosed double-quoted string
                    "'": hasUnclosedQuotes ? "'" : '&#39;',
                    '"': '&quot;'
                }[tag]));

            // Extract and process strings first
            let result = '';
            let lastIndex = 0;
            const stringRegex = /&quot;(.*?)&quot;/g;
            let match;

            while ((match = stringRegex.exec(processedLine)) !== null) {
                // Add the text before the string
                result += this.processNonStringContent(processedLine.substring(lastIndex, match.index), true);

                // Add the string with proper highlighting
                result += '<span class="ht-string">' + match[0] + '</span>';

                lastIndex = match.index + match[0].length;
            }

            // Add any remaining text
            if (lastIndex < processedLine.length) {
                result += this.processNonStringContent(processedLine.substring(lastIndex), true);
            }

            // If no strings were found, process the whole line
            if (result === '') {
                result = this.processNonStringContent(processedLine, true);
            }

            // Apply error highlighting after all other processing
            if (hasError) {
                result = '<span class="ht-error">' + result + '</span>';
            }

            processedLines.push(result);
        }

        // Join the lines back with <br> tags
        return processedLines.join('<br>');
    }

    static processNonStringContent(text, checkForComments = true) {
        // Highlight comments if requested (they take precedence)
        if (checkForComments) {
            const commentMatch = text.match(/--.*$/);
            if (commentMatch) {
                const commentIndex = text.indexOf('--');
                const beforeComment = text.substring(0, commentIndex);
                const comment = text.substring(commentIndex);
                return beforeComment + '<span class="ht-comment">' + comment + '</span>';
            }
        }

        // Process variables (the X of Y)
        text = text.replace(/\b(the\s+\w+(\s+of\s+\w+)*)\b/gi, function(match) {
            return '<span class="ht-variable">' + match + '</span>';
        });

        // Process keywords first (including 'on' and 'end')
        const keywords = [
            'on', 'end', 'if', 'then', 'else', 'repeat', 'with', 'while', 'until', 'put', 'get',
            'into', 'after', 'before', 'the', 'to', 'in', 'and', 'or', 'not', 'is', 'add',
            'subtract', 'multiply', 'divide', 'exit', 'pass', 'return', 'next', 'function',
            'try', 'catch'
        ];
        
        // Switch-related keywords get special highlighting
        const switchKeywords = ['switch', 'case', 'break'];

        // Process regular keywords
        for (const keyword of keywords) {
            const regex = new RegExp('\\b(' + keyword + ')\\b', 'gi');
            text = text.replace(regex, function(match) {
                return '<span class="ht-keyword">' + match + '</span>';
            });
        }
        
        // Process switch-related keywords with special highlighting
        for (const keyword of switchKeywords) {
            const regex = new RegExp('\\b(' + keyword + ')\\b', 'gi');
            text = text.replace(regex, function(match) {
                return '<span class="ht-switch-keyword">' + match + '</span>';
            });
        }
        
        // Special case for 'end switch' pattern
        text = text.replace(/(\bend\s+switch\b)/gi, function(match) {
            return '<span class="ht-switch-keyword">' + match + '</span>';
        });
        
        // Highlight handler parameters as variables
        // Look for patterns like 'on handlerName param1 param2' and highlight params
        const handlerParamRegex = /(<span class="ht-keyword">on<\/span>\s+\w+)(\s+[\w\s]+)/gi;
        text = text.replace(handlerParamRegex, function(match, handlerPart, params) {
            return handlerPart + '<span class="ht-variable">' + params + '</span>';
        });
        


        // Process functions
        const functions = [
            'abs', 'average', 'date', 'time', 'length', 'random', 'round', 'sqrt', 'sum', 'trunc',
            'word', 'char', 'item', 'line', 'offset', 'contains', 'number', 'min', 'max', 'sin',
            'cos', 'tan', 'exp', 'ln', 'log2', 'log10', 'pi', 'seconds', 'ticks', 'convert',
            'format', 'value', 'clickLoc', 'mouseLoc', 'screenRect', 'params'
        ];

        for (const func of functions) {
            const regex = new RegExp('\\b(' + func + ')\\b', 'gi');
            text = text.replace(regex, function(match) {
                return '<span class="ht-function">' + match + '</span>';
            });
        }

        // Process 'of' as operator when standalone
        text = text.replace(/\b(of)\b/gi, function(match) {
            // Only replace if not already inside a span
            if (match.indexOf('<span') === -1) {
                return '<span class="ht-operator">' + match + '</span>';
            }
            return match;
        });

        // Process numbers last to avoid interference
        text = text.replace(/\b(\d+(\.\d+)?)\b/g, function(match) {
            // Only replace if not already inside a span
            if (match.indexOf('<span') === -1) {
                return '<span class="ht-number">' + match + '</span>';
            }
            return match;
        });

        return text;
    }

    // Get the name of the current card
    static getCurrentCardName() {
        const currentCardId = window.currentCardId || 1;
        return currentCardId === 1 ? 'card' : `card-${currentCardId}`;
    }
    
    // Get all cards in the stack with their elements and IDs
    static getCards() {
        const cards = [];
        
        // Get cards in their DOM order from the stack container
        const stackContainer = document.getElementById('stack-container');
        if (stackContainer) {
            const cardElements = stackContainer.querySelectorAll('.card');
            cardElements.forEach((cardElement, index) => {
                // Get the card ID from the element's dataset or ID
                let cardId;
                if (cardElement.dataset.cardId) {
                    cardId = parseInt(cardElement.dataset.cardId);
                } else if (cardElement.id === 'card') {
                    cardId = 1;
                } else if (cardElement.id.startsWith('card-')) {
                    cardId = parseInt(cardElement.id.replace('card-', ''));
                } else {
                    cardId = index + 1; // Fallback to position-based ID
                }
                
                cards.push({ element: cardElement, id: cardId });
            });
        }
        
        return cards;
    }
    
    // Get a card by its logical number (position in DOM, 1-indexed)
    static getCardByNumber(cardNumber) {
        const stackContainer = document.getElementById('stack-container');
        if (!stackContainer) return null;
        
        const cardElements = stackContainer.querySelectorAll('.card');
        if (cardNumber < 1 || cardNumber > cardElements.length) return null;
        
        return cardElements[cardNumber - 1];
    }
    
    // Get the script key for a card by its logical number
    static getCardScriptKey(cardNumber) {
        const cardElement = this.getCardByNumber(cardNumber);
        if (!cardElement) return null;
        
        // Return the element ID which is used as the script key
        return cardElement.id;
    }
    
    // Get the current card ID
    static getCurrentCard() {
        return window.currentCardId || 1;
    }
    
    // Set the current card without triggering navigation events
    static setCurrentCard(cardId) {
        // Store current card ID globally
        window.currentCardId = cardId;
        
        // Get the card element
        let cardElement;
        if (cardId === 1) {
            cardElement = document.getElementById('card');
        } else {
            cardElement = document.getElementById(`card-${cardId}`);
        }
        
        // Store current card reference globally
        if (cardElement) {
            window.currentCard = cardElement;
            
            // Apply card-specific properties, especially background color
            const cardName = cardId === 1 ? 'card' : `card-${cardId}`;
            
            // Apply background color from custom properties if available
            if (WebTalkObjects.customProperties.has(cardName)) {
                const cardProps = WebTalkObjects.customProperties.get(cardName);
                
                // Apply background color if it exists in properties
                if (cardProps.has('backgroundColor')) {
                    const bgColor = cardProps.get('backgroundColor');
                    cardElement.style.backgroundColor = bgColor;
                    console.log(`Applied stored background color ${bgColor} to card ${cardId}`);
                } else if (cardProps.has('backgroundcolor')) {
                    const bgColor = cardProps.get('backgroundcolor');
                    cardElement.style.backgroundColor = bgColor;
                    console.log(`Applied stored background color ${bgColor} to card ${cardId}`);
                } else if (cardElement.dataset.backgroundColor) {
                    // Fallback to dataset attribute if available
                    cardElement.style.backgroundColor = cardElement.dataset.backgroundColor;
                    console.log(`Applied dataset background color ${cardElement.dataset.backgroundColor} to card ${cardId}`);
                }
            }
        }
    }
    
    // Add card context menu to a card element
    static addCardContextMenu(cardElement) {
        if (!cardElement) return;
        
        // Remove any existing context menu listener to avoid duplicates
        const oldContextMenu = cardElement._contextMenuListener;
        if (oldContextMenu) {
            cardElement.removeEventListener('contextmenu', oldContextMenu);
        }
        
        // Create new context menu listener
        const contextMenuListener = (e) => {
            e.preventDefault();
            e.stopPropagation();

            // Remove any existing context menus
            const existingMenu = document.querySelector('.context-menu');
            if (existingMenu) {
                existingMenu.remove();
            }

            // Create context menu
            const contextMenu = document.createElement('div');
            contextMenu.className = 'context-menu';
            contextMenu.style.top = `${e.clientY}px`;
            contextMenu.style.left = `${e.clientX}px`;

            // Get the current card ID
            const cardId = cardElement.id === 'card' ? 1 : parseInt(cardElement.id.replace('card-', ''));
            
            // Add edit script menu item
            const editScriptItem = document.createElement('div');
            editScriptItem.className = 'context-menu-item';
            editScriptItem.textContent = 'Edit Script';
            editScriptItem.addEventListener('click', () => {
                // Use the specific card ID for script editing
                const cardScriptName = cardId === 1 ? 'card' : `card-${cardId}`;
                WebTalkObjects.openScriptEditor(cardScriptName);
                contextMenu.remove();
            });

            contextMenu.appendChild(editScriptItem);
            
            // Add inspect card menu item
            const inspectCardItem = document.createElement('div');
            inspectCardItem.className = 'context-menu-item';
            inspectCardItem.textContent = 'Inspect this card';
            inspectCardItem.addEventListener('click', () => {
                // Select the card and show the inspector
                WebTalkObjects.selectObject(cardElement);
                if (window.WebTalkInspector) {
                    window.WebTalkInspector.showInspector(cardElement);
                } else if (WebTalkInspector) {
                    WebTalkInspector.showInspector(cardElement);
                }
                contextMenu.remove();
            });

            contextMenu.appendChild(inspectCardItem);

            // Add toggle fullscreen menu item
            const fullscreenItem = document.createElement('div');
            fullscreenItem.className = 'context-menu-item';
            fullscreenItem.textContent = 'Toggle fullscreen';
            fullscreenItem.addEventListener('click', () => {
                this.toggleFullscreen();
                contextMenu.remove();
            });

            contextMenu.appendChild(fullscreenItem);

            // Add context menu to document
            document.body.appendChild(contextMenu);

            // Close menu when clicking elsewhere
            const closeMenu = (e) => {
                if (!contextMenu.contains(e.target)) {
                    contextMenu.remove();
                    document.removeEventListener('mousedown', closeMenu);
                }
            };

            document.addEventListener('mousedown', closeMenu);
        };
        
        // Store the listener for potential future removal
        cardElement._contextMenuListener = contextMenuListener;
        
        // Add the context menu listener
        cardElement.addEventListener('contextmenu', contextMenuListener);
    }
    
    static initializeCard() {
        const card = document.getElementById('card');
        if (!card) return;

        // Set up card properties if not already done
        if (!this.customProperties.has('card')) {
            this.customProperties.set('card', new Map());
            
            // Set default visible property
            this.customProperties.get('card').set('visible', 'true');
            
            // Set default layer property for card (always the bottom)
            this.customProperties.get('card').set('layer', '1');
            card.style.zIndex = '1';
            
            // Get existing card name from dataset if it exists
            const existingCardName = card.dataset.name;
            console.log('initializeCard - Existing card name:', existingCardName);
            
            // Set default name property for the first card ONLY if it doesn't already have a name
            if (!existingCardName) {
                card.dataset.name = 'card 1';
                console.log('initializeCard - Setting default card name: card 1');
            }
            
            // Register the card in the objects and objectsById maps with its current name
            const cardId = '1';
            const cardName = card.dataset.name;
            console.log('initializeCard - Registering card with name:', cardName);
            
            this.objects.set(cardName, card);
            this.objectsById.set(cardId, cardName);
            
            // For backward compatibility, also register as 'card'
            this.objects.set('card', card);
            
            this.scripts.set('card', '');
            this.scripts.set(cardName, '');
            
            // Set the card ID as a data attribute
            card.dataset.id = cardId;

            // Add event handlers for the card
            card.addEventListener('mousedown', (e) => {
                // Prevent default browser behavior for all mouse buttons
                e.preventDefault();

                // Only execute in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Execute the mouseDown handler in the card's script
                    window.webTalkApp.interpreter.executeObjectScript('card', 'mouseDown', [e.button + 1]);
                } else if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'edit') {
                    // In edit mode, deselect the currently selected object when clicking on the card
                    this.deselectObject();
                }
            });

            card.addEventListener('mouseup', (e) => {
                // Only execute in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Execute the mouseUp handler in the card's script
                    window.webTalkApp.interpreter.executeObjectScript('card', 'mouseUp', [e.button + 1]);
                }
            });

            card.addEventListener('click', (e) => {
                // Only execute in browse mode
                if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.mode === 'browse') {
                    // Execute the mouseClick handler in the card's script
                    window.webTalkApp.interpreter.executeObjectScript('card', 'mouseClick');
                }
            });
        }
        
        // Add context menu to the card
        this.addCardContextMenu(card);
        
        // Add context menu to all other cards
        const cards = this.getCards();
        for (const cardInfo of cards) {
            if (cardInfo.id !== 1) { // Skip card 1 as we already handled it
                this.addCardContextMenu(cardInfo.element);
            }
        }
        
        // Add a MutationObserver to detect when new cards are added to the DOM
        if (!window._cardObserver) {
            window._cardObserver = new MutationObserver((mutations) => {
                mutations.forEach((mutation) => {
                    if (mutation.addedNodes) {
                        mutation.addedNodes.forEach((node) => {
                            // Check if the added node is a card
                            if (node.nodeType === 1 && node.classList && node.classList.contains('card')) {
                                // Add our context menu to the new card
                                WebTalkObjects.addCardContextMenu(node);
                            }
                        });
                    }
                });
            });
            
            // Start observing the document body for added nodes
            window._cardObserver.observe(document.body, { childList: true, subtree: true });
        }
    }

    // Handle mode changes
    static updateModeState(mode) {
        console.log('updateModeState called with mode:', mode);
        if (mode === 'browse') {
            console.log('Switching to browse mode, deselecting objects');
            this.deselectObject();
            
            // Disable dragging for ALL objects when switching to browse mode
            // This is critical to prevent objects from being draggable in browse mode
            document.querySelectorAll('[data-type]').forEach(element => {
                // Force isDraggable to false for all objects
                element.isDraggable = false;
                
                // Also add a visual indicator that the object is not draggable
                if (element.style.cursor === 'move') {
                    element.style.cursor = '';
                }
            });
        } else if (mode === 'edit') {
            console.log('Switching to edit mode');
            // Objects will be made draggable when clicked in edit mode
            // We don't set isDraggable here - it happens when objects are clicked in edit mode
        }
    }

    // Add selection handles to an object
    static selectObject(element) {
        console.log('selectObject called for:', element.dataset.name);

        // First, deselect any currently selected object
        this.deselectObject(true);

        // Only allow selection in edit mode
        if (!window.webTalkApp || !window.webTalkApp.interpreter || window.webTalkApp.interpreter.mode !== 'edit') {
            console.log('Not in edit mode, selection aborted');
            return;
        }

        console.log('Adding selection handles to:', element.dataset.name);

        // Mark the element as selected
        element.classList.add('object-selected');
        this.selectedObject = element;

        // Create and add resize handles
        const handlePositions = ['nw', 'n', 'ne', 'e', 'se', 's', 'sw', 'w'];
        handlePositions.forEach(position => {
            const handle = document.createElement('div');
            handle.className = `resize-handle handle-${position}`;
            handle.dataset.handle = position;
            element.appendChild(handle);
            console.log(`Added ${position} handle to ${element.dataset.name}`);

            // Add mouse event listeners for resizing
            handle.addEventListener('mousedown', this.startResize.bind(this));
        });
    }

    // Remove selection handles from the currently selected object
    static deselectObject(keepInspectorOpen = false) {
        // Remove 'object-selected' class from any element
        document.querySelectorAll('.object-selected').forEach(el => el.classList.remove('object-selected'));

        // Remove all resize handles from the DOM
        document.querySelectorAll('.resize-handle').forEach(handle => handle.remove());

        this.selectedObject = null;

        // Close the inspector when no object is selected, unless we're switching between objects
        if (WebTalkInspector && !keepInspectorOpen) {
            WebTalkInspector.hideInspector();
        }
    }

    // Start resize operation
    static startResize(e) {
        e.preventDefault();
        e.stopPropagation();

        if (!this.selectedObject) return;

        const handle = e.target;
        const element = this.selectedObject;
        const handlePosition = handle.dataset.handle;

        // Get initial element dimensions and position
        const rect = element.getBoundingClientRect();
        const startX = e.clientX;
        const startY = e.clientY;
        const startWidth = rect.width;
        const startHeight = rect.height;
        const startLeft = parseInt(element.style.left) || 0;
        const startTop = parseInt(element.style.top) || 0;

        // Store the object name for property updates
        const objectName = element.dataset.name;

        // Function to handle mouse movement during resize
        const handleResize = (moveEvent) => {
            moveEvent.preventDefault();

            // Calculate the delta movement
            const deltaX = moveEvent.clientX - startX;
            const deltaY = moveEvent.clientY - startY;

            // Apply changes based on which handle was grabbed
            let newWidth = startWidth;
            let newHeight = startHeight;
            let newLeft = startLeft;
            let newTop = startTop;

            // For player objects, always maintain aspect ratio regardless of Shift key
            const isPlayerObject = element.dataset.type === 'player';
            const maintainAspectRatio = moveEvent.shiftKey || isPlayerObject;
            const aspectRatio = startWidth / startHeight;

            // Calculate the center point of the original element
            const centerX = startLeft + startWidth / 2;
            const centerY = startTop + startHeight / 2;

            // Calculate minimum size based on gridSize
            const gridSize = (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.gridSize) || 0;
            const minSize = gridSize > 0 ? Math.max(20, gridSize) : 20;

            // Handle different resize directions
            if (handlePosition.includes('e')) {
                newWidth = Math.max(startWidth + deltaX, minSize); // Minimum width respects gridSize

                // Maintain aspect ratio if Shift is pressed
                if (maintainAspectRatio) {
                    newHeight = newWidth / aspectRatio;

                    // Adjust position to keep center point fixed
                    newLeft = centerX - newWidth / 2;
                    newTop = centerY - newHeight / 2;
                }
            }
            if (handlePosition.includes('w')) {
                const widthChange = Math.min(deltaX, startWidth - minSize); // Limit to maintain minimum width
                newWidth = startWidth - widthChange;
                newLeft = startLeft + widthChange;

                // Maintain aspect ratio if Shift is pressed
                if (maintainAspectRatio) {
                    newHeight = newWidth / aspectRatio;

                    // Adjust position to keep center point fixed
                    newLeft = centerX - newWidth / 2;
                    newTop = centerY - newHeight / 2;
                }
            }
            if (handlePosition.includes('s')) {
                newHeight = Math.max(startHeight + deltaY, minSize); // Minimum height respects gridSize

                // Maintain aspect ratio if Shift is pressed
                if (maintainAspectRatio) {
                    newWidth = newHeight * aspectRatio;

                    // Adjust position to keep center point fixed
                    newLeft = centerX - newWidth / 2;
                    newTop = centerY - newHeight / 2;
                }
            }
            if (handlePosition.includes('n')) {
                const heightChange = Math.min(deltaY, startHeight - minSize); // Limit to maintain minimum height
                newHeight = startHeight - heightChange;
                newTop = startTop + heightChange;

                // Maintain aspect ratio if Shift is pressed
                if (maintainAspectRatio) {
                    newWidth = newHeight * aspectRatio;

                    // Adjust position to keep center point fixed
                    newLeft = centerX - newWidth / 2;
                    newTop = centerY - newHeight / 2;
                }
            }

            // For corner handles, we need special handling to maintain aspect ratio
            if (maintainAspectRatio &&
                (handlePosition === 'nw' || handlePosition === 'ne' ||
                 handlePosition === 'sw' || handlePosition === 'se')) {

                // For corner resizing, prioritize the larger change to determine the scaling
                const absX = Math.abs(deltaX);
                const absY = Math.abs(deltaY);

                if (absX > absY) {
                    // X movement is dominant, calculate height based on width
                    newHeight = newWidth / aspectRatio;
                } else {
                    // Y movement is dominant, calculate width based on height
                    newWidth = newHeight * aspectRatio;
                }

                // When shift is pressed, always maintain the center point
                newLeft = centerX - newWidth / 2;
                newTop = centerY - newHeight / 2;
            }

            // Apply grid snapping and minimum size constraints if gridSize is set
            if (window.webTalkApp && window.webTalkApp.interpreter && window.webTalkApp.interpreter.gridSize > 0) {
                const gridSize = window.webTalkApp.interpreter.gridSize;
                
                // Define snapToGrid function for resize scope
                const snapToGrid = (value, gridSize) => {
                    if (gridSize <= 0) return value;
                    return Math.round(value / gridSize) * gridSize;
                };
                
                // Ensure minimum size is at least gridSize
                const minSize = Math.max(20, gridSize); // Keep 20px absolute minimum, but use gridSize if larger
                
                // Apply grid snapping to dimensions, but enforce minimum size
                newWidth = Math.max(snapToGrid(newWidth, gridSize), minSize);
                newHeight = Math.max(snapToGrid(newHeight, gridSize), minSize);
                
                // Apply grid snapping to position
                newLeft = snapToGrid(newLeft, gridSize);
                newTop = snapToGrid(newTop, gridSize);
            }

            // Apply the new dimensions and position
            element.style.width = `${newWidth}px`;
            element.style.height = `${newHeight}px`;
            element.style.left = `${newLeft}px`;
            element.style.top = `${newTop}px`;
        };

        // Function to handle mouse up event
const handleMouseUp = () => {
    document.removeEventListener('mousemove', handleResize);
    document.removeEventListener('mouseup', handleMouseUp);

    // Update the object properties in the interpreter
    if (objectName && window.webTalkApp && window.webTalkApp.interpreter) {
        const width = parseInt(element.style.width);
        const height = parseInt(element.style.height);
        const left = parseInt(element.style.left);
        const top = parseInt(element.style.top);

        // Calculate the new location (center point)
        const locLeft = left + width / 2;
        const locTop = top + height / 2;

        // Update the object properties
        window.webTalkApp.interpreter.interpret(`set the width of ${element.dataset.type} "${objectName}" to ${width}`);
        window.webTalkApp.interpreter.interpret(`set the height of ${element.dataset.type} "${objectName}" to ${height}`);
        window.webTalkApp.interpreter.interpret(`set the loc of ${element.dataset.type} "${objectName}" to ${locLeft},${locTop}`);

        // Handle resizing of SVG shapes
        if (element.dataset.type === 'graphic') {
            const graphicType = element.dataset.graphicType;
            const svg = element.querySelector('svg');

            // Handle polygon-based shapes (triangle, pentagon, hexagon, octagon, etc.)
            if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(graphicType)) {
                const polygon = svg.querySelector('polygon');
                if (polygon) {
                    // Get the original points
                    const originalPoints = polygon.getAttribute('points');
                    const pointsArray = originalPoints.split(' ').map(point => {
                        const [x, y] = point.split(',').map(Number);
                        return { x, y };
                    });

                    // Calculate the bounding box of the original points
                    let minX = Infinity, minY = Infinity, maxX = -Infinity, maxY = -Infinity;
                    pointsArray.forEach(point => {
                        minX = Math.min(minX, point.x);
                        minY = Math.min(minY, point.y);
                        maxX = Math.max(maxX, point.x);
                        maxY = Math.max(maxY, point.y);
                    });

                    const originalWidth = maxX - minX;
                    const originalHeight = maxY - minY;

                    // Calculate scaling factors
                    const scaleX = width / originalWidth;
                    const scaleY = height / originalHeight;

                    // Scale and center the points
                    const scaledPoints = pointsArray.map(point => {
                        // Scale relative to the center
                        const scaledX = (point.x - minX) * scaleX;
                        const scaledY = (point.y - minY) * scaleY;
                        return `${scaledX},${scaledY}`;
                    }).join(' ');

                    // Update the polygon points
                    polygon.setAttribute('points', scaledPoints);
                }
            }
            // Handle circle shape
            else if (graphicType === 'circle') {
                const circle = svg.querySelector('circle');
                if (circle) {
                    // For circles, we need to maintain aspect ratio
                    const minDimension = Math.min(width, height);
                    const radius = (minDimension * 0.45); // 45% of the smaller dimension

                    circle.setAttribute('r', `${radius}%`);
                }
            }
            // Handle path-based graphics (including rounded rectangles)
            else if (graphicType === 'path') {
                const path = svg.querySelector('path');
                if (path && window.ObjectScaler) {
                    // Get the current path data
                    const currentPathData = path.getAttribute('d');

                    // Get or store original points for scaling
                    let originalPoints = WebTalkObjects.customProperties.get(objectName).get('originalpoints');
                    let originalWidth = WebTalkObjects.customProperties.get(objectName).get('originalwidth');
                    let originalHeight = WebTalkObjects.customProperties.get(objectName).get('originalheight');

                    if (!originalPoints) {
                        originalPoints = currentPathData;
                        originalWidth = width;
                        originalHeight = height;

                        WebTalkObjects.customProperties.get(objectName).set('originalpoints', originalPoints);
                        WebTalkObjects.customProperties.get(objectName).set('originalwidth', originalWidth);
                        WebTalkObjects.customProperties.get(objectName).set('originalheight', originalHeight);
                    }

                    // Scale the path using ObjectScaler
                    const scaledPathData = window.ObjectScaler.scalePath(
                        originalPoints,
                        width,
                        height,
                        originalWidth,
                        originalHeight
                    );

                    // Update the path
                    path.setAttribute('d', scaledPathData);

                    // Update the points property
                    WebTalkObjects.customProperties.get(objectName).set('points', scaledPathData);
                }
            }
            // Handle oval shape
            else if (graphicType === 'oval') {
                const ellipse = svg.querySelector('ellipse');
                if (ellipse) {
                    // Update rx and ry based on new dimensions
                    ellipse.setAttribute('rx', '45%');
                    ellipse.setAttribute('ry', '45%');
                }
            }
            // Handle rectangle or square
            else if (graphicType === 'rectangle' || graphicType === 'square') {
                const rect = svg.querySelector('rect');
                if (rect) {
                    // Update width and height to maintain proportions
                    rect.setAttribute('width', '90%');
                    rect.setAttribute('height', '90%');
                }
            }
        }
    }
};


// Add event listeners for mousemove and mouseup
        document.addEventListener('mousemove', handleResize);
        document.addEventListener('mouseup', handleMouseUp);
    }

    static editScript(objectName, errorLine = null) {
        const scriptEditor = document.getElementById('scriptEditor');
        if (!scriptEditor) return;

        // Set the current object being edited
        scriptEditor.dataset.currentObject = objectName;

        // Load the script
        const script = this.scripts.get(objectName) || '';
        document.getElementById('scriptContent').value = script;

        // Show the script editor
        scriptEditor.style.display = 'block';
        
        // If an error line is specified, highlight it
        if (errorLine !== null && errorLine > 0) {
            // This method doesn't have direct access to the highlight layer
            // So we'll need to update openScriptEditor to handle this case
        }
    }

    static getCard() {
        // Get the current card ID from the interpreter or default to 1
        const currentCardId = window.currentCardId || 1;
        
        // Get the appropriate card element based on the current card ID
        let card;
        if (currentCardId === 1) {
            card = document.getElementById('card');
            // Ensure the card has an ID attribute set to 'card'
            if (card && !card.id) {
                card.id = 'card';
            }
        } else {
            card = document.getElementById(`card-${currentCardId}`);
        }
        
        return card;
    }

    static getCardById(cardId) {
        // Get the appropriate card element based on the card ID
        let card;
        if (cardId === 1) {
            card = document.getElementById('card');
            // Ensure the card has an ID attribute set to 'card'
            if (card && !card.id) {
                card.id = 'card';
            }
        } else {
            card = document.getElementById(`card-${cardId}`);
        }
        
        return card;
    }

    static getCorrectPropertyName(property) {
        // Ensure consistent property naming
        if (!property) return '';
        
        const lowerProperty = property.toLowerCase();
        
        // Map of standard property names with their correct casing
        const propertyMap = {
            'backgroundcolor': 'backgroundColor',
            'backgroundcolour': 'backgroundColor',
            'textcolor': 'textColor',
            'textcolour': 'textColor',
            'foregroundcolor': 'foregroundColor',
            'foregroundcolour': 'foregroundColor',
            'textalign': 'textAlign',
            'textstyle': 'textStyle',
            'fontsize': 'fontSize',
            'fontfamily': 'fontFamily',
            'lineheight': 'lineHeight',
            'borderstyle': 'borderStyle',
            'borderwidth': 'borderWidth',
            'bordercolor': 'borderColor',
            'bordercolour': 'borderColor'
        };
        
        return propertyMap[lowerProperty] || lowerProperty;
    }
    
    // Get the highest layer value among all objects
    static getHighestLayer() {
        let highest = 1; // Card is always layer 1
        
        // Check all objects
        for (const [name, element] of this.objects.entries()) {
            const zIndex = parseInt(element.style.zIndex) || 1;
            if (zIndex > highest) {
                highest = zIndex;
            }
        }
        
        return highest;
    }

    static duplicateObject(originalElement) {
        const objectType = originalElement.dataset.type; // Get the type, e.g., 'button'
        const objectName = originalElement.dataset.name; // Get the name of the right-clicked object
        if (!objectType) return; // Safeguard if type is missing
        
        // Build the duplication script using HyperTalk
        const duplicateScript = [
            `put "${objectName}" into myOldName`,
            `put myOldName & " copy " into myNewName`,
            `get the number of ${objectType}s of this card`,
            `put myNewName & it into myNewName`,
            `create ${objectType} myNewName`,
            `put the properties of ${objectType} myOldName into tCloneProps`,
            `put lineOffset("name",tCloneProps) into tSkipLine`,
            `put 0 into tClonePropertyLine`,
            `repeat the number of lines in tCloneProps`,
            `  add 1 to tClonePropertyLine`,
            `  if tClonePropertyLine is tSkipLine then`,
            `    -- skip name property`,
            `  else`,
            `    put word 1 of line tClonePropertyLine of tCloneProps into myPropertyName`,
            `    put the myPropertyName of ${objectType} myOldName into myPropertyValue`,
            `    if myPropertyName is not "id" and myPropertyName is not "type" and myPropertyName is not "visible" and myPropertyName is not "filename" then set the myPropertyName of ${objectType} myNewName to myPropertyValue`,
            `  end if`,
            `end repeat`,
            `put the type of control myOldName into myObjectType`,
            `if myObjectType contains "field" then`,
            `   put the text of field myOldName into field myNewName`,
            `end if`,
            `if myObjectType contains "image" then`,
            `   set the imagedata of image myNewName to the imagedata of image myOldName`,
            `end if`,
            `set the script of ${objectType} myNewName to the script of ${objectType} myOldName`,
            `put the right of ${objectType} myNewName into tRight`,
            `set the right of ${objectType} myNewName to tRight + 20`,
            `put the top of ${objectType} myNewName into tTop`,
            `set the top of ${objectType} myNewName to tTop + 20`
        ];
        
        // Run the script using the interpreter
        if (window.interpreter) {
            window.interpreter.runScript(duplicateScript.join('\n'));
        } else {
            console.error('Interpreter not available');
        }
    }
    
    static getCardProperty(property, cardId) {
        // If cardId is provided, get that specific card, otherwise get current card
        const card = cardId ? this.getCardById(cardId) : this.getCard();
        if (!card) {
            throw new Error('Card not found');
        }

        // Get the correct property name with consistent casing
        const propertyName = this.getCorrectPropertyName(property);
        
        // First check if it's a standard property
        switch (propertyName.toLowerCase()) {
            case 'id':
                return card.id || 'card';
            case 'name':
            case 'shortname':
                return card.dataset.name || card.id || 'card';
            case 'backgroundcolor':
            case 'backgroundcolour':
                return card.style.backgroundColor || '';
            case 'width':
                return card.offsetWidth;
            case 'height':
                return card.offsetHeight;
            case 'rect':
                return `0,0,${card.offsetWidth},${card.offsetHeight}`;
            case 'visible':
                return card.style.display !== 'none' ? 'true' : 'false';
            case 'layer':
                return card.style.zIndex || '0';
            case 'collisionrate':
                return this.collisionRate.toString();
            default:
                // Check custom properties
                const cardName = card.id === 'card' ? 'card 1' : `card ${card.id.replace('card-', '')}`;
                if (this.customProperties.has(cardName)) {
                    const customProps = this.customProperties.get(cardName);
                    if (customProps.has(propertyName)) {
                        return customProps.get(propertyName);
                    }
                }
                return '';
        }
    }

    // Physics loop methods for object movement and collision detection
    static startPhysicsLoop() {
        if (!this.isPhysicsLoopRunning) {
            console.log('Starting physics loop');
            this.isPhysicsLoopRunning = true;
            this.lastPhysicsFrameTime = performance.now();
            this.physicsLoop();
        }
    }

    static physicsLoop() {
        const currentTime = performance.now();
        const deltaTime = (currentTime - this.lastPhysicsFrameTime) / 1000; // Convert to seconds
        this.lastPhysicsFrameTime = currentTime;
        
        // Get the card dimensions for boundary checks
        const card = this.getCard();
        const cardRect = card ? card.getBoundingClientRect() : null;
        if (!cardRect) {
            console.error('Card element not found for boundary checks');
            return;
        }
        
        // Track objects that have active forces
        let hasActiveObjects = false;
        
        // Determine if we should check for collisions in this frame
        const shouldCheckCollisions = (currentTime - this.lastCollisionCheckTime) >= this.collisionRate;
        if (shouldCheckCollisions) {
            this.lastCollisionCheckTime = currentTime;
        }
        
        // Process all objects with active forces
        for (const [objectName, element] of this.objects.entries()) {
            // Skip if object doesn't have custom properties or no force applied
            if (!this.customProperties.has(objectName)) continue;
            
            const customProps = this.customProperties.get(objectName);
            let speed = customProps.get('force_speed');
            const direction = customProps.get('force_direction');
            const drag = customProps.get('force_drag');
            
            // Skip if no force or speed is zero
            if (speed === undefined || direction === undefined || speed <= 0) continue;
            
            // Apply drag if specified
            if (drag !== undefined && drag > 0) {
                // Calculate drag effect based on deltaTime
                // Higher drag values cause faster deceleration
                const dragFactor = drag / 100; // Convert to a 0-1 scale
                
                // Apply exponential deceleration (stronger at higher speeds, gentler at lower speeds)
                // Formula: speed * (1 - dragFactor)^deltaTime
                // This creates a more natural deceleration curve
                const decelerationFactor = Math.pow(1 - dragFactor, deltaTime * 2); // Multiply by 2 to make the effect more noticeable
                speed = speed * decelerationFactor;
                
                // If speed is very small (below threshold), set it to zero to prevent perpetual drift
                const SPEED_THRESHOLD = 1.5;
                if (speed < SPEED_THRESHOLD) {
                    speed = 0;
                }
                
                // Update the speed in custom properties
                customProps.set('force_speed', speed);
                
                // If speed reaches zero due to drag, stop the object
                if (speed <= 0) {
                    continue; // Skip to next object
                }
            }
            
            // Mark that we have at least one active object
            hasActiveObjects = true;
            
            // Get current position
            const left = parseInt(element.style.left) || 0;
            const top = parseInt(element.style.top) || 0;
            const width = parseInt(element.style.width) || 0;
            const height = parseInt(element.style.height) || 0;
            
            // Get stored direction vector components (if available) or calculate them
            let dirX = customProps.get('force_dir_x');
            let dirY = customProps.get('force_dir_y');
            
            // If direction vector components aren't stored, calculate them
            // This is a fallback for backward compatibility
            if (dirX === undefined || dirY === undefined) {
                // Direction: 0 = up, 90 = right, 180 = down, 270 = left
                const directionRadians = (direction * Math.PI) / 180;
                dirX = Math.sin(directionRadians);
                dirY = -Math.cos(directionRadians); // Negative because Y increases downward
                
                // Store the calculated direction vector for future frames
                customProps.set('force_dir_x', dirX);
                customProps.set('force_dir_y', dirY);
            }
            
            // Apply speed to direction vector (scaled for reasonable movement)
            const dx = speed * dirX * deltaTime * 60;
            const dy = speed * dirY * deltaTime * 60;
            
            // Update position
            let newLeft = left + dx;
            let newTop = top + dy;
            
            // Check for card boundary collisions before applying position
            let collisionWithCard = false;
            let collisionEdge = '';
            
            // Check left boundary
            if (newLeft < 0) {
                collisionWithCard = true;
                collisionEdge = 'left';
                newLeft = 0; // Stop at boundary
                console.log(`Object ${objectName} hit left boundary`);
            }
            // Check right boundary (subtract width to account for object size)
            else if (newLeft + width > cardRect.width) {
                collisionWithCard = true;
                collisionEdge = 'right';
                newLeft = cardRect.width - width; // Stop at boundary
                console.log(`Object ${objectName} hit right boundary`);
            }
            
            // Check top boundary
            if (newTop < 0) {
                collisionWithCard = true;
                collisionEdge = 'top';
                newTop = 0; // Stop at boundary
                console.log(`Object ${objectName} hit top boundary`);
            }
            // Check bottom boundary (subtract height to account for object size)
            else if (newTop + height > cardRect.height) {
                collisionWithCard = true;
                collisionEdge = 'bottom';
                newTop = cardRect.height - height; // Stop at boundary
                console.log(`Object ${objectName} hit bottom boundary`);
            }
            
            // Apply the new position
            element.style.left = `${newLeft}px`;
            element.style.top = `${newTop}px`;
            
            // Handle card boundary collision
            if (collisionWithCard && shouldCheckCollisions) {
                // Stop the object's movement to prevent endless loop
                customProps.set('force_speed', 0);
                customProps.set('force_direction', 0);
                
                // Send collision message to the card instead of the object
                this.sendCollisionMessageToCard(objectName);
            }
            
            // Check for collisions with other objects
            for (const [otherName, otherElement] of this.objects.entries()) {
                // Skip self-collision
                if (objectName === otherName) continue;
                
                // Get other object dimensions
                const otherLeft = parseInt(otherElement.style.left) || 0;
                const otherTop = parseInt(otherElement.style.top) || 0;
                const otherWidth = parseInt(otherElement.style.width) || 0;
                const otherHeight = parseInt(otherElement.style.height) || 0;
                
                // Simple bounding box collision detection
                if (shouldCheckCollisions && 
                    !(newLeft + width < otherLeft || 
                      newLeft > otherLeft + otherWidth || 
                      newTop + height < otherTop || 
                      newTop > otherTop + otherHeight)) {
                    
                    console.log(`Object ${objectName} collided with object ${otherName}`);
                    this.sendCollisionMessage(objectName, otherName);
                }
            }
        }
        
        // Continue the loop if there are objects with force applied
        if (hasActiveObjects) {
            requestAnimationFrame(() => this.physicsLoop());
        } else {
            console.log('Stopping physics loop - no objects with force');
            this.isPhysicsLoopRunning = false;
        }
    }

    static sendCollisionMessage(objectName, collidedWith) {
        // Get the object's script
        const script = this.scripts.get(objectName);
        console.log(`Checking collision handler for object ${objectName}`);
        
        if (!script) {
            console.log(`No script found for object ${objectName}`);
            return;
        }
        
        // Check if the script has an objectCollision handler
        const hasHandler = script.includes('on objectCollision') || script.includes('function objectCollision');
        console.log(`Object ${objectName} has objectCollision handler: ${hasHandler}`);
        
        if (hasHandler) {
            // Format the collided object parameter
            let param;
            // Get the type of the collided object
            const element = this.objects.get(collidedWith);
            if (element) {
                const type = element.dataset.type || 'object';
                param = `${type} with name ${collidedWith}`;
                console.log(`Collision with ${type} named ${collidedWith}`);
            } else {
                param = collidedWith;
                console.log(`Collision with unknown object ${collidedWith}`);
            }
            
            // Send the objectCollision message to the interpreter
            try {
                // Check if the interpreter is available
                if (window.webTalkApp && window.webTalkApp.interpreter && 
                    typeof window.webTalkApp.interpreter.executeObjectScript === 'function') {
                    
                    console.log(`Executing objectCollision handler for ${objectName} with param: ${param}`);
                    // Execute the objectCollision handler directly
                    window.webTalkApp.interpreter.executeObjectScript(objectName, 'objectCollision', [param]);
                    console.log(`Handler execution initiated`);
                } else {
                    console.log(`Cannot execute handler: interpreter not available`);
                }
            } catch (e) {
                console.log(`Error executing objectCollision handler for ${objectName}:`, e);
                // We don't rethrow the error, as per requirements, we just continue
            }
        }
    }
    
    static sendCollisionMessageToCard(objectName) {
        // Get the card
        const card = this.getCard();
        if (!card) {
            console.error('Card element not found for sending collision message');
            return;
        }
        
        // The card's ID is always 'card' in the WebTalk system
        const cardId = 'card';
        console.log(`Sending collision message to card ${cardId}`);
        
        // Get the card's script
        const cardScript = this.scripts.get(cardId);
        if (!cardScript) {
            console.log(`No script found for card ${cardId}`);
            return;
        }
        
        // Check if the card's script has an objectCollision handler
        const hasHandler = cardScript.includes('on objectCollision') || cardScript.includes('function objectCollision');
        console.log(`Card ${cardId} has objectCollision handler: ${hasHandler}`);
        
        if (hasHandler) {
            try {
                // Check if the interpreter is available
                if (window.webTalkApp && window.webTalkApp.interpreter && 
                    typeof window.webTalkApp.interpreter.executeObjectScript === 'function') {
                    
                    console.log(`Executing objectCollision handler for card ${cardId} with object: ${objectName}`);
                    // Execute the objectCollision handler on the card, passing the object ID as parameter
                    window.webTalkApp.interpreter.executeObjectScript(cardId, 'objectCollision', [objectName]);
                    console.log(`Card handler execution initiated with object: ${objectName}`);
                    
                    // Also log to the console for debugging
                    console.log(`Collision with ${objectName}`);
                } else {
                    console.log(`Cannot execute handler: interpreter not available`);
                }
            } catch (e) {
                console.log(`Error executing objectCollision handler for card ${cardId}:`, e);
                // We don't rethrow the error, as per requirements, we just continue
            }
        }
    }

    static toggleFullscreen() {
        if (!document.fullscreenElement) {
            // Enter fullscreen mode
            document.documentElement.requestFullscreen().catch(err => {
                console.error('Error attempting to enable fullscreen:', err);
            });
        } else {
            // Exit fullscreen mode
            document.exitFullscreen().catch(err => {
                console.error('Error attempting to exit fullscreen:', err);
            });
        }
    }
}

// Initialize the card when the page loads
document.addEventListener('DOMContentLoaded', () => {
    WebTalkObjects.initializeCard();
});
