// Main application
// Global objects to track key states
// Initialize as empty objects
window.keysPressed = {};
window.keysReleased = {};
window.keyHeldIntervals = {}; // Store interval IDs for held keys
window.keyHistory = []; // Track the last 20 keys pressed
window.MAX_KEY_HISTORY = 20; // Maximum number of keys to track

// Reset key states when window loses focus to prevent stuck keys
window.addEventListener('blur', function() {
    window.keysPressed = {};
    window.keysReleased = {};
    
    // Clear all key held intervals
    Object.keys(window.keyHeldIntervals).forEach(key => {
        clearInterval(window.keyHeldIntervals[key]);
        delete window.keyHeldIntervals[key];
    });
    
    console.log('Window lost focus, reset key states');
});

// Clear keysReleased periodically to ensure they don't stay active too long
setInterval(function() {
    window.keysReleased = {};
}, 100); // Clear every 100ms

window.WebTalkApp = class WebTalkApp {
    constructor() {
        this.interpreter = new InterpreterClass();
        // Make interpreter globally accessible for file operations
        window.interpreter = this.interpreter;
        this.initializeEventListeners();
        // Set up output handler
        this.interpreter.outputHandler = (text) => {
            this.appendOutput(text);
        };
        // Enable sounds by default in the main interface
        this.interpreter.playSounds = true;
        console.log('WebTalkApp initialized');
        
        // Flag to track whether preloading is complete
        this.preloadingComplete = false;
        
        // Check if shell.js exists and set hasShell flag
        this.checkShellJsExists();
        
        // Preload extended functions to ensure they're available immediately
        this.preloadExtendedFunctions();
    }

    // This function will preload the extended functions by running a simple command
    // that uses them, ensuring they're available for subsequent user commands
    preloadExtendedFunctions() {
        try {
            // Run a simple command that uses extended functions
            // We'll use baseConvert since that's one we know had issues
            // We'll run it silently without showing output
            const originalOutputHandler = this.interpreter.outputHandler;
            this.interpreter.outputHandler = () => {}; // Silent output handler
            
            // Try to execute a few extended functions to ensure they're loaded
            setTimeout(async () => {
                try {
                    await this.interpreter.interpret('put baseConvert(10, 10, 16) into tTemp');
                    await this.interpreter.interpret('put format(123.45, "#.#") into tTemp');
                    await this.interpreter.interpret('put pi() into tTemp');
                    console.clear();
                    console.log('Extended functions preloaded successfully');
                    // Mark preloading as complete
                    this.preloadingComplete = true;
                    // Restore the original output handler
                    this.interpreter.outputHandler = originalOutputHandler;
                    
                    // Run an empty put command to initialize output handling
                    // This ensures subsequent commands work correctly
                    setTimeout(() => {
                        this.interpreter.interpret('put ""');
                    }, 100);
                } catch (error) {
                    console.warn('Extended functions preload attempt 1 failed:', error);
                    // Try again after a short delay
                    setTimeout(async () => {
                        try {
                            await this.interpreter.interpret('put baseConvert(10, 10, 16) into tTemp');
                            console.clear();
                            console.log('Extended functions preloaded successfully on second attempt');
                        } catch (error) {
                            console.warn('Extended functions preload attempt 2 failed:', error);
                        } finally {
                            // Restore the original output handler
                            this.interpreter.outputHandler = originalOutputHandler;
                            // Mark preloading as complete even if it failed
                            this.preloadingComplete = true;
                            
                            // Run an empty put command to initialize output handling
                            // This ensures subsequent commands work correctly
                            setTimeout(() => {
                                this.interpreter.interpret('put ""');
                            }, 100);
                        }
                    }, 500);
                }
            }, 100);
        } catch (error) {
            console.warn('Error in preloadExtendedFunctions:', error);
            // Mark preloading as complete even if it failed
            this.preloadingComplete = true;
        }
    }

    processLineContinuations(script) {
        // Handle line continuation character ¬
        // Split script into lines and process continuations
        const lines = script.split('\n');
        const processedLines = [];
        
        for (let i = 0; i < lines.length; i++) {
            let line = lines[i];
            
            // Check if this line ends with the continuation character ¬
            while (line.endsWith('¬') && i + 1 < lines.length) {
                // Remove the continuation character and any trailing whitespace
                line = line.slice(0, -1).trimEnd();
                
                // Get the next line and trim leading whitespace
                i++;
                const nextLine = lines[i].trimStart();
                
                // Concatenate the lines - preserve spacing but don't add extra spaces
                // The continuation should join exactly as written, preserving any return keywords
                if (line.length > 0 && nextLine.length > 0 && !line.endsWith(' ') && !nextLine.startsWith(' ')) {
                    line += ' ' + nextLine;
                } else {
                    line += nextLine;
                }
            }
            
            processedLines.push(line);
        }
        
        return processedLines.join('\n');
    }

    initializeEventListeners() {
        const messageInput = document.getElementById('message-input');
        const runButton = document.getElementById('run-command');
        const clearButton = document.getElementById('clear-output');
        const copyButton = document.getElementById('copy-output');
        const messageOutput = document.getElementById('message-output');
        const clearAfterRunCheckbox = document.getElementById('clear-after-run');
        const userMessages = document.getElementById('user-messages');
        const card = document.getElementById('card');

        if (messageInput && runButton && clearButton && copyButton && messageOutput) {
            // Run command on button click
            runButton.addEventListener('click', () => {
                this.runCurrentCommand();
            });

            // Clear output on clear button click
            clearButton.addEventListener('click', () => {
                messageOutput.innerHTML = '';
            });

            // Copy output to clipboard on copy button click
            copyButton.addEventListener('click', async () => {
                try {
                    const outputText = messageOutput.textContent || messageOutput.innerText || '';
                    await navigator.clipboard.writeText(outputText);
                    
                    // Provide visual feedback
                    const originalText = copyButton.textContent;
                    copyButton.textContent = 'Copied!';
                    copyButton.style.backgroundColor = '#4CAF50';
                    
                    setTimeout(() => {
                        copyButton.textContent = originalText;
                        copyButton.style.backgroundColor = '';
                    }, 1000);
                } catch (err) {
                    console.error('Failed to copy text: ', err);
                    
                    // Fallback for older browsers
                    const textArea = document.createElement('textarea');
                    textArea.value = messageOutput.textContent || messageOutput.innerText || '';
                    document.body.appendChild(textArea);
                    textArea.select();
                    try {
                        document.execCommand('copy');
                        
                        // Provide visual feedback
                        const originalText = copyButton.textContent;
                        copyButton.textContent = 'Copied!';
                        copyButton.style.backgroundColor = '#4CAF50';
                        
                        setTimeout(() => {
                            copyButton.textContent = originalText;
                            copyButton.style.backgroundColor = '';
                        }, 1000);
                    } catch (fallbackErr) {
                        console.error('Fallback copy failed: ', fallbackErr);
                        alert('Copy failed. Please select and copy the text manually.');
                    }
                    document.body.removeChild(textArea);
                }
            });
        }

        // Global keyboard shortcuts
        document.addEventListener('keydown', async (e) => {
            // Track key state for the keydown condition
            const keyLower = e.key.toLowerCase();
            window.keysPressed[keyLower] = true;
            
            // Add key to history array
            window.keyHistory.push(e.key);
            // Keep only the last MAX_KEY_HISTORY keys
            if (window.keyHistory.length > window.MAX_KEY_HISTORY) {
                window.keyHistory.shift(); // Remove oldest key
            }
            
            console.log('Key pressed:', keyLower, 'Current keys pressed:', Object.keys(window.keysPressed));
            
            // Set up keyheld event for this key if not already set
            if (!window.keyHeldIntervals[keyLower] && this.interpreter && this.interpreter.mode === 'browse') {
                const app = this; // Store reference to this for use in interval
                
                // Create an interval that fires the keyheld event continuously while the key is held
                window.keyHeldIntervals[keyLower] = setInterval(async function() {
                    // Only process if still in browse mode and key is still pressed
                    if (app.interpreter.mode === 'browse' && window.keysPressed[keyLower]) {
                        // Don't capture keys when in input fields
                        if (!['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                            // Send keyheld event to card
                            const cardName = WebTalkObjects.getCurrentCardName();
                            const cardResult = await app.interpreter.executeObjectScript(cardName, 'keyheld', [e.key]);
                            
                            // If the card didn't handle it, try the selected object
                            if (cardResult && cardResult.includes('not found')) {
                                const selectedObject = WebTalkObjects.selectedObject;
                                if (selectedObject) {
                                    await app.interpreter.executeObjectScript(selectedObject.dataset.name, 'keyheld', [e.key]);
                                }
                            }
                        }
                    } else {
                        // If key is no longer pressed or mode changed, clear the interval
                        clearInterval(window.keyHeldIntervals[keyLower]);
                        delete window.keyHeldIntervals[keyLower];
                    }
                }, 16); // ~60 times per second (1000ms / 60 ≈ 16.67ms)
            }
            
            // Toggle message box visibility with Ctrl+M
            if (e.key === 'm' && e.ctrlKey && !e.altKey && !e.shiftKey) {
                e.preventDefault();
                if (userMessages) {
                    userMessages.classList.toggle('hidden');
                    // Adjust card height when user messages are hidden
                    if (card) {
                        card.style.flex = userMessages.classList.contains('hidden') ? '1 0 auto' : '1';
                    }
                }
            }
            
            // Global Ctrl+Enter to run command
            if (e.key === 'Enter' && (e.ctrlKey || e.metaKey) && !e.altKey && !e.shiftKey) {
                e.preventDefault();
                this.runCurrentCommand();
            }
            
            // Handle keydown events for keydown message handler
            // This must come before the arrow key handler to ensure all keys are captured
            if (this.interpreter.mode === 'browse') {
                // Don't capture keys when in input fields
                if (!['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    // First try to send to the current card
                    const cardName = WebTalkObjects.getCurrentCardName();
                    const cardResult = await this.interpreter.executeObjectScript(cardName, 'keydown', [e.key]);
                    
                    // If the card didn't handle it (no handler found), try the currently selected object if any
                    if (cardResult && cardResult.includes('not found')) {
                        const selectedObject = WebTalkObjects.selectedObject;
                        if (selectedObject) {
                            await this.interpreter.executeObjectScript(selectedObject.dataset.name, 'keydown', [e.key]);
                        }
                    }
                    
                    // Note: We don't prevent default behavior here to allow normal keyboard functionality
                    // This is similar to how the arrowkey handler works - it doesn't block other keyboard events
                }
            }
            
            // Handle rawKeyDown events for rawKeyDown message handler
            // This provides the raw key code instead of the key value
            if (this.interpreter.mode === 'browse') {
                // Don't capture keys when in input fields
                if (!['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    // First try to send to the current card
                    const cardName = WebTalkObjects.getCurrentCardName();
                    const cardResult = await this.interpreter.executeObjectScript(cardName, 'rawKeyDown', [e.keyCode || e.which]);
                    
                    // If the card didn't handle it (no handler found), try the currently selected object if any
                    if (cardResult && cardResult.includes('not found')) {
                        const selectedObject = WebTalkObjects.selectedObject;
                        if (selectedObject) {
                            await this.interpreter.executeObjectScript(selectedObject.dataset.name, 'rawKeyDown', [e.keyCode || e.which]);
                        }
                    }
                }
            }
            
            // Handle arrow keys for both object movement and arrowkey message handler
            if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'].includes(e.key) && !e.ctrlKey && !e.altKey) {
                // Don't capture arrow keys when in input fields
                if (['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    return; // Allow default behavior in input fields
                }
                
                // Map arrow keys to HyperTalk direction names
                let direction = '';
                switch(e.key) {
                    case 'ArrowUp': direction = 'up'; break;
                    case 'ArrowDown': direction = 'down'; break;
                    case 'ArrowLeft': direction = 'left'; break;
                    case 'ArrowRight': direction = 'right'; break;
                }
                
                // In browse mode, send arrowkey message to card and objects
                if (this.interpreter.mode === 'browse') {
                    // Prevent default scrolling behavior
                    e.preventDefault();
                    
                    // First try to send to the current card
                    const cardName = WebTalkObjects.getCurrentCardName();
                    const cardResult = await this.interpreter.executeObjectScript(cardName, 'arrowkey', [direction]);
                    
                    // If the card didn't handle it (no handler found), try the currently selected object if any
                    if (cardResult && cardResult.includes('not found')) {
                        const selectedObject = WebTalkObjects.selectedObject;
                        if (selectedObject) {
                            await this.interpreter.executeObjectScript(selectedObject.dataset.name, 'arrowkey', [direction]);
                        }
                    }
                    
                    return; // Don't proceed with object movement in browse mode
                }
                
                // In edit mode, move the selected object
                if (this.interpreter.mode === 'edit') {
                    const selectedObject = WebTalkObjects.selectedObject;
                    if (selectedObject) {
                        e.preventDefault();
                        
                        // Get current position
                        let currentTop = parseInt(selectedObject.style.top) || 0;
                        let currentLeft = parseInt(selectedObject.style.left) || 0;
                        
                        // Determine movement amount (1px normally, 10px with Shift key)
                        const moveAmount = e.shiftKey ? 10 : 1;
                        
                        // Apply grid snapping logic for arrow keys
                        if (this.interpreter && this.interpreter.gridSize > 0) {
                            // For grid snapping, move to the next/previous grid position
                            const gridSize = this.interpreter.gridSize;
                            const currentGridTop = Math.round(currentTop / gridSize) * gridSize;
                            const currentGridLeft = Math.round(currentLeft / gridSize) * gridSize;
                            
                            switch(e.key) {
                                case 'ArrowUp':
                                    currentTop = currentGridTop - gridSize;
                                    break;
                                case 'ArrowDown':
                                    currentTop = currentGridTop + gridSize;
                                    break;
                                case 'ArrowLeft':
                                    currentLeft = currentGridLeft - gridSize;
                                    break;
                                case 'ArrowRight':
                                    currentLeft = currentGridLeft + gridSize;
                                    break;
                            }
                        } else {
                            // Normal movement without grid snapping
                            switch(e.key) {
                                case 'ArrowUp':
                                    currentTop -= moveAmount;
                                    break;
                                case 'ArrowDown':
                                    currentTop += moveAmount;
                                    break;
                                case 'ArrowLeft':
                                    currentLeft -= moveAmount;
                                    break;
                                case 'ArrowRight':
                                    currentLeft += moveAmount;
                                    break;
                            }
                        }
                        
                        // Update position
                        selectedObject.style.top = `${currentTop}px`;
                        selectedObject.style.left = `${currentLeft}px`;
                        
                        // Update inspector if it's open
                        if (typeof WebTalkInspector !== 'undefined' && WebTalkInspector.updateInspectorContent) {
                            const objectName = selectedObject.dataset.name;
                            WebTalkInspector.updateInspectorContent(objectName);
                        }
                        
                        console.log(`Moved ${selectedObject.dataset.name} to left: ${currentLeft}px, top: ${currentTop}px`);
                    }
                }
            }
            
            // Delete selected object with Delete or Backspace key
            if ((e.key === 'Delete' || e.key === 'Backspace') && !e.ctrlKey && !e.altKey && !e.shiftKey) {
                // Only delete objects if in edit mode and not in an input field
                if (this.interpreter.mode === 'edit' && 
                    !['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    const selectedObject = WebTalkObjects.selectedObject;
                    if (selectedObject) {
                        e.preventDefault();
                        const objectName = selectedObject.dataset.name;
                        console.log(`Deleting selected object: ${objectName}`);
                        
                        // Determine object type and call appropriate delete method
                        const objectType = selectedObject.dataset.type;
                        if (objectType === 'button') {
                            this.interpreter.deleteButton(objectName);
                        } else if (objectType === 'field') {
                            this.interpreter.deleteField(objectName);
                        } else if (objectType === 'graphic') {
                            this.interpreter.deleteGraphic(objectName);
                        } else if (objectType === 'image') {
                            this.interpreter.deleteImage(objectName);
                        } else if (objectType === 'player') {
                            this.interpreter.deletePlayer(objectName);
                        } else if (objectType === 'scrollbar') {
                            this.interpreter.deleteScrollbar(objectName);
                        }
                        
                        // Deselect after deletion
                        WebTalkObjects.deselectObject();
                        WebTalkObjects.selectedObject = null;
                    }
                }
            }
            
            // Open script editor for selected object with Ctrl+E
            if (e.key === 'e' && e.ctrlKey && !e.altKey && !e.shiftKey) {
                // Only open script editor if in edit mode and not in an input field
                console.log('Ctrl+E detected'); // debug log
                if (this.interpreter.mode === 'edit' && 
                    !['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    const selectedObject = WebTalkObjects.selectedObject;
                    if (selectedObject) {
                        e.preventDefault();
                        const objectName = selectedObject.dataset.name;
                        console.log(`Opening script editor for: ${objectName}`);
                        
                        // Open the script editor for the selected object
                        WebTalkObjects.openScriptEditor(objectName);
                    }
                }
            }
        });

        // Handle keyUp events
        document.addEventListener('keyup', async (e) => {
            // Track key states for both keydown and keyup conditions
            const keyLower = e.key.toLowerCase();
            delete window.keysPressed[keyLower];
            
            // Mark this key as recently released
            window.keysReleased[keyLower] = true;
            console.log('Key released:', keyLower, 'Current keys pressed:', Object.keys(window.keysPressed));
            
            // Clear any keyheld interval for this key
            if (window.keyHeldIntervals[keyLower]) {
                clearInterval(window.keyHeldIntervals[keyLower]);
                delete window.keyHeldIntervals[keyLower];
            }
            // Only process keyUp events in browse mode
            if (this.interpreter.mode === 'browse') {
                // Don't capture keys when in input fields
                if (!['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement.tagName)) {
                    // First try to send to the current card
                    const cardName = WebTalkObjects.getCurrentCardName();
                    
                    // Handle keyUp event
                    const cardResult = await this.interpreter.executeObjectScript(cardName, 'keyUp', [e.key]);
                    
                    // If the card didn't handle it (no handler found), try the currently selected object if any
                    if (cardResult && cardResult.includes('not found')) {
                        const selectedObject = WebTalkObjects.selectedObject;
                        if (selectedObject) {
                            await this.interpreter.executeObjectScript(selectedObject.dataset.name, 'keyUp', [e.key]);
                        }
                    }
                    
                    // Handle rawKeyUp event with the raw key code
                    const rawCardResult = await this.interpreter.executeObjectScript(cardName, 'rawKeyUp', [e.keyCode || e.which]);
                    
                    // If the card didn't handle it (no handler found), try the currently selected object if any
                    if (rawCardResult && rawCardResult.includes('not found')) {
                        const selectedObject = WebTalkObjects.selectedObject;
                        if (selectedObject) {
                            await this.interpreter.executeObjectScript(selectedObject.dataset.name, 'rawKeyUp', [e.keyCode || e.which]);
                        }
                    }
                    
                    // Note: We don't prevent default behavior here to allow normal keyboard functionality
                }
            }
        });

        // Toggle edit/browse mode with middle mouse button on card
        if (card) {
            card.addEventListener('mousedown', (e) => {
                // Check if it's a middle mouse button click (button 1)
                if (e.button === 1) {
                    e.preventDefault();
                    
                    // Toggle between edit and browse modes
                    const currentMode = this.interpreter.mode;
                    const newMode = currentMode === 'browse' ? 'edit' : 'browse';
                    
                    // Run the appropriate command
                    this.interpreter.interpret(`set the mode to ${newMode}`);

                    if (newMode === "browse") {WebTalkObjects.deselectObject();}
                    
                    // Prevent default middle-click behavior (usually paste or scroll)
                    return false;
                }
            });
        }
    }

    appendOutput(content, isError = false) {
        const messageOutput = document.getElementById('message-output');
        if (!messageOutput) return;

        // Create a new line element
        const line = document.createElement('div');
        line.className = 'output-line';
        
        if (isError) {
            line.innerHTML = `<span style="color: #ff6b6b">${content}</span>`;
        } else {
            line.textContent = content;
        }
        
        // Append the new line
        messageOutput.appendChild(line);
        
        // Scroll to bottom
        messageOutput.scrollTop = messageOutput.scrollHeight;
    }

    async runCurrentCommand() {
        const messageInput = document.getElementById('message-input');
        const messageOutput = document.getElementById('message-output');
        const clearAfterRunCheckbox = document.getElementById('clear-after-run');
        
        if (!messageInput || !messageOutput) {
            console.error('Required elements not found');
            return;
        }
        
        let command = messageInput.value.trim();
        if (!command) {
            console.log('Empty command, ignoring');
            return;
        }
        
        console.log('Running command:', command);
        
        // Wait for preloading to complete if it's still in progress
        if (!this.preloadingComplete) {
            this.appendOutput("Initializing interpreter, please wait...");
            // Wait for preloading to complete with a timeout
            await new Promise(resolve => {
                const checkPreloading = () => {
                    if (this.preloadingComplete) {
                        resolve();
                    } else {
                        setTimeout(checkPreloading, 100);
                    }
                };
                checkPreloading();
                // Add a timeout to prevent infinite waiting
                setTimeout(resolve, 3000);
            });
        }
        
        // Process line continuations before splitting into commands
        command = this.processLineContinuations(command);
        
        // Split the command into lines and run each line
        const lines = command.split('\n');
        for (const line of lines) {
            if (line.trim()) {
                try {
                    console.log('Executing line:', line);
                    const result = await this.interpreter.interpret(line);
                    console.log('Result:', result);
                    
                    // Check for error messages like "Field X not found" and display them
                    if (result && typeof result === 'string' && 
                        (result.includes(" not found") || result.includes("Error:") || result.includes("error:")))
                    {
                        // This is an error message, display it in the message box
                        this.appendOutput(`⚠️ ${result}`);
                    }
                    // Only append non-error results if it's not from a put command
                    // Put commands already output via the outputHandler
                    else if (result !== undefined && result !== null && !line.trim().toLowerCase().startsWith('put ')) {
                        this.appendOutput(result);
                    }
                } catch (error) {
                    console.error('Command error:', error);
                    this.appendOutput(`Error: ${error.message}`, true);
                    return; // Stop on first error
                }
            }
        }
        
        // Only clear input if the checkbox is checked
        if (clearAfterRunCheckbox && clearAfterRunCheckbox.checked) {
            messageInput.value = '';
        }
    }
}

// Initialize the application when the DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.webTalkApp = new WebTalkApp();
    
    // Set initial palette button state based on interpreter's default mode
    if (typeof updatePaletteMode === 'function' && window.webTalkApp.interpreter) {
        updatePaletteMode(window.webTalkApp.interpreter.mode);
    }
    
    // Initialize current card context - set card 1 as default
    const mainCard = document.getElementById('card');
    if (mainCard) {
        window.currentCard = mainCard;
        window.currentCardId = 1;
        mainCard.dataset.cardId = '1';
        mainCard.style.display = 'block'; // Ensure card 1 is visible by default
    }
    
    // Check for stack parameter in URL
    const loadStackFromUrlParam = async () => {
        try {
            console.log('loadStackFromUrlParam called, URL:', window.location.href);
            // Get the URL parameters
            const urlParams = new URLSearchParams(window.location.search);
            console.log('URL params:', Array.from(urlParams.entries()));
            
            // Look for a stack parameter
            // The format could be ?stack=path/to/file.json or ?=path/to/file.json
            let stackPath = urlParams.get('stack');
            
            // If no stack parameter with a key, check for parameter without a key
            if (!stackPath) {
                // This handles the case where the URL is like ?=path/to/file.json
                stackPath = urlParams.get('');
            }
            
            if (stackPath) {
                console.log(`Loading stack from URL parameter: ${stackPath}`);
                
                // Make sure we have a .json or .wts extension
                if (!stackPath.endsWith('.json') && !stackPath.endsWith('.wts')) {
                    stackPath += '.json';
                }
                
                try {
                    let stackData;
                    
                    // For local files, we need to use a file input approach since browsers
                    // block direct file access for security reasons
                    if (window.location.protocol === 'file:') {
                        // Show a user-friendly message for local file loading
                        const message = `To load a stack from a local file, please use one of these methods:
1. Use the "Load Stack" button in the interface
2. Drag and drop the JSON file onto the page
3. Serve the files through a local web server (e.g., python -m http.server)

Local file URLs like "file://..." cannot be loaded directly due to browser security restrictions.`;
                        
                        alert(message);
                        throw new Error('Local file loading via URL parameter is not supported due to browser security restrictions');
                    } else {
                        // For HTTP/HTTPS URLs, use fetch as before
                        const response = await fetch(stackPath);
                        
                        if (!response.ok) {
                            throw new Error(`Failed to fetch stack file: ${response.status} ${response.statusText}`);
                        }
                        
                        // Check if this is a compressed .wts file
                        const isCompressed = stackPath.endsWith('.wts');
                        
                        // Read the file data
                        let fileData;
                        if (isCompressed) {
                            console.log('Decompressing .wts file from URL...');
                            // For compressed files, read as ArrayBuffer then decompress
                            const compressedData = await response.arrayBuffer();
                            console.log('ArrayBuffer size:', compressedData.byteLength);
                            console.log('LZMA available:', typeof LZMA !== 'undefined');
                            fileData = await WebTalkFileOperations.decompressData(compressedData);
                            console.log('Decompression complete, data length:', fileData.length);
                        } else {
                            // For JSON files, read as text to handle multi-card stacks
                            fileData = await response.text();
                        }
                        
                        // Check if this is a multi-card stack
                        if (WebTalkFileOperations.isMultiCardStack(fileData)) {
                            console.log('Loading multi-card stack from URL');
                            
                            // Parse card sections
                            const cardSections = WebTalkFileOperations.parseCardSections(fileData);
                            console.log(`Found ${cardSections.length} card sections`);
                            
                            // Clear the current stack and snapshot storage
                            WebTalkFileOperations.clearCurrentStack();
                            WebTalkFileOperations.originalCardSnapshots.clear();
                            
                            // Load each card
                            for (const cardSection of cardSections) {
                                const cardId = cardSection.cardId;
                                stackData = JSON.parse(cardSection.content);
                                const cardElement = WebTalkFileOperations.ensureCardElement(cardId);
                                
                                // Store the original JSON for this card
                                WebTalkFileOperations.originalCardSnapshots.set(cardId, cardSection.content);
                                
                                // Set display style based on card ID
                                if (cardId === 1) {
                                    cardElement.style.display = 'block';
                                } else {
                                    cardElement.style.display = 'none';
                                }
                                
                                // Ensure the card element has the correct cardId set
                                cardElement.dataset.cardId = cardId.toString();
                                
                                // Load objects for this specific card
                                await WebTalkFileOperations.loadStackObjects(stackData, cardElement);
                                
                                // Find card data and apply properties
                                const expectedCardObjectId = cardId === 1 ? 'card' : `card-${cardId}`;
                                const cardData = stackData.objects.find(obj => obj.id === expectedCardObjectId);
                                
                                if (cardData) {
                                    // Apply card background color
                                    if (cardData.properties) {
                                        let bgColor = cardData.properties.backgroundColor || cardData.properties.backgroundcolor;
                                        
                                        if (bgColor) {
                                            // Convert comma-separated RGB values to proper CSS format if needed
                                            let cssColor = bgColor;
                                            if (bgColor.match(/^\d+\s*,\s*\d+\s*,\s*\d+$/)) {
                                                const [r, g, b] = bgColor.split(',').map(v => v.trim());
                                                cssColor = `rgb(${r}, ${g}, ${b})`;
                                            }
                                            
                                            cardElement.style.backgroundColor = cssColor;
                                            
                                            // Store in custom properties
                                            const cardName = cardId === 1 ? 'card' : `card-${cardId}`;
                                            if (!WebTalkObjects.customProperties.has(cardName)) {
                                                WebTalkObjects.customProperties.set(cardName, new Map());
                                            }
                                            WebTalkObjects.customProperties.get(cardName).set('backgroundColor', bgColor);
                                            WebTalkObjects.customProperties.get(cardName).set('backgroundcolor', bgColor);
                                            cardElement.dataset.backgroundColor = bgColor;
                                        }
                                    }
                                    
                                    // Set card name
                                    if (cardData.name) {
                                        cardElement.dataset.name = cardData.name;
                                        WebTalkObjects.objects.set(cardData.name, cardElement);
                                        WebTalkObjects.objectsById.set(cardId.toString(), cardData.name);
                                        
                                        if (cardId > 1) {
                                            WebTalkObjects.objects.set(`card-${cardId}`, cardElement);
                                        }
                                    }
                                }
                            }
                            
                            // Set the current card to card 1
                            WebTalkObjects.setCurrentCard(1);
                            
                            // Skip the loadStackFromData call since we've already loaded everything
                            stackData = null;
                        } else {
                            // Single-card stack - parse as JSON
                            stackData = JSON.parse(fileData);
                        }
                    }
                    
                    // Wait for the interpreter to be fully initialized
                    await new Promise(resolve => {
                        const checkPreloading = () => {
                            if (window.webTalkApp.preloadingComplete) {
                                resolve();
                            } else {
                                setTimeout(checkPreloading, 100);
                            }
                        };
                        checkPreloading();
                        // Add a timeout to prevent infinite waiting
                        setTimeout(resolve, 3000);
                    });
                    
                    // Load the stack data (only if not already loaded as multi-card)
                    if (stackData) {
                        await WebTalkFileOperations.loadStackFromData(stackData);
                    }
                    console.log('Stack loaded successfully from URL parameter');
                } catch (error) {
                    console.error('Error loading stack from URL parameter:', error);
                    // Optionally show an error message to the user
                    setTimeout(() => {
                        alert(`Failed to load stack from URL: ${error.message}`);
                    }, 500);
                }
            }
        } catch (error) {
            console.error('Error in loadStackFromUrlParam:', error);
        }
    };
    
    // Execute the URL parameter loading
    loadStackFromUrlParam();
});

// Check if shell.js exists and set hasShell flag
window.WebTalkApp.prototype.checkShellJsExists = function() {
    // Create a script element to attempt to load shell.js
    const script = document.createElement('script');
    script.src = 'js/shell.js';
    script.async = true;
    
    // Set default value to false
    this.interpreter.setVariable('hasShell', 'false');
    // Also set the property directly on the interpreter for "the hasShell" access
    this.interpreter.hasShell = false;
    
    // If script loads successfully, set hasShell to true
    script.onload = () => {
        this.interpreter.setVariable('hasShell', 'true');
        // Also set the property directly on the interpreter for "the hasShell" access
        this.interpreter.hasShell = true;
        console.log('shell.js detected and loaded');
    };
    
    // If script fails to load, hasShell remains false (fail silently)
    script.onerror = () => {
        // No console error - silent failure for times we aren't in a wrapper.
    };
    
    // Append script to document to attempt loading
    document.head.appendChild(script);
};

// Drag logic for floating palette with persistent position
const palette = document.getElementById('edit-palette');
const handle = document.getElementById('palette-drag-handle');
let isDragging = false, offsetX = 0, offsetY = 0;

// Restore palette position from localStorage if present
if (palette) {
    const saved = localStorage.getItem('palettePosition');
    if (saved) {
        try {
            const { left, top } = JSON.parse(saved);
            if (typeof left === 'number' && typeof top === 'number') {
                palette.style.left = left + 'px';
                palette.style.top = top + 'px';
            }
        } catch (e) { /* ignore */ }
    }
}

if (palette && handle) {
    handle.addEventListener('mousedown', function(e) {
        isDragging = true;
        offsetX = e.clientX - palette.offsetLeft;
        offsetY = e.clientY - palette.offsetTop;
        document.body.style.userSelect = 'none';
        handle.style.cursor = 'grabbing';
    });
    // Add touch event handler for touchstart
    handle.addEventListener('touchstart', function(e) {
        isDragging = true;
        const touch = e.touches[0];
        offsetX = touch.clientX - palette.offsetLeft;
        offsetY = touch.clientY - palette.offsetTop;
        document.body.style.userSelect = 'none';
        handle.style.cursor = 'grabbing';
        e.preventDefault(); // Prevent scrolling while dragging
    });
    document.addEventListener('mousemove', function(e) {
        if (!isDragging) return;
        const left = e.clientX - offsetX;
        const top = e.clientY - offsetY;
        palette.style.left = left + 'px';
        palette.style.top = top + 'px';
        localStorage.setItem('palettePosition', JSON.stringify({ left, top }));
        
        // Update stack container padding during drag
        if (window.updatePaletteStackPadding) {
            window.updatePaletteStackPadding();
        }
    });
    // Add touch event handler for touchmove
    document.addEventListener('touchmove', function(e) {
        if (!isDragging) return;
        const touch = e.touches[0];
        const left = touch.clientX - offsetX;
        const top = touch.clientY - offsetY;
        palette.style.left = left + 'px';
        palette.style.top = top + 'px';
        localStorage.setItem('palettePosition', JSON.stringify({ left, top }));
        
        // Update stack container padding during touch drag
        if (window.updatePaletteStackPadding) {
            window.updatePaletteStackPadding();
        }
        
        e.preventDefault(); // Prevent scrolling while dragging
    });
    document.addEventListener('mouseup', function() {
        if (isDragging) {
            isDragging = false;
            document.body.style.userSelect = '';
            handle.style.cursor = 'grab';
        }
    });
    // Add touch event handler for touchend
    document.addEventListener('touchend', function() {
        if (isDragging) {
            isDragging = false;
            document.body.style.userSelect = '';
            handle.style.cursor = 'grab';
        }
    });
}
