// inspector.js - Handles the object inspector window

class WebTalkInspector {
    static inspectorWindow = null;
    static isDragging = false;
    static offsetX = 0;
    static offsetY = 0;
    static currentObject = null;
    static refreshInterval = null;

    // Initialize the inspector window
    static initialize() {
        // Create inspector window if it doesn't exist
        if (!this.inspectorWindow) {
            this.createInspectorWindow();
        }

        // Add global shortcut for Ctrl+I
        document.addEventListener('keydown', (e) => {
            if (e.key === 'i' && e.ctrlKey && !e.altKey && !e.shiftKey) {
                e.preventDefault();
                this.toggleInspector();
            }
        });

        // Restore inspector position from localStorage if present
        const saved = localStorage.getItem('inspectorPosition');
        if (saved && this.inspectorWindow) {
            try {
                const { left, top } = JSON.parse(saved);
                if (typeof left === 'number' && typeof top === 'number') {
                    this.inspectorWindow.style.left = left + 'px';
                    this.inspectorWindow.style.top = top + 'px';
                }
            } catch (e) { /* ignore */ }
        }

        // Listen for object deletion events
        document.addEventListener('keydown', (e) => {
            if ((e.key === 'Delete' || e.key === 'Backspace') &&
                !e.ctrlKey && !e.altKey && !e.shiftKey &&
                this.currentObject &&
                WebTalkObjects.selectedObject === this.currentObject) {

                // The selected object is about to be deleted, close the inspector
                setTimeout(() => {
                    // Check if the object still exists after deletion attempt
                    const objectName = this.currentObject ? this.currentObject.dataset.name : null;
                    if (!objectName || !WebTalkObjects.objects.has(objectName)) {
                        // If no object is selected, show card properties
                        const card = WebTalkObjects.getCard();
                        if (card && this.inspectorWindow.style.display !== 'none') {
                            this.showInspector(card);
                        } else {
                            this.hideInspector();
                        }
                    }
                }, 100); // Small delay to allow deletion to complete
            }
        });

        // Listen for object modification events
        this.setupObjectModificationListeners();

        console.log('WebTalkInspector initialized');
    }

    // Set up listeners for object modifications
    static setupObjectModificationListeners() {
        // Monitor for changes in the DOM that might affect the selected object
        const observer = new MutationObserver((mutations) => {
            if (this.currentObject && this.inspectorWindow && this.inspectorWindow.style.display !== 'none') {
                const objectName = this.currentObject.dataset.name;
                if (objectName) {
                    // Check if any mutation affects our current object
                    for (const mutation of mutations) {
                        if (mutation.target === this.currentObject ||
                            this.currentObject.contains(mutation.target) ||
                            (mutation.type === 'attributes' && mutation.attributeName === 'style')) {
                            this.updateInspectorContent(objectName);
                            break;
                        }
                    }
                } else if (this.currentObject.id === 'card') {
                    // For card, update on any mutation
                    this.updateCardInspectorContent();
                }
            }
        });

        // Observe the entire card for changes
        const card = document.getElementById('card');
        if (card) {
            observer.observe(card, {
                attributes: true,
                childList: true,
                subtree: true,
                attributeFilter: ['style', 'class', 'data-name']
            });
        }
    }

    // Create the inspector window DOM elements
    static createInspectorWindow() {
        const inspector = document.createElement('div');
        inspector.id = 'object-inspector';
        inspector.className = 'object-inspector';

        // Create drag handle
        const dragHandle = document.createElement('div');
        dragHandle.id = 'inspector-drag-handle';
        dragHandle.className = 'inspector-drag-handle';
        dragHandle.innerHTML = '<span>&#8942;</span><span class="inspector-title">Object Inspector</span>';

        // Create close button
        const closeButton = document.createElement('button');
        closeButton.className = 'inspector-close-button';
        closeButton.innerHTML = '&times;';
        closeButton.title = 'Close Inspector';
        closeButton.addEventListener('click', () => this.hideInspector());

        // Add drag handle and close button to header
        const header = document.createElement('div');
        header.className = 'inspector-header';
        header.appendChild(dragHandle);
        header.appendChild(closeButton);

        // Create content area
        const content = document.createElement('div');
        content.id = 'inspector-content';
        content.className = 'inspector-content';

        // Add header and content to inspector
        inspector.appendChild(header);
        inspector.appendChild(content);

        // Add inspector to document
        document.body.appendChild(inspector);

        // Set initial state
        inspector.style.display = 'none';

        // Store reference to inspector window
        this.inspectorWindow = inspector;

        // Make inspector draggable
        this.makeInspectorDraggable(dragHandle);
    }

    // Make the inspector window draggable
    static makeInspectorDraggable(handle) {
        handle.addEventListener('mousedown', (e) => {
            if (e.target === handle || e.target.parentNode === handle) {
                this.isDragging = true;
                this.offsetX = e.clientX - this.inspectorWindow.offsetLeft;
                this.offsetY = e.clientY - this.inspectorWindow.offsetTop;
                document.body.style.userSelect = 'none';
                handle.style.cursor = 'grabbing';
            }
        });

        // Add touch event handler for touchstart
        handle.addEventListener('touchstart', (e) => {
            if (e.target === handle || e.target.parentNode === handle) {
                this.isDragging = true;
                const touch = e.touches[0];
                this.offsetX = touch.clientX - this.inspectorWindow.offsetLeft;
                this.offsetY = touch.clientY - this.inspectorWindow.offsetTop;
                document.body.style.userSelect = 'none';
                handle.style.cursor = 'grabbing';
                e.preventDefault(); // Prevent scrolling while dragging
            }
        });

        document.addEventListener('mousemove', (e) => {
            if (!this.isDragging) return;
            const left = e.clientX - this.offsetX;
            const top = e.clientY - this.offsetY;
            this.inspectorWindow.style.left = left + 'px';
            this.inspectorWindow.style.top = top + 'px';
            localStorage.setItem('inspectorPosition', JSON.stringify({ left, top }));
        });

        // Add touch event handler for touchmove
        document.addEventListener('touchmove', (e) => {
            if (!this.isDragging) return;
            const touch = e.touches[0];
            const left = touch.clientX - this.offsetX;
            const top = touch.clientY - this.offsetY;
            this.inspectorWindow.style.left = left + 'px';
            this.inspectorWindow.style.top = top + 'px';
            localStorage.setItem('inspectorPosition', JSON.stringify({ left, top }));
            e.preventDefault(); // Prevent scrolling while dragging
        });

        document.addEventListener('mouseup', () => {
            if (this.isDragging) {
                this.isDragging = false;
                document.body.style.userSelect = '';
                handle.style.cursor = 'grab';
            }
        });

        // Add touch event handler for touchend
        document.addEventListener('touchend', () => {
            if (this.isDragging) {
                this.isDragging = false;
                document.body.style.userSelect = '';
                handle.style.cursor = 'grab';
            }
        });
    }

    // Toggle inspector visibility
    static toggleInspector() {
        if (!this.inspectorWindow) {
            this.createInspectorWindow();
        }

        const selectedObject = WebTalkObjects.selectedObject;

        if (this.inspectorWindow.style.display === 'none') {
            if (selectedObject) {
                this.showInspector(selectedObject);
            } else {
                // If no object is selected, show card properties
                const card = WebTalkObjects.getCard();
                if (card) {
                    this.showInspector(card);
                } else {
                    if (window.webTalkApp && window.webTalkApp.interpreter) {
                        window.webTalkApp.interpreter.outputHandler('No card found.');
                    }
                }
            }
        } else {
            this.hideInspector();
        }
    }

    // Show inspector for a specific object
    static showInspector(element) {
        if (!this.inspectorWindow) {
            this.createInspectorWindow();
        }

        // Store reference to current object
        this.currentObject = element;

        // Update inspector title
        const title = this.inspectorWindow.querySelector('.inspector-title');
        if (title) {
            // Check if this is any card (either 'card' or 'card-X')
            if (element.id === 'card' || element.id.startsWith('card-')) {
                // For multi-card stacks, show the card number
                const cardId = element.id === 'card' ? 1 : parseInt(element.id.replace('card-', ''));
                title.textContent = `Card ${cardId} Inspector`;
                // Set data attributes for card
                this.inspectorWindow.dataset.objectType = 'card';
                this.inspectorWindow.dataset.objectId = element.id;
            } else {
                title.textContent = `Object Inspector: ${element.dataset.name || 'Unknown'}`;
                // Set data attributes for object
                this.inspectorWindow.dataset.objectType = element.dataset.type || 'button';
                this.inspectorWindow.dataset.objectId = element.id;
            }
        }

        // Update inspector content
        if (element.id === 'card' || element.id.startsWith('card-')) {
            this.updateCardInspectorContent();
        } else {
            const objectName = element.dataset.name;
            if (objectName) {
                this.updateInspectorContent(objectName);
            }
        }

        // Show inspector
        this.inspectorWindow.style.display = 'block';

        // Start periodic refresh
        this.startPeriodicRefresh();
    }

    // Hide inspector
    static hideInspector() {
        if (this.inspectorWindow) {
            this.inspectorWindow.style.display = 'none';
            this.currentObject = null;
            this.stopPeriodicRefresh();
        }
    }

    // Start periodic refresh of inspector content
    static startPeriodicRefresh() {
        // Clear any existing interval
        this.stopPeriodicRefresh();

        // Set up a new interval to check for object existence and refresh content
        this.refreshInterval = setInterval(() => {
            if (this.currentObject) {
                if (this.currentObject.id === 'card') {
                    // For card, just refresh content
                    this.updateCardInspectorContent();
                } else {
                    const objectName = this.currentObject.dataset.name;

                    // Check if object still exists
                    if (!objectName || !WebTalkObjects.objects.has(objectName)) {
                        console.log('Object no longer exists, showing card inspector');
                        // If no object is selected, show card properties
                        const card = WebTalkObjects.getCard();
                        if (card) {
                            this.showInspector(card);
                        } else {
                            this.hideInspector();
                        }
                        return;
                    }

                    // Refresh content
                    this.updateInspectorContent(objectName);
                }
            } else {
                // No current object, stop refreshing
                this.stopPeriodicRefresh();
            }
        }, 500); // Check every 500ms
    }

    // Stop periodic refresh
    static stopPeriodicRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    // Update inspector content for the card
    static updateCardInspectorContent() {
        const content = document.getElementById('inspector-content');
        if (!content) return;

        // Clear existing content
        content.innerHTML = '';

        // Create table for properties
        const table = document.createElement('table');
        table.className = 'inspector-table';

        // Add header row
        const headerRow = document.createElement('tr');
        const propertyHeader = document.createElement('th');
        propertyHeader.textContent = 'Property';
        const valueHeader = document.createElement('th');
        valueHeader.textContent = 'Value';
        headerRow.appendChild(propertyHeader);
        headerRow.appendChild(valueHeader);
        table.appendChild(headerRow);

        // Get the current card element based on which card is being inspected
        let card;
        if (this.currentObject && (this.currentObject.id === 'card' || this.currentObject.id.startsWith('card-'))) {
            card = this.currentObject; // Use the currently selected card
        } else {
            // Fallback to card 1 if no specific card is selected
            card = document.getElementById('card');
        }
        
        if (!card) {
            content.textContent = 'Card not found';
            return;
        }

        // Add basic properties
        // Add card name as the first property (editable)
        // Use dataset.name as the primary source of truth, falling back to other methods if not available
        const cardId = card.id === 'card' ? 1 : parseInt(card.id.replace('card-', ''));
        const cardName = card.dataset.name || card.dataset.cardName || (cardId === 1 ? 'card' : `card-${cardId}`);
        this.addPropertyRow(table, 'name', cardName, true); // Make name editable
        
        this.addPropertyRow(table, 'id', card.id);
        this.addPropertyRow(table, 'width', card.offsetWidth + 'px');
        this.addPropertyRow(table, 'height', card.offsetHeight + 'px');
        
        // Get backgroundColor from custom properties first, then fall back to computed style
        let backgroundColor = '';
        
        // Check custom properties first
        if (WebTalkObjects.customProperties.has(cardName)) {
            const props = WebTalkObjects.customProperties.get(cardName);
            backgroundColor = props.get('backgroundColor') || props.get('backgroundcolor') || '';
        }
        
        // If no custom property, get from computed style
        if (!backgroundColor) {
            const computedBg = getComputedStyle(card).backgroundColor;
            if (computedBg && computedBg !== 'rgba(0, 0, 0, 0)' && computedBg !== 'transparent') {
                backgroundColor = computedBg;
            }
        }
        
        this.addPropertyRow(table, 'backgroundColor', backgroundColor, true); // Make backgroundColor editable

        // Count objects on card
        const buttons = card.querySelectorAll('.button').length;
        const fields = card.querySelectorAll('.field').length;
        const graphics = card.querySelectorAll('.graphic').length;

        this.addPropertyRow(table, 'buttons', buttons);
        this.addPropertyRow(table, 'fields', fields);
        this.addPropertyRow(table, 'graphics', graphics);
        this.addPropertyRow(table, 'totalObjects', buttons + fields + graphics);

        // Get interpreter mode
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            this.addPropertyRow(table, 'mode', window.webTalkApp.interpreter.mode || 'unknown');
        }

        // Add table to content
        content.appendChild(table);

        // Add footer with button and timestamp
        const footer = document.createElement('div');
        footer.className = 'inspector-footer';
        footer.style.display = 'flex';
        footer.style.justifyContent = 'space-between';
        footer.style.alignItems = 'center';
        footer.style.marginTop = '10px';
        footer.style.paddingTop = '5px';
        footer.style.borderTop = '1px solid #444';

        // Add overview button on the left
        const overviewButton = document.createElement('button');
        overviewButton.textContent = 'Overview';
        overviewButton.className = 'inspector-overview-button';
        overviewButton.style.background = '#666';
        overviewButton.style.border = 'none';
        overviewButton.style.color = 'white';
        overviewButton.style.padding = '4px 8px';
        overviewButton.style.fontSize = '12px';
        overviewButton.style.borderRadius = '3px';
        overviewButton.style.cursor = 'pointer';
        overviewButton.title = 'Toggle Overview';
        
        overviewButton.addEventListener('click', () => {
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.interpret('toggle overview');
            }
        });

        // Add hover and active effects to match clear-button
        overviewButton.addEventListener('mouseenter', () => {
            overviewButton.style.background = '#777';
        });
        overviewButton.addEventListener('mouseleave', () => {
            overviewButton.style.background = '#666';
        });
        overviewButton.addEventListener('mousedown', () => {
            overviewButton.style.background = '#555';
        });
        overviewButton.addEventListener('mouseup', () => {
            overviewButton.style.background = '#777';
        });

        // Add cards button next to overview
        const cardsButton = document.createElement('button');
        cardsButton.textContent = 'Cards';
        cardsButton.className = 'inspector-cards-button';
        cardsButton.style.background = '#666';
        cardsButton.style.border = 'none';
        cardsButton.style.color = 'white';
        cardsButton.style.padding = '4px 8px';
        cardsButton.style.fontSize = '12px';
        cardsButton.style.borderRadius = '3px';
        cardsButton.style.cursor = 'pointer';
        cardsButton.style.marginLeft = '5px';
        cardsButton.title = 'Show Card Navigator';
        
        cardsButton.addEventListener('click', () => {
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.interpret('show cardnav');
            }
        });

        // Add hover and active effects
        cardsButton.addEventListener('mouseenter', () => {
            cardsButton.style.background = '#777';
        });
        cardsButton.addEventListener('mouseleave', () => {
            cardsButton.style.background = '#666';
        });
        cardsButton.addEventListener('mousedown', () => {
            cardsButton.style.background = '#555';
        });
        cardsButton.addEventListener('mouseup', () => {
            cardsButton.style.background = '#777';
        });

        // Add last updated timestamp on the right
        const timestamp = document.createElement('div');
        timestamp.className = 'inspector-timestamp';
        timestamp.style.fontSize = '11px';
        timestamp.style.color = '#888';
        timestamp.style.fontStyle = 'italic';
        timestamp.style.margin = '0';
        timestamp.style.padding = '0';
        timestamp.style.border = 'none';
        timestamp.textContent = `Last updated: ${new Date().toLocaleTimeString()}`;

        footer.appendChild(overviewButton);
        footer.appendChild(cardsButton);
        footer.appendChild(timestamp);
        content.appendChild(footer);
    }

    // Update inspector content with object properties
    static updateInspectorContent(objectName) {
        const content = document.getElementById('inspector-content');
        if (!content) return;

        // Clear existing content
        content.innerHTML = '';

        // Create table for properties
        const table = document.createElement('table');
        table.className = 'inspector-table';

        // Add header row
        const headerRow = document.createElement('tr');
        const propertyHeader = document.createElement('th');
        propertyHeader.textContent = 'Property';
        const valueHeader = document.createElement('th');
        valueHeader.textContent = 'Value';
        headerRow.appendChild(propertyHeader);
        headerRow.appendChild(valueHeader);
        table.appendChild(headerRow);

        // Get object element
        const element = WebTalkObjects.objects.get(objectName);
        if (!element) {
            content.textContent = 'Object no longer exists';
            // If no object is selected, show card properties
            const card = WebTalkObjects.getCard();
            if (card) {
                this.showInspector(card);
            } else {
                this.hideInspector();
            }
            return;
        }

        // Update inspector data attributes
        if (this.inspectorWindow) {
            // Check if this is any card (either 'card' or 'card-X')
            if (element.id === 'card' || element.id.startsWith('card-')) {
                this.inspectorWindow.dataset.objectType = 'card';
                this.inspectorWindow.dataset.objectId = element.id;
            } else {
                this.inspectorWindow.dataset.objectType = element.dataset.type || 'button';
                this.inspectorWindow.dataset.objectId = element.id;
            }
        }

        // Add basic properties
        this.addPropertyRow(table, 'name', objectName);
        this.addPropertyRow(table, 'type', element.dataset.type);
        this.addPropertyRow(table, 'id', element.id);

        // Always add visible property
        const visibleValue = WebTalkObjects.getObjectProperty(element, 'visible');
        this.addPropertyRow(table, 'visible', visibleValue);

        // Add button-specific boolean properties
        if (element.dataset.type === 'button') {
            const hilightedValue = WebTalkObjects.getObjectProperty(element, 'hilighted');
            this.addPropertyRow(table, 'hilighted', hilightedValue);
            
            const disabledValue = WebTalkObjects.getObjectProperty(element, 'disabled');
            this.addPropertyRow(table, 'disabled', disabledValue);
        }

        // Add position and size properties
        this.addPropertyRow(table, 'left', element.style.left);
        this.addPropertyRow(table, 'top', element.style.top);
        this.addPropertyRow(table, 'width', element.style.width);
        this.addPropertyRow(table, 'height', element.style.height);

        // Add computed styles that might be relevant
        const computedStyle = window.getComputedStyle(element);
        this.addPropertyRow(table, 'backgroundColor', computedStyle.backgroundColor);
        this.addPropertyRow(table, 'foregroundColor', computedStyle.color);
        this.addPropertyRow(table, 'borderColor', computedStyle.borderColor);
        this.addPropertyRow(table, 'borderWidth', computedStyle.borderWidth);

        // Only add text-related properties for elements that can have text
        const textTypes = ['button', 'field', 'card', 'text'];
        if (textTypes.includes(element.dataset.type)) {
            this.addPropertyRow(table, 'fontFamily', computedStyle.fontFamily);
            this.addPropertyRow(table, 'fontSize', computedStyle.fontSize);
            this.addPropertyRow(table, 'textStyle', element.dataset.textStyle || 'plain');
        }

        // Add graphic-specific properties
        if (element.dataset.type === 'graphic') {
            // Get linesize from custom properties or default to 1
            const linesize = WebTalkObjects.getObjectProperty(element, 'linesize') || '1';
            this.addPropertyRow(table, 'linesize', linesize);
        }

        
        // Add custom properties based on object type
        const customProps = WebTalkObjects.customProperties.get(objectName);
        if (customProps) {
            // Define property sets for different object types
            const commonProps = ['visible', 'script', 'blendlevel'];
            const buttonProps = [...commonProps, 'visible', 'hilighted', 'disabled', 'isCheckbox', 'isMenu', 'angle', 'rotation', 'textAlign', 'textFont', 'textSize', 'textStyle', 'textColor', 'backgroundColor', 'borderColor', 'borderWidth', 'showName', 'foregroundColor'];
            const fieldProps = [...buttonProps, 'scrollable', 'editable', 'multiLine', 'autoTab', 'lockText', 'sharedText', 'showBorder', 'showFocusBorder', 'dontWrap', 'styledText'];
            const graphicProps = [...commonProps, 'angle', 'rotation', 'filled', 'linesize', 'pattern'];
            const lineProps = [...graphicProps, 'startArrow', 'endArrow', 'linesize', 'linePattern'];
            const imageProps = [...commonProps, 'angle', 'rotation', 'source'];
            const playerProps = [...commonProps, 'angle', 'rotation', 'filename'];
            const scrollbarProps = [...commonProps, 'thumbPosition', 'startValue', 'endValue', 'scrollType', 'progressColor', 'angle', 'rotation'];

            // Determine which property set to use based on object type
            let allowedProps = commonProps;
            switch(element.dataset.type) {
                case 'button':
                    allowedProps = buttonProps;
                    break;
                case 'field':
                    allowedProps = fieldProps;
                    break;
                case 'graphic':
                    // Further refine graphic types
                    if (element.classList.contains('line')) {
                        allowedProps = lineProps;
                    } else {
                        allowedProps = graphicProps;
                    }
                    break;
                case 'image':
                    allowedProps = imageProps;
                    break;
                case 'player':
                    allowedProps = playerProps;
                    break;
                case 'scrollbar':
                    allowedProps = scrollbarProps;
                    break;
            }

            // Add only the properties that apply to this object type
            customProps.forEach((value, key) => {
                // Skip internal properties, the 'points' property, formattingCommands, and properties already explicitly added
                const explicitlyAddedProps = ['visible', 'hilighted', 'disabled']; // Properties already added explicitly above
                if (!key.startsWith('_') &&
                    !key.startsWith('original') &&
                    key !== 'points' &&
                    key !== 'formattingCommands' &&
                    !explicitlyAddedProps.includes(key) &&
                    // Hide only the large data properties (data and backgroundPatternData)
                    !((element.dataset.type === 'image' || element.dataset.type === 'graphic' || element.id === 'card') && 
                      (key === 'data' || key === 'backgroundPatternData'))) {

                    // Always include basic properties (name, id, type)
                    const basicProps = ['name', 'id', 'type'];

                    // Check if this is a built-in property that should be filtered by object type
                    const builtInProps = [
                        'textAlign', 'textFont', 'textSize', 'textStyle', 'textColor',
                        'backgroundColor', 'borderColor', 'borderWidth', 'showName', 'foregroundColor',
                        'scrollable', 'editable', 'multiLine', 'autoTab', 'lockText', 'sharedText', 'showBorder', 'styledText',
                        'filled', 'linesize', 'pattern', 'startArrow', 'endArrow', 'linePattern',
                        'source', 'visible', 'script', 'angle', 'rotation', 'blendlevel',
                        'hilighted', 'highlighted', 'disabled', 'isCheckbox', 'isMenu'
                    ];

                    // If it's a built-in property, check if it's allowed for this object type
                    // If it's a custom property (not in builtInProps), always include it
                    if (basicProps.includes(key) ||
                        !builtInProps.includes(key) ||
                        allowedProps.includes(key)) {
                        this.addPropertyRow(table, key, value);
                    }
                }
            });
        }

        // Add table to content
        content.appendChild(table);

        // Add footer with button and timestamp
        const footer = document.createElement('div');
        footer.className = 'inspector-footer';
        footer.style.display = 'flex';
        footer.style.justifyContent = 'space-between';
        footer.style.alignItems = 'center';
        footer.style.marginTop = '10px';
        footer.style.paddingTop = '5px';
        footer.style.borderTop = '1px solid #444';

        // Add overview button on the left
        const overviewButton = document.createElement('button');
        overviewButton.textContent = 'Overview';
        overviewButton.className = 'inspector-overview-button';
        overviewButton.style.background = '#666';
        overviewButton.style.border = 'none';
        overviewButton.style.color = 'white';
        overviewButton.style.padding = '4px 8px';
        overviewButton.style.fontSize = '12px';
        overviewButton.style.borderRadius = '3px';
        overviewButton.style.cursor = 'pointer';
        overviewButton.title = 'Toggle Overview';
        
        overviewButton.addEventListener('click', () => {
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.interpret('toggle overview');
            }
        });

        // Add hover and active effects to match clear-button
        overviewButton.addEventListener('mouseenter', () => {
            overviewButton.style.background = '#777';
        });
        overviewButton.addEventListener('mouseleave', () => {
            overviewButton.style.background = '#666';
        });
        overviewButton.addEventListener('mousedown', () => {
            overviewButton.style.background = '#555';
        });
        overviewButton.addEventListener('mouseup', () => {
            overviewButton.style.background = '#777';
        });

        // Add cards button next to overview
        const cardsButton = document.createElement('button');
        cardsButton.textContent = 'Cards';
        cardsButton.className = 'inspector-cards-button';
        cardsButton.style.background = '#666';
        cardsButton.style.border = 'none';
        cardsButton.style.color = 'white';
        cardsButton.style.padding = '4px 8px';
        cardsButton.style.fontSize = '12px';
        cardsButton.style.borderRadius = '3px';
        cardsButton.style.cursor = 'pointer';
        cardsButton.style.marginLeft = '5px';
        cardsButton.title = 'Show Card Navigator';
        
        cardsButton.addEventListener('click', () => {
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                window.webTalkApp.interpreter.interpret('show cardnav');
            }
        });

        // Add hover and active effects
        cardsButton.addEventListener('mouseenter', () => {
            cardsButton.style.background = '#777';
        });
        cardsButton.addEventListener('mouseleave', () => {
            cardsButton.style.background = '#666';
        });
        cardsButton.addEventListener('mousedown', () => {
            cardsButton.style.background = '#555';
        });
        cardsButton.addEventListener('mouseup', () => {
            cardsButton.style.background = '#777';
        });

        // Add last updated timestamp on the right
        const timestamp = document.createElement('div');
        timestamp.className = 'inspector-timestamp';
        timestamp.style.fontSize = '11px';
        timestamp.style.color = '#888';
        timestamp.style.fontStyle = 'italic';
        timestamp.style.margin = '0';
        timestamp.style.padding = '0';
        timestamp.style.border = 'none';
        timestamp.textContent = `Last updated: ${new Date().toLocaleTimeString()}`;

        footer.appendChild(overviewButton);
        footer.appendChild(cardsButton);
        footer.appendChild(timestamp);
        content.appendChild(footer);
    }

    // Helper method to add a property row to the table
    static addPropertyRow(table, property, value, editable = false) {
        // Check if the property already exists in the table
        const existingRow = Array.from(table.querySelectorAll('tr')).find(row => {
            const nameCell = row.querySelector('.property-name');
            return nameCell && nameCell.textContent === property;
        });

        if (existingRow) {
            // Update existing row
            const valueCell = existingRow.querySelector('.property-value');
            if (valueCell) {
                // Special handling for boolean property checkboxes
                const booleanProps = ['visible', 'hilited', 'highlighted', 'ischeckbox', 'ismenu', 'disabled', 'scrollable', 'locktext', 'multiline', 'autotab', 'widemargins', 'sharedtext', 'dontwrap', 'showborder', 'hasslider', 'styledtext'];
                if (booleanProps.includes(property.toLowerCase())) {
                    const checkbox = valueCell.querySelector('input[type="checkbox"]');
                    if (checkbox) {
                        checkbox.checked = value.toString().toLowerCase() === 'true';
                        return;
                    }
                }
                
                // Format value based on type
                if (typeof value === 'object' && value !== null) {
                    valueCell.textContent = JSON.stringify(value);
                } else {
                    valueCell.textContent = value;
                }

                // Make property values clickable (except boolean properties which have checkboxes)
                const booleanProperties = ['visible', 'hilited', 'highlighted', 'ischeckbox', 'ismenu', 'disabled', 'scrollable', 'locktext', 'multiline', 'autotab', 'widemargins', 'sharedtext', 'dontwrap', 'showborder', 'hasslider', 'styledtext'];
                if (!booleanProperties.includes(property.toLowerCase())) {
                    this.makePropertyValueClickable(valueCell, property);
                }
            }
            return;
        }

        // Otherwise, create a new row
        const row = document.createElement('tr');

        const propertyCell = document.createElement('td');
        propertyCell.className = 'property-name';
        propertyCell.textContent = property;

        const valueCell = document.createElement('td');
        valueCell.className = 'property-value';

        // Special handling for boolean properties - create checkbox
        const booleanProps = ['visible', 'hilited', 'highlighted', 'ischeckbox', 'ismenu', 'disabled', 'scrollable', 'locktext', 'multiline', 'autotab', 'widemargins', 'sharedtext', 'dontwrap', 'showborder', 'hasslider', 'styledtext'];
        if (booleanProps.includes(property.toLowerCase())) {
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.checked = value.toString().toLowerCase() === 'true';
            checkbox.style.cursor = 'pointer';
            
            // Add event handler for checkbox toggle
            checkbox.addEventListener('change', () => {
                this.handleBooleanCheckboxToggle(property, checkbox.checked);
            });
            
            valueCell.appendChild(checkbox);
        } else {
            // Format value based on type
            if (typeof value === 'object' && value !== null) {
                valueCell.textContent = JSON.stringify(value);
            } else {
                valueCell.textContent = value;
            }

            // Make property values clickable (except boolean properties which have checkboxes)
            const booleanProperties = ['visible', 'hilited', 'highlighted', 'ischeckbox', 'ismenu', 'disabled', 'scrollable', 'locktext', 'multiline', 'autotab', 'widemargins', 'sharedtext', 'dontwrap', 'showborder', 'hasslider', 'styledtext'];
            if (!booleanProperties.includes(property.toLowerCase())) {
                this.makePropertyValueClickable(valueCell, property);
            }
        }

        row.appendChild(propertyCell);
        row.appendChild(valueCell);
        table.appendChild(row);
    }

    // Helper method to make property values clickable
    static makePropertyValueClickable(cell, property) {
        // Skip making certain properties clickable
        const nonClickableProps = ['id', 'type'];
        if (nonClickableProps.includes(property.toLowerCase())) {
            return;
        }

        // Add a visual indicator that it's clickable
        cell.style.cursor = 'pointer';

        // Style differently based on property type
        cell.style.color = '#aaddff';
        cell.style.textDecoration = 'none';


        // Add click event handler
        cell.addEventListener('click', () => {
            // Get the object type and ID from the inspector
            const inspector = document.querySelector('.object-inspector');
            if (!inspector) return;

            const objectType = inspector.dataset.objectType;
            const objectId = inspector.dataset.objectId;

            if (!objectType || !objectId) {
                console.error('Missing object type or ID in inspector');
                return;
            }

            // Get the current value
            const value = cell.textContent;

            // Pass to the InspectorTasks module to handle the click
            if (typeof InspectorTasks !== 'undefined' && InspectorTasks.handlePropertyClick) {
                InspectorTasks.handlePropertyClick(property, value, objectType, objectId);
            } else {
                console.error('InspectorTasks module not available');
            }
        });
    }

    // Handle checkbox toggle for boolean properties
    static handleBooleanCheckboxToggle(property, isChecked) {
        // Get the object type and ID from the inspector
        const inspector = document.querySelector('.object-inspector');
        if (!inspector) return;

        const objectType = inspector.dataset.objectType;
        const objectId = inspector.dataset.objectId;

        if (!objectType || !objectId) {
            console.error('Missing object type or ID in inspector');
            return;
        }

        // Convert checkbox state to string
        const propertyValue = isChecked ? 'true' : 'false';

        // Execute HyperTalk command to set the property
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            const command = `set the ${property} of ${objectType} id ${objectId} to ${propertyValue}`;
            console.log(`Executing ${property} command:`, command);
            window.webTalkApp.interpreter.interpret(command);
        }
    }

    // This method is kept for backward compatibility
    static makeNamePropertyClickable(cell) {
        this.makePropertyValueClickable(cell, 'name');
    }
}

// Initialize the inspector when the page loads
document.addEventListener('DOMContentLoaded', () => {
    WebTalkInspector.initialize();
});
