/**
 * HyperTalk Extender for Webtalk
 *
 * This module allows users to define extra HyperTalk scripts and call them
 * from HyperTalk scripts using the hte_ prefix.
 *
 * Example usage in a button script:
 *
 * on mouseUp
 *   get hte_myCommand
 *   put it
 * end mouseUp
 *
 * See line 28 in this file for the first sample script
 */

// Create a global HyperTalkExtender object
window.HyperTalkExtender = {
    // Store for extender scripts
    scripts: {},

    // Register a script with the extender
    registerScript(name, scriptLines) {
        this.scripts[name] = scriptLines;
    }
};

// myCommand sample script
HyperTalkExtender.registerScript('myCommand', [
    'get item 1 of the long date',
    'put it into tResult'
]);

// Add the hte_ function handler to the interpreter
(function() {
    // Get the original evaluateExpression method
    const originalEvaluateExpression = InterpreterClass.prototype.evaluateExpression;

    // Override the evaluateExpression method to handle hte_ function calls
    InterpreterClass.prototype.evaluateExpression = function(expr) {
        if (typeof expr === 'string' && expr.trim().startsWith('hte_')) {
            // Extract the function name (remove the hte_ prefix)
            const name = expr.trim().substring(4);

            // Check if the script exists
            if (!HyperTalkExtender.scripts[name]) {
                console.error(`HyperTalk extender script "${name}" not found`);
                return `Error: HyperTalk extender script "${name}" not found`;
            }

            // Get the script lines
            const scriptLines = HyperTalkExtender.scripts[name];

            // Save the current interpreter state
            const savedVariables = new Map();
            for (const [key, value] of this.variables.entries()) {
                savedVariables.set(key, value);
            }
            const savedIt = this.it;

            try {
                // Execute each line of the script
                for (const line of scriptLines) {
                    console.log(`Executing extender line: ${line}`);
                    this.interpret(line);
                }

                // Check for the special result variable 'tResult' first
                let result;
                if (this.variables.has('tResult')) {
                    result = this.variables.get('tResult');
                    console.log(`Using tResult: ${result}`);
                } else {
                    result = this.it;
                    console.log(`Using it: ${result}`);
                }

                return result;
            } catch (error) {
                console.error(`Error executing HyperTalk extender script "${name}":`, error);
                return `Error: ${error.message}`;
            } finally {
                // Restore the interpreter state, but keep the result
                this.variables = savedVariables;
                this.it = savedIt;
            }
        }

        // Call the original method for all other expressions
        return originalEvaluateExpression.call(this, expr);
    };

    // Add support for parameters in extender scripts
    InterpreterClass.prototype.param = function(index) {
        // Check if we have parameters in the current context
        if (this.currentParams && index <= this.currentParams.length) {
            return this.currentParams[index - 1]; // Convert to 0-based index
        }
        return '';
    };
})();

// Export the HyperTalkExtender
if (typeof module !== 'undefined' && module.exports) {
    module.exports = window.HyperTalkExtender;
}
