// file-operations.js - Handles saving and loading stacks

class WebTalkFileOperations {
    // RAM snapshot storage for loaded stack data
    static originalStackData = null; // Stores the original JSON data as loaded from file
    static originalCardSnapshots = new Map(); // Stores snapshots by card ID
    // Save the current stack to a file
    static async saveStackAs(filename) {
        try {
            // Check if filename is a variable name
            if (window.interpreter && window.interpreter.variables && window.interpreter.variables.has(filename)) {
                // Get the value from the variable
                const varValue = window.interpreter.variables.get(filename);
                if (varValue) {
                    filename = String(varValue);
                }
            }
            
            // Ensure filename has .wts extension (compressed format)
            if (!filename.endsWith('.wts') && !filename.endsWith('.json')) {
                filename += '.wts';
            }
            
            // Prepare the stack data
            const stackData = this.prepareStackData();
            
            // Convert to JSON - handle both string (multi-card) and object (single-card) formats
            let jsonData;
            if (typeof stackData === 'string') {
                // Multi-card format: stackData is already a formatted JSON string
                jsonData = stackData;
            } else {
                // Single-card format: stackData is an object that needs stringifying
                jsonData = JSON.stringify(stackData, null, 2);
            }
            
            // Check if we should use compression (for .wts files)
            const useCompression = filename.endsWith('.wts');
            
            // Save the file
            const success = await this.saveFile(filename, jsonData, useCompression);
            
            if (success) {
                console.log(`Stack saved as: ${filename}`);
                return true;
            } else {
                console.error(`Failed to save stack as: ${filename}`);
                return false;
            }
        } catch (error) {
            console.error('Error saving stack:', error);
            throw new Error(`Error saving stack: ${error.message}`);
        }
    }
    
    // Prepare stack data for saving, handling both single and multi-card stacks
    static prepareStackData() {
        // Get all cards in the stack
        const cards = WebTalkObjects.getCards();
        
        // For multi-card stacks, we need to create separate sections for each card
        if (cards.length > 1) {
            console.log(`Saving multi-card stack with ${cards.length} cards`);
            
            // Remember the current card ID to restore it later
            const originalCardId = window.currentCardId || 1;
            
            // Create first card data with only objects from the first card
            const firstCard = cards[0];
            const firstCardId = firstCard.id;
            window.currentCard = firstCard.element;
            window.currentCardId = firstCardId;
            
            // Prepare first card data with only its own objects
            const firstCardData = this.prepareCardDataForCard(firstCardId);
            
            // Remove duplicate card entries if they exist
            if (firstCardData.objects && Array.isArray(firstCardData.objects)) {
                // Find all card objects
                const cardObjects = firstCardData.objects.filter(obj => 
                    obj.type === 'card' || obj.id === 'card' || (obj.id && obj.id.startsWith('card-'))
                );
                
                // If we have more than one card object, keep only the first one
                if (cardObjects.length > 1) {
                    console.log(`Found ${cardObjects.length} card objects for card 1, removing duplicates`);
                    
                    // Keep only the first card object and remove others
                    const firstCardObject = cardObjects[0];
                    firstCardData.objects = firstCardData.objects.filter(obj => 
                        !(obj !== firstCardObject && (obj.type === 'card' || obj.id === 'card' || (obj.id && obj.id.startsWith('card-'))))
                    );
                }
            }
            
            let stackContent = JSON.stringify(firstCardData, null, 2);
            
            // Add markers and content for cards 2 and beyond
            for (let i = 1; i < cards.length; i++) {
                const card = cards[i];
                const cardId = card.id;
                
                // Set the current card for serialization purposes
                window.currentCard = card.element;
                window.currentCardId = cardId;
                
                // Prepare card data with only objects from this card
                const cardData = this.prepareCardDataForCard(cardId);
                const cardJson = JSON.stringify(cardData, null, 2);
                
                // Add markers for this additional card
                stackContent += `\n\n// CARD MARKER START: CARD ID [${cardId}]\n${cardJson}\n// CARD MARKER END: CARD ID [${cardId}]`;
            }
            
            // Restore the original current card
            WebTalkObjects.setCurrentCard(originalCardId);
            
            return stackContent;
        } else {
            // For single-card stacks, use the traditional method
            const singleCardData = this.prepareSingleCardData();
            return JSON.stringify(singleCardData, null, 2);
        }
    }
        
    // Prepare data for a specific card by collecting only objects that belong to that card
    static prepareCardDataForCard(cardId) {
        const stackData = {
            version: '1.0',
            timestamp: new Date().toISOString(),
            objects: []
        };
        
        // Get the card element
        const cardElementId = cardId === 1 ? 'card' : `card-${cardId}`;
        const cardElement = document.getElementById(cardElementId);
        if (!cardElement) {
            console.error(`Card element with ID ${cardElementId} not found`);
            return stackData;
        }
        
        // Add the card itself as the first object
        let cardName = cardElement.dataset.name || `card ${cardId}`;
        
        // Ensure card name is in proper format (not just a number)
        if (cardName.match(/^\d+$/)) {
            cardName = `card ${cardName}`;
            // Also fix it in the dataset for consistency
            cardElement.dataset.name = cardName;
        }
        
        stackData.objects.push({
            name: cardName,
            id: cardElementId,
            type: 'card', // Explicitly set the type property for card objects
            position: {
                left: '',
                top: '',
                width: '',
                height: ''
            },
            script: WebTalkObjects.scripts.get(cardElementId) || '',
            properties: {}
        });
        
        // Get custom properties for the card
        const cardProps = WebTalkObjects.customProperties.get(cardName);
        if (cardProps) {
            cardProps.forEach((value, key) => {
                // Skip backgroundColor if it's already set directly on the card element
                if (key === 'backgroundColor' && cardElement.style.backgroundColor) {
                    return;
                }
                stackData.objects[0].properties[key] = value;
            });
        }
        
        // Add card background color if set directly on the element
        if (cardElement.style.backgroundColor) {
            stackData.objects[0].properties.backgroundColor = cardElement.style.backgroundColor;
        }
        
        // Only include objects that belong to this card
        WebTalkObjects.objects.forEach((element, name) => {
            // Skip the card itself (already added) and objects that don't belong to this card
            if (name === cardName || element.closest(`#${cardElementId}`) !== cardElement) {
                return;
            }
            
            // Create object data
            const objectData = {
                name: name,
                id: element.id,
                type: element.dataset.type,
                position: {
                    left: element.style.left,
                    top: element.style.top,
                    width: element.style.width,
                    height: element.style.height
                },
                script: WebTalkObjects.scripts.get(name) || '',
                properties: {}
            };
            
            // Add custom properties
            if (WebTalkObjects.customProperties.has(name)) {
                const props = WebTalkObjects.customProperties.get(name);
                props.forEach((value, key) => {
                    objectData.properties[key] = value;
                });
            }
            
            // Handle image objects
            if (element.dataset.type === 'image' && element.dataset.imageData) {
                try {
                    const imageData = JSON.parse(element.dataset.imageData);
                    objectData.type = 'image';
                    objectData.imageData = imageData;
                } catch (e) {
                    console.error('Error parsing image data:', e);
                }
            }
            
            // For fields, save the content as plain text with \n line breaks
            if (element.dataset.type === 'field') {
                // Find the field-content div
                const fieldContent = element.querySelector('.field-content');
                if (fieldContent) {
                    // Use innerText to get plain text with \n line breaks
                    objectData.content = fieldContent.innerText;
                    
                    // Get formatting commands from customProperties if they exist
                    const customProps = WebTalkObjects.customProperties.get(name);
                    if (customProps && customProps.has('formattingCommands')) {
                        const formattingCommands = customProps.get('formattingCommands');
                        if (Array.isArray(formattingCommands) && formattingCommands.length > 0) {
                            objectData.formattingCommands = formattingCommands;
                        }
                    }
                }
            }
            
            stackData.objects.push(objectData);
        });
        
        return stackData;
    }
    
    static prepareSingleCardData() {
        const stackData = {
            version: '1.0',
            timestamp: new Date().toISOString(),
            objects: []
        };
        
        // Track which card element we've already added to avoid duplicates
        // For cards, we need to pick the best name (the one with the script)
        const cardElementMap = new Map(); // element -> {name, hasScript}
        
        // First pass: collect all card entries and check which has a script
        WebTalkObjects.objects.forEach((element, name) => {
            if (element.id === 'card' || element.id.startsWith('card-')) {
                const script = WebTalkObjects.scripts.get(name) || '';
                if (!cardElementMap.has(element) || script.length > 0) {
                    // Keep this entry if it's new or if it has a script
                    cardElementMap.set(element, { name, hasScript: script.length > 0 });
                }
            }
        });
        
        // Get all objects from WebTalkObjects
        WebTalkObjects.objects.forEach((element, name) => {
            // For card elements, only process the one we selected
            if (element.id === 'card' || element.id.startsWith('card-')) {
                const selectedEntry = cardElementMap.get(element);
                if (selectedEntry && selectedEntry.name !== name) {
                    console.log(`Skipping duplicate card entry for name: ${name}, using ${selectedEntry.name} instead`);
                    return; // Skip this duplicate
                }
            }
            
            // Preserve the original ID from the element
            const originalId = element.id;
            
            const objectData = {
                name: name,
                id: originalId,
                type: element.dataset.type,
                position: {
                    left: element.style.left,
                    top: element.style.top,
                    width: element.style.width,
                    height: element.style.height
                },
                script: WebTalkObjects.scripts.get(name) || '',
                properties: {}
            };
            
            // Get custom properties for this object
            const customProps = WebTalkObjects.customProperties.get(name);
            if (customProps) {
                // Convert the entire Map to a plain object
                customProps.forEach((value, key) => {
                    // Skip image-specific properties for image objects to avoid duplication
                    if (element.dataset.type === 'image' && (key === 'type' || key === 'data' || key === 'filename')) {
                        return;
                    }
                    
                    // Skip formattingCommands as they are saved separately at the top level
                    if (key === 'formattingCommands') {
                        return;
                    }
                    
                    // Save all custom properties automatically, preserving their original keys
                    // This ensures any new property added in the future gets saved without modification
                    objectData.properties[key] = value;
                });
            }
            
            // Also check for DOM properties that might not be in the customProperties map
            // This ensures we capture all properties set directly on the DOM element
            if (element.dataset.type === 'field' || element.dataset.type === 'button') {
                // Get computed style to check for applied properties
                const computedStyle = window.getComputedStyle(element);
                
                // Check for common style properties that might be set directly on the element
                const styleProps = [
                    { domProp: 'color', saveProp: 'foregroundColor' },
                    { domProp: 'backgroundColor', saveProp: 'backgroundColor' },
                    { domProp: 'borderColor', saveProp: 'borderColor' },
                    { domProp: 'borderWidth', saveProp: 'borderWidth' },
                    { domProp: 'textAlign', saveProp: 'textAlign' },
                    { domProp: 'fontFamily', saveProp: 'textFont' },
                    { domProp: 'fontSize', saveProp: 'textSize' },
                    { domProp: 'fontStyle', saveProp: 'textStyle' },
                    { domProp: 'fontWeight', saveProp: 'fontWeight' }
                ];
                
                // Save DOM properties that aren't already in the customProperties map
                styleProps.forEach(prop => {
                    const value = computedStyle[prop.domProp];
                    if (value && value !== 'initial' && value !== 'auto' && 
                        !objectData.properties[prop.saveProp]) {
                        objectData.properties[prop.saveProp] = value;
                    }
                });
            }
            
            // For fields, save the content as plain text with \n line breaks
            if (element.dataset.type === 'field') {
                // Find the field-content div
                const fieldContent = element.querySelector('.field-content');
                if (fieldContent) {
                    // Use innerText to get plain text with \n line breaks
                    objectData.content = fieldContent.innerText;
                    
                    // Get formatting commands from customProperties if they exist
                    const customProps = WebTalkObjects.customProperties.get(name);
                    if (customProps && customProps.has('formattingCommands')) {
                        const formattingCommands = customProps.get('formattingCommands');
                        if (Array.isArray(formattingCommands) && formattingCommands.length > 0) {
                            // Simply use the formatting commands as is
                            // Duplicates are now prevented in storeFormattingCommand
                            objectData.formattingCommands = formattingCommands;
                            console.log(`Saving ${formattingCommands.length} formatting commands for ${name}`);
                        }
                    }
                } else {
                    // Fallback to innerText if field-content div is not found
                    objectData.content = element.innerText;
                }
            }
            
            // For graphic objects
            if (element.dataset.type === 'graphic') {
                const graphicType = element.dataset.graphicType;
                if (graphicType === 'line') {
                    const svg = element.querySelector('svg');
                    const line = svg.querySelector('line');
                    
                    objectData.graphicType = 'line';
                    objectData.attributes = {
                        x1: parseInt(line.getAttribute('x1')),
                        y1: parseInt(line.getAttribute('y1')),
                        x2: parseInt(line.getAttribute('x2')),
                        y2: parseInt(line.getAttribute('y2')),
                        stroke: line.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(line.getAttribute('stroke-width')) || 1
                    };
                } else if (graphicType === 'oval') {
                    const svg = element.querySelector('svg');
                    const ellipse = svg.querySelector('ellipse');
                    
                    objectData.graphicType = 'oval';
                    objectData.attributes = {
                        cx: ellipse.getAttribute('cx'),
                        cy: ellipse.getAttribute('cy'),
                        rx: ellipse.getAttribute('rx'),
                        ry: ellipse.getAttribute('ry'),
                        fill: ellipse.getAttribute('fill') || 'none',
                        stroke: ellipse.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(ellipse.getAttribute('stroke-width')) || 2
                    };
                } else if (graphicType === 'circle') {
                    const svg = element.querySelector('svg');
                    const circle = svg.querySelector('circle');
                    
                    objectData.graphicType = 'circle';
                    objectData.attributes = {
                        cx: circle.getAttribute('cx'),
                        cy: circle.getAttribute('cy'),
                        r: circle.getAttribute('r'),
                        fill: circle.getAttribute('fill') || 'none',
                        stroke: circle.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(circle.getAttribute('stroke-width')) || 2
                    };
                } else if (graphicType === 'rectangle' || graphicType === 'square') {
                    const svg = element.querySelector('svg');
                    const rect = svg.querySelector('rect');
                    
                    objectData.graphicType = graphicType;
                    objectData.attributes = {
                        x: rect.getAttribute('x'),
                        y: rect.getAttribute('y'),
                        width: rect.getAttribute('width'),
                        height: rect.getAttribute('height'),
                        fill: rect.getAttribute('fill') || 'none',
                        stroke: rect.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(rect.getAttribute('stroke-width')) || 2
                    };
                } else if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(graphicType)) {
                    const svg = element.querySelector('svg');
                    const polygon = svg.querySelector('polygon');
                    
                    objectData.graphicType = graphicType;
                    objectData.attributes = {
                        points: polygon.getAttribute('points'),
                        fill: polygon.getAttribute('fill') || 'none',
                        stroke: polygon.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(polygon.getAttribute('stroke-width')) || 2
                    };
                }
            }
            
            // For image objects
            if (element.dataset.type === 'image') {
                // CRITICAL: Ensure the object type remains 'image' and doesn't get changed to 'image/png'
                objectData.type = 'image';
                
                // Create a separate imageData object (not inside properties)
                const originalType = customProps.get('type');
                
                // Check if we're loading an existing file with imageData already in properties
                // This handles backward compatibility with files that have flattened structure
                if (customProps.has('filename') && customProps.has('data')) {
                    // Move these properties from properties to imageData
                    objectData.imageData = {
                        filename: customProps.get('filename') || '',
                        type: originalType || 'image/png',
                        data: customProps.get('data') || ''
                    };
                    
                    // Remove from properties to avoid duplication
                    if (objectData.properties) {
                        delete objectData.properties.filename;
                        delete objectData.properties.type;
                        delete objectData.properties.data;
                    }
                } else {
                    // Normal case - create imageData from customProps
                    objectData.imageData = {
                        filename: customProps.get('filename') || '',
                        type: originalType || 'image/png',
                        data: customProps.get('data') || ''
                    };
                }
                
                // Ensure the type field in imageData is properly formatted for data URLs
                // BUT preserve SVG type if that's what it was originally
                if (objectData.imageData.type === 'image') {
                    // Before defaulting to PNG, check if the data is SVG content
                    if (objectData.imageData.data && typeof objectData.imageData.data === 'string') {
                        try {
                            // Try to detect SVG content in the data
                            const decodedData = atob(objectData.imageData.data);
                            if (decodedData.includes('<svg') || 
                                (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                objectData.imageData.type = 'image/svg+xml';
                            } else {
                                objectData.imageData.type = 'image/png';
                            }
                        } catch (e) {
                            // Not base64 data, default to PNG
                            objectData.imageData.type = 'image/png';
                        }
                    } else {
                        objectData.imageData.type = 'image/png';
                    }
                } else if (objectData.imageData.type && !objectData.imageData.type.includes('/')) {
                    // Special handling for SVG - preserve as image/svg+xml
                    if (objectData.imageData.type.toLowerCase() === 'svg') {
                        objectData.imageData.type = 'image/svg+xml';
                    } else {
                        objectData.imageData.type = `image/${objectData.imageData.type}`;
                    }
                }
                
                // Additional check: If data starts with SVG XML, ensure type is image/svg+xml
                if (objectData.imageData.data && 
                    (objectData.imageData.data.startsWith('<?xml') || 
                     objectData.imageData.data.startsWith('PD94bWw=') || // Base64 encoded <?xml
                     objectData.imageData.data.includes('<svg'))) {
                    objectData.imageData.type = 'image/svg+xml';
                }
            }
            
            // Handle soundData for any object type
            if (customProps && customProps.has('soundData')) {
                const soundBuffer = customProps.get('soundData');
                if (soundBuffer && soundBuffer.numberOfChannels) {
                    // Serialize AudioBuffer to a compressed format
                    const soundData = {
                        sampleRate: soundBuffer.sampleRate,
                        length: soundBuffer.length,
                        numberOfChannels: soundBuffer.numberOfChannels,
                        compressedChannels: []
                    };
                    
                    // Extract and compress channel data
                    for (let i = 0; i < soundBuffer.numberOfChannels; i++) {
                        const channelData = soundBuffer.getChannelData(i);
                        
                        // Convert Float32Array to Uint8Array for better compression
                        // Scale from [-1, 1] to [0, 255] range
                        const uint8Data = new Uint8Array(channelData.length);
                        for (let j = 0; j < channelData.length; j++) {
                            // Clamp to [-1, 1] range and convert to 8-bit
                            const clamped = Math.max(-1, Math.min(1, channelData[j]));
                            uint8Data[j] = Math.round((clamped + 1) * 127.5);
                        }
                        
                        // Convert to base64 for JSON storage
                        let binary = '';
                        for (let j = 0; j < uint8Data.length; j++) {
                            binary += String.fromCharCode(uint8Data[j]);
                        }
                        const base64Data = btoa(binary);
                        
                        soundData.compressedChannels.push(base64Data);
                    }
                    
                    objectData.soundData = soundData;
                    console.log(`Object ${name} soundData compressed:`, {
                        sampleRate: soundData.sampleRate,
                        length: soundData.length,
                        numberOfChannels: soundData.numberOfChannels,
                        compressionRatio: `${Math.round((soundData.compressedChannels[0].length / (soundData.length * 4)) * 100)}%`
                    });
                    
                    // Remove soundData from properties to avoid duplication
                    if (objectData.properties) {
                        delete objectData.properties.soundData;
                    }
                }
            }
            
            stackData.objects.push(objectData);
        });
        
        // Note: Card is already included in the objects array from the forEach loop above
        // No need to add it separately here
        
        return stackData;
    }
    
    // Prepare data for a specific card by collecting objects that belong to it
    // This is the original method that's still used for backward compatibility
    static prepareCardData(cardElement, cardId) {
        return this.prepareCardDataDirect(cardElement, cardId);
    }
    
    // Prepare data for a specific card by directly inspecting DOM and WebTalkObjects
    // This avoids navigating to the card and triggering opencard/closecard messages
    static prepareCardDataDirect(cardElement, cardId) {
        const stackData = {
            version: '1.0',
            timestamp: new Date().toISOString(),
            objects: []
        };
        
        // Get the card name from dataset or use default
        const cardName = cardElement.dataset.name || `card ${cardId}`;
        const cardElementId = cardId === 1 ? 'card' : `card-${cardId}`;
        
        // Add the card itself as the first object with explicit type
        const cardObject = {
            name: cardName,
            id: cardElementId,
            type: 'card',
            position: {
                left: '',
                top: '',
                width: '',
                height: ''
            },
            script: WebTalkObjects.scripts.get(cardElementId) || '',
            properties: {}
        };
        
        // Add custom properties for the card
        const cardProps = WebTalkObjects.customProperties.get(cardName);
        if (cardProps) {
            cardProps.forEach((value, key) => {
                // Skip certain properties that shouldn't be saved
                if (key !== 'formattingCommands' && key !== 'loadedFromJSON') {
                    cardObject.properties[key] = value;
                }
            });
        }
        
        // Add card background color if set
        if (cardElement.style.backgroundColor) {
            cardObject.properties.backgroundColor = cardElement.style.backgroundColor;
        }
        
        // Add the card object to the stack data
        stackData.objects.push(cardObject);
        
        // Get objects that belong to this specific card
        WebTalkObjects.objects.forEach((element, name) => {
            // Check if this object belongs to the specified card
            if (element.parentElement !== cardElement) {
                return; // Skip objects not on this card
            }
            
            // Use the same object serialization logic as prepareSingleCardData
            // Preserve the original ID from the element
            const originalId = element.id;
            
            const objectData = {
                name: name,
                id: originalId,
                type: element.dataset.type,
                position: {
                    left: element.style.left,
                    top: element.style.top,
                    width: element.style.width,
                    height: element.style.height
                },
                script: WebTalkObjects.scripts.get(name) || '',
                properties: {}
            };
            
            // Get custom properties for this object
            const customProps = WebTalkObjects.customProperties.get(name);
            if (customProps) {
                // Convert the entire Map to a plain object
                customProps.forEach((value, key) => {
                    // Skip image-specific properties for image objects to avoid duplication
                    if (element.dataset.type === 'image' && (key === 'type' || key === 'data' || key === 'filename')) {
                        return;
                    }
                    
                    // Skip formattingCommands as they are saved separately at the top level
                    if (key === 'formattingCommands') {
                        return;
                    }
                    
                    // Save all custom properties automatically, preserving their original keys
                    objectData.properties[key] = value;
                });
            }
            
            // Also check for DOM properties that might not be in the customProperties map
            // This ensures we capture all properties set directly on the DOM element
            if (element.dataset.type === 'field' || element.dataset.type === 'button') {
                // Get computed style to check for applied properties
                const computedStyle = window.getComputedStyle(element);
                
                // Check for common style properties that might be set directly on the element
                const styleProps = [
                    { domProp: 'color', saveProp: 'foregroundColor' },
                    { domProp: 'backgroundColor', saveProp: 'backgroundColor' },
                    { domProp: 'borderColor', saveProp: 'borderColor' },
                    { domProp: 'borderWidth', saveProp: 'borderWidth' },
                    { domProp: 'textAlign', saveProp: 'textAlign' },
                    { domProp: 'fontFamily', saveProp: 'textFont' },
                    { domProp: 'fontSize', saveProp: 'textSize' },
                    { domProp: 'fontStyle', saveProp: 'textStyle' },
                    { domProp: 'fontWeight', saveProp: 'fontWeight' }
                ];
                
                // Save DOM properties that aren't already in the customProperties map
                styleProps.forEach(prop => {
                    const value = computedStyle[prop.domProp];
                    if (value && value !== 'initial' && value !== 'auto' && 
                        !objectData.properties[prop.saveProp]) {
                        objectData.properties[prop.saveProp] = value;
                    }
                });
            }
            
            // Handle field content and formatting
            if (element.dataset.type === 'field') {
                const fieldContent = element.querySelector('.field-content');
                if (fieldContent) {
                    objectData.content = fieldContent.innerText;
                    
                    // Get formatting commands from customProperties if they exist
                    const customProps = WebTalkObjects.customProperties.get(name);
                    if (customProps && customProps.has('formattingCommands')) {
                        const formattingCommands = customProps.get('formattingCommands');
                        if (Array.isArray(formattingCommands) && formattingCommands.length > 0) {
                            objectData.formattingCommands = formattingCommands;
                            console.log(`Saving ${formattingCommands.length} formatting commands for ${name}`);
                        }
                    }
                } else {
                    objectData.content = element.innerText;
                }
            }
            
            // Handle graphic objects
            if (element.dataset.type === 'graphic') {
                const graphicType = element.dataset.graphicType;
                if (graphicType === 'line') {
                    const svg = element.querySelector('svg');
                    const line = svg.querySelector('line');
                    
                    objectData.graphicType = 'line';
                    objectData.attributes = {
                        x1: parseInt(line.getAttribute('x1')),
                        y1: parseInt(line.getAttribute('y1')),
                        x2: parseInt(line.getAttribute('x2')),
                        y2: parseInt(line.getAttribute('y2')),
                        stroke: line.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(line.getAttribute('stroke-width')) || 1
                    };
                } else if (graphicType === 'oval') {
                    const svg = element.querySelector('svg');
                    const ellipse = svg.querySelector('ellipse');
                    
                    objectData.graphicType = 'oval';
                    objectData.attributes = {
                        cx: ellipse.getAttribute('cx'),
                        cy: ellipse.getAttribute('cy'),
                        rx: ellipse.getAttribute('rx'),
                        ry: ellipse.getAttribute('ry'),
                        fill: ellipse.getAttribute('fill') || 'none',
                        stroke: ellipse.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(ellipse.getAttribute('stroke-width')) || 2
                    };
                } else if (graphicType === 'circle') {
                    const svg = element.querySelector('svg');
                    const circle = svg.querySelector('circle');
                    
                    objectData.graphicType = 'circle';
                    objectData.attributes = {
                        cx: circle.getAttribute('cx'),
                        cy: circle.getAttribute('cy'),
                        r: circle.getAttribute('r'),
                        fill: circle.getAttribute('fill') || 'none',
                        stroke: circle.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(circle.getAttribute('stroke-width')) || 2
                    };
                } else if (graphicType === 'rectangle' || graphicType === 'square') {
                    const svg = element.querySelector('svg');
                    const rect = svg.querySelector('rect');
                    
                    objectData.graphicType = graphicType;
                    objectData.attributes = {
                        x: rect.getAttribute('x'),
                        y: rect.getAttribute('y'),
                        width: rect.getAttribute('width'),
                        height: rect.getAttribute('height'),
                        fill: rect.getAttribute('fill') || 'none',
                        stroke: rect.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(rect.getAttribute('stroke-width')) || 2
                    };
                } else if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(graphicType)) {
                    const svg = element.querySelector('svg');
                    const polygon = svg.querySelector('polygon');
                    
                    objectData.graphicType = graphicType;
                    objectData.attributes = {
                        points: polygon.getAttribute('points'),
                        fill: polygon.getAttribute('fill') || 'none',
                        stroke: polygon.getAttribute('stroke') || 'black',
                        'stroke-width': parseInt(polygon.getAttribute('stroke-width')) || 2
                    };
                }
            }
            
            // Handle image objects
            if (element.dataset.type === 'image') {
                // CRITICAL: Ensure the object type remains 'image' and doesn't get changed to 'image/png'
                objectData.type = 'image';
                
                const img = element.querySelector('img');
                if (img && img.src) {
                    const customProps = WebTalkObjects.customProperties.get(name);
                    if (customProps) {
                        // Add console log to debug
                        console.log('Custom props for image in prepareCardData:', {
                            name: name,
                            type: customProps.get('type'),
                            dataLength: customProps.get('data') ? customProps.get('data').length : 0,
                            dataSample: customProps.get('data') ? customProps.get('data').substring(0, 50) + '...' : ''
                        });
                        
                        // Create a separate imageData object (not inside properties)
                        // Log the original type before any modifications
                        const originalType = customProps.get('type');
                        console.log(`Image ${name} original type from customProps: ${originalType}`);
                        
                        // Get the original type and ensure it's preserved correctly
                        let imageType = customProps.get('type') || 'image/png';
                        
                        // If the type is just 'image', convert it to 'image/png'
                        if (imageType === 'image') {
                            imageType = 'image/png';
                        }
                        
                        const imageData = {
                            filename: customProps.get('filename') || '',
                            type: imageType,
                            data: customProps.get('data') || ''
                        };
                        
                        console.log(`Image ${name} type after assignment: ${imageData.type}`);
                        
                        // Ensure the type field in imageData is properly formatted for data URLs
                        // BUT preserve SVG type if that's what it was originally
                        if (imageData.type === 'image') {
                            // Before defaulting to PNG, check if the data is SVG content
                            if (imageData.data && typeof imageData.data === 'string') {
                                try {
                                    // Try to detect SVG content in the data
                                    const decodedData = atob(imageData.data);
                                    if (decodedData.includes('<svg') || 
                                        (decodedData.includes('<?xml') && decodedData.includes('<svg'))) {
                                        imageData.type = 'image/svg+xml';
                                        console.log('SVG content detected and preserved in prepareCardData');
                                    } else {
                                        imageData.type = 'image/png';
                                    }
                                } catch (e) {
                                    // Not base64 data, default to PNG
                                    imageData.type = 'image/png';
                                }
                            } else {
                                imageData.type = 'image/png';
                            }
                        } else if (imageData.type && !imageData.type.includes('/')) {
                            // Special handling for SVG - preserve as image/svg+xml
                            if (imageData.type.toLowerCase() === 'svg') {
                                imageData.type = 'image/svg+xml';
                            } else {
                                imageData.type = `image/${imageData.type}`;
                            }
                        }
                        
                        // Additional check: If data starts with SVG XML, ensure type is image/svg+xml
                        if (imageData.data && 
                            (imageData.data.startsWith('<?xml') || 
                             imageData.data.startsWith('PD94bWw=') || // Base64 encoded <?xml
                             imageData.data.includes('<svg'))) {
                            imageData.type = 'image/svg+xml';
                        }
                        
                        objectData.imageData = imageData;
                        
                        // CRITICAL: Remove image-specific properties from the main properties object
                        // to prevent duplication and ensure they're only in the imageData object
                        if (objectData.properties) {
                            delete objectData.properties.filename;
                            delete objectData.properties.type;
                            delete objectData.properties.data;
                        }
                    }
                }
            }
            
            stackData.objects.push(objectData);
        });
        
        return stackData;
    }
    
    // Detect if a stack file contains multiple cards by checking for card markers
    static isMultiCardStack(fileContent) {
        if (typeof fileContent !== 'string') {
            return false;
        }
        
        // Look for card marker pattern starting from card 2
        const cardMarkerPattern = /\/\/ CARD MARKER START: CARD ID \[\d+\]/;
        return cardMarkerPattern.test(fileContent);
    }
    
    // Parse card sections from a multi-card stack file
    static parseCardSections(fileContent) {
        const cards = [];
        
        // Find the first card marker to determine where card 1 ends
        const firstMarkerMatch = fileContent.match(/\/\/ CARD MARKER START: CARD ID \[(\d+)\]/);
        
        if (!firstMarkerMatch) {
            // No markers found, treat as single card
            return [{ cardId: 1, content: fileContent }];
        }
        
        // Extract card 1 (everything before the first marker)
        const card1Content = fileContent.substring(0, firstMarkerMatch.index).trim();
        if (card1Content) {
            cards.push({ cardId: 1, content: card1Content });
        }
        
        // Extract remaining cards using markers
        const markerRegex = /\/\/ CARD MARKER START: CARD ID \[(\d+)\]([\s\S]*?)\/\/ CARD MARKER END: CARD ID \[\1\]/g;
        let match;
        
        while ((match = markerRegex.exec(fileContent)) !== null) {
            const cardId = parseInt(match[1]);
            const cardContent = match[2].trim();
            
            // Only add valid card content
            if (cardContent && cardId > 1) {
                cards.push({ cardId: cardId, content: cardContent });
            }
        }
        
        // Sort cards by ID to ensure they're processed in order
        cards.sort((a, b) => a.cardId - b.cardId);
        
        return cards;
    }
    
    // Create or ensure card DOM element exists for a specific card ID
    static ensureCardElement(cardId) {
        // Ensure cardId is treated as a number
        cardId = parseInt(cardId);
        
        console.log(`ensureCardElement called for cardId: ${cardId}`);
        
        // Generate the correct element ID based on card number
        const cardElementId = cardId === 1 ? 'card' : `card-${cardId}`;
        let cardElement = document.getElementById(cardElementId);
        
        if (!cardElement) {
            console.log(`Creating new card element with ID: ${cardElementId}`);
            
            // Create new card element
            cardElement = document.createElement('div');
            cardElement.id = cardElementId;
            cardElement.className = 'card';
            cardElement.dataset.cardId = cardId.toString();
            
            // Hide the new card initially - navigation will show it
            cardElement.style.display = 'none';
            
            // DO NOT set a default name here - it will be set later from JSON data if available
            
            // Add middle-mouse button event listener for mode switching
            cardElement.addEventListener('mousedown', (e) => {
                // Check if it's a middle mouse button click (button 1)
                if (e.button === 1) {
                    e.preventDefault();
                    
                    // Toggle between edit and browse modes
                    if (window.webTalkApp && window.webTalkApp.interpreter) {
                        const currentMode = window.webTalkApp.interpreter.mode;
                        const newMode = currentMode === 'browse' ? 'edit' : 'browse';
                        
                        // Run the appropriate command
                        window.webTalkApp.interpreter.interpret(`set the mode to ${newMode}`);

                        if (newMode === "browse") {WebTalkObjects.deselectObject();}
                        
                        // Prevent default middle-click behavior (usually paste or scroll)
                        return false;
                    }
                }
            });

            // Add context menu for script editing to the new card
            cardElement.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                e.stopPropagation();

                // Remove any existing context menus
                const existingMenu = document.querySelector('.context-menu');
                if (existingMenu) {
                    existingMenu.remove();
                }

                // Create context menu
                const contextMenu = document.createElement('div');
                contextMenu.className = 'context-menu';
                contextMenu.style.top = `${e.clientY}px`;
                contextMenu.style.left = `${e.clientX}px`;

                // Add edit script menu item
                const editScriptItem = document.createElement('div');
                editScriptItem.className = 'context-menu-item';
                editScriptItem.textContent = 'Edit Script';
                editScriptItem.addEventListener('click', () => {
                    // Get the card ID from the dataset
                    const cardId = parseInt(cardElement.dataset.cardId) || 1;
                    // Use the specific card ID for script editing
                    const cardScriptName = cardId === 1 ? 'card' : `card-${cardId}`;
                    WebTalkObjects.openScriptEditor(cardScriptName);
                    contextMenu.remove();
                });

                contextMenu.appendChild(editScriptItem);

                // Add context menu to document
                document.body.appendChild(contextMenu);

                // Close menu when clicking elsewhere
                const closeMenu = (e) => {
                    if (!contextMenu.contains(e.target)) {
                        contextMenu.remove();
                        document.removeEventListener('mousedown', closeMenu);
                    }
                };

                document.addEventListener('mousedown', closeMenu);
            });
            
            // Find the card container or create one
            const existingCard = document.getElementById('card');
            if (existingCard && cardId !== 1) {
                // Insert after the existing card
                existingCard.parentNode.insertBefore(cardElement, existingCard.nextSibling);
            } else if (cardId === 1) {
                // This is the main card, should already exist
                return existingCard;
            }
        } else {
            // Element exists, ensure its data-card-id attribute is set correctly
            cardElement.dataset.cardId = cardId.toString();
        }
        
        return cardElement;
    }
    
    // Load a text file and store content in textfileData variable
    static async loadTextFile() {
        try {
            const fileInput = document.createElement('input');
            fileInput.type = 'file';
            fileInput.accept = '.txt,text/plain';
            fileInput.style.display = 'none';
            
            return new Promise((resolve) => {
                fileInput.onchange = async (event) => {
                    const file = event.target.files[0];
                    if (file) {
                        try {
                            const content = await file.text();
                            // Store content in interpreter's textfileData variable
                            if (window.interpreter) {
                                window.interpreter.variables.set('textfileData', content);
                            }
                            resolve(file.name);
                        } catch (error) {
                            console.error('Error reading text file:', error);
                            if (window.interpreter && window.interpreter.createAnswerDialog) {
                                window.interpreter.createAnswerDialog(`Error reading file: ${error.message}`, ['OK']);
                            }
                            resolve('');
                        }
                    } else {
                        resolve('');
                    }
                    document.body.removeChild(fileInput);
                };
                
                fileInput.oncancel = () => {
                    resolve('');
                    document.body.removeChild(fileInput);
                };
                
                document.body.appendChild(fileInput);
                fileInput.click();
            });
        } catch (error) {
            console.error('Error creating file dialog:', error);
            if (window.interpreter && window.interpreter.createAnswerDialog) {
                window.interpreter.createAnswerDialog(`Error: ${error.message}`, ['OK']);
            }
            return '';
        }
    }
    
    // Save text content to a file
    static async saveTextFile(filename, content) {
        try {
            // Check if filename is a variable name
            if (window.interpreter && window.interpreter.variables && window.interpreter.variables.has(filename)) {
                const varValue = window.interpreter.variables.get(filename);
                if (varValue) {
                    filename = String(varValue);
                }
            }
            
            // Check if content is a variable name
            if (window.interpreter && window.interpreter.variables && window.interpreter.variables.has(content)) {
                const varValue = window.interpreter.variables.get(content);
                if (varValue !== undefined) {
                    content = String(varValue);
                }
            }
            
            // Ensure filename has .txt extension if not already present
            if (!filename.endsWith('.txt')) {
                filename += '.txt';
            }
            
            const success = await this.saveFile(filename, content);
            
            if (success) {
                console.log(`Text file saved as: ${filename}`);
                return true;
            } else {
                if (window.interpreter && window.interpreter.createAnswerDialog) {
                    window.interpreter.createAnswerDialog(`Failed to save text file: ${filename}`, ['OK']);
                }
                return false;
            }
        } catch (error) {
            console.error('Error saving text file:', error);
            if (window.interpreter && window.interpreter.createAnswerDialog) {
                window.interpreter.createAnswerDialog(`Error saving text file: ${error.message}`, ['OK']);
            }
            return false;
        }
    }
    
    // Compress data using LZMA
    static async compressData(data) {
        return new Promise((resolve, reject) => {
            try {
                // Check if we're running online (http/https)
                const isOnline = window.location.protocol === 'http:' || window.location.protocol === 'https:';
                
                if (isOnline) {
                    // Online: LZMA workers have compatibility issues, show answer dialog via HyperTalk
                    if (window.interpreter) {
                        window.interpreter.interpret('answer ".wts file compression is not supported when loading from a web server. Please save as .json instead." with "Cancel"');
                    }
                    // Don't reject to avoid double alert, just resolve with empty to cancel the operation
                    resolve(new Uint8Array());
                    return;
                }
                
                // Initialize LZMA if not already done
                if (typeof LZMA === 'undefined') {
                    reject(new Error('LZMA library not loaded'));
                    return;
                }
                
                // Offline (file://): Use LZMA with worker
                console.log('Using LZMA compression (local file mode)');
                const lzma = new LZMA('js/lzma_worker-min.js');
                
                // Convert string to byte array
                const encoder = new TextEncoder();
                const byteArray = encoder.encode(data);
                
                // Compress the data
                lzma.compress(Array.from(byteArray), 1, (result, error) => {
                    if (error) {
                        reject(new Error(`LZMA compression error: ${error}`));
                        return;
                    }
                    
                    // Convert result array to Uint8Array
                    const compressed = new Uint8Array(result);
                    resolve(compressed);
                });
            } catch (error) {
                reject(error);
            }
        });
    }
    
    // Decompress data using LZMA
    static async decompressData(compressedData) {
        return new Promise((resolve, reject) => {
            try {
                // Check if we're running online (http/https)
                const isOnline = window.location.protocol === 'http:' || window.location.protocol === 'https:';
                
                if (isOnline) {
                    // Online: LZMA workers have compatibility issues, show answer dialog via HyperTalk
                    if (window.interpreter) {
                        window.interpreter.interpret('answer ".wts files are not supported when loading from a web server. Please try using a .json file instead." with "Cancel"');
                    }
                    // Reject with a special error that can be caught silently
                    const error = new Error('WTS_NOT_SUPPORTED_ONLINE');
                    error.silent = true;
                    reject(error);
                    return;
                }
                
                // Initialize LZMA if not already done
                if (typeof LZMA === 'undefined') {
                    reject(new Error('LZMA library not loaded'));
                    return;
                }
                
                // Offline (file://): Use LZMA with worker
                console.log('Using LZMA decompression (local file mode)');
                const lzma = new LZMA('js/lzma_worker-min.js');
                
                // Convert Uint8Array to regular array for LZMA
                const compressedArray = Array.from(new Uint8Array(compressedData));
                console.log('Starting LZMA decompression, array length:', compressedArray.length);
                
                // Decompress the data
                lzma.decompress(compressedArray, (result, error) => {
                    console.log('LZMA decompress callback called, error:', error, 'result type:', typeof result);
                    if (error) {
                        console.error('LZMA decompression error:', error);
                        reject(new Error(`LZMA decompression error: ${error}`));
                        return;
                    }
                    
                    // Process the decompression result
                    let decompressed;
                    if (typeof result === 'string') {
                        // Result is already a string
                        decompressed = result;
                    } else if (Array.isArray(result) || result instanceof Uint8Array) {
                        // Result is an array of bytes, convert to string
                        const decoder = new TextDecoder();
                        decompressed = decoder.decode(new Uint8Array(result));
                    } else {
                        reject(new Error('Unexpected LZMA result type'));
                        return;
                    }
                    
                    // Successfully decompressed
                    resolve(decompressed);
                });
            } catch (error) {
                console.error('Exception in decompressData:', error);
                reject(error);
            }
        });
    }
    
    // Save a file using the Blob API and download attribute
    static async saveFile(filename, data, useCompression = false) {
        try {
            let blob;
            
            if (useCompression) {
                // Compress the data using LZMA
                console.log('Compressing data with LZMA...');
                const compressed = await this.compressData(data);
                blob = new Blob([compressed], { type: 'application/octet-stream' });
                console.log(`Compression complete. Original: ${data.length} bytes, Compressed: ${compressed.byteLength} bytes`);
            } else {
                // Create a Blob with the data as plain text
                blob = new Blob([data], { type: 'text/plain' });
            }
            
            // Create a URL for the Blob
            const url = URL.createObjectURL(blob);
            
            // Create a download link
            const a = document.createElement('a');
            a.href = url;
            a.download = filename;
            a.style.display = 'none';
            
            // Add to document, click, and remove
            document.body.appendChild(a);
            a.click();
            
            // Clean up
            setTimeout(() => {
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
            }, 100);
            
            return true;
        } catch (error) {
            console.error('Error in saveFile:', error);
            return false;
        }
    }
    
    // Load a stack from a file
    static async loadStack(filename) {
        try {
            // If filename is empty, we'll prompt for file selection via readFile
            // Otherwise, ensure filename has an extension (.json or .wts)
            if (filename && !filename.endsWith('.json') && !filename.endsWith('.wts')) {
                filename += '.wts';
            }
            
            // Show confirmation dialog only if we have a specific filename
            // If filename is empty, the user will select via file dialog
            if (filename) {
                const confirmMessage = `Are you sure you want to load a new stack? This will replace any stack content you currently have.`;
                const userChoice = await this.showConfirmDialog(confirmMessage, "Yes, load", "No, Cancel");
                
                if (userChoice === "No, Cancel") {
                    console.log("Stack loading cancelled by user");
                    return false;
                }
            } else {
                // For empty filename, show a simpler confirmation before opening file dialog
                const confirmMessage = `Are you sure you want to load a stack? This will replace any stack content you currently have.`;
                const userChoice = await this.showConfirmDialog(confirmMessage, "Yes, load", "No, Cancel");
                
                if (userChoice === "No, Cancel") {
                    console.log("Stack loading cancelled by user");
                    return false;
                }
            }
            
            // Attempt to load the file
            try {
                const fileData = await this.readFile(filename);
                if (!fileData) {
                    throw new Error(`Could not read file`);
                }
                
                // Store the original file data for future saves
                this.originalStackData = fileData;
                
                // Check if this is a multi-card stack
                if (this.isMultiCardStack(fileData)) {
                    console.log('Loading multi-card stack');
                    
                    // Parse card sections
                    const cardSections = this.parseCardSections(fileData);
                    console.log(`Found ${cardSections.length} card sections`);
                    
                    // Clear the current stack and snapshot storage
                    this.clearCurrentStack();
                    this.originalCardSnapshots.clear();
                    
                    // Load each card
                    for (const cardSection of cardSections) {
                        const cardId = cardSection.cardId;
                        const stackData = JSON.parse(cardSection.content);
                        const cardElement = this.ensureCardElement(cardId);
                        
                        // Store the original JSON for this card
                        this.originalCardSnapshots.set(cardId, cardSection.content);
                        
                        // Set display style based on card ID
                        if (cardId === 1) {
                            cardElement.style.display = 'block';
                        } else {
                            cardElement.style.display = 'none';
                        }
                        
                        // Ensure the card element has the correct cardId set
                        cardElement.dataset.cardId = cardId.toString();
                        console.log(`Loading card ${cardId} with element dataset.cardId: ${cardElement.dataset.cardId}`);
                        
                        // DEBUG: Log card objects found in this section
                        const cardObjects = stackData.objects.filter(obj => obj.type === 'card');
                        console.log(`Card ${cardId} section contains ${cardObjects.length} card objects:`, 
                            cardObjects.map(obj => ({ id: obj.id, script: obj.script ? obj.script.substring(0, 50) + '...' : 'EMPTY' })));
                        
                        // Load objects for this specific card
                        await this.loadStackObjects(stackData, cardElement);
                        
                        // Find card data by looking for the specific card object for this card ID
                        const expectedCardObjectId = cardId === 1 ? 'card' : `card-${cardId}`;
                        const cardData = stackData.objects.find(obj => obj.id === expectedCardObjectId);
                        console.log(`Looking for card object with ID '${expectedCardObjectId}' in card ${cardId} section, found:`, cardData ? cardData.id : 'NONE');
                        
                        if (cardData) {
                            // Apply card background color directly from cardData if available
                            if (cardData.properties) {
                                // Check for backgroundColor property (handle both camelCase and lowercase)
                                let bgColor = null;
                                if (cardData.properties.backgroundColor) {
                                    bgColor = cardData.properties.backgroundColor;
                                } else if (cardData.properties.backgroundcolor) {
                                    bgColor = cardData.properties.backgroundcolor;
                                }
                                
                                if (bgColor) {
                                    // Convert comma-separated RGB values to proper CSS format if needed
                                    let cssColor = bgColor;
                                    if (bgColor.match(/^\d+\s*,\s*\d+\s*,\s*\d+$/)) {
                                        const [r, g, b] = bgColor.split(',').map(v => v.trim());
                                        cssColor = `rgb(${r}, ${g}, ${b})`;
                                    }
                                    
                                    // Force apply the background color directly from the card data
                                    cardElement.style.backgroundColor = cssColor;
                                    console.log(`Applied card ${cardId} background color: ${bgColor} -> ${cssColor}`);
                                    
                                    // Store in custom properties for this specific card
                                    const cardName = cardId === 1 ? 'card' : `card-${cardId}`;
                                    if (!WebTalkObjects.customProperties.has(cardName)) {
                                        WebTalkObjects.customProperties.set(cardName, new Map());
                                    }
                                    WebTalkObjects.customProperties.get(cardName).set('backgroundColor', bgColor);
                                    WebTalkObjects.customProperties.get(cardName).set('backgroundcolor', bgColor);
                                    
                                    // Set a data attribute to track the background color
                                    cardElement.dataset.backgroundColor = bgColor;
                                }
                            }
                            
                            // Set card name if available
                            if (cardData.name) {
                                console.log(`Ensuring card ${cardId} name is set to: ${cardData.name}`);
                                
                                // Set the card name in the dataset
                                cardElement.dataset.name = cardData.name;
                                
                                // Update the objects map with the card name
                                WebTalkObjects.objects.set(cardData.name, cardElement);
                                
                                // Also update the objectsById map with the correct mapping
                                WebTalkObjects.objectsById.set(cardId.toString(), cardData.name);
                                
                                // For backward compatibility, also register as 'card-N'
                                if (cardId > 1) {
                                    WebTalkObjects.objects.set(`card-${cardId}`, cardElement);
                                }
                            }
                        }
                        
                        console.log(`Loaded card ${cardId}, name: ${cardElement.dataset.name}`);
                    }
                    
                    // Set the current card to card 1
                    WebTalkObjects.setCurrentCard(1);
                } else {
                    console.log('Loading single-card stack');
                    
                    // Parse the JSON data (single card)
                    const stackData = JSON.parse(fileData);
                    
                    // Store the original JSON for the single card
                    this.originalCardSnapshots.set(1, fileData);
                    
                    // Clear the current stack
                    this.clearCurrentStack();
                    
                    // Load the objects from the stack data
                    await this.loadStackObjects(stackData);
                }
                
                console.log(`Stack loaded from: ${filename}`);
                
                // Automatically run "go card 1" after loading the stack
                if (window.interpreter) {
                    try {
                        await window.interpreter.interpret('go card 1');
                        console.log('Automatically executed: go card 1');
                        console.log('Current mode after loading:', window.interpreter.mode);
                        console.log('Total objects loaded:', WebTalkObjects.objects.size);
                    } catch (error) {
                        console.error('Error executing "go card 1":', error);
                    }
                }
                
                return true;
            } catch (error) {
                // Check if this is a silent error (e.g., .wts not supported online)
                if (error.silent) {
                    console.log('Operation cancelled:', error.message);
                    return false;
                }
                
                const errorMessage = `Error loading stack from ${filename}: ${error.message}`;
                console.error(errorMessage);
                alert(errorMessage);
                throw new Error(errorMessage);
            }
        } catch (error) {
            // Check if this is a silent error
            if (error.silent) {
                return false;
            }
            console.error('Error in loadStack:', error);
            return false;
        }
    }
    
    // Show a confirmation dialog with custom buttons
    static async showConfirmDialog(message, confirmText, cancelText) {
        return new Promise((resolve) => {
            // Create dialog overlay
            const overlay = document.createElement('div');
            overlay.className = 'dialog-alert-overlay';
            
            // Create dialog box
            const dialog = document.createElement('div');
            dialog.className = 'dialog-alert';
            
            // Create message
            const messageElement = document.createElement('p');
            messageElement.textContent = message;
            messageElement.style.marginBottom = '20px';
            
            // Create button container
            const buttonContainer = document.createElement('div');
            buttonContainer.style.display = 'flex';
            buttonContainer.style.justifyContent = 'flex-end';
            buttonContainer.style.gap = '10px';
            
            // Create cancel button
            const cancelButton = document.createElement('button');
            cancelButton.textContent = cancelText;
            
            // Create confirm button
            const confirmButton = document.createElement('button');
            confirmButton.textContent = confirmText;
            confirmButton.style.backgroundColor = '#4CAF50'; // Keep the green color for the confirm button
            
            // Add event listeners
            cancelButton.addEventListener('click', () => {
                document.body.removeChild(overlay);
                resolve(cancelText);
            });
            
            confirmButton.addEventListener('click', () => {
                document.body.removeChild(overlay);
                resolve(confirmText);
            });
            
            // Assemble dialog
            buttonContainer.appendChild(cancelButton);
            buttonContainer.appendChild(confirmButton);
            dialog.appendChild(messageElement);
            dialog.appendChild(buttonContainer);
            overlay.appendChild(dialog);
            
            // Add to document
            document.body.appendChild(overlay);
        });
    }
    
    // Read a file using the FileReader API
    static async readFile(filename) {
        return new Promise((resolve, reject) => {
            // Create file input element
            const fileInput = document.createElement('input');
            fileInput.type = 'file';
            fileInput.style.display = 'none';
            fileInput.accept = '.json,.wts';
            
            // Add to document
            document.body.appendChild(fileInput);
            
            // Capture class reference for use in callbacks
            const FileOps = this;
            
            // Set up file reader
            fileInput.addEventListener('change', async (event) => {
                const file = event.target.files[0];
                if (!file) {
                    document.body.removeChild(fileInput);
                    reject(new Error('No file selected'));
                    return;
                }
                
                // Check if filename matches
                const selectedFilename = file.name.toLowerCase();
                const expectedFilename = filename.toLowerCase();
                
                // Allow the user to select any JSON or WTS file, but log a warning if names don't match
                if (selectedFilename !== expectedFilename) {
                    console.warn(`Selected file "${selectedFilename}" does not match requested file "${expectedFilename}"`);
                }
                
                // Determine if this is a compressed file
                const isCompressed = selectedFilename.endsWith('.wts');
                
                const reader = new FileReader();
                
                reader.onload = async (e) => {
                    try {
                        let result = e.target.result;
                        
                        // If it's a compressed file, decompress it
                        if (isCompressed) {
                            console.log('Decompressing .wts file...');
                            result = await FileOps.decompressData(result);
                            console.log('Decompression complete');
                        }
                        
                        document.body.removeChild(fileInput);
                        resolve(result);
                    } catch (error) {
                        console.error('Error in reader.onload:', error);
                        document.body.removeChild(fileInput);
                        // Preserve the silent flag if it exists
                        if (error.silent) {
                            reject(error);
                        } else {
                            reject(new Error(`Error processing file: ${error.message}`));
                        }
                    }
                };
                
                reader.onerror = (e) => {
                    document.body.removeChild(fileInput);
                    reject(new Error(`Error reading file: ${e.target.error}`));
                };
                
                // Read as ArrayBuffer for compressed files, as text for JSON files
                if (isCompressed) {
                    reader.readAsArrayBuffer(file);
                } else {
                    reader.readAsText(file);
                }
            });
            
            // Trigger file selection dialog
            fileInput.click();
        });
    }
    
    // Clear the current stack
    static clearCurrentStack() {
        const card = document.getElementById('card');
        if (!card) return;
        
        // Simply clear the card div - this removes all child elements and their event listeners
        card.innerHTML = '';
        
        // Reset WebTalkObjects state
        WebTalkObjects.objects = new Map();
        WebTalkObjects.objectsById = new Map();
        WebTalkObjects.scripts = new Map();
        WebTalkObjects.customProperties = new Map();
    }
    
    // Load a stack directly from JSON data without user prompts
    static async loadStackFromData(stackData) {
        try {
            // Clear the current stack
            this.clearCurrentStack();
            
            // Load the objects from the stack data
            await this.loadStackObjects(stackData);
            
            console.log('Stack loaded from data successfully');
            
            // Get all field objects that have formatting commands
            const fieldsWithFormatting = stackData.objects.filter(obj => 
                obj.type === 'field' && 
                obj.formattingCommands && 
                Array.isArray(obj.formattingCommands) && 
                obj.formattingCommands.length > 0
            );
            
            if (fieldsWithFormatting.length > 0) {
                console.log(`Found ${fieldsWithFormatting.length} fields with formatting commands, scheduling delayed restoration`);
                
                // Use a longer delay to ensure everything is ready
                setTimeout(() => {
                    // Try to apply formatting with multiple retries if needed
                    this.applyDelayedFormatting(fieldsWithFormatting);
                }, 1000); // 1000ms delay to ensure everything is ready
            }
            
            return true;
        } catch (error) {
            console.error('Error loading stack from data:', error);
            return false;
        }
    }
    
    // Helper method to apply formatting with retries
    static applyDelayedFormatting(fieldsWithFormatting, retryCount = 0) {
        const maxRetries = 5;
        
        // Check if interpreter is ready
        if (!window.webTalkApp || !window.webTalkApp.interpreter) {
            console.log(`Interpreter not ready for formatting, retry ${retryCount + 1} of ${maxRetries}`);
            if (retryCount < maxRetries) {
                setTimeout(() => {
                    this.applyDelayedFormatting(fieldsWithFormatting, retryCount + 1);
                }, 500 * Math.pow(1.5, retryCount));
            } else {
                console.error(`Failed to apply formatting after ${maxRetries} retries - interpreter not available`);
            }
            return;
        }
        
        // Apply formatting to each field
        fieldsWithFormatting.forEach(fieldData => {
            try {
                console.log(`Applying delayed formatting to field ${fieldData.name}`);
                
                // Use the same script as before
                const restoreScript = [
                    `put the arrayData("formattingCommands") of field "${fieldData.name}" into tCommands`,
                    'put the number of lines in tCommands into tRepeat',
                    'put 0 into tcount',
                    'repeat tRepeat',
                    '  add 1 to tcount',
                    '  put line tcount of tCommands into tVarToDo',
                    '  do tVarToDo',
                    'end repeat'
                ].join('\n');
                
                // Execute the script using the main interpreter instance
                window.webTalkApp.interpreter.runCommand(restoreScript);
                console.log(`Successfully applied formatting to ${fieldData.name}`);
            } catch (error) {
                console.error(`Error in delayed formatting restoration for ${fieldData.name}:`, error);
            }
        });
    }
    
    // Load objects from stack data
    static async loadStackObjects(stackData, targetCard = null) {
        if (!stackData || !stackData.objects || !Array.isArray(stackData.objects)) {
            throw new Error('Invalid stack data format');
        }
        
        // Use provided target card or default to main card
        const card = targetCard || document.getElementById('card');
        if (!card) {
            throw new Error('Card element not found');
        }
        
        // Store the previous current card and set the target card as current
        // This ensures objects are created on the correct card
        const previousCurrentCard = window.currentCard;
        window.currentCard = card;
        
        // Set a flag on the card to indicate we're loading from JSON
        // This will be used by createScrollbar to avoid setting default dimensions
        card.dataset.loadingFromJson = 'true';
        
        // Only set default background color if we're not loading a card with specific backgroundColor
        // Check if card data contains backgroundColor before defaulting to white
        const cardObject = stackData.objects.find(obj => 
            obj.type === 'card' || 
            obj.id === 'card' || 
            (obj.id && obj.id.startsWith('card-'))
        );
        
        // We'll set the background color from the card data later
        // Don't set a default white background here as it can override card-specific colors
        // Set card properties from stackData.properties if they exist
        if (stackData.properties) {
            // Create custom properties Map if it doesn't exist
            const cardId = targetCard ? (targetCard.dataset.cardId || 1) : 1;
            const cardName = cardId === 1 ? 'card' : `card-${cardId}`;
            
            if (!WebTalkObjects.customProperties.has(cardName)) {
                WebTalkObjects.customProperties.set(cardName, new Map());
            }
            
            // Set all card properties at once
            for (const [key, value] of Object.entries(stackData.properties)) {
                WebTalkObjects.setObjectProperty(cardName, key, value);
            }
            
            // Apply background color if it exists in properties
            if (stackData.properties.backgroundColor) {
                card.style.backgroundColor = stackData.properties.backgroundColor;
            } else if (stackData.properties.backgroundcolor) {
                card.style.backgroundColor = stackData.properties.backgroundcolor;
            }
        }
        
        // Process card data first to set background properties
        // Find the BEST card data by prioritizing objects with backgroundColor properties
        let cardData = null;
        
        // First, look for card objects that have backgroundColor properties
        const cardObjectsWithBg = stackData.objects.filter(obj => 
            (obj.type === 'card' || obj.id === 'card' || (obj.id && obj.id.startsWith('card-'))) &&
            obj.properties && 
            (obj.properties.backgroundColor || obj.properties.backgroundcolor)
        );
        
        if (cardObjectsWithBg.length > 0) {
            // Prefer the one with the most complete backgroundColor data
            cardData = cardObjectsWithBg.reduce((best, current) => {
                const bestBg = best.properties.backgroundColor || best.properties.backgroundcolor;
                const currentBg = current.properties.backgroundColor || current.properties.backgroundcolor;
                
                // Prefer RGB format over comma-separated values
                if (currentBg && currentBg.startsWith('rgb(') && (!bestBg || !bestBg.startsWith('rgb('))) {
                    return current;
                }
                return best;
            });
        }
        
        // If no card with backgroundColor found, fall back to any card object
        if (!cardData) {
            cardData = stackData.objects.find(obj => 
                obj.type === 'card' || 
                obj.id === 'card' || 
                (obj.id && obj.id.startsWith('card-'))
            );
        }
        
        // Get card ID from the target card or default to 1
        const cardId = targetCard ? (targetCard.dataset.cardId || 1) : 1;
        const cardName = cardId === 1 ? 'card' : `card-${cardId}`;
        
        // Apply card background color from card object properties if available
        if (cardData && cardData.properties) {
            console.log(`Found card ${cardId} properties:`, cardData.properties);
            
            // Check for backgroundColor property (handle both camelCase and lowercase)
            let bgColor = null;
            if (cardData.properties.backgroundColor) {
                bgColor = cardData.properties.backgroundColor;
            } else if (cardData.properties.backgroundcolor) {
                bgColor = cardData.properties.backgroundcolor;
            }
            
            if (bgColor) {
                // Convert comma-separated RGB values to proper CSS format if needed
                let cssColor = bgColor;
                if (bgColor.match(/^\d+\s*,\s*\d+\s*,\s*\d+$/)) {
                    const [r, g, b] = bgColor.split(',').map(v => v.trim());
                    cssColor = `rgb(${r}, ${g}, ${b})`;
                }
                
                // Force apply the background color directly from the card data
                card.style.backgroundColor = cssColor;
                console.log(`Applied card ${cardId} background color: ${cssColor} (original: ${bgColor})`);
                
                // Store in custom properties for this specific card
                if (!WebTalkObjects.customProperties.has(cardName)) {
                    WebTalkObjects.customProperties.set(cardName, new Map());
                }
                WebTalkObjects.customProperties.get(cardName).set('backgroundColor', cssColor);
                WebTalkObjects.customProperties.get(cardName).set('backgroundcolor', bgColor); // Keep original format too
                
                // Set a data attribute to track the background color
                card.dataset.backgroundColor = cssColor;
                
                // Schedule a delayed re-application to ensure it persists after all loading is complete
                setTimeout(() => {
                    if (card.style.backgroundColor !== cssColor) {
                        console.log(`Re-applying card ${cardId} background color after loading: ${cssColor}`);
                        card.style.backgroundColor = cssColor;
                    }
                }, 100);
                
                // Also schedule another check after a longer delay
                setTimeout(() => {
                    if (card.style.backgroundColor !== cssColor) {
                        console.log(`Final re-application of card ${cardId} background color: ${cssColor}`);
                        card.style.backgroundColor = cssColor;
                    }
                }, 500);
            }
        }
        
        // Set card name if it exists in the card data
        if (cardData && cardData.name) {
            console.log('Found card data with name:', cardData.name, 'for card element:', card.id);
            
            // Ensure card name is in proper format (not just a number)
            let cardName = cardData.name;
            if (cardName.match(/^\d+$/)) {
                cardName = `card ${cardName}`;
                console.log(`Fixed card name from "${cardData.name}" to "${cardName}"`);
            }
            
            // Set the card name in the dataset
            card.dataset.name = cardName;
            
            // Register the card with its name in WebTalkObjects
            const cardId = targetCard ? (targetCard.dataset.cardId || 1) : 1;
            const cardElementId = cardId === 1 ? 'card' : `card-${cardId}`;
            
            // Update the objects map with the corrected card name
            WebTalkObjects.objects.set(cardName, card);
            
            // Also register the card with its expected ID for script lookups
            const expectedCardId = cardId === 1 ? 'card' : `card-${cardId}`;
            WebTalkObjects.objects.set(expectedCardId, card);
            
            // Also update the objectsById map with the correct mapping
            WebTalkObjects.objectsById.set(cardId.toString(), cardName);
            
            console.log(`Set card name: ${cardData.name} for card ID: ${cardId}`);
            console.log(`Also registered card as: ${expectedCardId}`);
            console.log('Card dataset after setting name:', card.dataset);
        } else {
            console.log('No card name found in card data:', cardData);
        }
        
        // Handle card script from the cardScript property (new format)
        if (stackData.cardScript) {
            // For the main card or the specific card being loaded
            const cardId = targetCard ? (targetCard.dataset.cardId || 1) : 1;
            const cardScriptName = cardId === 1 ? 'card' : `card-${cardId}`;
            WebTalkObjects.scripts.set(cardScriptName, stackData.cardScript);
            console.log(`Set card script for ${cardScriptName}:`, stackData.cardScript);
            
            // Ensure the script is properly registered for execution
            if (cardScriptName !== 'card') {
                // Copy the script to the current card object name for proper execution
                WebTalkObjects.scripts.set(`card-${cardId}`, stackData.cardScript);
            }
        }
        
        if (cardData) {
            // Set card script from the card object (legacy format)
            if (cardData.script) {
                // Use the card object's ID to determine the correct script name
                let cardScriptName = 'card'; // Default for card 1
                if (cardData.id && cardData.id !== 'card') {
                    cardScriptName = cardData.id; // Use card-2, card-3, etc.
                }
                WebTalkObjects.scripts.set(cardScriptName, cardData.script);
                console.log(`Set card script for ${cardScriptName}:`, cardData.script.substring(0, 100) + '...');
                
                // Also register with the card's name if different from ID
                if (cardData.name && cardData.name !== cardScriptName) {
                    WebTalkObjects.scripts.set(cardData.name, cardData.script);
                    console.log(`Also set card script for name ${cardData.name}`);
                }
            }
               
            // Set card properties
            if (cardData.properties) {
                // Use the same card identifier as for scripts
                let cardPropertyName = 'card'; // Default for card 1
                if (cardData.id && cardData.id !== 'card') {
                    cardPropertyName = cardData.id; // Use card-2, card-3, etc.
                }
                
                // Create custom properties Map if it doesn't exist
                if (!WebTalkObjects.customProperties.has(cardPropertyName)) {
                    WebTalkObjects.customProperties.set(cardPropertyName, new Map());
                }
                
                // Set all card properties at once
                for (const [key, value] of Object.entries(cardData.properties)) {
                    WebTalkObjects.setObjectProperty(cardPropertyName, key, value);
                }
                
                // Apply background color if it exists in properties
                if (cardData.properties.backgroundColor) {
                    card.style.backgroundColor = cardData.properties.backgroundColor;
                }
                
                // Apply background pattern if it exists in properties
                if (cardData.properties.backgroundPattern && 
                    cardData.properties.backgroundPatternData && 
                    cardData.properties.backgroundPatternType) {
                    
                    // Set the background pattern on the card
                    card.style.backgroundImage = `url(data:${cardData.properties.backgroundPatternType};base64,${cardData.properties.backgroundPatternData})`;
                    card.style.backgroundRepeat = 'repeat';
                    
                    // Clear any background color when setting background pattern
                    card.style.backgroundColor = '';
                }
                
                // Apply foreground color if it exists
                if (cardData.properties.foregroundColor) {
                    card.style.color = cardData.properties.foregroundColor;
                }
            }
        }
        
        // Process other objects
        for (const objectData of stackData.objects) {
            try {
                // Skip card type objects - we already handled them
                if (objectData.type === 'card') continue;
                
                // Create the object
                // Map 'line' type to 'graphic' for object creation
                // CRITICAL FIX: Ensure image objects are created with type 'image', not 'image/png' or other MIME types
                let objectType = objectData.type;
                
                // Add null check to prevent startsWith error on undefined objectType
                if (!objectType) {
                    console.warn(`Object ${objectData.name} has undefined type, skipping object creation`);
                    continue;
                }
                
                if (objectType === 'line' || objectType === 'oval') {
                    objectType = 'graphic';
                } else if (objectType.startsWith('image/')) {
                    console.log(`Fixing object type for ${objectData.name}: was '${objectType}', setting to 'image'`);
                    objectType = 'image';
                }
                
                // Determine if we need to pass a shape parameter for graphic objects
                let shape = null;
                if (objectData.type === 'oval' || objectData.graphicType === 'oval') {
                    shape = 'oval';
                } else if (objectData.type === 'circle' || objectData.graphicType === 'circle') {
                    shape = 'circle';
                } else if (objectData.type === 'rectangle' || objectData.graphicType === 'rectangle') {
                    shape = 'rectangle';
                } else if (objectData.type === 'square' || objectData.graphicType === 'square') {
                    shape = 'square';
                } else if (objectData.type === 'triangle' || objectData.graphicType === 'triangle') {
                    shape = 'triangle';
                } else if (objectData.type === 'pentagon' || objectData.graphicType === 'pentagon') {
                    shape = 'pentagon';
                } else if (objectData.type === 'hexagon' || objectData.graphicType === 'hexagon') {
                    shape = 'hexagon';
                } else if (objectData.type === 'octagon' || objectData.graphicType === 'octagon') {
                    shape = 'octagon';
                } else if (objectData.type === 'decagon' || objectData.graphicType === 'decagon') {
                    shape = 'decagon';
                }
                
                // Pass the original ID from the JSON file to preserve IDs during loading
                const element = WebTalkObjects.createObject(objectType, objectData.name, shape, objectData.id);
                console.log(`Created ${objectType} "${objectData.name}" with ID ${objectData.id}`);                
                
                // For scrollbars, mark them as loaded from JSON to preserve dimensions BEFORE setting any properties
                if (objectData.type === 'scrollbar') {
                    // Set a flag to indicate this scrollbar was loaded from JSON
                    WebTalkObjects.customProperties.get(objectData.name).set('loadedFromJSON', true);
                    
                    // Store original dimensions to be applied after all properties are set
                    if (objectData.position) {
                        WebTalkObjects.customProperties.get(objectData.name).set('originalWidth', objectData.position.width);
                        WebTalkObjects.customProperties.get(objectData.name).set('originalHeight', objectData.position.height);
                        console.log(`Storing original dimensions for scrollbar ${objectData.name}: ${objectData.position.width} × ${objectData.position.height}`);
                    }
                }
                
                // Set position and size
                if (objectData.position) {
                    if (objectData.position.left) element.style.left = objectData.position.left;
                    if (objectData.position.top) element.style.top = objectData.position.top;
                    
                    // For scrollbars, ensure we set both width/height and min-width/min-height
                    if (objectData.type === 'scrollbar') {
                        if (objectData.position.width) {
                            element.style.width = objectData.position.width;
                            element.style.minWidth = objectData.position.width;
                            console.log(`Initial width set for ${objectData.name}: ${objectData.position.width}`);
                        }
                        if (objectData.position.height) {
                            element.style.height = objectData.position.height;
                            element.style.minHeight = objectData.position.height;
                            console.log(`Initial height set for ${objectData.name}: ${objectData.position.height}`);
                        }
                    } else {
                        // For non-scrollbar objects, set dimensions normally
                        if (objectData.position.width) element.style.width = objectData.position.width;
                        if (objectData.position.height) element.style.height = objectData.position.height;
                    }
                }
                
                // Set script - ensure this happens before applying properties
                if (objectData.script) {
                    WebTalkObjects.setScript(objectData.name, objectData.script);
                    console.log(`Set script for ${objectData.name}:`, objectData.script);
                }
                
                // Set content for fields
                if (objectData.type === 'field' && objectData.content) {
                    // Find the field-content div
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Always set content as plain text first
                        // Use textContent to preserve newlines with white-space: pre-wrap CSS
                        fieldContent.textContent = objectData.content;
                        
                        // Store formatting commands in customProperties if they exist
                        if (objectData.formattingCommands && Array.isArray(objectData.formattingCommands)) {
                            WebTalkObjects.customProperties.get(objectData.name).set('formattingCommands', objectData.formattingCommands);
                            
                            // Store the formatting commands for later application
                            const applyFormattingCommands = () => {
                                try {
                                    // Check if interpreter is ready
                                    if (!window.webTalkApp || 
                                        !window.webTalkApp.interpreter || 
                                        typeof window.webTalkApp.interpreter.runCommand !== 'function') {
                                        console.log(`Interpreter not ready for formatting ${objectData.name}`);
                                        return false;
                                    }
                                    
                                    // Check if WebTalkObjects is available
                                    if (!WebTalkObjects || !WebTalkObjects.elements) {
                                        console.log(`WebTalkObjects not ready for formatting ${objectData.name}`);
                                        return false;
                                    }
                                    
                                    // Check if the field element exists in the DOM
                                    const fieldElement = document.querySelector(`[data-name="${objectData.name}"]`);
                                    if (!fieldElement) {
                                        console.log(`Field element ${objectData.name} not found in DOM`);
                                        return false;
                                    }
                                    
                                    // Check if the field-content div exists
                                    const fieldContent = fieldElement.querySelector('.field-content');
                                    if (!fieldContent) {
                                        console.log(`Field content for ${objectData.name} not found`);
                                        return false;
                                    }
                                    
                                    // Use our new arrayData property to get the formatting commands
                                    // This is more reliable than accessing the stored array directly
                                    let formattingCommands = [];
                                    
                                    try {
                                        // Use the interpreter to get the arrayData property
                                        const script = `put the arrayData("formattingCommands") of field "${objectData.name}"`;                                        
                                        const result = window.webTalkApp.interpreter.runCommand(script);
                                        
                                        // If we got a result, parse it into commands
                                        if (result && typeof result === 'string') {
                                            formattingCommands = result.split('\n').filter(cmd => cmd.trim() !== '');
                                        } else if (Array.isArray(objectData.formattingCommands)) {
                                            // Fall back to the stored array if needed
                                            formattingCommands = objectData.formattingCommands;
                                        }
                                    } catch (error) {
                                        console.warn(`Error getting arrayData for ${objectData.name}, falling back to stored commands:`, error);
                                        // Fall back to the stored array if needed
                                        if (Array.isArray(objectData.formattingCommands)) {
                                            formattingCommands = objectData.formattingCommands;
                                        }
                                    }
                                    
                                    console.log(`Applying formatting commands to field ${objectData.name} using direct script execution`);
                                    
                                    // Use the same script as in inspector-tasks.js but execute it directly with window.webTalkApp.interpreter
                                    // This ensures we're using the same interpreter instance that's used during stack loading
                                    const restoreScript = [
                                        `put the arrayData("formattingCommands") of field "${objectData.name}" into tCommands`,
                                        'put the number of lines in tCommands into tRepeat',
                                        'put 0 into tcount',
                                        'repeat tRepeat',
                                        '  add 1 to tcount',
                                        '  put line tcount of tCommands into tVarToDo',
                                        '  do tVarToDo',
                                        'end repeat'
                                    ].join('\n');
                                    
                                    try {
                                        // Execute the script using the main interpreter instance
                                        window.webTalkApp.interpreter.runCommand(restoreScript);
                                    } catch (error) {
                                        console.error(`Error executing formatting restoration script for ${objectData.name}:`, error);
                                    }
                                    return true; // Successfully applied
                                } catch (error) {
                                    console.error(`Unexpected error when applying formatting commands to ${objectData.name}:`, error);
                                    return false;
                                }
                            };
                            
                            // Wait for the interpreter to be fully initialized using the preloadingComplete flag
                            const waitForInterpreter = () => {
                                if (window.webTalkApp && window.webTalkApp.preloadingComplete) {
                                    // Give a larger delay to ensure everything is ready
                                    setTimeout(() => {
                                        if (!applyFormattingCommands()) {
                                            console.warn(`Interpreter ready but couldn't apply formatting to ${objectData.name}. Will retry with increasing delays.`);
                                            
                                            // Multiple retries with increasing delays
                                            setTimeout(() => {
                                                if (!applyFormattingCommands()) {
                                                    console.warn(`First retry failed for ${objectData.name}, trying again...`);
                                                    setTimeout(() => {
                                                        if (!applyFormattingCommands()) {
                                                            console.error(`Failed to apply formatting commands to ${objectData.name} - interpreter or elements not available after retry.`);
                                                        }
                                                    }, 500); // Third attempt after 500ms
                                                }
                                            }, 300); // Second attempt after 300ms
                                        }
                                    }, 200); // First attempt after 200ms
                                } else {
                                    // Check again after a delay
                                    setTimeout(waitForInterpreter, 300);
                                }
                            };
                            
                            // Start waiting for the interpreter
                            waitForInterpreter();
                            
                            // Add a timeout to prevent infinite waiting
                            setTimeout(() => {
                                if (!window.webTalkApp || !window.webTalkApp.preloadingComplete) {
                                    console.warn(`Timeout waiting for interpreter initialization. Attempting to apply formatting to ${objectData.name} anyway.`);
                                    // Final attempt with a forced application
                                    setTimeout(() => {
                                        if (!applyFormattingCommands()) {
                                            console.error(`Final attempt to apply formatting to ${objectData.name} failed after timeout.`);
                                        }
                                    }, 1000); // Give one more second for things to settle
                                }
                            }, 5000); // 5 second timeout
                        }
                    } else {
                        // Fallback to innerText if field-content div is not found
                        element.innerText = objectData.content;
                    }
                }
                
                // Set the graphicType in the dataset if it exists
                if (objectData.graphicType && element.dataset.type === 'graphic') {
                    element.dataset.graphicType = objectData.graphicType;
                }
                
                // Special handling for line graphic objects
                if ((objectData.type === 'line' || objectData.graphicType === 'line') && objectData.attributes) {
                    const svg = element.querySelector('svg');
                    const line = svg.querySelector('line');
                    
                    if (objectData.attributes.x1 !== undefined) line.setAttribute('x1', objectData.attributes.x1);
                    if (objectData.attributes.y1 !== undefined) line.setAttribute('y1', objectData.attributes.y1);
                    if (objectData.attributes.x2 !== undefined) line.setAttribute('x2', objectData.attributes.x2);
                    if (objectData.attributes.y2 !== undefined) line.setAttribute('y2', objectData.attributes.y2);
                    if (objectData.attributes.stroke) line.setAttribute('stroke', objectData.attributes.stroke);
                    if (objectData.attributes['stroke-width']) line.setAttribute('stroke-width', objectData.attributes['stroke-width']);
                } else if ((objectData.type === 'oval' || objectData.graphicType === 'oval') && objectData.attributes) {
                    const svg = element.querySelector('svg');
                    const ellipse = svg.querySelector('ellipse');
                    
                    if (objectData.attributes.cx !== undefined) ellipse.setAttribute('cx', objectData.attributes.cx);
                    if (objectData.attributes.cy !== undefined) ellipse.setAttribute('cy', objectData.attributes.cy);
                    if (objectData.attributes.rx !== undefined) ellipse.setAttribute('rx', objectData.attributes.rx);
                    if (objectData.attributes.ry !== undefined) ellipse.setAttribute('ry', objectData.attributes.ry);
                    if (objectData.attributes.fill) ellipse.setAttribute('fill', objectData.attributes.fill);
                    if (objectData.attributes.stroke) ellipse.setAttribute('stroke', objectData.attributes.stroke);
                    if (objectData.attributes['stroke-width']) ellipse.setAttribute('stroke-width', objectData.attributes['stroke-width']);
                } else if ((objectData.type === 'circle' || objectData.graphicType === 'circle') && objectData.attributes) {
                    const svg = element.querySelector('svg');
                    const circle = svg.querySelector('circle');
                    
                    if (objectData.attributes.cx !== undefined) circle.setAttribute('cx', objectData.attributes.cx);
                    if (objectData.attributes.cy !== undefined) circle.setAttribute('cy', objectData.attributes.cy);
                    if (objectData.attributes.r !== undefined) circle.setAttribute('r', objectData.attributes.r);
                    if (objectData.attributes.fill) circle.setAttribute('fill', objectData.attributes.fill);
                    if (objectData.attributes.stroke) circle.setAttribute('stroke', objectData.attributes.stroke);
                    if (objectData.attributes['stroke-width']) circle.setAttribute('stroke-width', objectData.attributes['stroke-width']);
                } else if (((objectData.type === 'rectangle' || objectData.graphicType === 'rectangle') || 
                           (objectData.type === 'square' || objectData.graphicType === 'square')) && 
                           objectData.attributes) {
                    const svg = element.querySelector('svg');
                    const rect = svg.querySelector('rect');
                    
                    if (objectData.attributes.x !== undefined) rect.setAttribute('x', objectData.attributes.x);
                    if (objectData.attributes.y !== undefined) rect.setAttribute('y', objectData.attributes.y);
                    if (objectData.attributes.width !== undefined) rect.setAttribute('width', objectData.attributes.width);
                    if (objectData.attributes.height !== undefined) rect.setAttribute('height', objectData.attributes.height);
                    if (objectData.attributes.fill) rect.setAttribute('fill', objectData.attributes.fill);
                    if (objectData.attributes.stroke) rect.setAttribute('stroke', objectData.attributes.stroke);
                    if (objectData.attributes['stroke-width']) rect.setAttribute('stroke-width', objectData.attributes['stroke-width']);
                } else if (['triangle', 'pentagon', 'hexagon', 'octagon', 'decagon'].includes(objectData.graphicType) && 
                           objectData.attributes) {
                    const svg = element.querySelector('svg');
                    const polygon = svg.querySelector('polygon');
                    
                    if (objectData.attributes.points !== undefined) polygon.setAttribute('points', objectData.attributes.points);
                    if (objectData.attributes.fill) polygon.setAttribute('fill', objectData.attributes.fill);
                    if (objectData.attributes.stroke) polygon.setAttribute('stroke', objectData.attributes.stroke);
                    if (objectData.attributes['stroke-width']) polygon.setAttribute('stroke-width', objectData.attributes['stroke-width']);
                }
                
                // Special handling for image objects - use imageData instead of properties
                if (objectData.type === 'image') {
                    // No need to set filename from properties - we'll use imageData below
                }
                
                // Special handling for player objects
                if (objectData.type === 'player' && objectData.properties && objectData.properties.filename) {
                    // Set the player video source
                    WebTalkObjects.setObjectProperty(objectData.name, 'filename', objectData.properties.filename);
                }
                
                // CRITICAL: Set image data BEFORE setting properties to prevent file dialog
                const imgElement = element.querySelector('img');
                
                // Check for image data in properties (new format) or imageData (legacy format)
                let imageDataSource = null;
                if (objectData.properties && objectData.properties.data && objectData.properties.type) {
                    imageDataSource = {
                        data: objectData.properties.data,
                        type: objectData.properties.type,
                        filename: objectData.properties.filename || ''
                    };
                } else if (objectData.imageData && objectData.imageData.type && objectData.imageData.data) {
                    imageDataSource = objectData.imageData;
                }
                
                if (imgElement && imageDataSource) {
                    // Store the data in custom properties FIRST to prevent file dialog
                    const customProps = WebTalkObjects.customProperties.get(objectData.name);
                    if (customProps) {
                        // CRITICAL: Store image data in custom properties BEFORE setting filename
                        customProps.set('data', imageDataSource.data || '');
                        customProps.set('type', imageDataSource.type || 'image/png');
                        customProps.set('filename', imageDataSource.filename || '');
                        
                        console.log(`Pre-stored image data for ${objectData.name} to prevent file dialog`);
                    }
                    
                    // Use the MIME type for the image source but NOT for the data-type attribute
                    imgElement.src = `data:${imageDataSource.type};base64,${imageDataSource.data}`;
                    
                    // Remove placeholder styling
                    element.style.backgroundColor = '';
                    element.style.border = '';
                    
                    // CRITICAL FIX: Ensure the data-type attribute is ALWAYS set to 'image'
                    if (element.dataset.type !== 'image') {
                        console.log(`Fixing data-type for ${objectData.name}: was '${element.dataset.type}', setting to 'image'`);
                        element.dataset.type = 'image';
                    }
                    
                    // CRITICAL FIX: Also force the objectData.type to be 'image' to ensure consistency
                    if (objectData.type !== 'image') {
                        console.log(`Fixing objectData.type for ${objectData.name}: was '${objectData.type}', setting to 'image'`);
                        objectData.type = 'image';
                    }
                }
                
                // Handle soundData if present
                if (objectData.soundData) {
                    console.log(`Loading soundData for ${objectData.name}:`, {
                        sampleRate: objectData.soundData.sampleRate,
                        length: objectData.soundData.length,
                        numberOfChannels: objectData.soundData.numberOfChannels
                    });
                    
                    // Recreate AudioBuffer from compressed data
                    const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
                    const audioBuffer = audioCtx.createBuffer(
                        objectData.soundData.numberOfChannels,
                        objectData.soundData.length,
                        objectData.soundData.sampleRate
                    );
                    
                    // Restore channel data from compressed format
                    for (let i = 0; i < objectData.soundData.numberOfChannels; i++) {
                        const channelData = audioBuffer.getChannelData(i);
                        
                        // Handle both old uncompressed format and new compressed format
                        if (objectData.soundData.compressedChannels) {
                            // New compressed format
                            const base64Data = objectData.soundData.compressedChannels[i];
                            
                            // Decode base64 to binary
                            const binaryString = atob(base64Data);
                            const uint8Data = new Uint8Array(binaryString.length);
                            for (let j = 0; j < binaryString.length; j++) {
                                uint8Data[j] = binaryString.charCodeAt(j);
                            }
                            
                            // Convert back from [0, 255] to [-1, 1] range
                            for (let j = 0; j < uint8Data.length; j++) {
                                channelData[j] = (uint8Data[j] / 127.5) - 1;
                            }
                        } else if (objectData.soundData.channels) {
                            // Old uncompressed format (backward compatibility)
                            const savedChannelData = objectData.soundData.channels[i];
                            for (let j = 0; j < savedChannelData.length; j++) {
                                channelData[j] = savedChannelData[j];
                            }
                        }
                    }
                    
                    // Store sound data in custom properties
                    const customProps = WebTalkObjects.customProperties.get(objectData.name);
                    if (customProps) {
                        customProps.set('soundData', audioBuffer);
                        customProps.set('audioContext', audioCtx);
                        
                        console.log(`Restored soundData for ${objectData.name}`);
                    }
                }
                
                // Set custom properties
                if (objectData.properties) {
                    // Create custom properties Map if it doesn't exist
                    if (!WebTalkObjects.customProperties.has(objectData.name)) {
                        WebTalkObjects.customProperties.set(objectData.name, new Map());
                    }
                    
                    // Special handling for backgroundPattern on graphic objects
                    if (objectData.type === 'graphic' && 
                        (objectData.properties.backgroundPattern || 
                         objectData.properties.backgroundpattern)) {
                        
                        // Handle both uppercase and lowercase property names
                        const patternData = objectData.properties.backgroundPatternData || 
                                           objectData.properties.backgroundpatterndata;
                        const patternType = objectData.properties.backgroundPatternType || 
                                           objectData.properties.backgroundpatterntype;
                        
                        // Make sure we have all the necessary data
                        if (patternData && patternType) {
                            // Get the SVG element
                            const svg = element.querySelector('svg');
                            if (svg) {
                                // Check the graphic type from the element's dataset
                                const graphicType = element.dataset.graphicType || 'line';
                                
                                // Create a pattern definition in the SVG
                                let patternId = `pattern_${objectData.name.replace(/\s+/g, '_')}`;
                                
                                // Remove any existing pattern with this ID
                                let existingDefs = svg.querySelector('defs');
                                if (existingDefs) {
                                    let existingPattern = existingDefs.querySelector(`#${patternId}`);
                                    if (existingPattern) {
                                        existingDefs.removeChild(existingPattern);
                                    }
                                } else {
                                    // Create defs element if it doesn't exist
                                    existingDefs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
                                    svg.insertBefore(existingDefs, svg.firstChild);
                                }
                                
                                // Create the pattern element
                                const pattern = document.createElementNS('http://www.w3.org/2000/svg', 'pattern');
                                pattern.setAttribute('id', patternId);
                                pattern.setAttribute('patternUnits', 'userSpaceOnUse');
                                pattern.setAttribute('width', '20'); // Default size, will be overridden by image
                                pattern.setAttribute('height', '20');
                                
                                // Create an image element inside the pattern
                                const image = document.createElementNS('http://www.w3.org/2000/svg', 'image');
                                image.setAttribute('href', `data:${patternType};base64,${patternData}`);
                                image.setAttribute('x', '0');
                                image.setAttribute('y', '0');
                                image.setAttribute('width', '20');
                                image.setAttribute('height', '20');
                                
                                // Add the image to the pattern
                                pattern.appendChild(image);
                                
                                // Add the pattern to the defs
                                existingDefs.appendChild(pattern);
                                
                                // Apply the pattern to the appropriate shape element based on graphic type
                                if (graphicType === 'path') {
                                    const path = svg.querySelector('path');
                                    if (path) {
                                        path.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'oval') {
                                    const ellipse = svg.querySelector('ellipse');
                                    if (ellipse) {
                                        ellipse.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'rectangle') {
                                    const rect = svg.querySelector('rect');
                                    if (rect) {
                                        rect.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else if (graphicType === 'polygon') {
                                    const polygon = svg.querySelector('polygon');
                                    if (polygon) {
                                        polygon.setAttribute('fill', `url(#${patternId})`);
                                    }
                                } else {
                                    // For other types, try to find any shape element
                                    const shape = svg.querySelector('rect, circle, ellipse, polygon, line, path');
                                    if (shape) {
                                        shape.setAttribute('fill', `url(#${patternId})`);
                                    }
                                }
                                
                                // Load the image to get its dimensions
                                const tempImg = new Image();
                                tempImg.onload = function() {
                                    // Set the pattern dimensions based on the loaded image
                                    pattern.setAttribute('width', this.width);
                                    pattern.setAttribute('height', this.height);
                                    image.setAttribute('width', this.width);
                                    image.setAttribute('height', this.height);
                                    
                                    // Force a redraw of the SVG to ensure the pattern is applied
                                    const svgContent = svg.innerHTML;
                                    svg.innerHTML = svgContent;
                                    
                                    // Ensure the pattern is applied to the shape element again after redraw
                                    const shapeElement = getShapeElementForGraphicType(svg, graphicType);
                                    if (shapeElement) {
                                        shapeElement.setAttribute('fill', `url(#${patternId})`);
                                    }
                                    
                                    console.log(`Updated pattern dimensions for ${objectData.name}: ${this.width}x${this.height}`);
                                    
                                    // Helper function to get the appropriate shape element based on graphic type
                                    function getShapeElementForGraphicType(svg, graphicType) {
                                        if (graphicType === 'path') {
                                            return svg.querySelector('path');
                                        } else if (graphicType === 'oval') {
                                            return svg.querySelector('ellipse');
                                        } else if (graphicType === 'rectangle') {
                                            return svg.querySelector('rect');
                                        } else if (graphicType === 'polygon') {
                                            return svg.querySelector('polygon');
                                        } else {
                                            // Fallback for any other graphic types
                                            return svg.querySelector('rect, circle, ellipse, polygon, path');
                                        }
                                    }
                                };
                                tempImg.src = `data:${patternType};base64,${patternData}`;
                                
                                // Store in custom properties to ensure they're available
                                if (!WebTalkObjects.customProperties.has(objectData.name)) {
                                    WebTalkObjects.customProperties.set(objectData.name, new Map());
                                }
                                WebTalkObjects.customProperties.get(objectData.name).set('backgroundPattern', objectData.properties.backgroundPattern || objectData.properties.backgroundpattern || 'custom');
                                WebTalkObjects.customProperties.get(objectData.name).set('backgroundPatternData', patternData);
                                WebTalkObjects.customProperties.get(objectData.name).set('backgroundPatternType', patternType);
                                
                                // Log success for debugging
                                console.log(`Applied background pattern to graphic ${objectData.name}`);
                            }
                        }
                    }
                    
                    // First, check if this object has a backgroundPattern
                    const hasBackgroundPattern = objectData.properties.backgroundPattern || objectData.properties.backgroundpattern;
                    
                    // Apply all properties, except backgroundPattern which will be handled last
                    // and backgroundColor/fill if a backgroundPattern is present (to avoid overriding the pattern)
                                        for (const [key, value] of Object.entries(objectData.properties)) {
                        // Skip backgroundPattern properties as they were already handled specially
                        if (key.toLowerCase() === 'backgroundpattern' || 
                            key.toLowerCase() === 'backgroundpatterndata' || 
                            key.toLowerCase() === 'backgroundpatterntype') {
                            continue;
                        }
                        
                        // Skip backgroundColor and fill if a backgroundPattern is present to avoid overriding the pattern
                        if (hasBackgroundPattern && 
                            (key.toLowerCase() === 'backgroundcolor' || key.toLowerCase() === 'backgroundcolour' || 
                             key.toLowerCase() === 'fill' || key.toLowerCase() === 'fillcolor' || key.toLowerCase() === 'fillcolour') && 
                            objectData.type === 'graphic') {
                            console.log(`Skipping ${key} on ${objectData.name} because backgroundPattern is present`);
                            continue;
                        }
                        
                        // CRITICAL: Skip image data properties for image objects - they were already handled above
                        if (objectData.type === 'image' && (key === 'data' || key === 'type' || key === 'filename')) {
                            console.log(`Skipping image property ${key} for ${objectData.name} - already handled`);
                            continue;
                        }
                        
                        try {
                            WebTalkObjects.setObjectProperty(objectData.name, key, value);
                        } catch (error) {
                            console.warn(`Error setting property ${key} on ${objectData.name}: ${error.message}`);
                            // Continue with other properties
                        }
                    }
                    
                    // Special handling for showborder to ensure it's properly applied
                    if (objectData.properties.hasOwnProperty('showborder') || objectData.properties.hasOwnProperty('showBorder')) {
                        try {
                            const showBorderValue = objectData.properties.showborder || objectData.properties.showBorder;
                            const showBorder = showBorderValue === true || showBorderValue === 'true';
                            
                            // Safely get the element using getObject
                            const elem = WebTalkObjects.getObject(objectData.name);
                            if (elem && (elem.dataset.type === 'field' || elem.dataset.type === 'graphic')) {
                                // Apply border style directly with !important to override any default styles
                                elem.style.setProperty('border', showBorder ? '1px solid black' : 'none', 'important');
                                
                                // Store in custom properties
                                if (!WebTalkObjects.customProperties.has(objectData.name)) {
                                    WebTalkObjects.customProperties.set(objectData.name, new Map());
                                }
                                WebTalkObjects.customProperties.get(objectData.name).set('showborder', showBorder);
                                
                                console.log(`Applied showborder=${showBorder} to ${objectData.name}`);
                            }
                        } catch (error) {
                            console.warn(`Error applying showborder to ${objectData.name}: ${error.message}`);
                        }
                    }
                    
                    // Special handling for foregroundColor to ensure it's applied correctly
                    // Note: We don't call setObjectProperty again to avoid duplicate dialogs
                    if (objectData.properties.foregroundColor || objectData.properties.foregroundcolor) {
                        try {
                            const colorValue = objectData.properties.foregroundColor || objectData.properties.foregroundcolor;
                            
                            // Safely get the element using getObject instead of elements map
                            const elem = WebTalkObjects.getObject(objectData.name);
                            if (elem) {
                                // Parse the color value if needed
                                const parsedColor = WebTalkObjects.parseColor ? 
                                    WebTalkObjects.parseColor(colorValue) : colorValue;
                                
                                console.log(`Applying foregroundColor ${parsedColor} to ${objectData.name}`);
                                
                                // Apply with !important to override any default styles
                                elem.style.setProperty('color', parsedColor, 'important');
                                
                                // For fields, also apply to the field-content element
                                if (elem.dataset.type === 'field') {
                                    const fieldContent = elem.querySelector('.field-content');
                                    if (fieldContent) {
                                        fieldContent.style.setProperty('color', parsedColor, 'important');
                                        console.log(`Applied color to field-content for ${objectData.name}`);
                                    } else {
                                        console.log(`No field-content found for ${objectData.name}`);
                                    }
                                    
                                    // Try a different approach - create a style element specifically for this field
                                    const styleId = `field-color-${elem.id}`;
                                    let styleEl = document.getElementById(styleId);
                                    if (!styleEl) {
                                        styleEl = document.createElement('style');
                                        styleEl.id = styleId;
                                        document.head.appendChild(styleEl);
                                    }
                                    
                                    // Set a rule with very high specificity
                                    styleEl.textContent = `
                                        #${elem.id} { color: ${parsedColor} !important; }
                                        #${elem.id} * { color: ${parsedColor} !important; }
                                    `;
                                    console.log(`Created style element for ${objectData.name} with ID ${styleId}`);
                                }
                                
                                // Force a style recalculation
                                void elem.offsetHeight;
                            } else {
                                console.warn(`Element not found for ${objectData.name} when applying foregroundColor`);
                            }
                        } catch (error) {
                            console.warn(`Error applying foregroundColor to ${objectData.name}: ${error.message}`);
                            // Continue with other properties
                        }
                    }
                    
                    // Special handling for textFont/fontFamily to ensure it's applied correctly
                    console.log(`Font properties for ${objectData.name}:`, objectData.properties);
                    if (objectData.properties.textFont || objectData.properties.fontFamily) {
                        try {
                            const fontValue = objectData.properties.textFont || objectData.properties.fontFamily;
                            
                            // Safely get the element using getObject instead of elements map
                            const elem = WebTalkObjects.getObject(objectData.name);
                            if (elem) {
                                console.log(`Applying font ${fontValue} to ${objectData.name}`);
                                
                                // Apply with !important to override any default styles
                                elem.style.setProperty('fontFamily', fontValue, 'important');
                                
                                // Also set the style directly to ensure it takes effect
                                elem.style.fontFamily = fontValue;
                                
                                // For fields, also apply to the field-content element
                                if (elem.dataset.type === 'field') {
                                    const fieldContent = elem.querySelector('.field-content');
                                    if (fieldContent) {
                                        fieldContent.style.setProperty('fontFamily', fontValue, 'important');
                                        fieldContent.style.fontFamily = fontValue;
                                        console.log(`Applied font to field-content for ${objectData.name}`);
                                    }
                                }
                            } else {
                                console.warn(`Element not found for ${objectData.name} when applying font`);
                            }
                        } catch (error) {
                            console.warn(`Error applying font to ${objectData.name}: ${error.message}`);
                            // Continue with other properties
                        }
                    }
                    
                    // Apply visual properties after setting all properties
                    this.applyVisualProperties(objectData.name);
                    
                    // For scrollbars, reapply the original dimensions AFTER all properties are set
                    // This ensures the dimensions from JSON are preserved even after updateScrollbarThumb is called
                    if (objectData.type === 'scrollbar') {
                        const originalWidth = WebTalkObjects.customProperties.get(objectData.name).get('originalWidth');
                        const originalHeight = WebTalkObjects.customProperties.get(objectData.name).get('originalHeight');
                        const scrollbarElement = WebTalkObjects.getObject(objectData.name);
                        
                        if (scrollbarElement && originalWidth && originalHeight) {
                            // Force the dimensions to match what was in the JSON
                            scrollbarElement.style.width = originalWidth;
                            scrollbarElement.style.height = originalHeight;
                            
                            // Also set min-width and min-height to override CSS defaults
                            scrollbarElement.style.minWidth = originalWidth;
                            scrollbarElement.style.minHeight = originalHeight;
                            
                            // For round scrollbars, ensure the dimensions are applied correctly
                            const scrollType = WebTalkObjects.customProperties.get(objectData.name).get('scrollType');
                            if (scrollType === 'round') {
                                // Force a reflow to ensure the dimensions are applied
                                void scrollbarElement.offsetHeight;
                                
                                // Apply dimensions again after reflow
                                setTimeout(() => {
                                    scrollbarElement.style.width = originalWidth;
                                    scrollbarElement.style.height = originalHeight;
                                    scrollbarElement.style.minWidth = originalWidth;
                                    scrollbarElement.style.minHeight = originalHeight;
                                    console.log(`Delayed reapplication of dimensions for round scrollbar ${objectData.name}: ${originalWidth} × ${originalHeight}`);
                                }, 0);
                            }
                            
                            console.log(`Reapplied original dimensions to scrollbar ${objectData.name}: ${originalWidth} × ${originalHeight}`);
                        }
                    }
                    
                    // Re-apply backgroundPattern if it exists, to ensure it's not overridden by other properties
                    if (hasBackgroundPattern) {
                        // Find the SVG element
                        const svg = element.querySelector('svg');
                        if (svg) {
                            // Get the pattern ID
                            const patternId = `pattern_${objectData.name.replace(/\s+/g, '_')}`;
                            
                            // Find the pattern element
                            const pattern = svg.querySelector(`#${patternId}`);
                            if (pattern) {
                                // Get the graphic type
                                const graphicType = element.dataset.graphicType || 'line';
                                
                                // Apply the pattern to the appropriate shape element
                                if (graphicType === 'path') {
                                    const path = svg.querySelector('path');
                                    if (path) path.setAttribute('fill', `url(#${patternId})`);
                                } else if (graphicType === 'oval') {
                                    const ellipse = svg.querySelector('ellipse');
                                    if (ellipse) ellipse.setAttribute('fill', `url(#${patternId})`);
                                } else if (graphicType === 'rectangle') {
                                    const rect = svg.querySelector('rect');
                                    if (rect) rect.setAttribute('fill', `url(#${patternId})`);
                                } else if (graphicType === 'polygon') {
                                    const polygon = svg.querySelector('polygon');
                                    if (polygon) polygon.setAttribute('fill', `url(#${patternId})`);
                                } else {
                                    // Fallback for any other graphic types
                                    const shape = svg.querySelector('rect, circle, ellipse, polygon, path');
                                    if (shape) shape.setAttribute('fill', `url(#${patternId})`);
                                }
                                
                                console.log(`Re-applied background pattern to ${objectData.name} after all properties`);
                            }
                        }
                    }
                    
                    // Apply line-specific lineHeight properties for fields
                    if (objectData.type === 'field') {
                        const fieldElement = WebTalkObjects.getObject(objectData.name);
                        if (fieldElement) {
                            const fieldContent = fieldElement.querySelector('.field-content');
                            if (fieldContent) {
                                const customProps = WebTalkObjects.customProperties.get(objectData.name);
                                if (customProps) {
                                    // Find all lineHeight_line_X properties
                                    const lineHeightProps = [];
                                    customProps.forEach((value, key) => {
                                        if (key.startsWith('lineHeight_line_')) {
                                            const lineNum = parseInt(key.replace('lineHeight_line_', ''));
                                            if (!isNaN(lineNum)) {
                                                lineHeightProps.push({ lineNum, value });
                                            }
                                        }
                                    });
                                    
                                    if (lineHeightProps.length > 0) {
                                        // Get the field content HTML
                                        const fieldContentHTML = fieldContent.innerHTML;
                                        
                                        // Check if the content is already split into divs or if it's plain text
                                        let lines;
                                        let isContentWrappedInDivs = /<div[^>]*>[^<]*<\/div>/.test(fieldContentHTML);
                                        
                                        if (isContentWrappedInDivs) {
                                            // Content already has divs, extract them
                                            const divMatches = fieldContentHTML.match(/<div[^>]*>([^<]*)<\/div>/g) || [];
                                            lines = divMatches;
                                        } else {
                                            // Content is plain text, split by newlines and wrap each line in a div
                                            lines = fieldContentHTML.split('\n').map(line => `<div>${line}</div>`);
                                        }
                                        
                                        // Apply lineHeight to each specified line
                                        lineHeightProps.forEach(prop => {
                                            if (prop.lineNum > 0 && prop.lineNum <= lines.length) {
                                                const targetLineIndex = prop.lineNum - 1;
                                                const currentLine = lines[targetLineIndex];
                                                
                                                // Check if the line already has a style attribute
                                                const styleRegex = /<div([^>]*)>([^<]*)<\/div>/;
                                                const styleMatch = currentLine.match(styleRegex);
                                                
                                                if (styleMatch) {
                                                    // Extract the existing attributes and content
                                                    const existingAttrs = styleMatch[1] || '';
                                                    const content = styleMatch[2] || '';
                                                    
                                                    // Check if there's already a style attribute
                                                    if (existingAttrs.includes('style=')) {
                                                        // Update the existing style attribute with the new line-height
                                                        const updatedAttrs = existingAttrs.replace(/style="([^"]*)"/, (match, styles) => {
                                                            // If line-height already exists, update it
                                                            if (styles.includes('line-height:')) {
                                                                return `style="${styles.replace(/line-height:[^;]*;?/, `line-height: ${prop.value}px;`)}"`;  
                                                            } else {
                                                                // Add line-height to existing styles
                                                                return `style="${styles}line-height: ${prop.value}px;"`;  
                                                            }
                                                        });
                                                        lines[targetLineIndex] = `<div${updatedAttrs}>${content}</div>`;
                                                    } else {
                                                        // Add a new style attribute
                                                        lines[targetLineIndex] = `<div${existingAttrs} style="line-height: ${prop.value}px;">${content}</div>`;
                                                    }
                                                } else {
                                                    // This shouldn't happen with our preparation, but just in case
                                                    lines[targetLineIndex] = `<div style="line-height: ${prop.value}px;">${currentLine.replace(/<\/?div[^>]*>/g, '')}</div>`;
                                                }
                                            }
                                        });
                                        
                                        // Update the field content
                                        fieldContent.innerHTML = lines.join('');
                                    }
                                }
                            }
                        }
                    }
                }
            } catch (error) {
                console.error(`Error creating object ${objectData.name}:`, error);
            }
        }
        
        // Initialize the card
        WebTalkObjects.initializeCard();
        
        // Apply all formatting commands to fields
        this.applyAllFieldFormattingCommands();
        
        // CRITICAL FIX: Post-load sweep to ensure all image objects have data-type="image"
        console.log('Running post-load sweep to fix image data-type attributes...');
        document.querySelectorAll('[data-type^="image/"]').forEach(element => {
            console.log(`Post-load fix: Changing data-type from '${element.dataset.type}' to 'image' for element:`, element);
            element.dataset.type = 'image';
        });
        
        // Restore the scroll position if available
        if (stackData.scrollPosition) {
            window.scrollTo(stackData.scrollPosition.x, stackData.scrollPosition.y);
        }
        
        // Get the card script
        const cardScript = WebTalkObjects.scripts.get('card') || '';
        
        // Check for openstack and opencard handlers
        // We're looking for the full handler pattern with 'on openstack' and 'end openstack'
        const hasOpenstackHandler = (
            (cardScript.includes('on openstack') && cardScript.includes('end openstack')) || 
            (cardScript.includes('on OpenStack') && cardScript.includes('end OpenStack'))
        );
        const hasOpencardHandler = (
            cardScript.includes('on opencard') || 
            cardScript.includes('on OpenCard')
        );
        
        // Only proceed if we have at least one handler
        if (hasOpenstackHandler || hasOpencardHandler) {
            try {
                // Wait to ensure JSON stack data is fully loaded
                setTimeout(() => {
                    if (window.webTalkApp && window.webTalkApp.interpreter) {
                        // If we have an openstack handler, execute it first
                        if (hasOpenstackHandler) {
                            console.log('Executing openstack handler');
                            window.webTalkApp.interpreter.executeObjectScript('card', 'openstack');
                            
                            // Then execute opencard handler after a small delay
                            if (hasOpencardHandler) {
                                setTimeout(() => {
                                    console.log('Executing opencard handler');
                                    window.webTalkApp.interpreter.executeObjectScript('card', 'opencard');
                                }, 50); // Small delay to ensure openstack completes first
                            }
                        } else if (hasOpencardHandler) {
                            // If no openstack handler but we have opencard, just execute that
                            console.log('Executing opencard handler');
                            window.webTalkApp.interpreter.executeObjectScript('card', 'opencard');
                        }
                    }
                }, 500); // Wait for everything to be fully loaded
            } catch (error) {
                console.error('Error executing stack/card handlers:', error);
            }
        }
        
        // Restore the previous current card context
        window.currentCard = previousCurrentCard;

        // Refresh overview palette if visible
        if (window.refreshObjectOverview) {
            window.refreshObjectOverview();
        }
    }
    
    // Helper method to apply all formatting commands to all fields
    static applyAllFieldFormattingCommands() {
        console.log('Applying formatting commands to all fields...');
        
        // Get all field names from the customProperties Map
        for (const [objectName, properties] of WebTalkObjects.customProperties.entries()) {
            // Check if this object has formattingCommands
            if (properties.has('formattingCommands')) {
                const formattingCommands = properties.get('formattingCommands');
                
                // Skip if no formatting commands
                if (!formattingCommands || formattingCommands.length === 0) continue;
                
                // Get the element to verify it's a field
                const element = WebTalkObjects.objects.get(objectName);
                if (!element || element.dataset.type !== 'field') continue;
                
                console.log(`Applying ${formattingCommands.length} formatting commands to field "${objectName}"`);
                
                // Execute each formatting command
                for (const cmd of formattingCommands) {
                    try {
                        // Use the interpreter to execute the command
                        if (window.webTalkApp && window.webTalkApp.interpreter) {
                            window.webTalkApp.interpreter.interpret(cmd);
                        }
                    } catch (error) {
                        console.error(`Error executing formatting command for field "${objectName}": ${cmd}`, error);
                    }
                }
            }
        }
    }
    
    // Helper method to apply visual properties
    static applyVisualProperties(objectName) {
        const element = WebTalkObjects.objects.get(objectName);
        const customProps = WebTalkObjects.customProperties.get(objectName);
        
        if (!element || !customProps) return;
        
        // No special handling for foregroundColor here - it's handled in getObjectProperty
        
        // Define special properties that need custom handling
        const specialProperties = {
            'textContent': (value) => {
                if (element.dataset.type === 'field') {
                    element.querySelector('.field-content').textContent = value;
                }
            },
            'visibleLines': (value) => {
                if (element.dataset.type === 'field') {
                    const fieldContent = element.querySelector('.field-content');
                    fieldContent.style.height = `${parseInt(value) * 20}px`;
                }
            },
            'buttonText': (value) => {
                if (element.dataset.type === 'button') {
                    element.querySelector('.button-label').textContent = value;
                }
            },
            'lineHeight': (value) => {
                if (element.dataset.type === 'field') {
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Ensure value is treated as a string and handle both number and string formats
                        const valueStr = value.toString();
                        fieldContent.style.lineHeight = valueStr.includes('px') ? valueStr : `${valueStr}px`;
                    }
                }
            },
            // Also handle lowercase version for compatibility
            'lineheight': (value) => {
                if (element.dataset.type === 'field') {
                    const fieldContent = element.querySelector('.field-content');
                    if (fieldContent) {
                        // Ensure value is treated as a string and handle both number and string formats
                        const valueStr = value.toString();
                        fieldContent.style.lineHeight = valueStr.includes('px') ? valueStr : `${valueStr}px`;
                    }
                }
            }
        };
        
        // Define CSS property mappings (property name to CSS property)
        const cssPropertyMappings = {
            // Support both camelCase and lowercase versions for backward compatibility
            'foregroundColor': 'color',
            'foregroundcolor': 'color',
            'backgroundColor': 'backgroundColor',
            'backgroundcolor': 'backgroundColor',
            'borderColor': 'borderColor',
            'bordercolor': 'borderColor',
            'borderWidth': 'borderWidth',
            'borderwidth': 'borderWidth',
            'borderStyle': 'borderStyle',
            'borderstyle': 'borderStyle',
            'fontFamily': 'fontFamily',
            'fontfamily': 'fontFamily',
            'fontSize': 'fontSize',
            'fontsize': 'fontSize',
            'fontStyle': 'fontStyle',
            'fontstyle': 'fontStyle',
            'fontWeight': 'fontWeight',
            'fontweight': 'fontWeight',
            'textAlign': 'textAlign',
            'textalign': 'textAlign',
            'textDecoration': 'textDecoration',
            'textdecoration': 'textDecoration',
            'textTransform': 'textTransform',
            'texttransform': 'textTransform',
            'lineHeight': 'lineHeight',
            'lineheight': 'lineHeight',
            'opacity': 'opacity',
            'zIndex': 'zIndex',
            'zindex': 'zIndex',
            'paddingTop': 'paddingTop',
            'paddingtop': 'paddingTop',
            'paddingRight': 'paddingRight',
            'paddingright': 'paddingRight',
            'paddingBottom': 'paddingBottom',
            'paddingbottom': 'paddingBottom',
            'paddingLeft': 'paddingLeft',
            'paddingleft': 'paddingLeft',
            'marginTop': 'marginTop',
            'margintop': 'marginTop',
            'marginRight': 'marginRight',
            'marginright': 'marginRight',
            'marginBottom': 'marginBottom',
            'marginbottom': 'marginBottom',
            'marginLeft': 'marginLeft',
            'marginleft': 'marginLeft'
        };
        
        // Process all properties in the customProps map
        customProps.forEach((value, key) => {
            // Skip null or undefined values
            if (value === null || value === undefined) return;
            
            // Convert key to lowercase for case-insensitive comparison
            const lowerKey = key.toLowerCase();
            
            // Handle special properties first
            if (specialProperties[lowerKey]) {
                specialProperties[lowerKey](value);
                return;
            }
            
            // Special handling for foregroundColor to ensure it overrides defaults
            if (lowerKey === 'foregroundcolor') {
                // For all element types, apply color with !important to override defaults
                element.style.setProperty('color', value, 'important');
                
                // For fields, we need extra handling to ensure the content gets the color
                if (element.dataset.type === 'field') {
                    // If this is a field, force a redraw by toggling a property
                    const currentDisplay = element.style.display;
                    element.style.display = 'none';
                    // Force a reflow
                    void element.offsetHeight;
                    element.style.display = currentDisplay;
                }
                return;
            }
            
            // Handle CSS properties
            // First check if we have a mapping for this property
            let cssProperty = cssPropertyMappings[lowerKey];
            
            // If no mapping exists, try to use the original key as a CSS property
            if (!cssProperty) {
                // Convert from camelCase to CSS property if needed
                // e.g., 'textAlign' becomes 'text-align'
                cssProperty = key.replace(/([A-Z])/g, '-$1').toLowerCase();
                
                // Check if this is a valid CSS property
                if (cssProperty in document.documentElement.style ||
                    key in document.documentElement.style) {
                    // Use the original key as it might be a valid camelCase CSS property
                    cssProperty = key;
                } else {
                    // Not a CSS property, try to handle as a custom property
                    try {
                        // For any other property, try to set it as a custom attribute
                        element.setAttribute(`data-${lowerKey}`, value);
                    } catch (e) {
                        console.warn(`Could not apply property ${key} to element ${objectName}`);
                    }
                    return;
                }
            }
            
            // Apply the CSS property
            try {
                element.style[cssProperty] = value;
            } catch (e) {
                console.warn(`Error applying CSS property ${cssProperty} with value ${value} to element ${objectName}`);
            }
        });
    }
}

// Extend the interpreter to handle the 'save stack as' command
(function() {
    // Wait for the interpreter to be available
    const extendInterpreter = function() {
        if (window.InterpreterClass && InterpreterClass.prototype) {
            // Store the original interpret method
            const originalInterpret = InterpreterClass.prototype.interpret;
            
            // Override the interpret method to handle file operations
            InterpreterClass.prototype.interpret = async function(command) {
                // Add a new command specifically for variable-based saving
                // Special command to list all variables (for debugging)
                if (command.toLowerCase() === 'list variables') {
                    let varList = 'Variables:\n';
                    for (const [key, value] of this.variables.entries()) {
                        varList += `${key} = "${value}"\n`;
                    }
                    return varList;
                }
                
                // Special command to prompt for filename and save stack
                // this is when used via the floating "edit-palette" to workaround the 'save stack as [variable]' problem
                // REMEMBER: DO NOT REMOVE THIS FUNCTION. It is not a duplicate.
                if (command.toLowerCase() === 'save stack prompt') {
                    // Use the built-in ask dialog to get the filename
                    const originalInterpret = this.interpret;
                    try {
                        // Check if we're running online (http/https)
                        const isOnline = window.location.protocol === 'http:' || window.location.protocol === 'https:';
                        
                        // First run the ask command with appropriate message
                        if (isOnline) {
                            // Online: only JSON format is supported
                            await originalInterpret.call(this, 'ask "Save json stack as:"');
                        } else {
                            // Local: both formats supported, WTS is default
                            await originalInterpret.call(this, 'ask "Save stack as:" & return & "(hold shift to save as older JSON format)"');
                        }
                        
                        // Get the result from 'it'
                        if (this.it && this.it.trim() !== '') {
                            let filename = String(this.it);
                            
                            // Determine format based on environment and shift key
                            if (isOnline) {
                                // Online: always save as JSON
                                if (!filename.endsWith('.json') && !filename.endsWith('.wts')) {
                                    filename += '.json';
                                } else if (filename.endsWith('.wts')) {
                                    filename = filename.replace(/\.wts$/, '.json');
                                }
                                console.log('Saving as JSON (online mode):', filename);
                            } else {
                                // Local: check shift key to determine format
                                // If shift is down, save as JSON; otherwise save as WTS (default)
                                console.log('Shift key state:', this.shiftKeyDown);
                                if (this.shiftKeyDown) {
                                    // Ensure .json extension
                                    if (!filename.endsWith('.json') && !filename.endsWith('.wts')) {
                                        filename += '.json';
                                    } else if (filename.endsWith('.wts')) {
                                        filename = filename.replace(/\.wts$/, '.json');
                                    }
                                    console.log('Saving as JSON:', filename);
                                } else {
                                    // Ensure .wts extension (default)
                                    if (!filename.endsWith('.wts') && !filename.endsWith('.json')) {
                                        filename += '.wts';
                                    } else if (filename.endsWith('.json')) {
                                        filename = filename.replace(/\.json$/, '.wts');
                                    }
                                    console.log('Saving as WTS:', filename);
                                }
                            }
                            
                            await WebTalkFileOperations.saveStackAs(filename);
                            return `Stack saved as: ${filename}`;
                        } else {
                            return 'Save cancelled';
                        }
                    } catch (error) {
                        throw new Error(`Failed to save stack: ${error.message}`);
                    }
                }
                
                // Command to clear the message output area
                // First remove any comments (anything after --)
                let cleanCommand = command;
                const commentIndex = command.indexOf('--');
                if (commentIndex !== -1) {
                    cleanCommand = command.substring(0, commentIndex).trim();
                }
                
                if (cleanCommand.toLowerCase() === 'clear message box') {
                    // Get the message output element
                    const messageOutput = document.getElementById('message-output');
                    if (messageOutput) {
                        // Clear the content (same as clicking the clear-output button)
                        messageOutput.innerHTML = '';
                        return ''; // Return empty string to avoid adding output
                    } else {
                        throw new Error('Message output element not found');
                    }
                }

                const saveWithVarMatch = command.match(/^save\s+stack\s+with\s+(\w+)$/i);
                if (saveWithVarMatch) {
                    const varName = saveWithVarMatch[1];
                    
                    // Check if the variable exists
                    if (this.variables.has(varName)) {
                        const filename = String(this.variables.get(varName));
                        try {
                            await WebTalkFileOperations.saveStackAs(filename);
                            return `Stack saved as: ${filename}`;
                        } catch (error) {
                            throw new Error(`Failed to save stack: ${error.message}`);
                        }
                    } else {
                        // Try to get the variable from the global interpreter
                        if (window.interpreter && window.interpreter.variables && 
                            window.interpreter.variables.has(varName)) {
                            const filename = String(window.interpreter.variables.get(varName));
                            try {
                                await WebTalkFileOperations.saveStackAs(filename);
                                return `Stack saved as: ${filename}`;
                            } catch (error) {
                                throw new Error(`Failed to save stack: ${error.message}`);
                            }
                        } else {
                            throw new Error(`Variable '${varName}' not found`);
                        }
                    }
                }
                
                // Special test command to check variable values
                const testVarMatch = command.match(/^test\s+var\s+(\w+)$/i);
                if (testVarMatch) {
                    const varName = testVarMatch[1];
                    if (this.variables.has(varName)) {
                        return `Variable ${varName} = "${this.variables.get(varName)}"`;
                    } else {
                        return `Variable ${varName} not found`;
                    }
                }
                
                // Check for 'save stack as' command
                const saveStackMatch = command.match(/^save\s+stack\s+as\s+(?:"([^"]+)"|'([^']+)'|(\S+))$/i);
                if (saveStackMatch) {
                    // Get the filename from the match
                    let filename = saveStackMatch[1] || saveStackMatch[2] || saveStackMatch[3];
                    
                    // Special handling for 'it' variable
                    if (filename === 'it' && this.it !== undefined) {
                        filename = String(this.it);
                    }
                    // Check if it's a variable name
                    else if (this.variables && this.variables.has(filename)) {
                        filename = String(this.variables.get(filename));
                    }
                    
                    try {
                        await WebTalkFileOperations.saveStackAs(filename);
                        return `Stack saved as: ${filename}`;
                    } catch (error) {
                        throw new Error(`Failed to save stack: ${error.message}`);
                    }
                }
                
                // Check for 'load stack' command
                const loadStackMatch = command.match(/^load\s+stack\s+(?:"([^"]+)"|'([^']+)'|(\S+))$/i);
                if (loadStackMatch) {
                    const filename = loadStackMatch[1] || loadStackMatch[2] || loadStackMatch[3];
                    try {
                        await WebTalkFileOperations.loadStack(filename);
                        return `Stack loaded from: ${filename}`;
                    } catch (error) {
                        throw new Error(`Failed to load stack: ${error.message}`);
                    }
                }
                
                // If not a file operation command, use the original interpret method
                return originalInterpret.call(this, command);
            };
            
            console.log('Successfully extended the InterpreterClass with file operations');
            return true;
        }
        return false;
    };
    
    // Try to extend immediately
    if (!extendInterpreter()) {
        // If not successful, wait for the interpreter to be loaded
        let attempts = 0;
        const maxAttempts = 50;
        const interval = setInterval(function() {
            attempts++;
            if (extendInterpreter() || attempts >= maxAttempts) {
                clearInterval(interval);
                if (attempts >= maxAttempts) {
                    console.error('Failed to extend InterpreterClass with file operations after maximum attempts');
                }
            }
        }, 10);
        
        // Also try to extend on DOMContentLoaded and window.onload events
        document.addEventListener('DOMContentLoaded', function() {
            extendInterpreter();
        });
        
        window.addEventListener('load', function() {
            extendInterpreter();
        });
        
        // Immediate retry
        setTimeout(function() {
            extendInterpreter();
        }, 0);
    }
})();
