/**
 * Extended HyperTalk Functions
 * This file contains additional functions that extend the core HyperTalk interpreter.
 */

// Create a namespace for extended functions
if (typeof window !== 'undefined' && !window.HyperTalkExtended) {
    window.HyperTalkExtended = {};
}

// Preload battery information when the script loads
(function() {
    if ('getBattery' in navigator && !window.batteryInfo) {
        navigator.getBattery().then(battery => {
            if (!battery) {
                window.batteryInfo = "false";
                return;
            }
            
            // Store battery information
            const chargingStatus = battery.charging ? "charging" : "discharging";
            const batteryLevel = Math.round(battery.level * 100) + "%";
            window.batteryInfo = `true, ${chargingStatus}, ${batteryLevel}`;
            
            // Set up event listeners
            battery.addEventListener('levelchange', function() {
                const newLevel = Math.round(battery.level * 100) + "%";
                const chargingState = battery.charging ? "charging" : "discharging";
                window.batteryInfo = `true, ${chargingState}, ${newLevel}`;
            });
            
            battery.addEventListener('chargingchange', function() {
                const level = Math.round(battery.level * 100) + "%";
                const newChargingState = battery.charging ? "charging" : "discharging";
                window.batteryInfo = `true, ${newChargingState}, ${level}`;
            });
        }).catch(error => {
            console.log("Battery preload error: " + error);
            window.batteryInfo = "false";
        });
    }
})();

// Define the extended expression evaluator
HyperTalkExtended.evaluateExtendedExpression = function(expr) {
    if (!expr) return undefined;
    expr = expr.trim();
    
    // Handle format function
    const formatMatch = expr.match(/^format\s*\(\s*(.+?)\s*,\s*(.+?)\s*\)$/i);
    if (formatMatch) {
        const [_, number, formatStr] = formatMatch;
        const numberValue = Number(this.evaluateExpression(number));
        const formatValue = this.getTextValue(formatStr);
        
        if (isNaN(numberValue)) {
            throw new Error('format: first argument must be a number');
        }
        
        try {
            // Parse the format string
            const decimalPlaces = formatValue.match(/^#*\.#*$/i) ? 
                (formatValue.split('.')[1] || '').length : 
                0;
            
            // Format the number with specified decimal places
            return numberValue.toFixed(decimalPlaces);
        } catch (e) {
            throw new Error('format: invalid format string');
        }
    }

    // Handle baseConvert function
    const baseConvertMatch = expr.match(/^baseConvert\s*\(\s*(.+?)\s*,\s*(.+?)\s*,\s*(.+?)\s*\)$/i);
    if (baseConvertMatch) {
        const [_, number, fromBase, toBase] = baseConvertMatch;
        const numberValue = this.getTextValue(number);
        const fromBaseValue = Number(this.evaluateExpression(fromBase));
        const toBaseValue = Number(this.evaluateExpression(toBase));
        
        if (fromBaseValue < 2 || fromBaseValue > 36) {
            throw new Error('baseConvert: fromBase must be between 2 and 36');
        }
        
        if (toBaseValue < 2 || toBaseValue > 36) {
            throw new Error('baseConvert: toBase must be between 2 and 36');
        }
        
        try {
            // Convert from source base to decimal, then to target base
            const decimal = parseInt(numberValue, fromBaseValue);
            if (isNaN(decimal)) {
                throw new Error('baseConvert: invalid number for the given base');
            }
            return decimal.toString(toBaseValue);
        } catch (e) {
            throw new Error('baseConvert: ' + e.message);
        }
    }

    // Handle pi function
    const piMatch = expr.match(/^pi\s*\(\s*\)$/i);
    if (piMatch) {
        return Math.PI.toString();
    }
    // Handle variance function
    const varianceMatch = expr.match(/^variance\s*\(\s*(.+?)\s*\)$/i);
    if (varianceMatch) {
        const [_, list] = varianceMatch;
        const listValue = this.getTextValue(list);
        const numbers = listValue.split(',').map(item => {
            const num = Number(item.trim());
            if (isNaN(num)) {
                throw new Error('variance: all items must be numbers');
            }
            return num;
        });
        
        if (numbers.length < 2) {
            throw new Error('variance: need at least two numbers');
        }
        
        // Calculate mean
        const mean = numbers.reduce((sum, num) => sum + num, 0) / numbers.length;
        
        // Calculate variance
        const variance = numbers.reduce((sum, num) => sum + Math.pow(num - mean, 2), 0) / numbers.length;
        
        return variance.toString();
    }

    // Handle removeHTML function
    const removeHTMLMatch = expr.match(/^removeHTML\s*\(\s*(.+?)\s*\)$/i);
    if (removeHTMLMatch) {
        const [_, target] = removeHTMLMatch;
        const targetValue = this.getTextValue(target);
        
        // Create a temporary div to handle the HTML parsing
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = targetValue;
        
        // Return the text content without HTML tags
        return tempDiv.textContent;
    }
    
    // Handle lastKeys function - returns the last N keys pressed as a comma-separated list
    const lastKeysMatch = expr.match(/^lastKeys\s*\(\s*(.+?)\s*\)$/i);
    if (lastKeysMatch) {
        const [_, countStr] = lastKeysMatch;
        let count = Number(this.evaluateExpression(countStr));
        
        // Validate count
        if (isNaN(count) || count < 1) {
            throw new Error('lastKeys: count must be a positive number');
        }
        
        // Cap at MAX_KEY_HISTORY (default 20)
        count = Math.min(count, window.MAX_KEY_HISTORY || 20);
        
        // Get the last N keys from the history array
        if (!window.keyHistory) {
            // If keyHistory doesn't exist yet, return empty string
            return '';
        }
        
        // Get the last N keys (or fewer if not enough keys pressed)
        const startIndex = Math.max(0, window.keyHistory.length - count);
        const keys = window.keyHistory.slice(startIndex);
        
        // Return as comma-separated list
        return keys.join(',');
    }
    
    // Handle removeDuplicates function
    const removeDuplicatesMatch = expr.match(/^removeDuplicates\s*\(\s*(.+?)\s*\)$/i);
    if (removeDuplicatesMatch) {
        const [_, target] = removeDuplicatesMatch;
        const targetValue = this.getTextValue(target);
        
        // Split the input by newlines
        const lines = targetValue.split('\n');
        
        // Create a map to track unique lines and their original order
        const uniqueLines = [];
        const seenLines = new Set();
        
        // Process each line
        lines.forEach(line => {
            if (!seenLines.has(line)) {
                // First occurrence of this line
                seenLines.add(line);
                uniqueLines.push(line);
            }
            // Duplicates are ignored
        });
        
        // Return only unique lines
        return uniqueLines.join('\n');
    }
    
    // Handle standardDeviation function
    const stdDevMatch = expr.match(/^standardDeviation\s*\(\s*(.+?)\s*\)$/i);
    if (stdDevMatch) {
        const [_, list] = stdDevMatch;
        const listValue = this.getTextValue(list);
        const numbers = listValue.split(',').map(item => {
            const num = Number(item.trim());
            if (isNaN(num)) {
                throw new Error('standardDeviation: all items must be numbers');
            }
            return num;
        });
        
        if (numbers.length < 2) {
            throw new Error('standardDeviation: need at least two numbers');
        }
        
        // Calculate mean
        const mean = numbers.reduce((sum, num) => sum + num, 0) / numbers.length;
        
        // Calculate variance
        const variance = numbers.reduce((sum, num) => sum + Math.pow(num - mean, 2), 0) / numbers.length;
        
        // Calculate standard deviation (square root of variance)
        return Math.sqrt(variance).toString();
    }
    // Handle union function
    const unionMatch = expr.match(/^union\s*\(\s*("[^"]*"|[^,)]+)\s*,\s*("[^"]*"|[^,)]+)\s*\)$/i);
    if (unionMatch) {
        const [_, list1, list2] = unionMatch;
        const list1Value = this.getTextValue(list1);
        const list2Value = this.getTextValue(list2);
        
        const items1 = list1Value.split(',').map(item => item.trim());
        const items2 = list2Value.split(',').map(item => item.trim());
        
        // Create a union of the two lists (no duplicates)
        const unionSet = new Set([...items1, ...items2]);
        
        return Array.from(unionSet).join(',');
    }

    // Handle intersection function
    const intersectionMatch = expr.match(/^intersection\s*\(\s*("[^"]*"|[^,)]+)\s*,\s*("[^"]*"|[^,)]+)\s*\)$/i);
    if (intersectionMatch) {
        const [_, list1, list2] = intersectionMatch;
        const list1Value = this.getTextValue(list1);
        const list2Value = this.getTextValue(list2);
        
        const items1 = list1Value.split(',').map(item => item.trim());
        const items2 = list2Value.split(',').map(item => item.trim());
        
        // Create a set for faster lookups
        const set2 = new Set(items2);
        
        // Find items that exist in both lists
        const intersection = items1.filter(item => set2.has(item));
        
        return intersection.join(',');
    }

    // Handle split function
    const splitMatch = expr.match(/^split\s*\(\s*(.+?)\s*,\s*(.+?)\s*\)$/i);
    if (splitMatch) {
        const [_, text, delimiter] = splitMatch;
        const textValue = this.getTextValue(text);
        const delimValue = this.getTextValue(delimiter);
        
        // Split the text by the delimiter
        return textValue.split(delimValue).join(',');
    }

    // Handle join/combine function (combine is a synonym for join)
    const joinMatch = expr.match(/^(?:join|combine)\s*\(\s*("[^"]*"|[^,)]+)\s*,\s*("[^"]*"|[^,)]+)\s*\)$/i);
    if (joinMatch) {
        const [_, list, delimiter] = joinMatch;
        const listValue = this.getTextValue(list);
        const delimValue = this.getTextValue(delimiter);
        
        // Join the list items with the delimiter
        return listValue.split(',').join(delimValue);
    }
    // Handle base64Encode function
    const base64EncodeMatch = expr.match(/^base64Encode\s*\(\s*(.+?)\s*\)$/i);
    if (base64EncodeMatch) {
        const [_, text] = base64EncodeMatch;
        const textValue = this.getTextValue(text);
        
        try {
            // Use btoa for base64 encoding
            return btoa(textValue);
        } catch (e) {
            throw new Error('base64Encode: ' + e.message);
        }
    }

    // Handle base64Decode function
    const base64DecodeMatch = expr.match(/^base64Decode\s*\(\s*(.+?)\s*\)$/i);
    if (base64DecodeMatch) {
        const [_, text] = base64DecodeMatch;
        const textValue = this.getTextValue(text);
        
        try {
            // Use atob for base64 decoding
            return atob(textValue);
        } catch (e) {
            throw new Error('base64Decode: invalid base64 string');
        }
    }

    // Handle numToCodepoint function
    const numToCodepointMatch = expr.match(/^numToCodepoint\s*\(\s*(.+?)\s*\)$/i);
    if (numToCodepointMatch) {
        const [_, number] = numToCodepointMatch;
        const numValue = Number(this.evaluateExpression(number));
        
        // Check if the number is within valid Unicode range (0x0 to 0x10FFFF)
        if (isNaN(numValue) || numValue < 0 || numValue > 0x10FFFF) {
            throw new Error('numToCodepoint: value must be a number between 0x000000 and 0x10FFFF');
        }
        
        // Convert the number to its Unicode character
        return String.fromCodePoint(numValue);
    }
    
    // Handle codepointToNum function
    const codepointToNumMatch = expr.match(/^codepointToNum\s*\(\s*(.+?)\s*\)$/i);
    if (codepointToNumMatch) {
        const [_, charExpr] = codepointToNumMatch;
        const charValue = this.evaluateExpression(charExpr);
        
        // Check if input is a string
        if (typeof charValue !== 'string') {
            throw new Error('codepointToNum: argument must be a character');
        }
        
        // Check if input contains exactly one codepoint
        if ([...charValue].length !== 1) {
            throw new Error('codepointToNum: argument must contain exactly one character');
        }
        
        // Get the codepoint value
        const codepoint = charValue.codePointAt(0);
        
        // Return the integer value
        return codepoint.toString();
    }
    
    // Handle md5Digest function
    const md5DigestMatch = expr.match(/^md5Digest\s*\(\s*(.+?)\s*\)$/i);
    if (md5DigestMatch) {
        const [_, text] = md5DigestMatch;
        const textValue = this.getTextValue(text);
        
        // MD5 implementation
        function md5(string) {
            function cmn(q, a, b, x, s, t) {
                a = add32(add32(a, q), add32(x, t));
                return add32((a << s) | (a >>> (32 - s)), b);
            }

            function ff(a, b, c, d, x, s, t) {
                return cmn((b & c) | ((~b) & d), a, b, x, s, t);
            }

            function gg(a, b, c, d, x, s, t) {
                return cmn((b & d) | (c & (~d)), a, b, x, s, t);
            }

            function hh(a, b, c, d, x, s, t) {
                return cmn(b ^ c ^ d, a, b, x, s, t);
            }

            function ii(a, b, c, d, x, s, t) {
                return cmn(c ^ (b | (~d)), a, b, x, s, t);
            }

            function md5cycle(x, k) {
                let a = x[0], b = x[1], c = x[2], d = x[3];

                a = ff(a, b, c, d, k[0], 7, -680876936);
                d = ff(d, a, b, c, k[1], 12, -389564586);
                c = ff(c, d, a, b, k[2], 17, 606105819);
                b = ff(b, c, d, a, k[3], 22, -1044525330);
                a = ff(a, b, c, d, k[4], 7, -176418897);
                d = ff(d, a, b, c, k[5], 12, 1200080426);
                c = ff(c, d, a, b, k[6], 17, -1473231341);
                b = ff(b, c, d, a, k[7], 22, -45705983);
                a = ff(a, b, c, d, k[8], 7, 1770035416);
                d = ff(d, a, b, c, k[9], 12, -1958414417);
                c = ff(c, d, a, b, k[10], 17, -42063);
                b = ff(b, c, d, a, k[11], 22, -1990404162);
                a = ff(a, b, c, d, k[12], 7, 1804603682);
                d = ff(d, a, b, c, k[13], 12, -40341101);
                c = ff(c, d, a, b, k[14], 17, -1502002290);
                b = ff(b, c, d, a, k[15], 22, 1236535329);

                a = gg(a, b, c, d, k[1], 5, -165796510);
                d = gg(d, a, b, c, k[6], 9, -1069501632);
                c = gg(c, d, a, b, k[11], 14, 643717713);
                b = gg(b, c, d, a, k[0], 20, -373897302);
                a = gg(a, b, c, d, k[5], 5, -701558691);
                d = gg(d, a, b, c, k[10], 9, 38016083);
                c = gg(c, d, a, b, k[15], 14, -660478335);
                b = gg(b, c, d, a, k[4], 20, -405537848);
                a = gg(a, b, c, d, k[9], 5, 568446438);
                d = gg(d, a, b, c, k[14], 9, -1019803690);
                c = gg(c, d, a, b, k[3], 14, -187363961);
                b = gg(b, c, d, a, k[8], 20, 1163531501);
                a = gg(a, b, c, d, k[13], 5, -1444681467);
                d = gg(d, a, b, c, k[2], 9, -51403784);
                c = gg(c, d, a, b, k[7], 14, 1735328473);
                b = gg(b, c, d, a, k[12], 20, -1926607734);

                a = hh(a, b, c, d, k[5], 4, -378558);
                d = hh(d, a, b, c, k[8], 11, -2022574463);
                c = hh(c, d, a, b, k[11], 16, 1839030562);
                b = hh(b, c, d, a, k[14], 23, -35309556);
                a = hh(a, b, c, d, k[1], 4, -1530992060);
                d = hh(d, a, b, c, k[4], 11, 1272893353);
                c = hh(c, d, a, b, k[7], 16, -155497632);
                b = hh(b, c, d, a, k[10], 23, -1094730640);
                a = hh(a, b, c, d, k[13], 4, 681279174);
                d = hh(d, a, b, c, k[0], 11, -358537222);
                c = hh(c, d, a, b, k[3], 16, -722521979);
                b = hh(b, c, d, a, k[6], 23, 76029189);
                a = hh(a, b, c, d, k[9], 4, -640364487);
                d = hh(d, a, b, c, k[12], 11, -421815835);
                c = hh(c, d, a, b, k[15], 16, 530742520);
                b = hh(b, c, d, a, k[2], 23, -995338651);

                a = ii(a, b, c, d, k[0], 6, -198630844);
                d = ii(d, a, b, c, k[7], 10, 1126891415);
                c = ii(c, d, a, b, k[14], 15, -1416354905);
                b = ii(b, c, d, a, k[5], 21, -57434055);
                a = ii(a, b, c, d, k[12], 6, 1700485571);
                d = ii(d, a, b, c, k[3], 10, -1894986606);
                c = ii(c, d, a, b, k[10], 15, -1051523);
                b = ii(b, c, d, a, k[1], 21, -2054922799);
                a = ii(a, b, c, d, k[8], 6, 1873313359);
                d = ii(d, a, b, c, k[15], 10, -30611744);
                c = ii(c, d, a, b, k[6], 15, -1560198380);
                b = ii(b, c, d, a, k[13], 21, 1309151649);
                a = ii(a, b, c, d, k[4], 6, -145523070);
                d = ii(d, a, b, c, k[11], 10, -1120210379);
                c = ii(c, d, a, b, k[2], 15, 718787259);
                b = ii(b, c, d, a, k[9], 21, -343485551);

                x[0] = add32(a, x[0]);
                x[1] = add32(b, x[1]);
                x[2] = add32(c, x[2]);
                x[3] = add32(d, x[3]);
            }

            function md5blk(s) {
                let i, md5blks = [];
                for (i = 0; i < 64; i += 4) {
                    md5blks[i >> 2] = s.charCodeAt(i) + (s.charCodeAt(i + 1) << 8) + (s.charCodeAt(i + 2) << 16) + (s.charCodeAt(i + 3) << 24);
                }
                return md5blks;
            }

            function md5blk_array(a) {
                let i, md5blks = [];
                for (i = 0; i < 64; i += 4) {
                    md5blks[i >> 2] = a[i] + (a[i + 1] << 8) + (a[i + 2] << 16) + (a[i + 3] << 24);
                }
                return md5blks;
            }

            function md51(s) {
                let n = s.length,
                    state = [1732584193, -271733879, -1732584194, 271733878],
                    i;
                for (i = 64; i <= s.length; i += 64) {
                    md5cycle(state, md5blk(s.substring(i - 64, i)));
                }
                s = s.substring(i - 64);
                let tail = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                for (i = 0; i < s.length; i++)
                    tail[i >> 2] |= s.charCodeAt(i) << ((i % 4) << 3);
                tail[i >> 2] |= 0x80 << ((i % 4) << 3);
                if (i > 55) {
                    md5cycle(state, tail);
                    for (i = 0; i < 16; i++) tail[i] = 0;
                }
                tail[14] = n * 8;
                md5cycle(state, tail);
                return state;
            }

            function md51_array(a) {
                let n = a.length,
                    state = [1732584193, -271733879, -1732584194, 271733878],
                    i;
                for (i = 64; i <= a.length; i += 64) {
                    md5cycle(state, md5blk_array(a.subarray(i - 64, i)));
                }
                a = (i - 64) < a.length ? a.subarray(i - 64) : new Uint8Array(0);
                let tail = new Uint8Array(64),
                    len = a.length;
                for (i = 0; i < len; i++)
                    tail[i] = a[i];
                tail[len] = 0x80;
                if (len > 55) {
                    md5cycle(state, tail.buffer);
                    for (i = 0; i < 16; i++) tail[i] = 0;
                }
                tail[14] = n * 8;
                md5cycle(state, tail);
                return state;
            }

            function hex_chr(val) {
                let v = val.toString(16);
                return v.length < 2 ? '0' + v : v;
            }

            function rhex(n) {
                let s = '',
                    j;
                for (j = 0; j < 4; j++)
                    s += hex_chr((n >> (j * 8 + 4)) & 0x0F) + hex_chr((n >> (j * 8)) & 0x0F);
                return s;
            }

            function hex(x) {
                let i;
                for (i = 0; i < x.length; i++)
                    x[i] = rhex(x[i]);
                return x.join('');
            }

            function add32(a, b) {
                return (a + b) & 0xFFFFFFFF;
            }

            function utf8Encode(string) {
                let utftext = "";
                for (let n = 0; n < string.length; n++) {
                    let c = string.charCodeAt(n);
                    if (c < 128) {
                        utftext += String.fromCharCode(c);
                    } else if ((c > 127) && (c < 2048)) {
                        utftext += String.fromCharCode((c >> 6) | 192);
                        utftext += String.fromCharCode((c & 63) | 128);
                    } else {
                        utftext += String.fromCharCode((c >> 12) | 224);
                        utftext += String.fromCharCode(((c >> 6) & 63) | 128);
                        utftext += String.fromCharCode((c & 63) | 128);
                    }
                }
                return utftext;
            }

            return hex(md51(utf8Encode(string)));
        }
        
        return md5(textValue);
    }

    // Handle sha1Digest function
    const sha1DigestMatch = expr.match(/^sha1Digest\s*\(\s*(.+?)\s*\)$/i);
    if (sha1DigestMatch) {
        const [_, text] = sha1DigestMatch;
        const textValue = this.getTextValue(text);
        
        // SHA1 implementation
        function sha1(str) {
            function utf8Encode(str) {
                let utftext = "";
                for (let n = 0; n < str.length; n++) {
                    let c = str.charCodeAt(n);
                    if (c < 128) {
                        utftext += String.fromCharCode(c);
                    } else if ((c > 127) && (c < 2048)) {
                        utftext += String.fromCharCode((c >> 6) | 192);
                        utftext += String.fromCharCode((c & 63) | 128);
                    } else {
                        utftext += String.fromCharCode((c >> 12) | 224);
                        utftext += String.fromCharCode(((c >> 6) & 63) | 128);
                        utftext += String.fromCharCode((c & 63) | 128);
                    }
                }
                return utftext;
            }

            function rotate_left(n, s) {
                return (n << s) | (n >>> (32 - s));
            }

            let blockstart;
            let i, j;
            let W = new Array(80);
            let H0 = 0x67452301;
            let H1 = 0xEFCDAB89;
            let H2 = 0x98BADCFE;
            let H3 = 0x10325476;
            let H4 = 0xC3D2E1F0;
            let A, B, C, D, E;
            let temp;

            str = utf8Encode(str);
            let str_len = str.length;

            let word_array = [];
            for (i = 0; i < str_len - 3; i += 4) {
                j = str.charCodeAt(i) << 24 | str.charCodeAt(i + 1) << 16 |
                    str.charCodeAt(i + 2) << 8 | str.charCodeAt(i + 3);
                word_array.push(j);
            }

            switch (str_len % 4) {
                case 0:
                    i = 0x080000000;
                    break;
                case 1:
                    i = str.charCodeAt(str_len - 1) << 24 | 0x0800000;
                    break;
                case 2:
                    i = str.charCodeAt(str_len - 2) << 24 | str.charCodeAt(str_len - 1) << 16 | 0x08000;
                    break;
                case 3:
                    i = str.charCodeAt(str_len - 3) << 24 | str.charCodeAt(str_len - 2) << 16 | str.charCodeAt(str_len - 1) << 8 | 0x80;
                    break;
            }

            word_array.push(i);

            while ((word_array.length % 16) != 14) word_array.push(0);

            word_array.push(str_len >>> 29);
            word_array.push((str_len << 3) & 0x0ffffffff);

            for (blockstart = 0; blockstart < word_array.length; blockstart += 16) {
                for (i = 0; i < 16; i++) W[i] = word_array[blockstart + i];
                for (i = 16; i < 80; i++) W[i] = rotate_left(W[i - 3] ^ W[i - 8] ^ W[i - 14] ^ W[i - 16], 1);

                A = H0;
                B = H1;
                C = H2;
                D = H3;
                E = H4;

                for (i = 0; i < 20; i++) {
                    temp = (rotate_left(A, 5) + ((B & C) | (~B & D)) + E + W[i] + 0x5A827999) & 0x0ffffffff;
                    E = D;
                    D = C;
                    C = rotate_left(B, 30);
                    B = A;
                    A = temp;
                }

                for (i = 20; i < 40; i++) {
                    temp = (rotate_left(A, 5) + (B ^ C ^ D) + E + W[i] + 0x6ED9EBA1) & 0x0ffffffff;
                    E = D;
                    D = C;
                    C = rotate_left(B, 30);
                    B = A;
                    A = temp;
                }

                for (i = 40; i < 60; i++) {
                    temp = (rotate_left(A, 5) + ((B & C) | (B & D) | (C & D)) + E + W[i] + 0x8F1BBCDC) & 0x0ffffffff;
                    E = D;
                    D = C;
                    C = rotate_left(B, 30);
                    B = A;
                    A = temp;
                }

                for (i = 60; i < 80; i++) {
                    temp = (rotate_left(A, 5) + (B ^ C ^ D) + E + W[i] + 0xCA62C1D6) & 0x0ffffffff;
                    E = D;
                    D = C;
                    C = rotate_left(B, 30);
                    B = A;
                    A = temp;
                }

                H0 = (H0 + A) & 0x0ffffffff;
                H1 = (H1 + B) & 0x0ffffffff;
                H2 = (H2 + C) & 0x0ffffffff;
                H3 = (H3 + D) & 0x0ffffffff;
                H4 = (H4 + E) & 0x0ffffffff;
            }

            temp = H0.toString(16).padStart(8, '0') +
                   H1.toString(16).padStart(8, '0') +
                   H2.toString(16).padStart(8, '0') +
                   H3.toString(16).padStart(8, '0') +
                   H4.toString(16).padStart(8, '0');

            return temp.toLowerCase();
        }
        
        return sha1(textValue);
    }
    
    // Handle byteOffset function
    const byteOffsetMatch = expr.match(/^byteOffset\s*\(\s*(.+?)\s*,\s*(.+?)(?:\s*,\s*(.+?))?\s*\)$/i);
    if (byteOffsetMatch) {
        const [_, searchByte, sourceString, skipCount] = byteOffsetMatch;
        const searchByteValue = this.getTextValue(searchByte);
        const sourceStringValue = this.getTextValue(sourceString);
        const skipCountValue = skipCount ? Number(this.evaluateExpression(skipCount)) : 0;
        
        if (skipCountValue < 0) {
            throw new Error('byteOffset: skip count must be non-negative');
        }
        
        // Skip the specified number of characters if provided
        const startIndex = Math.min(skipCountValue, sourceStringValue.length);
        const searchString = sourceStringValue.substring(startIndex);
        
        // Find the position of the search byte in the source string
        const position = searchString.indexOf(searchByteValue);
        
        // Return the position (1-based) or 0 if not found
        return position === -1 ? "0" : (position + 1).toString();
    }
    
    // Handle isuppercase function
    const isUppercaseMatch = expr.match(/^isuppercase\s*\(\s*(.+?)\s*\)$/i);
    if (isUppercaseMatch) {
        const [_, text] = isUppercaseMatch;
        const textValue = this.getTextValue(text);
        
        // If the string is empty, return false
        if (!textValue) {
            return "false";
        }
        
        // Check if the string contains only uppercase letters
        const hasUpperCase = /[A-Z]/.test(textValue);
        const hasLowerCase = /[a-z]/.test(textValue);
        
        if (hasUpperCase && !hasLowerCase) {
            return "true";
        } else if (!hasUpperCase && hasLowerCase) {
            return "false";
        } else if (hasUpperCase && hasLowerCase) {
            return "mixed";
        } else {
            // String contains no letters (only numbers, symbols, etc.)
            return "false";
        }
    }
    
    // Handle pixelColor function (US spelling) with optional 'the' prefix
    const pixelColorMatch = expr.match(/^(?:the\s+)?pixelColor\s*\(\s*(.+?)\s*,\s*(.+?)\s*\)$/i);
    if (pixelColorMatch) {
        const [_, xExpr, yExpr] = pixelColorMatch;
        const x = Math.round(Number(this.evaluateExpression(xExpr)));
        const y = Math.round(Number(this.evaluateExpression(yExpr)));
        
        if (isNaN(x) || isNaN(y)) {
            throw new Error('pixelColor: coordinates must be numbers');
        }
        
        return this.getPixelColorAt(x, y);
    }
    
    // Handle pixelColour function (UK spelling) with optional 'the' prefix
    const pixelColourMatch = expr.match(/^(?:the\s+)?pixelColour\s*\(\s*(.+?)\s*,\s*(.+?)\s*\)$/i);
    if (pixelColourMatch) {
        const [_, xExpr, yExpr] = pixelColourMatch;
        const x = Math.round(Number(this.evaluateExpression(xExpr)));
        const y = Math.round(Number(this.evaluateExpression(yExpr)));
        
        if (isNaN(x) || isNaN(y)) {
            throw new Error('pixelColour: coordinates must be numbers');
        }
        
        return this.getPixelColorAt(x, y);
    }
    
    // Handle hasBattery property with optional 'the' prefix
    const hasBatteryMatch = expr.match(/^(?:the\s+)?hasBattery$/i);

    // Handle angleAndDistance function with flexible parameter handling
    // This pattern matches both 4-parameter (x1,y1,x2,y2) and 2-parameter (obj1,obj2) formats
    const angleAndDistanceMatch = expr.match(/^(?:the\s+)?angleAndDistance\s*\(\s*(.+?)(?:\s*,\s*(.+?))?(?:\s*,\s*(.+?))?(?:\s*,\s*(.+?))?(?:\s*,\s*(.+?))?\s*\)$/i);
    if (angleAndDistanceMatch) {
        const [_, param1, param2, param3, param4, param5] = angleAndDistanceMatch;
        
        // Helper function to extract coordinates from an object reference or location
        const extractCoordinates = (objExpr) => {
            if (!objExpr) return null;
            
            // Try to evaluate as a direct expression first
            const directValue = this.evaluateExpression(objExpr);
            
            // Check if it's a numeric value
            if (!isNaN(Number(directValue))) {
                return Number(directValue);
            }
            
            // Check if the evaluated value is a location string (e.g., "100,200")
            if (typeof directValue === 'string' && directValue.includes(',')) {
                const coords = directValue.split(',').map(Number);
                if (coords.length === 2 && !isNaN(coords[0]) && !isNaN(coords[1])) {
                    return coords;
                }
            }
            
            // If not a number or location string, try to get the loc property of the object
            try {
                // Check if it's 'the mouseloc'
                if (objExpr.toLowerCase() === 'the mouseloc') {
                    const mouseLocValue = this.evaluateExpression('the mouseloc');
                    if (mouseLocValue) {
                        return mouseLocValue.split(',').map(Number);
                    }
                }
                
                // Extract object type and name from expressions like 'graphic "A"'
                const objMatch = objExpr.match(/^(\w+)\s+"([^"]+)"$/i);
                if (objMatch) {
                    const [_, objType, objName] = objMatch;
                    // Use WebTalkObjects.getObjectProperty to access object properties
                    const locValue = window.WebTalkObjects.getObjectProperty(objName, 'loc', objType);
                    if (locValue) {
                        // The loc property typically returns "x,y" format
                        return locValue.split(',').map(Number);
                    }
                }
            } catch (e) {
                console.error('Error extracting coordinates:', e);
                // If any error occurs during object property access, just return null
                return null;
            }
            
            return null;
        };
        
        let x1, y1, x2, y2, diagonalFlag;
        
        // Check if we have 2 parameters (objects) or 4 parameters (coordinates)
        if (param3 === undefined || param4 === undefined) {
            // 2-parameter format: angleAndDistance(obj1, obj2, [diagonal])
            
            // First try to handle direct object references
            let coords1, coords2;
            
            // Try to handle direct object references like "graphic \"A\"" or "button \"B\""
            const obj1Match = param1.match(/^(\w+)\s+"([^"]+)"$/i);
            const obj2Match = param2.match(/^(\w+)\s+"([^"]+)"$/i);
            
            if (obj1Match) {
                const [_, objType1, objName1] = obj1Match;
                try {
                    const locValue = window.WebTalkObjects.getObjectProperty(objName1, 'loc', objType1);
                    if (locValue) {
                        coords1 = locValue.split(',').map(Number);
                    }
                } catch (e) {
                    console.error('Error getting location for', objType1, objName1, e);
                }
            }
            
            if (obj2Match) {
                const [_, objType2, objName2] = obj2Match;
                try {
                    const locValue = window.WebTalkObjects.getObjectProperty(objName2, 'loc', objType2);
                    if (locValue) {
                        coords2 = locValue.split(',').map(Number);
                    }
                } catch (e) {
                    console.error('Error getting location for', objType2, objName2, e);
                }
            }
            
            // If direct object reference handling didn't work, try the general approach
            if (!coords1) coords1 = extractCoordinates(param1);
            if (!coords2) coords2 = extractCoordinates(param2);
            
            // Check if we got valid coordinates
            if (!coords1 || !coords2) {
                throw new Error('angleAndDistance: could not extract valid coordinates from objects');
            }
            
            // If coords are arrays, they're from loc properties
            if (Array.isArray(coords1) && Array.isArray(coords2)) {
                [x1, y1] = coords1;
                [x2, y2] = coords2;
                diagonalFlag = param3; // The third parameter would be the diagonal flag
            } else {
                throw new Error('angleAndDistance: invalid object references');
            }
        } else {
            // 4-parameter format: angleAndDistance(x1, y1, x2, y2, [diagonal])
            x1 = Number(this.evaluateExpression(param1));
            y1 = Number(this.evaluateExpression(param2));
            x2 = Number(this.evaluateExpression(param3));
            y2 = Number(this.evaluateExpression(param4));
            diagonalFlag = param5;
        }
        
        // Check if inputs are valid numbers
        if (isNaN(x1) || isNaN(y1) || isNaN(x2) || isNaN(y2)) {
            throw new Error('angleAndDistance: all coordinates must be numbers');
        }
        
        // Calculate differences
        const xDiff = x2 - x1;
        const yDiff = y2 - y1;
        
        // Calculate angle in degrees with 0 at top (north), increasing clockwise
        // In screen coordinates, Y increases downward
        // First get the standard atan2 angle (I think!) - should be counterclockwise from east
        let angle = Math.atan2(-yDiff, xDiff) * (180 / Math.PI);
        
        // Convert to clockwise from north (0 at top)
        // 1. Negate the angle to make it clockwise
        // 2. Add 90 to shift the origin to north
        angle = (-angle + 90) % 360;
        
        // Ensure angle is between 0 and 360
        if (angle < 0) angle += 360;
        
        // Round angle to nearest integer
        angle = Math.round(angle);
        
        // Format the output based on whether diagonal flag is present
        if (diagonalFlag && this.getTextValue(diagonalFlag).toLowerCase() === 'diagonal') {
            // Calculate diagonal distance using Pythagorean theorem
            const diagonalDistance = Math.sqrt(xDiff * xDiff + yDiff * yDiff);
            // Round to 2 decimal places
            const roundedDiagonal = Math.round(diagonalDistance * 100) / 100;
            
            return `${angle},${xDiff},${yDiff},${roundedDiagonal}`;
        } else {
            return `${angle},${xDiff},${yDiff}`;
        }
    }

    // Handle closestControl function with optional 'the' prefix and two syntax forms:
    // 1. closestControl(object)
    // 2. closestControl to object
    const closestControlParenMatch = expr.match(/^(?:the\s+)?closestControl\s*\(\s*(.+?)\s*\)$/i);
    const closestControlToMatch = expr.match(/^(?:the\s+)?closestControl\s+to\s+(.+?)$/i);
    const closestControlMatch = closestControlParenMatch || closestControlToMatch;
    
    if (closestControlMatch) {
        const [_, objExpr] = closestControlMatch;
        console.log(`closestControl: Processing reference object: ${objExpr}`);
        
        // Parse the reference object type and name
        const objMatch = objExpr.match(/^(\w+)\s+"([^"]+)"$/i);
        if (!objMatch) {
            console.error('closestControl: Invalid object reference format');
            return '';
        }
        
        let refObjectType = objMatch[1].toLowerCase();
        // Map shorthand types to full types
        if (refObjectType === 'btn') refObjectType = 'button';
        if (refObjectType === 'fld') refObjectType = 'field';
        
        const refObjectName = objMatch[2];
        console.log(`closestControl: Reference object parsed as: ${refObjectType} "${refObjectName}"`);
        
        // Get the current card element
        const card = document.getElementById('card');
        if (!card) {
            console.error('closestControl: Current card not found');
            return '';
        }
        console.log('closestControl: Found card element:', card.id);
        
        // Find the reference object in the DOM
        const refSelector = `[data-type="${refObjectType}"][data-name="${refObjectName}"]`;
        const refElement = card.querySelector(refSelector);
        if (!refElement) {
            console.error(`closestControl: Reference object ${refObjectType} "${refObjectName}" not found in DOM`);
            return '';
        }
        
        // Get reference object position
        const refRect = refElement.getBoundingClientRect();
        const refCenterX = refRect.left + refRect.width / 2;
        const refCenterY = refRect.top + refRect.height / 2;
        console.log(`closestControl: Reference object center at ${refCenterX},${refCenterY}`);
        
        // Find all controls on the card
        const controls = card.querySelectorAll('[data-type="button"], [data-type="field"], [data-type="graphic"], [data-type="image"], [data-type="player"], [data-type="scrollbar"]');
        console.log(`closestControl: Found ${controls.length} total controls on card`);
        
        // Variables to track the closest control
        let closestDistance = Infinity;
        let closestControl = null;
        
        // Check each control to find the closest one
        for (let i = 0; i < controls.length; i++) {
            const control = controls[i];
            const controlType = control.dataset.type;
            const controlName = control.dataset.name;
            
            // Skip the reference object itself
            if (controlType === refObjectType && controlName === refObjectName) {
                console.log(`closestControl: Skipping reference object: ${controlType} "${controlName}"`);
                continue;
            }
            
            console.log(`closestControl: Checking distance to: ${controlType} "${controlName}"`);
            
            try {
                // Get control position and calculate distance directly from DOM
                const controlRect = control.getBoundingClientRect();
                const controlCenterX = controlRect.left + controlRect.width / 2;
                const controlCenterY = controlRect.top + controlRect.height / 2;
                
                // Calculate Euclidean distance between centers
                const xDiff = controlCenterX - refCenterX;
                const yDiff = controlCenterY - refCenterY;
                const distance = Math.sqrt(xDiff * xDiff + yDiff * yDiff);
                
                console.log(`closestControl: Distance to ${controlType} "${controlName}": ${distance}`);
                
                // Update closest object if this one is closer
                if (!isNaN(distance) && distance < closestDistance) {
                    closestDistance = distance;
                    closestControl = { type: controlType, name: controlName };
                    console.log(`closestControl: New closest: ${controlType} "${controlName}" at ${distance}`);
                }
            } catch (e) {
                console.log(`closestControl: Error checking distance to ${controlType} "${controlName}": ${e.message}`);
            }
        }
        
        // Return the closest object in the format "type "name""
        if (closestControl) {
            const result = `${closestControl.type} "${closestControl.name}"`;
            console.log(`closestControl: Returning result: ${result}`);
            return result;
        } else {
            console.log('closestControl: No closest object found, returning empty string');
            return '';
        }
    }
    
    if (hasBatteryMatch) {
        // Use the preloaded battery information if available
        if (window.batteryInfo) {
            return window.batteryInfo;
        }
        
        // If battery info isn't preloaded yet, use a synchronous approach
        // This is a fallback that should only happen if the page just loaded
        if ('getBattery' in navigator) {
            // For MacOS, we need to return a reliable result immediately
            const userAgent = navigator.userAgent.toLowerCase();
            const isMacOS = userAgent.includes('macintosh') || userAgent.includes('mac os x');
            const isMobile = /android|webos|iphone|ipad|ipod|blackberry|iemobile|opera mini/i.test(userAgent);
            
            // If this is MacOS or a mobile device, we can be confident there's a battery
            if (isMacOS || isMobile) {
                // Trigger the battery info loading for next time if not already in progress
                if (!window._batteryLoadStarted) {
                    window._batteryLoadStarted = true;
                    
                    // Load the actual battery info for future calls
                    navigator.getBattery().then(battery => {
                        if (!battery) {
                            window.batteryInfo = "false";
                            return;
                        }
                        
                        // Store the actual battery information
                        const chargingStatus = battery.charging ? "charging" : "discharging";
                        const batteryLevel = Math.round(battery.level * 100) + "%";
                        window.batteryInfo = `true, ${chargingStatus}, ${batteryLevel}`;
                        
                        // Set up event listeners for future updates
                        battery.addEventListener('levelchange', function() {
                            const newLevel = Math.round(battery.level * 100) + "%";
                            const chargingState = battery.charging ? "charging" : "discharging";
                            window.batteryInfo = `true, ${chargingState}, ${newLevel}`;
                        });
                        
                        battery.addEventListener('chargingchange', function() {
                            const level = Math.round(battery.level * 100) + "%";
                            const newChargingState = battery.charging ? "charging" : "discharging";
                            window.batteryInfo = `true, ${newChargingState}, ${level}`;
                        });
                    }).catch(error => {
                        console.log("Battery API error: " + error);
                        // Keep the default value if there's an error
                    });
                }
                
                // Return specific information for MacOS
                if (isMacOS) {
                    // Most MacOS devices have batteries and are likely to be charging when plugged in
                    // This is a reasonable default until we get the actual data
                    return "true, charging, 100%";
                } else if (isMobile) {
                    // Mobile devices have batteries but charge state is less predictable
                    return "true, unknown, unknown";
                }
            }
            
            // For other devices, we'll need to check more carefully
            try {
                // Try to access battery directly if available through older APIs
                const nav = window.navigator;
                const batteryObj = nav.battery || nav.mozBattery;
                
                if (batteryObj) {
                    const chargingStatus = batteryObj.charging ? "charging" : "discharging";
                    const batteryLevel = Math.round(batteryObj.level * 100) + "%";
                    window.batteryInfo = `true, ${chargingStatus}, ${batteryLevel}`;
                    return window.batteryInfo;
                }
            } catch (e) {
                console.log("Error in synchronous battery detection: " + e);
            }
            
            // If we can't determine battery status synchronously, return a default
            return "false";
        } else {
            // If getBattery API is not available
            console.log("Battery Status API not supported on this device.");
            return "false";
        }
    }
    
    // If no extended function matched, return undefined
    return undefined;
};

// Directly modify the InterpreterClass prototype to add our extended functionality
(function() {
    // Function to extend the interpreter
    function extendInterpreter() {
        if (typeof InterpreterClass !== 'undefined' && InterpreterClass.prototype) {
            // Store a reference to the original evaluateExpression method
            const originalEvaluateExpression = InterpreterClass.prototype.evaluateExpression;
            
            // Add the getPixelColorAt helper method to the prototype
            InterpreterClass.prototype.getPixelColorAt = function(x, y) {
                try {
                    // Get the card element
                    const card = document.getElementById('card');
                    if (!card) {
                        throw new Error('Card not found');
                    }
                    
                    // Get the card's position and dimensions
                    const cardRect = card.getBoundingClientRect();
                    
                    // Calculate the absolute position on the page
                    const pageX = cardRect.left + x;
                    const pageY = cardRect.top + y;
                    
                    // Use elementsFromPoint to get all elements at that position
                    const elements = document.elementsFromPoint(pageX, pageY);
                    
                    // Find the topmost visible element that's part of the card
                    for (const element of elements) {
                        if (card.contains(element) || element === card) {
                            // Get the computed style of the element
                            const style = window.getComputedStyle(element);
                            
                            // Check if the element has a background color
                            const bgColor = style.backgroundColor;
                            if (bgColor && bgColor !== 'transparent' && bgColor !== 'rgba(0, 0, 0, 0)') {
                                // Extract RGB values from the color string
                                const rgbMatch = bgColor.match(/rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)/);
                                if (rgbMatch) {
                                    return `${rgbMatch[1]},${rgbMatch[2]},${rgbMatch[3]}`;
                                }
                                return bgColor; // Fallback to original format if parsing fails
                            }
                            
                            // If it's an image, try to get the pixel color from the image
                            if (element.tagName === 'IMG' && element.complete && element.naturalHeight !== 0) {
                                try {
                                    // Create a canvas to draw the image
                                    const canvas = document.createElement('canvas');
                                    const context = canvas.getContext('2d');
                                    
                                    // Calculate the position within the image
                                    const imgRect = element.getBoundingClientRect();
                                    const imgX = pageX - imgRect.left;
                                    const imgY = pageY - imgRect.top;
                                    
                                    // Set canvas size to 1x1 pixel
                                    canvas.width = 1;
                                    canvas.height = 1;
                                    
                                    // Draw just the pixel we need
                                    context.drawImage(element, imgX, imgY, 1, 1, 0, 0, 1, 1);
                                    
                                    // Get the pixel data
                                    const pixelData = context.getImageData(0, 0, 1, 1).data;
                                    return `${pixelData[0]},${pixelData[1]},${pixelData[2]}`;
                                } catch (imgError) {
                                    console.warn('Error getting pixel from image:', imgError);
                                    // Continue to next element
                                }
                            }
                        }
                    }
                    
                    // If no specific color found, return the card's background color
                    const cardStyle = window.getComputedStyle(card);
                    const cardBgColor = cardStyle.backgroundColor || 'rgb(255, 255, 255)';
                    
                    // Extract RGB values from the color string
                    const rgbMatch = cardBgColor.match(/rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)/);
                    if (rgbMatch) {
                        return `${rgbMatch[1]},${rgbMatch[2]},${rgbMatch[3]}`;
                    }
                    
                    return '255,255,255'; // Default if parsing fails
                } catch (e) {
                    console.error('getPixelColorAt error:', e);
                    return '255,255,255'; // Default to white if all else fails
                }
            };
            
            // Add goURL command to the interpreter
            const originalInterpret = InterpreterClass.prototype.interpret;
            InterpreterClass.prototype.interpret = async function(script) {
                if (!script) return;
                
                script = script.trim();
                
                // Handle goURL command - supports both function-style syntax with parentheses and HyperTalk-style syntax without parentheses
                const goURLFunctionMatch = script.match(/^goURL\s*\(\s*(?:"([^"]+)"|'([^']+)'|([^)]+))\s*\)$/i);
                const goURLCommandMatch = script.match(/^goURL\s+(?:"([^"]+)"|'([^']+)'|(\S+))$/i);
                
                if (goURLFunctionMatch || goURLCommandMatch) {
                    const match = goURLFunctionMatch || goURLCommandMatch;
                    const url = match[1] || match[2] || match[3];
                    
                    // Validate URL
                    let validatedUrl = url;
                    if (!url.match(/^https?:\/\//i)) {
                        validatedUrl = 'https://' + url;
                    }
                    
                    try {
                        // Open URL in a new tab
                        window.open(validatedUrl, '_blank');
                        return '';
                    } catch (error) {
                        return `Error opening URL: ${error.message}`;
                    }
                }
                
                // If not a goURL command, use the original interpret method
                return await originalInterpret.call(this, script);
            };
            
            // Override the evaluateExpression method
            InterpreterClass.prototype.evaluateExpression = function(expr) {
                try {
                    // First, try to evaluate using our extended functions
                    // Make sure to bind 'this' context to the current interpreter instance
                    const result = HyperTalkExtended.evaluateExtendedExpression.call(this, expr);
                    
                    // If the extended functions didn't handle it, use the original method
                    if (result === undefined) {
                        return originalEvaluateExpression.call(this, expr);
                    }
                    
                    return result;
                } catch (error) {
                    // If there's an error in extended functions that might be due to context issues,
                    // fall back to the original method
                    if (error.message.includes('getTextValue is not a function')) {
                        return originalEvaluateExpression.call(this, expr);
                    }
                    // Otherwise rethrow the error
                    throw error;
                }
            };
            
            console.log('Successfully extended the InterpreterClass prototype');
            return true;
        }
        return false;
    }
    
    // Try to extend immediately
    if (!extendInterpreter()) {
        // If not successful, wait for the interpreter to be loaded
        let attempts = 0;
        const maxAttempts = 200; // Increased from 80 to 200 for more retries
        const interval = setInterval(function() {
            attempts++;
            if (extendInterpreter() || attempts >= maxAttempts) {
                clearInterval(interval);
                if (attempts >= maxAttempts) {
                    console.error('Failed to extend InterpreterClass after maximum attempts');
                } else if (attempts > 1) {
                    console.log(`Successfully extended InterpreterClass after ${attempts} attempts`);
                }
            }
        }, 25); // Increased from 10ms to 25ms for longer intervals
        
        // Also try to extend on DOMContentLoaded and window.onload events
        document.addEventListener('DOMContentLoaded', function() {
            extendInterpreter();
        });
        
        window.addEventListener('load', function() {
            extendInterpreter();
        });
        
        // Add a more aggressive immediate retry
        setTimeout(function() {
            extendInterpreter();
        }, 0);
    }
})();
