/**
 * card-navigator.js - Card Navigator Palette for WebTalk
 * 
 * This file implements a floating palette that displays thumbnails of all cards
 * in the stack, allowing for visual navigation between cards.
 */

(function() {
    // Card Navigator state
    let isCardNavigatorVisible = false;
    let cardNavigatorElement = null;
    let cardThumbnailsContainer = null;
    let isHorizontalLayout = false;
    let minThumbnailSize = 64; // Minimum thumbnail size in pixels
    let maxThumbnailSize = 256; // Maximum thumbnail size in pixels
    let currentThumbnailSize = maxThumbnailSize;
    
    /**
     * Creates the card navigator palette if it doesn't exist
     */
    function createCardNavigator() {
        if (cardNavigatorElement) return;
        
        // Create the main container
        cardNavigatorElement = document.createElement('div');
        cardNavigatorElement.id = 'card-navigator';
        cardNavigatorElement.className = 'object-inspector'; // Use inspector styling
        cardNavigatorElement.style.width = '200px'; // Minimum width
        cardNavigatorElement.style.height = '200px'; // Minimum height
        cardNavigatorElement.style.display = 'none';
        
        // Create header with drag handle
        const header = document.createElement('div');
        header.className = 'inspector-header';
        
        const dragHandle = document.createElement('div');
        dragHandle.className = 'inspector-drag-handle';
        dragHandle.id = 'card-navigator-drag-handle';
        
        const handleIcon = document.createElement('span');
        handleIcon.innerHTML = '&#8942;'; // Vertical ellipsis
        
        const title = document.createElement('div');
        title.className = 'inspector-title';
        title.textContent = 'Card Navigator';
        
        dragHandle.appendChild(handleIcon);
        dragHandle.appendChild(title);
        
        const closeButton = document.createElement('button');
        closeButton.className = 'inspector-close-button';
        closeButton.innerHTML = '&times;';
        closeButton.title = 'Close';
        closeButton.addEventListener('click', hideCardNavigator);
        
        header.appendChild(dragHandle);
        header.appendChild(closeButton);
        
        // Create content area for thumbnails
        const content = document.createElement('div');
        content.className = 'inspector-content';
        content.style.padding = '10px';
        content.style.height = 'calc(100% - 40px)';
        content.style.overflow = 'auto';
        
        // Create thumbnails container
        cardThumbnailsContainer = document.createElement('div');
        cardThumbnailsContainer.className = 'card-thumbnails';
        cardThumbnailsContainer.style.display = 'flex';
        cardThumbnailsContainer.style.flexDirection = 'column';
        cardThumbnailsContainer.style.gap = '10px';
        cardThumbnailsContainer.style.alignItems = 'center';
        
        content.appendChild(cardThumbnailsContainer);
        
        // Create resize handle
        const resizeHandle = document.createElement('div');
        resizeHandle.className = 'resize-handle';
        resizeHandle.style.position = 'absolute';
        resizeHandle.style.bottom = '0';
        resizeHandle.style.right = '0';
        resizeHandle.style.width = '16px';
        resizeHandle.style.height = '16px';
        resizeHandle.style.cursor = 'ns-resize'; // Vertical resize cursor
        // Use resizer.png image
        resizeHandle.style.backgroundImage = 'url("images/resizer.png")';
        resizeHandle.style.backgroundSize = 'contain';
        resizeHandle.style.backgroundRepeat = 'no-repeat';
        
        // Assemble the palette
        cardNavigatorElement.appendChild(header);
        cardNavigatorElement.appendChild(content);
        cardNavigatorElement.appendChild(resizeHandle);
        
        // Add to document
        document.body.appendChild(cardNavigatorElement);
        
        // Set up drag functionality
        setupDrag(dragHandle);
        
        // Set up resize functionality
        setupResize(resizeHandle);
    }
    
    /**
     * Sets up drag functionality for the card navigator
     * @param {HTMLElement} handle - The drag handle element
     */
    function setupDrag(handle) {
        let offsetX, offsetY;
        let isDragging = false;
        
        const startDrag = (clientX, clientY) => {
            isDragging = true;
            offsetX = clientX - cardNavigatorElement.offsetLeft;
            offsetY = clientY - cardNavigatorElement.offsetTop;
        };
        
        const doDrag = (clientX, clientY) => {
            if (!isDragging) return;
            
            const left = Math.max(0, clientX - offsetX);
            const top = Math.max(0, clientY - offsetY);
            
            cardNavigatorElement.style.left = left + 'px';
            cardNavigatorElement.style.top = top + 'px';
            
            // Save position to localStorage
            localStorage.setItem('cardNavigatorPosition', JSON.stringify({ left, top }));
        };
        
        const endDrag = () => {
            isDragging = false;
        };
        
        // Mouse events
        handle.addEventListener('mousedown', (e) => {
            e.preventDefault();
            startDrag(e.clientX, e.clientY);
        });
        
        document.addEventListener('mousemove', (e) => {
            doDrag(e.clientX, e.clientY);
        });
        
        document.addEventListener('mouseup', endDrag);
        
        // Touch events
        handle.addEventListener('touchstart', (e) => {
            const touch = e.touches[0];
            startDrag(touch.clientX, touch.clientY);
        });
        
        document.addEventListener('touchmove', (e) => {
            if (!isDragging) return;
            const touch = e.touches[0];
            doDrag(touch.clientX, touch.clientY);
        });
        
        document.addEventListener('touchend', endDrag);
        
        // Restore position from localStorage if present
        const savedPosition = localStorage.getItem('cardNavigatorPosition');
        if (savedPosition) {
            try {
                const { left, top } = JSON.parse(savedPosition);
                cardNavigatorElement.style.left = left + 'px';
                cardNavigatorElement.style.top = top + 'px';
            } catch (e) {
                console.error('Error restoring card navigator position:', e);
            }
        }
    }
    
    /**
     * Sets up resize functionality for the card navigator
     * @param {HTMLElement} handle - The resize handle element
     */
    function setupResize(handle) {
        let startX, startY, startWidth, startHeight;
        let isResizing = false;
        
        const startResize = (clientX, clientY) => {
            isResizing = true;
            startX = clientX;
            startY = clientY;
            startWidth = cardNavigatorElement.offsetWidth;
            startHeight = cardNavigatorElement.offsetHeight;
        };
        
        const doResize = (clientX, clientY) => {
            if (!isResizing) return;
            
            // Only allow vertical resizing (height changes, width remains fixed)
            const newHeight = Math.max(200, startHeight + (clientY - startY));
            
            // Keep width fixed
            cardNavigatorElement.style.height = newHeight + 'px';
            
            // Save dimensions to localStorage (keep width fixed at 200px)
            localStorage.setItem('cardNavigatorSize', JSON.stringify({ width: 200, height: newHeight }));
            
            // Update layout based on new dimensions
            updateLayout();
        };
        
        const endResize = () => {
            isResizing = false;
        };
        
        // Mouse events
        handle.addEventListener('mousedown', (e) => {
            e.preventDefault();
            startResize(e.clientX, e.clientY);
        });
        
        document.addEventListener('mousemove', (e) => {
            doResize(e.clientX, e.clientY);
        });
        
        document.addEventListener('mouseup', endResize);
        
        // Touch events
        handle.addEventListener('touchstart', (e) => {
            const touch = e.touches[0];
            startResize(touch.clientX, touch.clientY);
        });
        
        document.addEventListener('touchmove', (e) => {
            if (!isResizing) return;
            const touch = e.touches[0];
            doResize(touch.clientX, touch.clientY);
        });
        
        document.addEventListener('touchend', endResize);
        
        // Restore size from localStorage if present
        const savedSize = localStorage.getItem('cardNavigatorSize');
        if (savedSize) {
            try {
                const { width, height } = JSON.parse(savedSize);
                cardNavigatorElement.style.width = width + 'px';
                cardNavigatorElement.style.height = height + 'px';
                updateLayout();
            } catch (e) {
                console.error('Error restoring card navigator size:', e);
            }
        }
    }
    
    /**
     * Updates the layout of thumbnails based on the palette dimensions
     */
    function updateLayout() {
        if (!cardNavigatorElement || !cardThumbnailsContainer) return;
        
        const width = cardNavigatorElement.offsetWidth;
        const height = cardNavigatorElement.offsetHeight;
        
        // Always use column layout since we only resize vertically
        isHorizontalLayout = false;
        
        cardThumbnailsContainer.style.flexDirection = 'column';
        cardThumbnailsContainer.style.flexWrap = 'nowrap';
        cardThumbnailsContainer.style.justifyContent = 'flex-start';
        
        // Keep thumbnail size fixed at minThumbnailSize
        currentThumbnailSize = minThumbnailSize;
        
        // Update all thumbnails with fixed size
        const thumbnails = cardThumbnailsContainer.querySelectorAll('.card-thumbnail');
        thumbnails.forEach(thumbnail => {
            thumbnail.style.width = currentThumbnailSize + 'px';
            thumbnail.style.height = (currentThumbnailSize * 0.75) + 'px';
        });
    }
    
    /**
     * Shows the card navigator palette
     */
    function showCardNavigator() {
        createCardNavigator();
        cardNavigatorElement.style.display = 'block';
        isCardNavigatorVisible = true;
        
        // Use a longer delay to ensure the palette is fully visible and has dimensions before rendering thumbnails
        setTimeout(() => {
            updateCardThumbnails();
            
            // Add additional update attempts with increasing delays to ensure proper rendering
            // This mimics what happens when clicking a thumbnail but without navigation
            setTimeout(() => {
                updateCardThumbnails();
            }, 100);
            
            setTimeout(() => {
                updateCardThumbnails();
            }, 300);
        }, 100);
    }
    
    /**
     * Hides the card navigator palette
     */
    function hideCardNavigator() {
        if (cardNavigatorElement) {
            cardNavigatorElement.style.display = 'none';
        }
        isCardNavigatorVisible = false;
    }
    
    /**
     * Creates a thumbnail for a single card
     * @param {Object} card - The card object
     * @param {number} index - The card index (1-based)
     * @returns {HTMLElement} - The thumbnail element
     */
    function createCardThumbnail(card, index) {
        const thumbnail = document.createElement('div');
        thumbnail.className = 'card-thumbnail';
        thumbnail.dataset.cardIndex = index;
        thumbnail.style.width = currentThumbnailSize + 'px';
        thumbnail.style.height = (currentThumbnailSize * 0.75) + 'px';
        thumbnail.style.border = '1px solid #444';
        thumbnail.style.borderRadius = '4px';
        thumbnail.style.overflow = 'hidden';
        thumbnail.style.position = 'relative';
        thumbnail.style.backgroundColor = '#222';
        thumbnail.style.cursor = 'pointer';
        
        // Add card number/name label
        const label = document.createElement('div');
        label.className = 'card-thumbnail-label';
        label.textContent = card.name || `Card ${index}`;
        label.style.position = 'absolute';
        label.style.bottom = '0';
        label.style.left = '0';
        label.style.right = '0';
        label.style.padding = '4px';
        label.style.backgroundColor = 'rgba(0,0,0,0.7)';
        label.style.color = '#fff';
        label.style.fontSize = '12px';
        label.style.textAlign = 'center';
        label.style.whiteSpace = 'nowrap';
        label.style.overflow = 'hidden';
        label.style.textOverflow = 'ellipsis';
        
        // Create the thumbnail content (will be populated later)
        const content = document.createElement('div');
        content.className = 'card-thumbnail-content';
        content.style.width = '100%';
        content.style.height = '100%';
        content.style.position = 'relative';
        // Remove transform from here as it will be applied to the card clone
        
        thumbnail.appendChild(content);
        thumbnail.appendChild(label);
        
        // Add click handler to navigate to the card
        thumbnail.addEventListener('click', (e) => {
            // Prevent default to avoid any unwanted behavior
            e.preventDefault();
            e.stopPropagation();
            
            if (window.webTalkApp && window.webTalkApp.interpreter) {
                // Navigate to the card
                window.webTalkApp.interpreter.interpret(`go card ${index}`);
                
                // Update thumbnails after a short delay to ensure the card change is complete
                setTimeout(() => {
                    updateCardThumbnails();
                }, 100);
            }
        });
        
        // Add context menu for right-click actions
        thumbnail.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            showThumbnailContextMenu(e, index);
        });
        
        return thumbnail;
    }
    
    /**
     * Shows the context menu for a card thumbnail
     * @param {Event} event - The triggering event
     * @param {number} cardIndex - The card index
     */
    function showThumbnailContextMenu(event, cardIndex) {
        // Remove any existing context menu
        const existingMenu = document.getElementById('card-thumbnail-context-menu');
        if (existingMenu) {
            existingMenu.remove();
        }
        
        // Create context menu
        const menu = document.createElement('div');
        menu.id = 'card-thumbnail-context-menu';
        menu.style.position = 'fixed';
        menu.style.zIndex = '1000';
        menu.style.backgroundColor = '#333';
        menu.style.border = '1px solid #555';
        menu.style.borderRadius = '4px';
        menu.style.boxShadow = '0 2px 10px rgba(0,0,0,0.3)';
        menu.style.padding = '4px 0';
        menu.style.minWidth = '120px';
        
        // Position the menu
        menu.style.left = event.clientX + 'px';
        menu.style.top = event.clientY + 'px';
        
        // Add menu items
        const menuItems = [
            { label: 'Go to Card', action: () => goToCard(cardIndex) },
            { label: 'New Card After', action: () => createCardAfter(cardIndex) },
            { label: 'Delete Card', action: () => deleteCard(cardIndex) }
        ];
        
        menuItems.forEach(item => {
            const menuItem = document.createElement('div');
            menuItem.className = 'context-menu-item';
            menuItem.textContent = item.label;
            menuItem.style.padding = '6px 12px';
            menuItem.style.cursor = 'pointer';
            menuItem.style.color = '#fff';
            
            menuItem.addEventListener('mouseover', () => {
                menuItem.style.backgroundColor = '#555';
            });
            
            menuItem.addEventListener('mouseout', () => {
                menuItem.style.backgroundColor = 'transparent';
            });
            
            menuItem.addEventListener('click', () => {
                item.action();
                menu.remove();
            });
            
            menu.appendChild(menuItem);
        });
        
        // Add the menu to the document
        document.body.appendChild(menu);
        
        // Close the menu when clicking outside
        const closeMenu = (e) => {
            if (!menu.contains(e.target)) {
                menu.remove();
                document.removeEventListener('click', closeMenu);
            }
        };
        
        // Use setTimeout to avoid immediate closing
        setTimeout(() => {
            document.addEventListener('click', closeMenu);
        }, 0);
    }
    
    /**
     * Navigates to a specific card
     * @param {number} cardIndex - The card index
     */
    function goToCard(cardIndex) {
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            window.webTalkApp.interpreter.interpret(`go card ${cardIndex}`);
        }
    }
    
    /**
     * Creates a new card after the specified card
     * @param {number} cardIndex - The card index
     */
    function createCardAfter(cardIndex) {
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            const script = `
                put randomstring(6) into myNewName
                create card myNewName
            `;
            window.webTalkApp.interpreter.interpret(script);
        }
    }
    
    /**
     * Deletes the specified card
     * @param {number} cardIndex - The card index
     */
    function deleteCard(cardIndex) {
        if (window.webTalkApp && window.webTalkApp.interpreter) {
            window.webTalkApp.interpreter.interpret(`delete card ${cardIndex}`);
        }
    }
    
    /**
     * Updates all card thumbnails in the navigator
     */
    function updateCardThumbnails() {
        if (!isCardNavigatorVisible || !cardThumbnailsContainer) return;
        
        // Clear existing thumbnails
        cardThumbnailsContainer.innerHTML = '';
        
        // Get all cards from the interpreter
        if (!window.webTalkApp || !window.webTalkApp.interpreter) return;
        
        const interpreter = window.webTalkApp.interpreter;
        
        // Find all card elements in the DOM
        const cardElements = document.querySelectorAll('[data-card-id]');
        
        if (!cardElements || cardElements.length === 0) {
            // No cards found, use the default card
            const defaultCard = {
                name: 'Card 1',
                element: document.getElementById('card')
            };
            
            const thumbnail = createCardThumbnail(defaultCard, 1);
            cardThumbnailsContainer.appendChild(thumbnail);
            renderThumbnailContent(thumbnail, defaultCard);
        } else {
            // Multi-card mode - create thumbnails for all cards
            let index = 1;
            cardElements.forEach(cardElement => {
                const cardId = cardElement.getAttribute('data-card-id');
                const cardName = cardElement.getAttribute('data-card-name') || `Card ${index}`;
                
                const card = {
                    name: cardName,
                    element: cardElement,
                    id: cardId
                };
                
                const thumbnail = createCardThumbnail(card, index);
                cardThumbnailsContainer.appendChild(thumbnail);
                renderThumbnailContent(thumbnail, card);
                index++;
            });
        }
        
        // Update layout
        updateLayout();
    }
    
    /**
     * Renders the content of a card into its thumbnail
     * @param {HTMLElement} thumbnail - The thumbnail element
     * @param {Object} card - The card object
     */
    function renderThumbnailContent(thumbnail, card) {
        const content = thumbnail.querySelector('.card-thumbnail-content');
        if (!content) return;
        
        // Clear existing content
        content.innerHTML = '';
        
        // Get the card element
        const cardElement = card.element || document.getElementById('card');
        if (!cardElement) return;
        
        // Create a clone of the card element for the thumbnail
        const clone = cardElement.cloneNode(true);
        
        // Calculate scale factor based on thumbnail size vs card size
        const thumbnailWidth = content.offsetWidth || 200;
        const thumbnailHeight = content.offsetHeight || 150;
        const cardWidth = cardElement.offsetWidth || 800;
        const cardHeight = cardElement.offsetHeight || 600;
        
        const scaleX = thumbnailWidth / cardWidth;
        const scaleY = thumbnailHeight / cardHeight;
        const scale = Math.min(scaleX, scaleY);
        
        // Apply scaling to the clone
        clone.style.transform = `scale(${scale})`;
        clone.style.transformOrigin = 'top left';
        clone.style.width = cardWidth + 'px';
        clone.style.height = cardHeight + 'px';
        clone.style.position = 'absolute';
        clone.style.top = '0';
        clone.style.left = '0';
        
        // Copy background styles
        clone.style.backgroundColor = getComputedStyle(cardElement).backgroundColor;
        
        // Remove any scripts
        const scripts = clone.querySelectorAll('script');
        for (let i = 0; i < scripts.length; i++) {
            const script = scripts[i];
            if (script.parentNode) {
                script.parentNode.removeChild(script);
            }
        }
        
        // Remove any event listeners by replacing interactive elements with clones
        const interactiveElements = clone.querySelectorAll('button, input, select, textarea');
        for (let i = 0; i < interactiveElements.length; i++) {
            const el = interactiveElements[i];
            const elClone = el.cloneNode(true);
            if (el.parentNode) {
                el.parentNode.replaceChild(elClone, el);
            }
        }
        
        // Add the clone to the content
        content.appendChild(clone);
    }
    
    /**
     * Registers the card navigator commands with the interpreter
     */
    function registerCardNavigatorCommands() {
        if (!window.webTalkApp || !window.webTalkApp.interpreter) {
            console.error('WebTalk interpreter not found');
            return;
        }
        
        const interpreter = window.webTalkApp.interpreter;
        
        // Register the show/hide card navigator commands
        interpreter.registerCommand('show card navigator', () => {
            showCardNavigator();
            return '';
        });
        
        interpreter.registerCommand('show cardnav', () => {
            showCardNavigator();
            return '';
        });
        
        interpreter.registerCommand('hide card navigator', () => {
            hideCardNavigator();
            return '';
        });
        
        interpreter.registerCommand('hide cardnav', () => {
            hideCardNavigator();
            return '';
        });
    }
    
    /**
     * Initializes the card navigator
     */
    function initCardNavigator() {
        // Register commands
        registerCardNavigatorCommands();
        
        // Listen for card changes
        document.addEventListener('cardChanged', () => {
            if (isCardNavigatorVisible) {
                updateCardThumbnails();
            }
        });
        
        // Listen for card creation/deletion
        document.addEventListener('cardCreated', () => {
            if (isCardNavigatorVisible) {
                updateCardThumbnails();
            }
        });
        
        document.addEventListener('cardDeleted', () => {
            if (isCardNavigatorVisible) {
                updateCardThumbnails();
            }
        });
        
        // Add CSS for the card navigator
        addCardNavigatorStyles();
    }
    
    /**
     * Adds CSS styles for the card navigator
     */
    function addCardNavigatorStyles() {
        const style = document.createElement('style');
        style.textContent = `
            .resize-handle {
                position: absolute;
                bottom: 0;
                right: 0;
                width: 10px;
                height: 10px;
                cursor: nwse-resize;
                background-color: #555;
                border-radius: 0 0 4px 0;
            }
            
            .card-thumbnails {
                display: flex;
                flex-direction: column;
                gap: 10px;
                align-items: center;
                width: 100%;
                height: 100%;
            }
            
            .card-thumbnail {
                transition: transform 0.1s ease-in-out, box-shadow 0.1s ease-in-out;
            }
            
            .card-thumbnail:hover {
                transform: scale(1.05);
                box-shadow: 0 0 10px rgba(0,0,0,0.5);
            }
            
            #card-thumbnail-context-menu {
                user-select: none;
            }
            
            .context-menu-item:hover {
                background-color: #555;
            }
        `;
        
        document.head.appendChild(style);
    }
    
    // Initialize when the document is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initCardNavigator);
    } else {
        initCardNavigator();
    }
    
    // Expose functions to global scope
    window.cardNavigator = {
        show: showCardNavigator,
        hide: hideCardNavigator,
        update: updateCardThumbnails,
        isVisible: () => isCardNavigatorVisible
    };
})();
